#ifndef _vuDVector_H_
#define _vuDVector_H_

#include "vuSimpleTypes.h"

#include <iostream.h>

/** vuDVector is a templatized dynamic vector class for the volume tree.
    That is lightweight and doesn't conflict with wxWindows or the standard library.
*/
template <class T>
class vuDVector
{
public:

    //!Default constructor
    vuDVector()
    {
        m_Size     = 4;
        m_Length   = 0;
        m_StepSize = 4;
        createData();
    }

    //!Copy constructor that does a deep copy.
    vuDVector(const vuDVector<T>& v)
    {
        m_Size     = v.m_Size;
        m_Length   = v.m_Length;
        m_StepSize = v.m_StepSize;
        createData();
        copyData(v);
    }

    //!Destructor.
    ~vuDVector()
    {
        destroyData();
    }


    //!Sets the stepsize of the vector.
    /*!The vector is always clamped to a multiple
       of stepsize when it is increased.
    */
    void setStepSize(dword s)
    {
        m_StepSize = s;
    }

    //!Returns the step size of the vector.
    dword getStepSize(void) const
    {
        return m_StepSize;
    }

    //!Sets the size of the vector.
    /*!Vector size is the total allocated length of the vector.
    */
    void setSize(dword s)
    {
        resizeData(s);
    }

    //!Returns the size of the vector.
    dword getSize(void) const
    {
        return m_Size;
    }

    //!Returns the length of the vector.
    /*!Vector length is the total number of data
       elements, which will be <= vector size.
    */
    dword getLength(void) const
    {
        return m_Length;
    }

    //!Returns true if the element is stored in the vector.
    bool isMember(const T& elem) const
    {
		for (dword i=0;i<m_Length;++i)
			if (m_Data[i] == elem) return true;

		return false;
    }

    //!Returns the position of the element within the vector.
	dword findIndex(const T& elem) const
	{
		dword i;
      for (i=0;i<m_Length;++i)
          if (m_Data[i] == elem) return i;
      
      return (dword)(-1);
	}

    //!Removes the indexed element from the vector.
    void remove(dword index)
    {
        removeRange(index,index);
    }

    //!Removes all elements from the vector.
    void removeAll()
    {
        removeRange(0,m_Length);
    }

    //!Removes all the elements in the vector from start to end, inclusive.
    /*!The indices of all elements with indices j, such
       that (j>end) become (j-(end-start+1)).  Length is
       decreased by (j-(end-start+1)), but the size of
       the vector is not altered.
    */
    void removeRange(dword start, dword end)
    {
        dword diff;
        dword i;

        // Make sure start is <= end.
        if (start > end)
        {
            dword temp = end;
            end = start;
            start = temp;
        }

        // If start is out of bounds, then return.
        if (start >= m_Length)
            return;

        // If end is out of bounds, then shorten it.
        if (end >= m_Length)
            end = m_Length-1;

        // Move the data in the array.
        diff = end-start+1;
        for(i=end+1;i<m_Length;++i)
            m_Data[i-diff] = m_Data[i];

        // Change the length.
        m_Length -= diff;
    }

    //!Resize the vector to be the same size as the number of data elements stored in it.
    void sizeToLength(void)
    {
        resizeData(m_Length);
    }

    //!Resize the vector to be the multiple of the stepsize that is nearest to the length.
    void stepToLength(void)
    {
        resizeData((((m_Length-1)/m_StepSize)+1)*m_StepSize);
    }

    //!Assignment operator which does a deep copy.
    vuDVector<T>& operator=(const vuDVector<T>& rhs)
    {
        if (this != &rhs)
        {
            destroyData();
            m_Size     = rhs.m_Size;
            m_Length   = rhs.m_Length;
            m_StepSize = rhs.m_StepSize;
            createData();
            copyData(rhs);
        }
        return *this;
    }

    //!Access operator allowing access and modification of elements.
    T& operator[](dword index)
    {
        if (index >= m_Size)
            resizeData(((index/m_StepSize)+1)*m_StepSize);
        if (index >= m_Length)
            m_Length = index+1;
        return m_Data[index];
    }

    //!Const Access operator allowing const access.
    const T& operator[](dword index) const
    {
        return m_Data[index];
    }

    //!Adds the element to the end of the vector.
    void add(const T& elem)
    {
        (*this)[m_Length] = elem;
    }

    //!Inserts an element at the given position in the vector.
    /*!If inserting past the end of the list, the element is just added
       to the end.
    */
    void insert(dword index, const T&elem)
    {
        if (index >= m_Length)
        {
            (*this)[index] = elem;
            return;
        }

        //If the array is too small, then resize it
        if (m_Length == m_Size)
            resizeData(m_Size+m_StepSize);

        // Move the data in the array.
        for(int i=m_Length-1; (i>=0) && (dword(i)>=index); i--)
            m_Data[i+1] = m_Data[i];

        //Note that there's an extra element
        m_Length++;

        // Add the new element.
        (*this)[index] = elem;
    }

private:
    //!Helper function for the creation of the dynamic data in the list.
    void createData(void)
    {
        if (m_Size)
            m_Data = new T[m_Size];
    }

    //!Helper function for the destruction of the dynamic data in the list.
    void destroyData(void)
    {
    	/* These couts were to test for a seg fault that was originating around the time
		that this function was being called. */
//    	cout << "destroying: " << int (m_Data) << endl;
        if (m_Data)
        {
            delete [] m_Data;
            m_Data = 0;
        }
//	cout << "done destruction: " << endl;
    }

    //!Copies the elements from the vector v to the instance.
    void copyData(const vuDVector& v)
    {
        dword i;

        if (m_Size != v.m_Size)
            throw "vuDVector: vectors not the same size.";

        for(i=0;i<m_Length;++i)
            m_Data[i] = v.m_Data[i];
    }

    //!Resizes the vector to the new size.
    void resizeData(dword size)
    {
    	int tsize = size / 10;

	if (tsize > 3)
		m_StepSize = tsize;
        if (size == m_Size)
        {
            return;
        }
        else if (size > m_Size)
        {
            T* temp;
            dword i;

            temp = m_Data;
            m_Data = new T[size];

            for(i=0;i<m_Size;++i)
                m_Data[i] = temp[i];

            delete [] temp;
            m_Size = size;
        }
        else
        {
            T* temp;
            dword i;

            temp = m_Data;
            m_Data = new T[size];

            for(i=0;i<size;++i)
                m_Data[i] = temp[i];

            delete [] temp;
            m_Size = size;

            if (m_Size < m_Length)
                m_Length = m_Size;
        }
    }

private:
    dword m_Size; /**< the current size of the vector (including preallocated but
    	not necessarily used elements). */
    dword m_Length; //!< the current length of the vector (last used element)
    dword m_StepSize; /**< the current stepsize that is being used to expand the vector
    			when the vector is expanded (note, this will change without warning).*/
    T*    m_Data; //!< the data buffer of the vector
};

#endif
