/**
 * @file	debug.h
 * @brief	Defines some debugging macros.
 *
 * @author	Markus Trenkwalder
 * @date	14. Jun 2004, 16:04:41
 * @version	0.0.1
 *
 * $id : $
 *
 */


// {{{ Documentation:

/**
 * \page MCdebug Debugging in Marching Cubes
 *
 * Debugging in the marching cubes engine works by including the file debug.h into the files where
 * debug output is needed. Befor including the file the macro \ref DEBUG_MODE should be defined to
 * the desired debug level. The debug macros such as \ref DEBUG0 or \ref DEBUG1 have the level in
 * their name and will be executed if the actual debug level is smaller or equal to the level of the
 * macro. The macros are designed to introduce no extra code if they are disabled. There exists a
 * spezial debug macros called \ref TODO. It will be executed as long as the debug level is samller
 * or equal than \em 99. Taking this aspect into account the debug level of \em 99 is very special
 * because is disables all nurmal debug output but does not deisable the \ref TODO output. Every
 * debug macro prints it's output to stderr.
 *
 * <h2>Debugging OpenGL</h2>
 *
 * I have included marginal support for debugging OpenGL function calls. It is realised by
 * redefining the OpenGL funcions such that first the signature is printed to stderr and then the
 * function itself is called. This has two big implication:
 * \li	The return values of the functions can not be used. If this is necessary, the debugging
 *	macro should be removed or commented out.
 * \li	The debugging header file must be included after any OpenGL include file.
 *
 * \todo	Remove all macros for functions that return pointers or can be identified as
 *		functions with important return values.
 */

/**
 * \def DEBUG_MODE
 *
 * Defines the actual debug level of the file. Every debug statement with a level equal or grater
 * than DEBUG_MODE will be executed. A spezial debug level is level zero. This level should be used
 * to display the function calls.
 */

/**
 * \def DEBUG0(msg)
 *
 * Defines a debug output for level \c 0. \p msg will be written stderr. This level is a spezial one
 * because it is designed to print the function calls. The preceeding text befor \p msg includes the
 * word \c DEBUG0 followed by the filename where the function can be found and the line number of
 * debug output (see \ref DEBUG_LOCATION).
 */

/**
 * \def DEBUG1(msg)
 *
 * Defines a debug output for level \c 1. The \c DEBUG macros for all subsequent debug levels look
 * like this macro. The macro prints the word \c DEBUG<level> followed by the filename, the line
 * number in parentheses and the function name in square brackets and after a colon the \p msg is
 * printed.
 */

/**
 * \def DEBUG2(msg)
 *
 * See \ref DEBUG1.
 */

/**
 * \def DEBUG3(msg)
 *
 * See \ref DEBUG1.
 */

/**
 * \def DEBUG4(msg)
 *
 * See \ref DEBUG1.
 */

/**
 * \def DEBUG5(msg)
 *
 * See \ref DEBUG1.
 */

/**
 * \def DEBUG6(msg)
 *
 * See \ref DEBUG1.
 */

/**
 * \def TODO(msg)
 *
 * This debug macro should be used for todos. It will be executed as long as the debug level is
 * samler or equal than \a 99. The output of this macro looks like the one in \ref DEBUG1 except
 * that the first word is \c TODO!!.
 */

/**
 * \def DEBUG_LOCATION
 *
 * Defines the prepended text in the debug output. It is composed of the file name, the line number
 * in parentheses and the function name in square brackets followed by a colon.
 */

/**
 * \def DEBUG_LOCATION_SHORT
 *
 * Defines the prepended text in \ref DEBUG0. It starts with the file name, followed by the line
 * number in parentheses and ends with a colon.
 */

// }}}


#ifndef __DEBUG_H__	// {{{
#define __DEBUG_H__

#ifdef NDEBUG
# undef DEBUG_CONST
# undef DEBUG_DEST
# undef DEBUG_GL
# define DEBUG_MODE 1000
#endif

#include <iostream>

#define DEBUG_LOCATION_SHORT __FILE__ << "(" << __LINE__ << "): "
#define DEBUG_LOCATION __FILE__ << "(" << __LINE__ << ")[" << __FUNCTION__ << "]: "

// {{{ DEBUG_CONST
#ifdef DEBUG_CONST
# define DEBUGC(msg)	std::cerr << "DEBUGC|" << DEBUG_LOCATION_SHORT << msg
#else
# define DEBUGC(msg)
#endif
// }}}

// {{{ DEBUG_DEST
#ifdef DEBUG_DEST
# define DEBUGD(msg)	std::cerr << "DEBUGD|" << DEBUG_LOCATION_SHORT << msg
#else
# define DEBUGD(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 0
#if DEBUG_MODE <= 0
# define DEBUG0(msg)	std::cerr << "DEBUG0|" << DEBUG_LOCATION_SHORT << msg
#else
# define DEBUG0(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 1
#if DEBUG_MODE <= 1
# define DEBUG1(msg)	std::cerr << "DEBUG1|" << DEBUG_LOCATION << msg
#else
# define DEBUG1(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 2
#if DEBUG_MODE <= 2
# define DEBUG2(msg)	std::cerr << "DEBUG2|" << DEBUG_LOCATION << msg
#else
# define DEBUG2(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 3
#if DEBUG_MODE <= 3
# define DEBUG3(msg)	std::cerr << "DEBUG3|" << DEBUG_LOCATION << msg
#else
# define DEBUG3(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 4
#if DEBUG_MODE <= 4
# define DEBUG4(msg)	std::cerr << "DEBUG4|" << DEBUG_LOCATION << msg
#else
# define DEBUG4(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 5
#if DEBUG_MODE <= 5
# define DEBUG5(msg)	std::cerr << "DEBUG5|" << DEBUG_LOCATION << msg
#else
# define DEBUG5(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 6
#if DEBUG_MODE <= 6
# define DEBUG6(msg)	std::cerr << "DEBUG6|" << DEBUG_LOCATION << msg
#else
# define DEBUG6(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 7
#if DEBUG_MODE <= 7
# define DEBUG7(msg)	std::cerr << "DEBUG7|" << DEBUG_LOCATION << msg
#else
# define DEBUG7(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 8
#if DEBUG_MODE <= 8
# define DEBUG8(msg)	std::cerr << "DEBUG8|" << DEBUG_LOCATION << msg
#else
# define DEBUG8(msg)
#endif
// }}}

// {{{ DEBUG_MODE <= 9
#if DEBUG_MODE <= 9
# define DEBUG9(msg)	std::cerr << "DEBUG9|" << DEBUG_LOCATION << msg
#else
# define DEBUG9(msg)
#endif
// }}}

// {{{ DEBUG_GL
#ifdef DEBUG_GL
# define GL_DEBUG	"GLDEBUG|" << DEBUG_LOCATION

# define DEBUGGL(arg)			std::cerr << GL_DEBUG << arg

//	{{{ gl-Functions
# define glAccum(args...)	std::cerr << GL_DEBUG << "glAccum("#args")\n"; glAccum(args);
# define glAlphaFunc(args...)	std::cerr << GL_DEBUG << "glAlphaFunc("#args")\n"; glAlphaFunc(args);
# define glAreTexturesResident(args...)	std::cerr << GL_DEBUG << "glAreTexturesResident("#args")\n"; glAreTexturesResident(args);
# define glArrayElement(args...)	std::cerr << GL_DEBUG << "glArrayElement("#args")\n"; glArrayElement(args);
# define glBegin(args...)	std::cerr << GL_DEBUG << "glBegin("#args")\n"; glBegin(args);
# define glBindTexture(args...)	std::cerr << GL_DEBUG << "glBindTexture("#args")\n"; glBindTexture(args);
# define glBitmap(args...)	std::cerr << GL_DEBUG << "glBitmap("#args")\n"; glBitmap(args);
# define glBlendFunc(args...)	std::cerr << GL_DEBUG << "glBlendFunc("#args")\n"; glBlendFunc(args);
# define glCallList(args...)	std::cerr << GL_DEBUG << "glCallList("#args")\n"; glCallList(args);
# define glCallLists(args...)	std::cerr << GL_DEBUG << "glCallLists("#args")\n"; glCallLists(args);
# define glClear(args...)	std::cerr << GL_DEBUG << "glClear("#args")\n"; glClear(args);
# define glClearAccum(args...)	std::cerr << GL_DEBUG << "glClearAccum("#args")\n"; glClearAccum(args);
# define glClearColor(args...)	std::cerr << GL_DEBUG << "glClearColor("#args")\n"; glClearColor(args);
# define glClearDepth(args...)	std::cerr << GL_DEBUG << "glClearDepth("#args")\n"; glClearDepth(args);
# define glClearIndex(args...)	std::cerr << GL_DEBUG << "glClearIndex("#args")\n"; glClearIndex(args);
# define glClearStencil(args...)	std::cerr << GL_DEBUG << "glClearStencil("#args")\n"; glClearStencil(args);
# define glClipPlane(args...)	std::cerr << GL_DEBUG << "glClipPlane("#args")\n"; glClipPlane(args);
# define glColor3b(args...)	std::cerr << GL_DEBUG << "glColor3b("#args")\n"; glColor3b(args);
# define glColor3bv(args...)	std::cerr << GL_DEBUG << "glColor3bv("#args")\n"; glColor3bv(args);
# define glColor3d(args...)	std::cerr << GL_DEBUG << "glColor3d("#args")\n"; glColor3d(args);
# define glColor3dv(args...)	std::cerr << GL_DEBUG << "glColor3dv("#args")\n"; glColor3dv(args);
# define glColor3f(args...)	std::cerr << GL_DEBUG << "glColor3f("#args")\n"; glColor3f(args);
# define glColor3fv(args...)	std::cerr << GL_DEBUG << "glColor3fv("#args")\n"; glColor3fv(args);
# define glColor3i(args...)	std::cerr << GL_DEBUG << "glColor3i("#args")\n"; glColor3i(args);
# define glColor3iv(args...)	std::cerr << GL_DEBUG << "glColor3iv("#args")\n"; glColor3iv(args);
# define glColor3s(args...)	std::cerr << GL_DEBUG << "glColor3s("#args")\n"; glColor3s(args);
# define glColor3sv(args...)	std::cerr << GL_DEBUG << "glColor3sv("#args")\n"; glColor3sv(args);
# define glColor3ub(args...)	std::cerr << GL_DEBUG << "glColor3ub("#args")\n"; glColor3ub(args);
# define glColor3ubv(args...)	std::cerr << GL_DEBUG << "glColor3ubv("#args")\n"; glColor3ubv(args);
# define glColor3ui(args...)	std::cerr << GL_DEBUG << "glColor3ui("#args")\n"; glColor3ui(args);
# define glColor3uiv(args...)	std::cerr << GL_DEBUG << "glColor3uiv("#args")\n"; glColor3uiv(args);
# define glColor3us(args...)	std::cerr << GL_DEBUG << "glColor3us("#args")\n"; glColor3us(args);
# define glColor3usv(args...)	std::cerr << GL_DEBUG << "glColor3usv("#args")\n"; glColor3usv(args);
# define glColor4b(args...)	std::cerr << GL_DEBUG << "glColor4b("#args")\n"; glColor4b(args);
# define glColor4bv(args...)	std::cerr << GL_DEBUG << "glColor4bv("#args")\n"; glColor4bv(args);
# define glColor4d(args...)	std::cerr << GL_DEBUG << "glColor4d("#args")\n"; glColor4d(args);
# define glColor4dv(args...)	std::cerr << GL_DEBUG << "glColor4dv("#args")\n"; glColor4dv(args);
# define glColor4f(args...)	std::cerr << GL_DEBUG << "glColor4f("#args")\n"; glColor4f(args);
# define glColor4fv(args...)	std::cerr << GL_DEBUG << "glColor4fv("#args")\n"; glColor4fv(args);
# define glColor4i(args...)	std::cerr << GL_DEBUG << "glColor4i("#args")\n"; glColor4i(args);
# define glColor4iv(args...)	std::cerr << GL_DEBUG << "glColor4iv("#args")\n"; glColor4iv(args);
# define glColor4s(args...)	std::cerr << GL_DEBUG << "glColor4s("#args")\n"; glColor4s(args);
# define glColor4sv(args...)	std::cerr << GL_DEBUG << "glColor4sv("#args")\n"; glColor4sv(args);
# define glColor4ub(args...)	std::cerr << GL_DEBUG << "glColor4ub("#args")\n"; glColor4ub(args);
# define glColor4ubv(args...)	std::cerr << GL_DEBUG << "glColor4ubv("#args")\n"; glColor4ubv(args);
# define glColor4ui(args...)	std::cerr << GL_DEBUG << "glColor4ui("#args")\n"; glColor4ui(args);
# define glColor4uiv(args...)	std::cerr << GL_DEBUG << "glColor4uiv("#args")\n"; glColor4uiv(args);
# define glColor4us(args...)	std::cerr << GL_DEBUG << "glColor4us("#args")\n"; glColor4us(args);
# define glColor4usv(args...)	std::cerr << GL_DEBUG << "glColor4usv("#args")\n"; glColor4usv(args);
# define glColorMask(args...)	std::cerr << GL_DEBUG << "glColorMask("#args")\n"; glColorMask(args);
# define glColorMaterial(args...)	std::cerr << GL_DEBUG << "glColorMaterial("#args")\n"; glColorMaterial(args);
# define glColorPointer(args...)	std::cerr << GL_DEBUG << "glColorPointer("#args")\n"; glColorPointer(args);
# define glCopyPixels(args...)	std::cerr << GL_DEBUG << "glCopyPixels("#args")\n"; glCopyPixels(args);
# define glCopyTexImage1D(args...)	std::cerr << GL_DEBUG << "glCopyTexImage1D("#args")\n"; glCopyTexImage1D(args);
# define glCopyTexImage2D(args...)	std::cerr << GL_DEBUG << "glCopyTexImage2D("#args")\n"; glCopyTexImage2D(args);
# define glCopyTexSubImage1D(args...)	std::cerr << GL_DEBUG << "glCopyTexSubImage1D("#args")\n"; glCopyTexSubImage1D(args);
# define glCopyTexSubImage2D(args...)	std::cerr << GL_DEBUG << "glCopyTexSubImage2D("#args")\n"; glCopyTexSubImage2D(args);
# define glCullFace(args...)	std::cerr << GL_DEBUG << "glCullFace("#args")\n"; glCullFace(args);
# define glDeleteLists(args...)	std::cerr << GL_DEBUG << "glDeleteLists("#args")\n"; glDeleteLists(args);
# define glDeleteTextures(args...)	std::cerr << GL_DEBUG << "glDeleteTextures("#args")\n"; glDeleteTextures(args);
# define glDepthFunc(args...)	std::cerr << GL_DEBUG << "glDepthFunc("#args")\n"; glDepthFunc(args);
# define glDepthMask(args...)	std::cerr << GL_DEBUG << "glDepthMask("#args")\n"; glDepthMask(args);
# define glDepthRange(args...)	std::cerr << GL_DEBUG << "glDepthRange("#args")\n"; glDepthRange(args);
# define glDisable(args...)	std::cerr << GL_DEBUG << "glDisable("#args")\n"; glDisable(args);
# define glDisableClientState(args...)	std::cerr << GL_DEBUG << "glDisableClientState("#args")\n"; glDisableClientState(args);
# define glDrawArrays(args...)	std::cerr << GL_DEBUG << "glDrawArrays("#args")\n"; glDrawArrays(args);
# define glDrawBuffer(args...)	std::cerr << GL_DEBUG << "glDrawBuffer("#args")\n"; glDrawBuffer(args);
# define glDrawElements(args...)	std::cerr << GL_DEBUG << "glDrawElements("#args")\n"; glDrawElements(args);
# define glDrawPixels(args...)	std::cerr << GL_DEBUG << "glDrawPixels("#args")\n"; glDrawPixels(args);
# define glEdgeFlag(args...)	std::cerr << GL_DEBUG << "glEdgeFlag("#args")\n"; glEdgeFlag(args);
# define glEdgeFlagPointer(args...)	std::cerr << GL_DEBUG << "glEdgeFlagPointer("#args")\n"; glEdgeFlagPointer(args);
# define glEdgeFlagv(args...)	std::cerr << GL_DEBUG << "glEdgeFlagv("#args")\n"; glEdgeFlagv(args);
# define glEnable(args...)	std::cerr << GL_DEBUG << "glEnable("#args")\n"; glEnable(args);
# define glEnableClientState(args...)	std::cerr << GL_DEBUG << "glEnableClientState("#args")\n"; glEnableClientState(args);
# define glEnd(args...)	std::cerr << GL_DEBUG << "glEnd("#args")\n"; glEnd(args);
# define glEndList(args...)	std::cerr << GL_DEBUG << "glEndList("#args")\n"; glEndList(args);
# define glEvalCoord1d(args...)	std::cerr << GL_DEBUG << "glEvalCoord1d("#args")\n"; glEvalCoord1d(args);
# define glEvalCoord1dv(args...)	std::cerr << GL_DEBUG << "glEvalCoord1dv("#args")\n"; glEvalCoord1dv(args);
# define glEvalCoord1f(args...)	std::cerr << GL_DEBUG << "glEvalCoord1f("#args")\n"; glEvalCoord1f(args);
# define glEvalCoord1fv(args...)	std::cerr << GL_DEBUG << "glEvalCoord1fv("#args")\n"; glEvalCoord1fv(args);
# define glEvalCoord2d(args...)	std::cerr << GL_DEBUG << "glEvalCoord2d("#args")\n"; glEvalCoord2d(args);
# define glEvalCoord2dv(args...)	std::cerr << GL_DEBUG << "glEvalCoord2dv("#args")\n"; glEvalCoord2dv(args);
# define glEvalCoord2f(args...)	std::cerr << GL_DEBUG << "glEvalCoord2f("#args")\n"; glEvalCoord2f(args);
# define glEvalCoord2fv(args...)	std::cerr << GL_DEBUG << "glEvalCoord2fv("#args")\n"; glEvalCoord2fv(args);
# define glEvalMesh1(args...)	std::cerr << GL_DEBUG << "glEvalMesh1("#args")\n"; glEvalMesh1(args);
# define glEvalMesh2(args...)	std::cerr << GL_DEBUG << "glEvalMesh2("#args")\n"; glEvalMesh2(args);
# define glEvalPoint1(args...)	std::cerr << GL_DEBUG << "glEvalPoint1("#args")\n"; glEvalPoint1(args);
# define glEvalPoint2(args...)	std::cerr << GL_DEBUG << "glEvalPoint2("#args")\n"; glEvalPoint2(args);
# define glFeedbackBuffer(args...)	std::cerr << GL_DEBUG << "glFeedbackBuffer("#args")\n"; glFeedbackBuffer(args);
# define glFinish(args...)	std::cerr << GL_DEBUG << "glFinish("#args")\n"; glFinish(args);
# define glFlush(args...)	std::cerr << GL_DEBUG << "glFlush("#args")\n"; glFlush(args);
# define glFogf(args...)	std::cerr << GL_DEBUG << "glFogf("#args")\n"; glFogf(args);
# define glFogfv(args...)	std::cerr << GL_DEBUG << "glFogfv("#args")\n"; glFogfv(args);
# define glFogi(args...)	std::cerr << GL_DEBUG << "glFogi("#args")\n"; glFogi(args);
# define glFogiv(args...)	std::cerr << GL_DEBUG << "glFogiv("#args")\n"; glFogiv(args);
# define glFrontFace(args...)	std::cerr << GL_DEBUG << "glFrontFace("#args")\n"; glFrontFace(args);
# define glFrustum(args...)	std::cerr << GL_DEBUG << "glFrustum("#args")\n"; glFrustum(args);
# define glGenLists(args...)	std::cerr << GL_DEBUG << "glGenLists("#args")\n"; glGenLists(args);
# define glGenTextures(args...)	std::cerr << GL_DEBUG << "glGenTextures("#args")\n"; glGenTextures(args);
# define glGetBooleanv(args...)	std::cerr << GL_DEBUG << "glGetBooleanv("#args")\n"; glGetBooleanv(args);
# define glGetClipPlane(args...)	std::cerr << GL_DEBUG << "glGetClipPlane("#args")\n"; glGetClipPlane(args);
# define glGetDoublev(args...)	std::cerr << GL_DEBUG << "glGetDoublev("#args")\n"; glGetDoublev(args);
# define glGetError(args...)	std::cerr << GL_DEBUG << "glGetError("#args")\n"; glGetError(args);
# define glGetFloatv(args...)	std::cerr << GL_DEBUG << "glGetFloatv("#args")\n"; glGetFloatv(args);
# define glGetIntegerv(args...)	std::cerr << GL_DEBUG << "glGetIntegerv("#args")\n"; glGetIntegerv(args);
# define glGetLightfv(args...)	std::cerr << GL_DEBUG << "glGetLightfv("#args")\n"; glGetLightfv(args);
# define glGetLightiv(args...)	std::cerr << GL_DEBUG << "glGetLightiv("#args")\n"; glGetLightiv(args);
# define glGetMapdv(args...)	std::cerr << GL_DEBUG << "glGetMapdv("#args")\n"; glGetMapdv(args);
# define glGetMapfv(args...)	std::cerr << GL_DEBUG << "glGetMapfv("#args")\n"; glGetMapfv(args);
# define glGetMapiv(args...)	std::cerr << GL_DEBUG << "glGetMapiv("#args")\n"; glGetMapiv(args);
# define glGetMaterialfv(args...)	std::cerr << GL_DEBUG << "glGetMaterialfv("#args")\n"; glGetMaterialfv(args);
# define glGetMaterialiv(args...)	std::cerr << GL_DEBUG << "glGetMaterialiv("#args")\n"; glGetMaterialiv(args);
# define glGetPixelMapfv(args...)	std::cerr << GL_DEBUG << "glGetPixelMapfv("#args")\n"; glGetPixelMapfv(args);
# define glGetPixelMapuiv(args...)	std::cerr << GL_DEBUG << "glGetPixelMapuiv("#args")\n"; glGetPixelMapuiv(args);
# define glGetPixelMapusv(args...)	std::cerr << GL_DEBUG << "glGetPixelMapusv("#args")\n"; glGetPixelMapusv(args);
# define glGetPointerv(args...)	std::cerr << GL_DEBUG << "glGetPointerv("#args")\n"; glGetPointerv(args);
# define glGetPolygonStipple(args...)	std::cerr << GL_DEBUG << "glGetPolygonStipple("#args")\n"; glGetPolygonStipple(args);
# define glGetTexEnvfv(args...)	std::cerr << GL_DEBUG << "glGetTexEnvfv("#args")\n"; glGetTexEnvfv(args);
# define glGetTexEnviv(args...)	std::cerr << GL_DEBUG << "glGetTexEnviv("#args")\n"; glGetTexEnviv(args);
# define glGetTexGendv(args...)	std::cerr << GL_DEBUG << "glGetTexGendv("#args")\n"; glGetTexGendv(args);
# define glGetTexGenfv(args...)	std::cerr << GL_DEBUG << "glGetTexGenfv("#args")\n"; glGetTexGenfv(args);
# define glGetTexGeniv(args...)	std::cerr << GL_DEBUG << "glGetTexGeniv("#args")\n"; glGetTexGeniv(args);
# define glGetTexImage(args...)	std::cerr << GL_DEBUG << "glGetTexImage("#args")\n"; glGetTexImage(args);
# define glGetTexLevelParameterfv(args...)	std::cerr << GL_DEBUG << "glGetTexLevelParameterfv("#args")\n"; glGetTexLevelParameterfv(args);
# define glGetTexLevelParameteriv(args...)	std::cerr << GL_DEBUG << "glGetTexLevelParameteriv("#args")\n"; glGetTexLevelParameteriv(args);
# define glGetTexParameterfv(args...)	std::cerr << GL_DEBUG << "glGetTexParameterfv("#args")\n"; glGetTexParameterfv(args);
# define glGetTexParameteriv(args...)	std::cerr << GL_DEBUG << "glGetTexParameteriv("#args")\n"; glGetTexParameteriv(args);
# define glHint(args...)	std::cerr << GL_DEBUG << "glHint("#args")\n"; glHint(args);
# define glIndexMask(args...)	std::cerr << GL_DEBUG << "glIndexMask("#args")\n"; glIndexMask(args);
# define glIndexPointer(args...)	std::cerr << GL_DEBUG << "glIndexPointer("#args")\n"; glIndexPointer(args);
# define glIndexd(args...)	std::cerr << GL_DEBUG << "glIndexd("#args")\n"; glIndexd(args);
# define glIndexdv(args...)	std::cerr << GL_DEBUG << "glIndexdv("#args")\n"; glIndexdv(args);
# define glIndexf(args...)	std::cerr << GL_DEBUG << "glIndexf("#args")\n"; glIndexf(args);
# define glIndexfv(args...)	std::cerr << GL_DEBUG << "glIndexfv("#args")\n"; glIndexfv(args);
# define glIndexi(args...)	std::cerr << GL_DEBUG << "glIndexi("#args")\n"; glIndexi(args);
# define glIndexiv(args...)	std::cerr << GL_DEBUG << "glIndexiv("#args")\n"; glIndexiv(args);
# define glIndexs(args...)	std::cerr << GL_DEBUG << "glIndexs("#args")\n"; glIndexs(args);
# define glIndexsv(args...)	std::cerr << GL_DEBUG << "glIndexsv("#args")\n"; glIndexsv(args);
# define glIndexub(args...)	std::cerr << GL_DEBUG << "glIndexub("#args")\n"; glIndexub(args);
# define glIndexubv(args...)	std::cerr << GL_DEBUG << "glIndexubv("#args")\n"; glIndexubv(args);
# define glInitNames(args...)	std::cerr << GL_DEBUG << "glInitNames("#args")\n"; glInitNames(args);
# define glInterleavedArrays(args...)	std::cerr << GL_DEBUG << "glInterleavedArrays("#args")\n"; glInterleavedArrays(args);
# define glIsEnabled(args...)	std::cerr << GL_DEBUG << "glIsEnabled("#args")\n"; glIsEnabled(args);
# define glIsList(args...)	std::cerr << GL_DEBUG << "glIsList("#args")\n"; glIsList(args);
# define glIsTexture(args...)	std::cerr << GL_DEBUG << "glIsTexture("#args")\n"; glIsTexture(args);
# define glLightModelf(args...)	std::cerr << GL_DEBUG << "glLightModelf("#args")\n"; glLightModelf(args);
# define glLightModelfv(args...)	std::cerr << GL_DEBUG << "glLightModelfv("#args")\n"; glLightModelfv(args);
# define glLightModeli(args...)	std::cerr << GL_DEBUG << "glLightModeli("#args")\n"; glLightModeli(args);
# define glLightModeliv(args...)	std::cerr << GL_DEBUG << "glLightModeliv("#args")\n"; glLightModeliv(args);
# define glLightf(args...)	std::cerr << GL_DEBUG << "glLightf("#args")\n"; glLightf(args);
# define glLightfv(args...)	std::cerr << GL_DEBUG << "glLightfv("#args")\n"; glLightfv(args);
# define glLighti(args...)	std::cerr << GL_DEBUG << "glLighti("#args")\n"; glLighti(args);
# define glLightiv(args...)	std::cerr << GL_DEBUG << "glLightiv("#args")\n"; glLightiv(args);
# define glLineStipple(args...)	std::cerr << GL_DEBUG << "glLineStipple("#args")\n"; glLineStipple(args);
# define glLineWidth(args...)	std::cerr << GL_DEBUG << "glLineWidth("#args")\n"; glLineWidth(args);
# define glListBase(args...)	std::cerr << GL_DEBUG << "glListBase("#args")\n"; glListBase(args);
# define glLoadIdentity(args...)	std::cerr << GL_DEBUG << "glLoadIdentity("#args")\n"; glLoadIdentity(args);
# define glLoadMatrixd(args...)	std::cerr << GL_DEBUG << "glLoadMatrixd("#args")\n"; glLoadMatrixd(args);
# define glLoadMatrixf(args...)	std::cerr << GL_DEBUG << "glLoadMatrixf("#args")\n"; glLoadMatrixf(args);
# define glLoadName(args...)	std::cerr << GL_DEBUG << "glLoadName("#args")\n"; glLoadName(args);
# define glLogicOp(args...)	std::cerr << GL_DEBUG << "glLogicOp("#args")\n"; glLogicOp(args);
# define glMap1d(args...)	std::cerr << GL_DEBUG << "glMap1d("#args")\n"; glMap1d(args);
# define glMap1f(args...)	std::cerr << GL_DEBUG << "glMap1f("#args")\n"; glMap1f(args);
# define glMap2d(args...)	std::cerr << GL_DEBUG << "glMap2d("#args")\n"; glMap2d(args);
# define glMap2f(args...)	std::cerr << GL_DEBUG << "glMap2f("#args")\n"; glMap2f(args);
# define glMapGrid1d(args...)	std::cerr << GL_DEBUG << "glMapGrid1d("#args")\n"; glMapGrid1d(args);
# define glMapGrid1f(args...)	std::cerr << GL_DEBUG << "glMapGrid1f("#args")\n"; glMapGrid1f(args);
# define glMapGrid2d(args...)	std::cerr << GL_DEBUG << "glMapGrid2d("#args")\n"; glMapGrid2d(args);
# define glMapGrid2f(args...)	std::cerr << GL_DEBUG << "glMapGrid2f("#args")\n"; glMapGrid2f(args);
# define glMaterialf(args...)	std::cerr << GL_DEBUG << "glMaterialf("#args")\n"; glMaterialf(args);
# define glMaterialfv(args...)	std::cerr << GL_DEBUG << "glMaterialfv("#args")\n"; glMaterialfv(args);
# define glMateriali(args...)	std::cerr << GL_DEBUG << "glMateriali("#args")\n"; glMateriali(args);
# define glMaterialiv(args...)	std::cerr << GL_DEBUG << "glMaterialiv("#args")\n"; glMaterialiv(args);
# define glMatrixMode(args...)	std::cerr << GL_DEBUG << "glMatrixMode("#args")\n"; glMatrixMode(args);
# define glMultMatrixd(args...)	std::cerr << GL_DEBUG << "glMultMatrixd("#args")\n"; glMultMatrixd(args);
# define glMultMatrixf(args...)	std::cerr << GL_DEBUG << "glMultMatrixf("#args")\n"; glMultMatrixf(args);
# define glNewList(args...)	std::cerr << GL_DEBUG << "glNewList("#args")\n"; glNewList(args);
# define glNormal3b(args...)	std::cerr << GL_DEBUG << "glNormal3b("#args")\n"; glNormal3b(args);
# define glNormal3bv(args...)	std::cerr << GL_DEBUG << "glNormal3bv("#args")\n"; glNormal3bv(args);
# define glNormal3d(args...)	std::cerr << GL_DEBUG << "glNormal3d("#args")\n"; glNormal3d(args);
# define glNormal3dv(args...)	std::cerr << GL_DEBUG << "glNormal3dv("#args")\n"; glNormal3dv(args);
# define glNormal3f(args...)	std::cerr << GL_DEBUG << "glNormal3f("#args")\n"; glNormal3f(args);
# define glNormal3fv(args...)	std::cerr << GL_DEBUG << "glNormal3fv("#args")\n"; glNormal3fv(args);
# define glNormal3i(args...)	std::cerr << GL_DEBUG << "glNormal3i("#args")\n"; glNormal3i(args);
# define glNormal3iv(args...)	std::cerr << GL_DEBUG << "glNormal3iv("#args")\n"; glNormal3iv(args);
# define glNormal3s(args...)	std::cerr << GL_DEBUG << "glNormal3s("#args")\n"; glNormal3s(args);
# define glNormal3sv(args...)	std::cerr << GL_DEBUG << "glNormal3sv("#args")\n"; glNormal3sv(args);
# define glNormalPointer(args...)	std::cerr << GL_DEBUG << "glNormalPointer("#args")\n"; glNormalPointer(args);
# define glOrtho(args...)	std::cerr << GL_DEBUG << "glOrtho("#args")\n"; glOrtho(args);
# define glPassThrough(args...)	std::cerr << GL_DEBUG << "glPassThrough("#args")\n"; glPassThrough(args);
# define glPixelMapfv(args...)	std::cerr << GL_DEBUG << "glPixelMapfv("#args")\n"; glPixelMapfv(args);
# define glPixelMapuiv(args...)	std::cerr << GL_DEBUG << "glPixelMapuiv("#args")\n"; glPixelMapuiv(args);
# define glPixelMapusv(args...)	std::cerr << GL_DEBUG << "glPixelMapusv("#args")\n"; glPixelMapusv(args);
# define glPixelStoref(args...)	std::cerr << GL_DEBUG << "glPixelStoref("#args")\n"; glPixelStoref(args);
# define glPixelStorei(args...)	std::cerr << GL_DEBUG << "glPixelStorei("#args")\n"; glPixelStorei(args);
# define glPixelTransferf(args...)	std::cerr << GL_DEBUG << "glPixelTransferf("#args")\n"; glPixelTransferf(args);
# define glPixelTransferi(args...)	std::cerr << GL_DEBUG << "glPixelTransferi("#args")\n"; glPixelTransferi(args);
# define glPixelZoom(args...)	std::cerr << GL_DEBUG << "glPixelZoom("#args")\n"; glPixelZoom(args);
# define glPointSize(args...)	std::cerr << GL_DEBUG << "glPointSize("#args")\n"; glPointSize(args);
# define glPolygonMode(args...)	std::cerr << GL_DEBUG << "glPolygonMode("#args")\n"; glPolygonMode(args);
# define glPolygonOffset(args...)	std::cerr << GL_DEBUG << "glPolygonOffset("#args")\n"; glPolygonOffset(args);
# define glPolygonStipple(args...)	std::cerr << GL_DEBUG << "glPolygonStipple("#args")\n"; glPolygonStipple(args);
# define glPopAttrib(args...)	std::cerr << GL_DEBUG << "glPopAttrib("#args")\n"; glPopAttrib(args);
# define glPopClientAttrib(args...)	std::cerr << GL_DEBUG << "glPopClientAttrib("#args")\n"; glPopClientAttrib(args);
# define glPopMatrix(args...)	std::cerr << GL_DEBUG << "glPopMatrix("#args")\n"; glPopMatrix(args);
# define glPopName(args...)	std::cerr << GL_DEBUG << "glPopName("#args")\n"; glPopName(args);
# define glPrioritizeTextures(args...)	std::cerr << GL_DEBUG << "glPrioritizeTextures("#args")\n"; glPrioritizeTextures(args);
# define glPushAttrib(args...)	std::cerr << GL_DEBUG << "glPushAttrib("#args")\n"; glPushAttrib(args);
# define glPushClientAttrib(args...)	std::cerr << GL_DEBUG << "glPushClientAttrib("#args")\n"; glPushClientAttrib(args);
# define glPushMatrix(args...)	std::cerr << GL_DEBUG << "glPushMatrix("#args")\n"; glPushMatrix(args);
# define glPushName(args...)	std::cerr << GL_DEBUG << "glPushName("#args")\n"; glPushName(args);
# define glRasterPos2d(args...)	std::cerr << GL_DEBUG << "glRasterPos2d("#args")\n"; glRasterPos2d(args);
# define glRasterPos2dv(args...)	std::cerr << GL_DEBUG << "glRasterPos2dv("#args")\n"; glRasterPos2dv(args);
# define glRasterPos2f(args...)	std::cerr << GL_DEBUG << "glRasterPos2f("#args")\n"; glRasterPos2f(args);
# define glRasterPos2fv(args...)	std::cerr << GL_DEBUG << "glRasterPos2fv("#args")\n"; glRasterPos2fv(args);
# define glRasterPos2i(args...)	std::cerr << GL_DEBUG << "glRasterPos2i("#args")\n"; glRasterPos2i(args);
# define glRasterPos2iv(args...)	std::cerr << GL_DEBUG << "glRasterPos2iv("#args")\n"; glRasterPos2iv(args);
# define glRasterPos2s(args...)	std::cerr << GL_DEBUG << "glRasterPos2s("#args")\n"; glRasterPos2s(args);
# define glRasterPos2sv(args...)	std::cerr << GL_DEBUG << "glRasterPos2sv("#args")\n"; glRasterPos2sv(args);
# define glRasterPos3d(args...)	std::cerr << GL_DEBUG << "glRasterPos3d("#args")\n"; glRasterPos3d(args);
# define glRasterPos3dv(args...)	std::cerr << GL_DEBUG << "glRasterPos3dv("#args")\n"; glRasterPos3dv(args);
# define glRasterPos3f(args...)	std::cerr << GL_DEBUG << "glRasterPos3f("#args")\n"; glRasterPos3f(args);
# define glRasterPos3fv(args...)	std::cerr << GL_DEBUG << "glRasterPos3fv("#args")\n"; glRasterPos3fv(args);
# define glRasterPos3i(args...)	std::cerr << GL_DEBUG << "glRasterPos3i("#args")\n"; glRasterPos3i(args);
# define glRasterPos3iv(args...)	std::cerr << GL_DEBUG << "glRasterPos3iv("#args")\n"; glRasterPos3iv(args);
# define glRasterPos3s(args...)	std::cerr << GL_DEBUG << "glRasterPos3s("#args")\n"; glRasterPos3s(args);
# define glRasterPos3sv(args...)	std::cerr << GL_DEBUG << "glRasterPos3sv("#args")\n"; glRasterPos3sv(args);
# define glRasterPos4d(args...)	std::cerr << GL_DEBUG << "glRasterPos4d("#args")\n"; glRasterPos4d(args);
# define glRasterPos4dv(args...)	std::cerr << GL_DEBUG << "glRasterPos4dv("#args")\n"; glRasterPos4dv(args);
# define glRasterPos4f(args...)	std::cerr << GL_DEBUG << "glRasterPos4f("#args")\n"; glRasterPos4f(args);
# define glRasterPos4fv(args...)	std::cerr << GL_DEBUG << "glRasterPos4fv("#args")\n"; glRasterPos4fv(args);
# define glRasterPos4i(args...)	std::cerr << GL_DEBUG << "glRasterPos4i("#args")\n"; glRasterPos4i(args);
# define glRasterPos4iv(args...)	std::cerr << GL_DEBUG << "glRasterPos4iv("#args")\n"; glRasterPos4iv(args);
# define glRasterPos4s(args...)	std::cerr << GL_DEBUG << "glRasterPos4s("#args")\n"; glRasterPos4s(args);
# define glRasterPos4sv(args...)	std::cerr << GL_DEBUG << "glRasterPos4sv("#args")\n"; glRasterPos4sv(args);
# define glReadBuffer(args...)	std::cerr << GL_DEBUG << "glReadBuffer("#args")\n"; glReadBuffer(args);
# define glReadPixels(args...)	std::cerr << GL_DEBUG << "glReadPixels("#args")\n"; glReadPixels(args);
# define glRectd(args...)	std::cerr << GL_DEBUG << "glRectd("#args")\n"; glRectd(args);
# define glRectdv(args...)	std::cerr << GL_DEBUG << "glRectdv("#args")\n"; glRectdv(args);
# define glRectf(args...)	std::cerr << GL_DEBUG << "glRectf("#args")\n"; glRectf(args);
# define glRectfv(args...)	std::cerr << GL_DEBUG << "glRectfv("#args")\n"; glRectfv(args);
# define glRecti(args...)	std::cerr << GL_DEBUG << "glRecti("#args")\n"; glRecti(args);
# define glRectiv(args...)	std::cerr << GL_DEBUG << "glRectiv("#args")\n"; glRectiv(args);
# define glRects(args...)	std::cerr << GL_DEBUG << "glRects("#args")\n"; glRects(args);
# define glRectsv(args...)	std::cerr << GL_DEBUG << "glRectsv("#args")\n"; glRectsv(args);
# define glRenderMode(args...)	std::cerr << GL_DEBUG << "glRenderMode("#args")\n"; glRenderMode(args);
# define glRotated(args...)	std::cerr << GL_DEBUG << "glRotated("#args")\n"; glRotated(args);
# define glRotatef(args...)	std::cerr << GL_DEBUG << "glRotatef("#args")\n"; glRotatef(args);
# define glScaled(args...)	std::cerr << GL_DEBUG << "glScaled("#args")\n"; glScaled(args);
# define glScalef(args...)	std::cerr << GL_DEBUG << "glScalef("#args")\n"; glScalef(args);
# define glScissor(args...)	std::cerr << GL_DEBUG << "glScissor("#args")\n"; glScissor(args);
# define glSelectBuffer(args...)	std::cerr << GL_DEBUG << "glSelectBuffer("#args")\n"; glSelectBuffer(args);
# define glShadeModel(args...)	std::cerr << GL_DEBUG << "glShadeModel("#args")\n"; glShadeModel(args);
# define glStencilFunc(args...)	std::cerr << GL_DEBUG << "glStencilFunc("#args")\n"; glStencilFunc(args);
# define glStencilMask(args...)	std::cerr << GL_DEBUG << "glStencilMask("#args")\n"; glStencilMask(args);
# define glStencilOp(args...)	std::cerr << GL_DEBUG << "glStencilOp("#args")\n"; glStencilOp(args);
# define glTexCoord1d(args...)	std::cerr << GL_DEBUG << "glTexCoord1d("#args")\n"; glTexCoord1d(args);
# define glTexCoord1dv(args...)	std::cerr << GL_DEBUG << "glTexCoord1dv("#args")\n"; glTexCoord1dv(args);
# define glTexCoord1f(args...)	std::cerr << GL_DEBUG << "glTexCoord1f("#args")\n"; glTexCoord1f(args);
# define glTexCoord1fv(args...)	std::cerr << GL_DEBUG << "glTexCoord1fv("#args")\n"; glTexCoord1fv(args);
# define glTexCoord1i(args...)	std::cerr << GL_DEBUG << "glTexCoord1i("#args")\n"; glTexCoord1i(args);
# define glTexCoord1iv(args...)	std::cerr << GL_DEBUG << "glTexCoord1iv("#args")\n"; glTexCoord1iv(args);
# define glTexCoord1s(args...)	std::cerr << GL_DEBUG << "glTexCoord1s("#args")\n"; glTexCoord1s(args);
# define glTexCoord1sv(args...)	std::cerr << GL_DEBUG << "glTexCoord1sv("#args")\n"; glTexCoord1sv(args);
# define glTexCoord2d(args...)	std::cerr << GL_DEBUG << "glTexCoord2d("#args")\n"; glTexCoord2d(args);
# define glTexCoord2dv(args...)	std::cerr << GL_DEBUG << "glTexCoord2dv("#args")\n"; glTexCoord2dv(args);
# define glTexCoord2f(args...)	std::cerr << GL_DEBUG << "glTexCoord2f("#args")\n"; glTexCoord2f(args);
# define glTexCoord2fv(args...)	std::cerr << GL_DEBUG << "glTexCoord2fv("#args")\n"; glTexCoord2fv(args);
# define glTexCoord2i(args...)	std::cerr << GL_DEBUG << "glTexCoord2i("#args")\n"; glTexCoord2i(args);
# define glTexCoord2iv(args...)	std::cerr << GL_DEBUG << "glTexCoord2iv("#args")\n"; glTexCoord2iv(args);
# define glTexCoord2s(args...)	std::cerr << GL_DEBUG << "glTexCoord2s("#args")\n"; glTexCoord2s(args);
# define glTexCoord2sv(args...)	std::cerr << GL_DEBUG << "glTexCoord2sv("#args")\n"; glTexCoord2sv(args);
# define glTexCoord3d(args...)	std::cerr << GL_DEBUG << "glTexCoord3d("#args")\n"; glTexCoord3d(args);
# define glTexCoord3dv(args...)	std::cerr << GL_DEBUG << "glTexCoord3dv("#args")\n"; glTexCoord3dv(args);
# define glTexCoord3f(args...)	std::cerr << GL_DEBUG << "glTexCoord3f("#args")\n"; glTexCoord3f(args);
# define glTexCoord3fv(args...)	std::cerr << GL_DEBUG << "glTexCoord3fv("#args")\n"; glTexCoord3fv(args);
# define glTexCoord3i(args...)	std::cerr << GL_DEBUG << "glTexCoord3i("#args")\n"; glTexCoord3i(args);
# define glTexCoord3iv(args...)	std::cerr << GL_DEBUG << "glTexCoord3iv("#args")\n"; glTexCoord3iv(args);
# define glTexCoord3s(args...)	std::cerr << GL_DEBUG << "glTexCoord3s("#args")\n"; glTexCoord3s(args);
# define glTexCoord3sv(args...)	std::cerr << GL_DEBUG << "glTexCoord3sv("#args")\n"; glTexCoord3sv(args);
# define glTexCoord4d(args...)	std::cerr << GL_DEBUG << "glTexCoord4d("#args")\n"; glTexCoord4d(args);
# define glTexCoord4dv(args...)	std::cerr << GL_DEBUG << "glTexCoord4dv("#args")\n"; glTexCoord4dv(args);
# define glTexCoord4f(args...)	std::cerr << GL_DEBUG << "glTexCoord4f("#args")\n"; glTexCoord4f(args);
# define glTexCoord4fv(args...)	std::cerr << GL_DEBUG << "glTexCoord4fv("#args")\n"; glTexCoord4fv(args);
# define glTexCoord4i(args...)	std::cerr << GL_DEBUG << "glTexCoord4i("#args")\n"; glTexCoord4i(args);
# define glTexCoord4iv(args...)	std::cerr << GL_DEBUG << "glTexCoord4iv("#args")\n"; glTexCoord4iv(args);
# define glTexCoord4s(args...)	std::cerr << GL_DEBUG << "glTexCoord4s("#args")\n"; glTexCoord4s(args);
# define glTexCoord4sv(args...)	std::cerr << GL_DEBUG << "glTexCoord4sv("#args")\n"; glTexCoord4sv(args);
# define glTexCoordPointer(args...)	std::cerr << GL_DEBUG << "glTexCoordPointer("#args")\n"; glTexCoordPointer(args);
# define glTexEnvf(args...)	std::cerr << GL_DEBUG << "glTexEnvf("#args")\n"; glTexEnvf(args);
# define glTexEnvfv(args...)	std::cerr << GL_DEBUG << "glTexEnvfv("#args")\n"; glTexEnvfv(args);
# define glTexEnvi(args...)	std::cerr << GL_DEBUG << "glTexEnvi("#args")\n"; glTexEnvi(args);
# define glTexEnviv(args...)	std::cerr << GL_DEBUG << "glTexEnviv("#args")\n"; glTexEnviv(args);
# define glTexGend(args...)	std::cerr << GL_DEBUG << "glTexGend("#args")\n"; glTexGend(args);
# define glTexGendv(args...)	std::cerr << GL_DEBUG << "glTexGendv("#args")\n"; glTexGendv(args);
# define glTexGenf(args...)	std::cerr << GL_DEBUG << "glTexGenf("#args")\n"; glTexGenf(args);
# define glTexGenfv(args...)	std::cerr << GL_DEBUG << "glTexGenfv("#args")\n"; glTexGenfv(args);
# define glTexGeni(args...)	std::cerr << GL_DEBUG << "glTexGeni("#args")\n"; glTexGeni(args);
# define glTexGeniv(args...)	std::cerr << GL_DEBUG << "glTexGeniv("#args")\n"; glTexGeniv(args);
# define glTexImage1D(args...)	std::cerr << GL_DEBUG << "glTexImage1D("#args")\n"; glTexImage1D(args);
# define glTexImage2D(args...)	std::cerr << GL_DEBUG << "glTexImage2D("#args")\n"; glTexImage2D(args);
# define glTexParameterf(args...)	std::cerr << GL_DEBUG << "glTexParameterf("#args")\n"; glTexParameterf(args);
# define glTexParameterfv(args...)	std::cerr << GL_DEBUG << "glTexParameterfv("#args")\n"; glTexParameterfv(args);
# define glTexParameteri(args...)	std::cerr << GL_DEBUG << "glTexParameteri("#args")\n"; glTexParameteri(args);
# define glTexParameteriv(args...)	std::cerr << GL_DEBUG << "glTexParameteriv("#args")\n"; glTexParameteriv(args);
# define glTexSubImage1D(args...)	std::cerr << GL_DEBUG << "glTexSubImage1D("#args")\n"; glTexSubImage1D(args);
# define glTexSubImage2D(args...)	std::cerr << GL_DEBUG << "glTexSubImage2D("#args")\n"; glTexSubImage2D(args);
# define glTranslated(args...)	std::cerr << GL_DEBUG << "glTranslated("#args")\n"; glTranslated(args);
# define glTranslatef(args...)	std::cerr << GL_DEBUG << "glTranslatef("#args")\n"; glTranslatef(args);
# define glVertex2d(args...)	std::cerr << GL_DEBUG << "glVertex2d("#args")\n"; glVertex2d(args);
# define glVertex2dv(args...)	std::cerr << GL_DEBUG << "glVertex2dv("#args")\n"; glVertex2dv(args);
# define glVertex2f(args...)	std::cerr << GL_DEBUG << "glVertex2f("#args")\n"; glVertex2f(args);
# define glVertex2fv(args...)	std::cerr << GL_DEBUG << "glVertex2fv("#args")\n"; glVertex2fv(args);
# define glVertex2i(args...)	std::cerr << GL_DEBUG << "glVertex2i("#args")\n"; glVertex2i(args);
# define glVertex2iv(args...)	std::cerr << GL_DEBUG << "glVertex2iv("#args")\n"; glVertex2iv(args);
# define glVertex2s(args...)	std::cerr << GL_DEBUG << "glVertex2s("#args")\n"; glVertex2s(args);
# define glVertex2sv(args...)	std::cerr << GL_DEBUG << "glVertex2sv("#args")\n"; glVertex2sv(args);
# define glVertex3d(args...)	std::cerr << GL_DEBUG << "glVertex3d("#args")\n"; glVertex3d(args);
# define glVertex3dv(args...)	std::cerr << GL_DEBUG << "glVertex3dv("#args")\n"; glVertex3dv(args);
# define glVertex3f(args...)	std::cerr << GL_DEBUG << "glVertex3f("#args")\n"; glVertex3f(args);
# define glVertex3fv(args...)	std::cerr << GL_DEBUG << "glVertex3fv("#args")\n"; glVertex3fv(args);
# define glVertex3i(args...)	std::cerr << GL_DEBUG << "glVertex3i("#args")\n"; glVertex3i(args);
# define glVertex3iv(args...)	std::cerr << GL_DEBUG << "glVertex3iv("#args")\n"; glVertex3iv(args);
# define glVertex3s(args...)	std::cerr << GL_DEBUG << "glVertex3s("#args")\n"; glVertex3s(args);
# define glVertex3sv(args...)	std::cerr << GL_DEBUG << "glVertex3sv("#args")\n"; glVertex3sv(args);
# define glVertex4d(args...)	std::cerr << GL_DEBUG << "glVertex4d("#args")\n"; glVertex4d(args);
# define glVertex4dv(args...)	std::cerr << GL_DEBUG << "glVertex4dv("#args")\n"; glVertex4dv(args);
# define glVertex4f(args...)	std::cerr << GL_DEBUG << "glVertex4f("#args")\n"; glVertex4f(args);
# define glVertex4fv(args...)	std::cerr << GL_DEBUG << "glVertex4fv("#args")\n"; glVertex4fv(args);
# define glVertex4i(args...)	std::cerr << GL_DEBUG << "glVertex4i("#args")\n"; glVertex4i(args);
# define glVertex4iv(args...)	std::cerr << GL_DEBUG << "glVertex4iv("#args")\n"; glVertex4iv(args);
# define glVertex4s(args...)	std::cerr << GL_DEBUG << "glVertex4s("#args")\n"; glVertex4s(args);
# define glVertex4sv(args...)	std::cerr << GL_DEBUG << "glVertex4sv("#args")\n"; glVertex4sv(args);
# define glVertexPointer(args...)	std::cerr << GL_DEBUG << "glVertexPointer("#args")\n"; glVertexPointer(args);
# define glViewport(args...)	std::cerr << GL_DEBUG << "glViewport("#args")\n"; glViewport(args);
//	}}} gl-Functions

//	{{{ glext-Functions
# define glBlendColor(args...)	std::cerr << GL_DEBUG << "glBlendColor("#args")\n"; glBlendColor(args);
# define glBlendEquation(args...)	std::cerr << GL_DEBUG << "glBlendEquation("#args")\n"; glBlendEquation(args);
# define glDrawRangeElements(args...)	std::cerr << GL_DEBUG << "glDrawRangeElements("#args")\n"; glDrawRangeElements(args);
# define glColorTable(args...)	std::cerr << GL_DEBUG << "glColorTable("#args")\n"; glColorTable(args);
# define glColorTableParameterfv(args...)	std::cerr << GL_DEBUG << "glColorTableParameterfv("#args")\n"; glColorTableParameterfv(args);
# define glColorTableParameteriv(args...)	std::cerr << GL_DEBUG << "glColorTableParameteriv("#args")\n"; glColorTableParameteriv(args);
# define glCopyColorTable(args...)	std::cerr << GL_DEBUG << "glCopyColorTable("#args")\n"; glCopyColorTable(args);
# define glGetColorTable(args...)	std::cerr << GL_DEBUG << "glGetColorTable("#args")\n"; glGetColorTable(args);
# define glGetColorTableParameterfv(args...)	std::cerr << GL_DEBUG << "glGetColorTableParameterfv("#args")\n"; glGetColorTableParameterfv(args);
# define glGetColorTableParameteriv(args...)	std::cerr << GL_DEBUG << "glGetColorTableParameteriv("#args")\n"; glGetColorTableParameteriv(args);
# define glColorSubTable(args...)	std::cerr << GL_DEBUG << "glColorSubTable("#args")\n"; glColorSubTable(args);
# define glCopyColorSubTable(args...)	std::cerr << GL_DEBUG << "glCopyColorSubTable("#args")\n"; glCopyColorSubTable(args);
# define glConvolutionFilter1D(args...)	std::cerr << GL_DEBUG << "glConvolutionFilter1D("#args")\n"; glConvolutionFilter1D(args);
# define glConvolutionFilter2D(args...)	std::cerr << GL_DEBUG << "glConvolutionFilter2D("#args")\n"; glConvolutionFilter2D(args);
# define glConvolutionParameterf(args...)	std::cerr << GL_DEBUG << "glConvolutionParameterf("#args")\n"; glConvolutionParameterf(args);
# define glConvolutionParameterfv(args...)	std::cerr << GL_DEBUG << "glConvolutionParameterfv("#args")\n"; glConvolutionParameterfv(args);
# define glConvolutionParameteri(args...)	std::cerr << GL_DEBUG << "glConvolutionParameteri("#args")\n"; glConvolutionParameteri(args);
# define glConvolutionParameteriv(args...)	std::cerr << GL_DEBUG << "glConvolutionParameteriv("#args")\n"; glConvolutionParameteriv(args);
# define glCopyConvolutionFilter1D(args...)	std::cerr << GL_DEBUG << "glCopyConvolutionFilter1D("#args")\n"; glCopyConvolutionFilter1D(args);
# define glCopyConvolutionFilter2D(args...)	std::cerr << GL_DEBUG << "glCopyConvolutionFilter2D("#args")\n"; glCopyConvolutionFilter2D(args);
# define glGetConvolutionFilter(args...)	std::cerr << GL_DEBUG << "glGetConvolutionFilter("#args")\n"; glGetConvolutionFilter(args);
# define glGetConvolutionParameterfv(args...)	std::cerr << GL_DEBUG << "glGetConvolutionParameterfv("#args")\n"; glGetConvolutionParameterfv(args);
# define glGetConvolutionParameteriv(args...)	std::cerr << GL_DEBUG << "glGetConvolutionParameteriv("#args")\n"; glGetConvolutionParameteriv(args);
# define glGetSeparableFilter(args...)	std::cerr << GL_DEBUG << "glGetSeparableFilter("#args")\n"; glGetSeparableFilter(args);
# define glSeparableFilter2D(args...)	std::cerr << GL_DEBUG << "glSeparableFilter2D("#args")\n"; glSeparableFilter2D(args);
# define glGetHistogram(args...)	std::cerr << GL_DEBUG << "glGetHistogram("#args")\n"; glGetHistogram(args);
# define glGetHistogramParameterfv(args...)	std::cerr << GL_DEBUG << "glGetHistogramParameterfv("#args")\n"; glGetHistogramParameterfv(args);
# define glGetHistogramParameteriv(args...)	std::cerr << GL_DEBUG << "glGetHistogramParameteriv("#args")\n"; glGetHistogramParameteriv(args);
# define glGetMinmax(args...)	std::cerr << GL_DEBUG << "glGetMinmax("#args")\n"; glGetMinmax(args);
# define glGetMinmaxParameterfv(args...)	std::cerr << GL_DEBUG << "glGetMinmaxParameterfv("#args")\n"; glGetMinmaxParameterfv(args);
# define glGetMinmaxParameteriv(args...)	std::cerr << GL_DEBUG << "glGetMinmaxParameteriv("#args")\n"; glGetMinmaxParameteriv(args);
# define glHistogram(args...)	std::cerr << GL_DEBUG << "glHistogram("#args")\n"; glHistogram(args);
# define glMinmax(args...)	std::cerr << GL_DEBUG << "glMinmax("#args")\n"; glMinmax(args);
# define glResetHistogram(args...)	std::cerr << GL_DEBUG << "glResetHistogram("#args")\n"; glResetHistogram(args);
# define glResetMinmax(args...)	std::cerr << GL_DEBUG << "glResetMinmax("#args")\n"; glResetMinmax(args);
# define glTexImage3D(args...)	std::cerr << GL_DEBUG << "glTexImage3D("#args")\n"; glTexImage3D(args);
# define glTexSubImage3D(args...)	std::cerr << GL_DEBUG << "glTexSubImage3D("#args")\n"; glTexSubImage3D(args);
# define glCopyTexSubImage3D(args...)	std::cerr << GL_DEBUG << "glCopyTexSubImage3D("#args")\n"; glCopyTexSubImage3D(args);
# define glActiveTexture(args...)	std::cerr << GL_DEBUG << "glActiveTexture("#args")\n"; glActiveTexture(args);
# define glClientActiveTexture(args...)	std::cerr << GL_DEBUG << "glClientActiveTexture("#args")\n"; glClientActiveTexture(args);
# define glMultiTexCoord1d(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1d("#args")\n"; glMultiTexCoord1d(args);
# define glMultiTexCoord1dv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1dv("#args")\n"; glMultiTexCoord1dv(args);
# define glMultiTexCoord1f(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1f("#args")\n"; glMultiTexCoord1f(args);
# define glMultiTexCoord1fv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1fv("#args")\n"; glMultiTexCoord1fv(args);
# define glMultiTexCoord1i(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1i("#args")\n"; glMultiTexCoord1i(args);
# define glMultiTexCoord1iv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1iv("#args")\n"; glMultiTexCoord1iv(args);
# define glMultiTexCoord1s(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1s("#args")\n"; glMultiTexCoord1s(args);
# define glMultiTexCoord1sv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1sv("#args")\n"; glMultiTexCoord1sv(args);
# define glMultiTexCoord2d(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2d("#args")\n"; glMultiTexCoord2d(args);
# define glMultiTexCoord2dv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2dv("#args")\n"; glMultiTexCoord2dv(args);
# define glMultiTexCoord2f(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2f("#args")\n"; glMultiTexCoord2f(args);
# define glMultiTexCoord2fv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2fv("#args")\n"; glMultiTexCoord2fv(args);
# define glMultiTexCoord2i(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2i("#args")\n"; glMultiTexCoord2i(args);
# define glMultiTexCoord2iv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2iv("#args")\n"; glMultiTexCoord2iv(args);
# define glMultiTexCoord2s(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2s("#args")\n"; glMultiTexCoord2s(args);
# define glMultiTexCoord2sv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2sv("#args")\n"; glMultiTexCoord2sv(args);
# define glMultiTexCoord3d(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3d("#args")\n"; glMultiTexCoord3d(args);
# define glMultiTexCoord3dv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3dv("#args")\n"; glMultiTexCoord3dv(args);
# define glMultiTexCoord3f(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3f("#args")\n"; glMultiTexCoord3f(args);
# define glMultiTexCoord3fv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3fv("#args")\n"; glMultiTexCoord3fv(args);
# define glMultiTexCoord3i(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3i("#args")\n"; glMultiTexCoord3i(args);
# define glMultiTexCoord3iv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3iv("#args")\n"; glMultiTexCoord3iv(args);
# define glMultiTexCoord3s(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3s("#args")\n"; glMultiTexCoord3s(args);
# define glMultiTexCoord3sv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3sv("#args")\n"; glMultiTexCoord3sv(args);
# define glMultiTexCoord4d(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4d("#args")\n"; glMultiTexCoord4d(args);
# define glMultiTexCoord4dv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4dv("#args")\n"; glMultiTexCoord4dv(args);
# define glMultiTexCoord4f(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4f("#args")\n"; glMultiTexCoord4f(args);
# define glMultiTexCoord4fv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4fv("#args")\n"; glMultiTexCoord4fv(args);
# define glMultiTexCoord4i(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4i("#args")\n"; glMultiTexCoord4i(args);
# define glMultiTexCoord4iv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4iv("#args")\n"; glMultiTexCoord4iv(args);
# define glMultiTexCoord4s(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4s("#args")\n"; glMultiTexCoord4s(args);
# define glMultiTexCoord4sv(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4sv("#args")\n"; glMultiTexCoord4sv(args);
# define glLoadTransposeMatrixf(args...)	std::cerr << GL_DEBUG << "glLoadTransposeMatrixf("#args")\n"; glLoadTransposeMatrixf(args);
# define glLoadTransposeMatrixd(args...)	std::cerr << GL_DEBUG << "glLoadTransposeMatrixd("#args")\n"; glLoadTransposeMatrixd(args);
# define glMultTransposeMatrixf(args...)	std::cerr << GL_DEBUG << "glMultTransposeMatrixf("#args")\n"; glMultTransposeMatrixf(args);
# define glMultTransposeMatrixd(args...)	std::cerr << GL_DEBUG << "glMultTransposeMatrixd("#args")\n"; glMultTransposeMatrixd(args);
# define glSampleCoverage(args...)	std::cerr << GL_DEBUG << "glSampleCoverage("#args")\n"; glSampleCoverage(args);
# define glCompressedTexImage3D(args...)	std::cerr << GL_DEBUG << "glCompressedTexImage3D("#args")\n"; glCompressedTexImage3D(args);
# define glCompressedTexImage2D(args...)	std::cerr << GL_DEBUG << "glCompressedTexImage2D("#args")\n"; glCompressedTexImage2D(args);
# define glCompressedTexImage1D(args...)	std::cerr << GL_DEBUG << "glCompressedTexImage1D("#args")\n"; glCompressedTexImage1D(args);
# define glCompressedTexSubImage3D(args...)	std::cerr << GL_DEBUG << "glCompressedTexSubImage3D("#args")\n"; glCompressedTexSubImage3D(args);
# define glCompressedTexSubImage2D(args...)	std::cerr << GL_DEBUG << "glCompressedTexSubImage2D("#args")\n"; glCompressedTexSubImage2D(args);
# define glCompressedTexSubImage1D(args...)	std::cerr << GL_DEBUG << "glCompressedTexSubImage1D("#args")\n"; glCompressedTexSubImage1D(args);
# define glGetCompressedTexImage(args...)	std::cerr << GL_DEBUG << "glGetCompressedTexImage("#args")\n"; glGetCompressedTexImage(args);
# define glBlendFuncSeparate(args...)	std::cerr << GL_DEBUG << "glBlendFuncSeparate("#args")\n"; glBlendFuncSeparate(args);
# define glFogCoordf(args...)	std::cerr << GL_DEBUG << "glFogCoordf("#args")\n"; glFogCoordf(args);
# define glFogCoordfv(args...)	std::cerr << GL_DEBUG << "glFogCoordfv("#args")\n"; glFogCoordfv(args);
# define glFogCoordd(args...)	std::cerr << GL_DEBUG << "glFogCoordd("#args")\n"; glFogCoordd(args);
# define glFogCoorddv(args...)	std::cerr << GL_DEBUG << "glFogCoorddv("#args")\n"; glFogCoorddv(args);
# define glFogCoordPointer(args...)	std::cerr << GL_DEBUG << "glFogCoordPointer("#args")\n"; glFogCoordPointer(args);
# define glMultiDrawArrays(args...)	std::cerr << GL_DEBUG << "glMultiDrawArrays("#args")\n"; glMultiDrawArrays(args);
# define glMultiDrawElements(args...)	std::cerr << GL_DEBUG << "glMultiDrawElements("#args")\n"; glMultiDrawElements(args);
# define glPointParameterf(args...)	std::cerr << GL_DEBUG << "glPointParameterf("#args")\n"; glPointParameterf(args);
# define glPointParameterfv(args...)	std::cerr << GL_DEBUG << "glPointParameterfv("#args")\n"; glPointParameterfv(args);
# define glPointParameteri(args...)	std::cerr << GL_DEBUG << "glPointParameteri("#args")\n"; glPointParameteri(args);
# define glPointParameteriv(args...)	std::cerr << GL_DEBUG << "glPointParameteriv("#args")\n"; glPointParameteriv(args);
# define glSecondaryColor3b(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3b("#args")\n"; glSecondaryColor3b(args);
# define glSecondaryColor3bv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3bv("#args")\n"; glSecondaryColor3bv(args);
# define glSecondaryColor3d(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3d("#args")\n"; glSecondaryColor3d(args);
# define glSecondaryColor3dv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3dv("#args")\n"; glSecondaryColor3dv(args);
# define glSecondaryColor3f(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3f("#args")\n"; glSecondaryColor3f(args);
# define glSecondaryColor3fv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3fv("#args")\n"; glSecondaryColor3fv(args);
# define glSecondaryColor3i(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3i("#args")\n"; glSecondaryColor3i(args);
# define glSecondaryColor3iv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3iv("#args")\n"; glSecondaryColor3iv(args);
# define glSecondaryColor3s(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3s("#args")\n"; glSecondaryColor3s(args);
# define glSecondaryColor3sv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3sv("#args")\n"; glSecondaryColor3sv(args);
# define glSecondaryColor3ub(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3ub("#args")\n"; glSecondaryColor3ub(args);
# define glSecondaryColor3ubv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3ubv("#args")\n"; glSecondaryColor3ubv(args);
# define glSecondaryColor3ui(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3ui("#args")\n"; glSecondaryColor3ui(args);
# define glSecondaryColor3uiv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3uiv("#args")\n"; glSecondaryColor3uiv(args);
# define glSecondaryColor3us(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3us("#args")\n"; glSecondaryColor3us(args);
# define glSecondaryColor3usv(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3usv("#args")\n"; glSecondaryColor3usv(args);
# define glSecondaryColorPointer(args...)	std::cerr << GL_DEBUG << "glSecondaryColorPointer("#args")\n"; glSecondaryColorPointer(args);
# define glWindowPos2d(args...)	std::cerr << GL_DEBUG << "glWindowPos2d("#args")\n"; glWindowPos2d(args);
# define glWindowPos2dv(args...)	std::cerr << GL_DEBUG << "glWindowPos2dv("#args")\n"; glWindowPos2dv(args);
# define glWindowPos2f(args...)	std::cerr << GL_DEBUG << "glWindowPos2f("#args")\n"; glWindowPos2f(args);
# define glWindowPos2fv(args...)	std::cerr << GL_DEBUG << "glWindowPos2fv("#args")\n"; glWindowPos2fv(args);
# define glWindowPos2i(args...)	std::cerr << GL_DEBUG << "glWindowPos2i("#args")\n"; glWindowPos2i(args);
# define glWindowPos2iv(args...)	std::cerr << GL_DEBUG << "glWindowPos2iv("#args")\n"; glWindowPos2iv(args);
# define glWindowPos2s(args...)	std::cerr << GL_DEBUG << "glWindowPos2s("#args")\n"; glWindowPos2s(args);
# define glWindowPos2sv(args...)	std::cerr << GL_DEBUG << "glWindowPos2sv("#args")\n"; glWindowPos2sv(args);
# define glWindowPos3d(args...)	std::cerr << GL_DEBUG << "glWindowPos3d("#args")\n"; glWindowPos3d(args);
# define glWindowPos3dv(args...)	std::cerr << GL_DEBUG << "glWindowPos3dv("#args")\n"; glWindowPos3dv(args);
# define glWindowPos3f(args...)	std::cerr << GL_DEBUG << "glWindowPos3f("#args")\n"; glWindowPos3f(args);
# define glWindowPos3fv(args...)	std::cerr << GL_DEBUG << "glWindowPos3fv("#args")\n"; glWindowPos3fv(args);
# define glWindowPos3i(args...)	std::cerr << GL_DEBUG << "glWindowPos3i("#args")\n"; glWindowPos3i(args);
# define glWindowPos3iv(args...)	std::cerr << GL_DEBUG << "glWindowPos3iv("#args")\n"; glWindowPos3iv(args);
# define glWindowPos3s(args...)	std::cerr << GL_DEBUG << "glWindowPos3s("#args")\n"; glWindowPos3s(args);
# define glWindowPos3sv(args...)	std::cerr << GL_DEBUG << "glWindowPos3sv("#args")\n"; glWindowPos3sv(args);
# define glGenQueries(args...)	std::cerr << GL_DEBUG << "glGenQueries("#args")\n"; glGenQueries(args);
# define glDeleteQueries(args...)	std::cerr << GL_DEBUG << "glDeleteQueries("#args")\n"; glDeleteQueries(args);
# define glIsQuery(args...)	std::cerr << GL_DEBUG << "glIsQuery("#args")\n"; glIsQuery(args);
# define glBeginQuery(args...)	std::cerr << GL_DEBUG << "glBeginQuery("#args")\n"; glBeginQuery(args);
# define glEndQuery(args...)	std::cerr << GL_DEBUG << "glEndQuery("#args")\n"; glEndQuery(args);
# define glGetQueryiv(args...)	std::cerr << GL_DEBUG << "glGetQueryiv("#args")\n"; glGetQueryiv(args);
# define glGetQueryObjectiv(args...)	std::cerr << GL_DEBUG << "glGetQueryObjectiv("#args")\n"; glGetQueryObjectiv(args);
# define glGetQueryObjectuiv(args...)	std::cerr << GL_DEBUG << "glGetQueryObjectuiv("#args")\n"; glGetQueryObjectuiv(args);
# define glBindBuffer(args...)	std::cerr << GL_DEBUG << "glBindBuffer("#args")\n"; glBindBuffer(args);
# define glDeleteBuffers(args...)	std::cerr << GL_DEBUG << "glDeleteBuffers("#args")\n"; glDeleteBuffers(args);
# define glGenBuffers(args...)	std::cerr << GL_DEBUG << "glGenBuffers("#args")\n"; glGenBuffers(args);
# define glIsBuffer(args...)	std::cerr << GL_DEBUG << "glIsBuffer("#args")\n"; glIsBuffer(args);
# define glBufferData(args...)	std::cerr << GL_DEBUG << "glBufferData("#args")\n"; glBufferData(args);
# define glBufferSubData(args...)	std::cerr << GL_DEBUG << "glBufferSubData("#args")\n"; glBufferSubData(args);
# define glGetBufferSubData(args...)	std::cerr << GL_DEBUG << "glGetBufferSubData("#args")\n"; glGetBufferSubData(args);
# define glMapBuffer(args...)	std::cerr << GL_DEBUG << "glMapBuffer("#args")\n"; glMapBuffer(args);
# define glUnmapBuffer(args...)	std::cerr << GL_DEBUG << "glUnmapBuffer("#args")\n"; glUnmapBuffer(args);
# define glGetBufferParameteriv(args...)	std::cerr << GL_DEBUG << "glGetBufferParameteriv("#args")\n"; glGetBufferParameteriv(args);
# define glGetBufferPointerv(args...)	std::cerr << GL_DEBUG << "glGetBufferPointerv("#args")\n"; glGetBufferPointerv(args);
# define glActiveTextureARB(args...)	std::cerr << GL_DEBUG << "glActiveTextureARB("#args")\n"; glActiveTextureARB(args);
# define glClientActiveTextureARB(args...)	std::cerr << GL_DEBUG << "glClientActiveTextureARB("#args")\n"; glClientActiveTextureARB(args);
# define glMultiTexCoord1dARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1dARB("#args")\n"; glMultiTexCoord1dARB(args);
# define glMultiTexCoord1dvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1dvARB("#args")\n"; glMultiTexCoord1dvARB(args);
# define glMultiTexCoord1fARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1fARB("#args")\n"; glMultiTexCoord1fARB(args);
# define glMultiTexCoord1fvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1fvARB("#args")\n"; glMultiTexCoord1fvARB(args);
# define glMultiTexCoord1iARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1iARB("#args")\n"; glMultiTexCoord1iARB(args);
# define glMultiTexCoord1ivARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1ivARB("#args")\n"; glMultiTexCoord1ivARB(args);
# define glMultiTexCoord1sARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1sARB("#args")\n"; glMultiTexCoord1sARB(args);
# define glMultiTexCoord1svARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1svARB("#args")\n"; glMultiTexCoord1svARB(args);
# define glMultiTexCoord2dARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2dARB("#args")\n"; glMultiTexCoord2dARB(args);
# define glMultiTexCoord2dvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2dvARB("#args")\n"; glMultiTexCoord2dvARB(args);
# define glMultiTexCoord2fARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2fARB("#args")\n"; glMultiTexCoord2fARB(args);
# define glMultiTexCoord2fvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2fvARB("#args")\n"; glMultiTexCoord2fvARB(args);
# define glMultiTexCoord2iARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2iARB("#args")\n"; glMultiTexCoord2iARB(args);
# define glMultiTexCoord2ivARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2ivARB("#args")\n"; glMultiTexCoord2ivARB(args);
# define glMultiTexCoord2sARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2sARB("#args")\n"; glMultiTexCoord2sARB(args);
# define glMultiTexCoord2svARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2svARB("#args")\n"; glMultiTexCoord2svARB(args);
# define glMultiTexCoord3dARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3dARB("#args")\n"; glMultiTexCoord3dARB(args);
# define glMultiTexCoord3dvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3dvARB("#args")\n"; glMultiTexCoord3dvARB(args);
# define glMultiTexCoord3fARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3fARB("#args")\n"; glMultiTexCoord3fARB(args);
# define glMultiTexCoord3fvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3fvARB("#args")\n"; glMultiTexCoord3fvARB(args);
# define glMultiTexCoord3iARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3iARB("#args")\n"; glMultiTexCoord3iARB(args);
# define glMultiTexCoord3ivARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3ivARB("#args")\n"; glMultiTexCoord3ivARB(args);
# define glMultiTexCoord3sARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3sARB("#args")\n"; glMultiTexCoord3sARB(args);
# define glMultiTexCoord3svARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3svARB("#args")\n"; glMultiTexCoord3svARB(args);
# define glMultiTexCoord4dARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4dARB("#args")\n"; glMultiTexCoord4dARB(args);
# define glMultiTexCoord4dvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4dvARB("#args")\n"; glMultiTexCoord4dvARB(args);
# define glMultiTexCoord4fARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4fARB("#args")\n"; glMultiTexCoord4fARB(args);
# define glMultiTexCoord4fvARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4fvARB("#args")\n"; glMultiTexCoord4fvARB(args);
# define glMultiTexCoord4iARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4iARB("#args")\n"; glMultiTexCoord4iARB(args);
# define glMultiTexCoord4ivARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4ivARB("#args")\n"; glMultiTexCoord4ivARB(args);
# define glMultiTexCoord4sARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4sARB("#args")\n"; glMultiTexCoord4sARB(args);
# define glMultiTexCoord4svARB(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4svARB("#args")\n"; glMultiTexCoord4svARB(args);
# define glLoadTransposeMatrixfARB(args...)	std::cerr << GL_DEBUG << "glLoadTransposeMatrixfARB("#args")\n"; glLoadTransposeMatrixfARB(args);
# define glLoadTransposeMatrixdARB(args...)	std::cerr << GL_DEBUG << "glLoadTransposeMatrixdARB("#args")\n"; glLoadTransposeMatrixdARB(args);
# define glMultTransposeMatrixfARB(args...)	std::cerr << GL_DEBUG << "glMultTransposeMatrixfARB("#args")\n"; glMultTransposeMatrixfARB(args);
# define glMultTransposeMatrixdARB(args...)	std::cerr << GL_DEBUG << "glMultTransposeMatrixdARB("#args")\n"; glMultTransposeMatrixdARB(args);
# define glSampleCoverageARB(args...)	std::cerr << GL_DEBUG << "glSampleCoverageARB("#args")\n"; glSampleCoverageARB(args);
# define glCompressedTexImage3DARB(args...)	std::cerr << GL_DEBUG << "glCompressedTexImage3DARB("#args")\n"; glCompressedTexImage3DARB(args);
# define glCompressedTexImage2DARB(args...)	std::cerr << GL_DEBUG << "glCompressedTexImage2DARB("#args")\n"; glCompressedTexImage2DARB(args);
# define glCompressedTexImage1DARB(args...)	std::cerr << GL_DEBUG << "glCompressedTexImage1DARB("#args")\n"; glCompressedTexImage1DARB(args);
# define glCompressedTexSubImage3DARB(args...)	std::cerr << GL_DEBUG << "glCompressedTexSubImage3DARB("#args")\n"; glCompressedTexSubImage3DARB(args);
# define glCompressedTexSubImage2DARB(args...)	std::cerr << GL_DEBUG << "glCompressedTexSubImage2DARB("#args")\n"; glCompressedTexSubImage2DARB(args);
# define glCompressedTexSubImage1DARB(args...)	std::cerr << GL_DEBUG << "glCompressedTexSubImage1DARB("#args")\n"; glCompressedTexSubImage1DARB(args);
# define glGetCompressedTexImageARB(args...)	std::cerr << GL_DEBUG << "glGetCompressedTexImageARB("#args")\n"; glGetCompressedTexImageARB(args);
# define glPointParameterfARB(args...)	std::cerr << GL_DEBUG << "glPointParameterfARB("#args")\n"; glPointParameterfARB(args);
# define glPointParameterfvARB(args...)	std::cerr << GL_DEBUG << "glPointParameterfvARB("#args")\n"; glPointParameterfvARB(args);
# define glWeightbvARB(args...)	std::cerr << GL_DEBUG << "glWeightbvARB("#args")\n"; glWeightbvARB(args);
# define glWeightsvARB(args...)	std::cerr << GL_DEBUG << "glWeightsvARB("#args")\n"; glWeightsvARB(args);
# define glWeightivARB(args...)	std::cerr << GL_DEBUG << "glWeightivARB("#args")\n"; glWeightivARB(args);
# define glWeightfvARB(args...)	std::cerr << GL_DEBUG << "glWeightfvARB("#args")\n"; glWeightfvARB(args);
# define glWeightdvARB(args...)	std::cerr << GL_DEBUG << "glWeightdvARB("#args")\n"; glWeightdvARB(args);
# define glWeightubvARB(args...)	std::cerr << GL_DEBUG << "glWeightubvARB("#args")\n"; glWeightubvARB(args);
# define glWeightusvARB(args...)	std::cerr << GL_DEBUG << "glWeightusvARB("#args")\n"; glWeightusvARB(args);
# define glWeightuivARB(args...)	std::cerr << GL_DEBUG << "glWeightuivARB("#args")\n"; glWeightuivARB(args);
# define glWeightPointerARB(args...)	std::cerr << GL_DEBUG << "glWeightPointerARB("#args")\n"; glWeightPointerARB(args);
# define glVertexBlendARB(args...)	std::cerr << GL_DEBUG << "glVertexBlendARB("#args")\n"; glVertexBlendARB(args);
# define glCurrentPaletteMatrixARB(args...)	std::cerr << GL_DEBUG << "glCurrentPaletteMatrixARB("#args")\n"; glCurrentPaletteMatrixARB(args);
# define glMatrixIndexubvARB(args...)	std::cerr << GL_DEBUG << "glMatrixIndexubvARB("#args")\n"; glMatrixIndexubvARB(args);
# define glMatrixIndexusvARB(args...)	std::cerr << GL_DEBUG << "glMatrixIndexusvARB("#args")\n"; glMatrixIndexusvARB(args);
# define glMatrixIndexuivARB(args...)	std::cerr << GL_DEBUG << "glMatrixIndexuivARB("#args")\n"; glMatrixIndexuivARB(args);
# define glMatrixIndexPointerARB(args...)	std::cerr << GL_DEBUG << "glMatrixIndexPointerARB("#args")\n"; glMatrixIndexPointerARB(args);
# define glWindowPos2dARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2dARB("#args")\n"; glWindowPos2dARB(args);
# define glWindowPos2dvARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2dvARB("#args")\n"; glWindowPos2dvARB(args);
# define glWindowPos2fARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2fARB("#args")\n"; glWindowPos2fARB(args);
# define glWindowPos2fvARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2fvARB("#args")\n"; glWindowPos2fvARB(args);
# define glWindowPos2iARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2iARB("#args")\n"; glWindowPos2iARB(args);
# define glWindowPos2ivARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2ivARB("#args")\n"; glWindowPos2ivARB(args);
# define glWindowPos2sARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2sARB("#args")\n"; glWindowPos2sARB(args);
# define glWindowPos2svARB(args...)	std::cerr << GL_DEBUG << "glWindowPos2svARB("#args")\n"; glWindowPos2svARB(args);
# define glWindowPos3dARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3dARB("#args")\n"; glWindowPos3dARB(args);
# define glWindowPos3dvARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3dvARB("#args")\n"; glWindowPos3dvARB(args);
# define glWindowPos3fARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3fARB("#args")\n"; glWindowPos3fARB(args);
# define glWindowPos3fvARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3fvARB("#args")\n"; glWindowPos3fvARB(args);
# define glWindowPos3iARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3iARB("#args")\n"; glWindowPos3iARB(args);
# define glWindowPos3ivARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3ivARB("#args")\n"; glWindowPos3ivARB(args);
# define glWindowPos3sARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3sARB("#args")\n"; glWindowPos3sARB(args);
# define glWindowPos3svARB(args...)	std::cerr << GL_DEBUG << "glWindowPos3svARB("#args")\n"; glWindowPos3svARB(args);
# define glVertexAttrib1dARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1dARB("#args")\n"; glVertexAttrib1dARB(args);
# define glVertexAttrib1dvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1dvARB("#args")\n"; glVertexAttrib1dvARB(args);
# define glVertexAttrib1fARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1fARB("#args")\n"; glVertexAttrib1fARB(args);
# define glVertexAttrib1fvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1fvARB("#args")\n"; glVertexAttrib1fvARB(args);
# define glVertexAttrib1sARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1sARB("#args")\n"; glVertexAttrib1sARB(args);
# define glVertexAttrib1svARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1svARB("#args")\n"; glVertexAttrib1svARB(args);
# define glVertexAttrib2dARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2dARB("#args")\n"; glVertexAttrib2dARB(args);
# define glVertexAttrib2dvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2dvARB("#args")\n"; glVertexAttrib2dvARB(args);
# define glVertexAttrib2fARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2fARB("#args")\n"; glVertexAttrib2fARB(args);
# define glVertexAttrib2fvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2fvARB("#args")\n"; glVertexAttrib2fvARB(args);
# define glVertexAttrib2sARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2sARB("#args")\n"; glVertexAttrib2sARB(args);
# define glVertexAttrib2svARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2svARB("#args")\n"; glVertexAttrib2svARB(args);
# define glVertexAttrib3dARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3dARB("#args")\n"; glVertexAttrib3dARB(args);
# define glVertexAttrib3dvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3dvARB("#args")\n"; glVertexAttrib3dvARB(args);
# define glVertexAttrib3fARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3fARB("#args")\n"; glVertexAttrib3fARB(args);
# define glVertexAttrib3fvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3fvARB("#args")\n"; glVertexAttrib3fvARB(args);
# define glVertexAttrib3sARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3sARB("#args")\n"; glVertexAttrib3sARB(args);
# define glVertexAttrib3svARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3svARB("#args")\n"; glVertexAttrib3svARB(args);
# define glVertexAttrib4NbvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NbvARB("#args")\n"; glVertexAttrib4NbvARB(args);
# define glVertexAttrib4NivARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NivARB("#args")\n"; glVertexAttrib4NivARB(args);
# define glVertexAttrib4NsvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NsvARB("#args")\n"; glVertexAttrib4NsvARB(args);
# define glVertexAttrib4NubARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NubARB("#args")\n"; glVertexAttrib4NubARB(args);
# define glVertexAttrib4NubvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NubvARB("#args")\n"; glVertexAttrib4NubvARB(args);
# define glVertexAttrib4NuivARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NuivARB("#args")\n"; glVertexAttrib4NuivARB(args);
# define glVertexAttrib4NusvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4NusvARB("#args")\n"; glVertexAttrib4NusvARB(args);
# define glVertexAttrib4bvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4bvARB("#args")\n"; glVertexAttrib4bvARB(args);
# define glVertexAttrib4dARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4dARB("#args")\n"; glVertexAttrib4dARB(args);
# define glVertexAttrib4dvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4dvARB("#args")\n"; glVertexAttrib4dvARB(args);
# define glVertexAttrib4fARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4fARB("#args")\n"; glVertexAttrib4fARB(args);
# define glVertexAttrib4fvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4fvARB("#args")\n"; glVertexAttrib4fvARB(args);
# define glVertexAttrib4ivARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4ivARB("#args")\n"; glVertexAttrib4ivARB(args);
# define glVertexAttrib4sARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4sARB("#args")\n"; glVertexAttrib4sARB(args);
# define glVertexAttrib4svARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4svARB("#args")\n"; glVertexAttrib4svARB(args);
# define glVertexAttrib4ubvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4ubvARB("#args")\n"; glVertexAttrib4ubvARB(args);
# define glVertexAttrib4uivARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4uivARB("#args")\n"; glVertexAttrib4uivARB(args);
# define glVertexAttrib4usvARB(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4usvARB("#args")\n"; glVertexAttrib4usvARB(args);
# define glVertexAttribPointerARB(args...)	std::cerr << GL_DEBUG << "glVertexAttribPointerARB("#args")\n"; glVertexAttribPointerARB(args);
# define glEnableVertexAttribArrayARB(args...)	std::cerr << GL_DEBUG << "glEnableVertexAttribArrayARB("#args")\n"; glEnableVertexAttribArrayARB(args);
# define glDisableVertexAttribArrayARB(args...)	std::cerr << GL_DEBUG << "glDisableVertexAttribArrayARB("#args")\n"; glDisableVertexAttribArrayARB(args);
# define glProgramStringARB(args...)	std::cerr << GL_DEBUG << "glProgramStringARB("#args")\n"; glProgramStringARB(args);
# define glBindProgramARB(args...)	std::cerr << GL_DEBUG << "glBindProgramARB("#args")\n"; glBindProgramARB(args);
# define glDeleteProgramsARB(args...)	std::cerr << GL_DEBUG << "glDeleteProgramsARB("#args")\n"; glDeleteProgramsARB(args);
# define glGenProgramsARB(args...)	std::cerr << GL_DEBUG << "glGenProgramsARB("#args")\n"; glGenProgramsARB(args);
# define glProgramEnvParameter4dARB(args...)	std::cerr << GL_DEBUG << "glProgramEnvParameter4dARB("#args")\n"; glProgramEnvParameter4dARB(args);
# define glProgramEnvParameter4dvARB(args...)	std::cerr << GL_DEBUG << "glProgramEnvParameter4dvARB("#args")\n"; glProgramEnvParameter4dvARB(args);
# define glProgramEnvParameter4fARB(args...)	std::cerr << GL_DEBUG << "glProgramEnvParameter4fARB("#args")\n"; glProgramEnvParameter4fARB(args);
# define glProgramEnvParameter4fvARB(args...)	std::cerr << GL_DEBUG << "glProgramEnvParameter4fvARB("#args")\n"; glProgramEnvParameter4fvARB(args);
# define glProgramLocalParameter4dARB(args...)	std::cerr << GL_DEBUG << "glProgramLocalParameter4dARB("#args")\n"; glProgramLocalParameter4dARB(args);
# define glProgramLocalParameter4dvARB(args...)	std::cerr << GL_DEBUG << "glProgramLocalParameter4dvARB("#args")\n"; glProgramLocalParameter4dvARB(args);
# define glProgramLocalParameter4fARB(args...)	std::cerr << GL_DEBUG << "glProgramLocalParameter4fARB("#args")\n"; glProgramLocalParameter4fARB(args);
# define glProgramLocalParameter4fvARB(args...)	std::cerr << GL_DEBUG << "glProgramLocalParameter4fvARB("#args")\n"; glProgramLocalParameter4fvARB(args);
# define glGetProgramEnvParameterdvARB(args...)	std::cerr << GL_DEBUG << "glGetProgramEnvParameterdvARB("#args")\n"; glGetProgramEnvParameterdvARB(args);
# define glGetProgramEnvParameterfvARB(args...)	std::cerr << GL_DEBUG << "glGetProgramEnvParameterfvARB("#args")\n"; glGetProgramEnvParameterfvARB(args);
# define glGetProgramLocalParameterdvARB(args...)	std::cerr << GL_DEBUG << "glGetProgramLocalParameterdvARB("#args")\n"; glGetProgramLocalParameterdvARB(args);
# define glGetProgramLocalParameterfvARB(args...)	std::cerr << GL_DEBUG << "glGetProgramLocalParameterfvARB("#args")\n"; glGetProgramLocalParameterfvARB(args);
# define glGetProgramivARB(args...)	std::cerr << GL_DEBUG << "glGetProgramivARB("#args")\n"; glGetProgramivARB(args);
# define glGetProgramStringARB(args...)	std::cerr << GL_DEBUG << "glGetProgramStringARB("#args")\n"; glGetProgramStringARB(args);
# define glGetVertexAttribdvARB(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribdvARB("#args")\n"; glGetVertexAttribdvARB(args);
# define glGetVertexAttribfvARB(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribfvARB("#args")\n"; glGetVertexAttribfvARB(args);
# define glGetVertexAttribivARB(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribivARB("#args")\n"; glGetVertexAttribivARB(args);
# define glGetVertexAttribPointervARB(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribPointervARB("#args")\n"; glGetVertexAttribPointervARB(args);
# define glIsProgramARB(args...)	std::cerr << GL_DEBUG << "glIsProgramARB("#args")\n"; glIsProgramARB(args);
# define glBindBufferARB(args...)	std::cerr << GL_DEBUG << "glBindBufferARB("#args")\n"; glBindBufferARB(args);
# define glDeleteBuffersARB(args...)	std::cerr << GL_DEBUG << "glDeleteBuffersARB("#args")\n"; glDeleteBuffersARB(args);
# define glGenBuffersARB(args...)	std::cerr << GL_DEBUG << "glGenBuffersARB("#args")\n"; glGenBuffersARB(args);
# define glIsBufferARB(args...)	std::cerr << GL_DEBUG << "glIsBufferARB("#args")\n"; glIsBufferARB(args);
# define glBufferDataARB(args...)	std::cerr << GL_DEBUG << "glBufferDataARB("#args")\n"; glBufferDataARB(args);
# define glBufferSubDataARB(args...)	std::cerr << GL_DEBUG << "glBufferSubDataARB("#args")\n"; glBufferSubDataARB(args);
# define glGetBufferSubDataARB(args...)	std::cerr << GL_DEBUG << "glGetBufferSubDataARB("#args")\n"; glGetBufferSubDataARB(args);
# define glMapBufferARB(args...)	std::cerr << GL_DEBUG << "glMapBufferARB("#args")\n"; glMapBufferARB(args);
# define glUnmapBufferARB(args...)	std::cerr << GL_DEBUG << "glUnmapBufferARB("#args")\n"; glUnmapBufferARB(args);
# define glGetBufferParameterivARB(args...)	std::cerr << GL_DEBUG << "glGetBufferParameterivARB("#args")\n"; glGetBufferParameterivARB(args);
# define glGetBufferPointervARB(args...)	std::cerr << GL_DEBUG << "glGetBufferPointervARB("#args")\n"; glGetBufferPointervARB(args);
# define glGenQueriesARB(args...)	std::cerr << GL_DEBUG << "glGenQueriesARB("#args")\n"; glGenQueriesARB(args);
# define glDeleteQueriesARB(args...)	std::cerr << GL_DEBUG << "glDeleteQueriesARB("#args")\n"; glDeleteQueriesARB(args);
# define glIsQueryARB(args...)	std::cerr << GL_DEBUG << "glIsQueryARB("#args")\n"; glIsQueryARB(args);
# define glBeginQueryARB(args...)	std::cerr << GL_DEBUG << "glBeginQueryARB("#args")\n"; glBeginQueryARB(args);
# define glEndQueryARB(args...)	std::cerr << GL_DEBUG << "glEndQueryARB("#args")\n"; glEndQueryARB(args);
# define glGetQueryivARB(args...)	std::cerr << GL_DEBUG << "glGetQueryivARB("#args")\n"; glGetQueryivARB(args);
# define glGetQueryObjectivARB(args...)	std::cerr << GL_DEBUG << "glGetQueryObjectivARB("#args")\n"; glGetQueryObjectivARB(args);
# define glGetQueryObjectuivARB(args...)	std::cerr << GL_DEBUG << "glGetQueryObjectuivARB("#args")\n"; glGetQueryObjectuivARB(args);
# define glDeleteObjectARB(args...)	std::cerr << GL_DEBUG << "glDeleteObjectARB("#args")\n"; glDeleteObjectARB(args);
# define glGetHandleARB(args...)	std::cerr << GL_DEBUG << "glGetHandleARB("#args")\n"; glGetHandleARB(args);
# define glDetachObjectARB(args...)	std::cerr << GL_DEBUG << "glDetachObjectARB("#args")\n"; glDetachObjectARB(args);
# define glCreateShaderObjectARB(args...)	std::cerr << GL_DEBUG << "glCreateShaderObjectARB("#args")\n"; glCreateShaderObjectARB(args);
# define glShaderSourceARB(args...)	std::cerr << GL_DEBUG << "glShaderSourceARB("#args")\n"; glShaderSourceARB(args);
# define glCompileShaderARB(args...)	std::cerr << GL_DEBUG << "glCompileShaderARB("#args")\n"; glCompileShaderARB(args);
# define glCreateProgramObjectARB(args...)	std::cerr << GL_DEBUG << "glCreateProgramObjectARB("#args")\n"; glCreateProgramObjectARB(args);
# define glAttachObjectARB(args...)	std::cerr << GL_DEBUG << "glAttachObjectARB("#args")\n"; glAttachObjectARB(args);
# define glLinkProgramARB(args...)	std::cerr << GL_DEBUG << "glLinkProgramARB("#args")\n"; glLinkProgramARB(args);
# define glUseProgramObjectARB(args...)	std::cerr << GL_DEBUG << "glUseProgramObjectARB("#args")\n"; glUseProgramObjectARB(args);
# define glValidateProgramARB(args...)	std::cerr << GL_DEBUG << "glValidateProgramARB("#args")\n"; glValidateProgramARB(args);
# define glUniform1fARB(args...)	std::cerr << GL_DEBUG << "glUniform1fARB("#args")\n"; glUniform1fARB(args);
# define glUniform2fARB(args...)	std::cerr << GL_DEBUG << "glUniform2fARB("#args")\n"; glUniform2fARB(args);
# define glUniform3fARB(args...)	std::cerr << GL_DEBUG << "glUniform3fARB("#args")\n"; glUniform3fARB(args);
# define glUniform4fARB(args...)	std::cerr << GL_DEBUG << "glUniform4fARB("#args")\n"; glUniform4fARB(args);
# define glUniform1iARB(args...)	std::cerr << GL_DEBUG << "glUniform1iARB("#args")\n"; glUniform1iARB(args);
# define glUniform2iARB(args...)	std::cerr << GL_DEBUG << "glUniform2iARB("#args")\n"; glUniform2iARB(args);
# define glUniform3iARB(args...)	std::cerr << GL_DEBUG << "glUniform3iARB("#args")\n"; glUniform3iARB(args);
# define glUniform4iARB(args...)	std::cerr << GL_DEBUG << "glUniform4iARB("#args")\n"; glUniform4iARB(args);
# define glUniform1fvARB(args...)	std::cerr << GL_DEBUG << "glUniform1fvARB("#args")\n"; glUniform1fvARB(args);
# define glUniform2fvARB(args...)	std::cerr << GL_DEBUG << "glUniform2fvARB("#args")\n"; glUniform2fvARB(args);
# define glUniform3fvARB(args...)	std::cerr << GL_DEBUG << "glUniform3fvARB("#args")\n"; glUniform3fvARB(args);
# define glUniform4fvARB(args...)	std::cerr << GL_DEBUG << "glUniform4fvARB("#args")\n"; glUniform4fvARB(args);
# define glUniform1ivARB(args...)	std::cerr << GL_DEBUG << "glUniform1ivARB("#args")\n"; glUniform1ivARB(args);
# define glUniform2ivARB(args...)	std::cerr << GL_DEBUG << "glUniform2ivARB("#args")\n"; glUniform2ivARB(args);
# define glUniform3ivARB(args...)	std::cerr << GL_DEBUG << "glUniform3ivARB("#args")\n"; glUniform3ivARB(args);
# define glUniform4ivARB(args...)	std::cerr << GL_DEBUG << "glUniform4ivARB("#args")\n"; glUniform4ivARB(args);
# define glUniformMatrix2fvARB(args...)	std::cerr << GL_DEBUG << "glUniformMatrix2fvARB("#args")\n"; glUniformMatrix2fvARB(args);
# define glUniformMatrix3fvARB(args...)	std::cerr << GL_DEBUG << "glUniformMatrix3fvARB("#args")\n"; glUniformMatrix3fvARB(args);
# define glUniformMatrix4fvARB(args...)	std::cerr << GL_DEBUG << "glUniformMatrix4fvARB("#args")\n"; glUniformMatrix4fvARB(args);
# define glGetObjectParameterfvARB(args...)	std::cerr << GL_DEBUG << "glGetObjectParameterfvARB("#args")\n"; glGetObjectParameterfvARB(args);
# define glGetObjectParameterivARB(args...)	std::cerr << GL_DEBUG << "glGetObjectParameterivARB("#args")\n"; glGetObjectParameterivARB(args);
# define glGetInfoLogARB(args...)	std::cerr << GL_DEBUG << "glGetInfoLogARB("#args")\n"; glGetInfoLogARB(args);
# define glGetAttachedObjectsARB(args...)	std::cerr << GL_DEBUG << "glGetAttachedObjectsARB("#args")\n"; glGetAttachedObjectsARB(args);
# define glGetUniformLocationARB(args...)	std::cerr << GL_DEBUG << "glGetUniformLocationARB("#args")\n"; glGetUniformLocationARB(args);
# define glGetActiveUniformARB(args...)	std::cerr << GL_DEBUG << "glGetActiveUniformARB("#args")\n"; glGetActiveUniformARB(args);
# define glGetUniformfvARB(args...)	std::cerr << GL_DEBUG << "glGetUniformfvARB("#args")\n"; glGetUniformfvARB(args);
# define glGetUniformivARB(args...)	std::cerr << GL_DEBUG << "glGetUniformivARB("#args")\n"; glGetUniformivARB(args);
# define glGetShaderSourceARB(args...)	std::cerr << GL_DEBUG << "glGetShaderSourceARB("#args")\n"; glGetShaderSourceARB(args);
# define glBindAttribLocationARB(args...)	std::cerr << GL_DEBUG << "glBindAttribLocationARB("#args")\n"; glBindAttribLocationARB(args);
# define glGetActiveAttribARB(args...)	std::cerr << GL_DEBUG << "glGetActiveAttribARB("#args")\n"; glGetActiveAttribARB(args);
# define glGetAttribLocationARB(args...)	std::cerr << GL_DEBUG << "glGetAttribLocationARB("#args")\n"; glGetAttribLocationARB(args);
# define glDrawBuffersARB(args...)	std::cerr << GL_DEBUG << "glDrawBuffersARB("#args")\n"; glDrawBuffersARB(args);
# define glBlendColorEXT(args...)	std::cerr << GL_DEBUG << "glBlendColorEXT("#args")\n"; glBlendColorEXT(args);
# define glPolygonOffsetEXT(args...)	std::cerr << GL_DEBUG << "glPolygonOffsetEXT("#args")\n"; glPolygonOffsetEXT(args);
# define glTexImage3DEXT(args...)	std::cerr << GL_DEBUG << "glTexImage3DEXT("#args")\n"; glTexImage3DEXT(args);
# define glTexSubImage3DEXT(args...)	std::cerr << GL_DEBUG << "glTexSubImage3DEXT("#args")\n"; glTexSubImage3DEXT(args);
# define glGetTexFilterFuncSGIS(args...)	std::cerr << GL_DEBUG << "glGetTexFilterFuncSGIS("#args")\n"; glGetTexFilterFuncSGIS(args);
# define glTexFilterFuncSGIS(args...)	std::cerr << GL_DEBUG << "glTexFilterFuncSGIS("#args")\n"; glTexFilterFuncSGIS(args);
# define glTexSubImage1DEXT(args...)	std::cerr << GL_DEBUG << "glTexSubImage1DEXT("#args")\n"; glTexSubImage1DEXT(args);
# define glTexSubImage2DEXT(args...)	std::cerr << GL_DEBUG << "glTexSubImage2DEXT("#args")\n"; glTexSubImage2DEXT(args);
# define glCopyTexImage1DEXT(args...)	std::cerr << GL_DEBUG << "glCopyTexImage1DEXT("#args")\n"; glCopyTexImage1DEXT(args);
# define glCopyTexImage2DEXT(args...)	std::cerr << GL_DEBUG << "glCopyTexImage2DEXT("#args")\n"; glCopyTexImage2DEXT(args);
# define glCopyTexSubImage1DEXT(args...)	std::cerr << GL_DEBUG << "glCopyTexSubImage1DEXT("#args")\n"; glCopyTexSubImage1DEXT(args);
# define glCopyTexSubImage2DEXT(args...)	std::cerr << GL_DEBUG << "glCopyTexSubImage2DEXT("#args")\n"; glCopyTexSubImage2DEXT(args);
# define glCopyTexSubImage3DEXT(args...)	std::cerr << GL_DEBUG << "glCopyTexSubImage3DEXT("#args")\n"; glCopyTexSubImage3DEXT(args);
# define glGetHistogramEXT(args...)	std::cerr << GL_DEBUG << "glGetHistogramEXT("#args")\n"; glGetHistogramEXT(args);
# define glGetHistogramParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glGetHistogramParameterfvEXT("#args")\n"; glGetHistogramParameterfvEXT(args);
# define glGetHistogramParameterivEXT(args...)	std::cerr << GL_DEBUG << "glGetHistogramParameterivEXT("#args")\n"; glGetHistogramParameterivEXT(args);
# define glGetMinmaxEXT(args...)	std::cerr << GL_DEBUG << "glGetMinmaxEXT("#args")\n"; glGetMinmaxEXT(args);
# define glGetMinmaxParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glGetMinmaxParameterfvEXT("#args")\n"; glGetMinmaxParameterfvEXT(args);
# define glGetMinmaxParameterivEXT(args...)	std::cerr << GL_DEBUG << "glGetMinmaxParameterivEXT("#args")\n"; glGetMinmaxParameterivEXT(args);
# define glHistogramEXT(args...)	std::cerr << GL_DEBUG << "glHistogramEXT("#args")\n"; glHistogramEXT(args);
# define glMinmaxEXT(args...)	std::cerr << GL_DEBUG << "glMinmaxEXT("#args")\n"; glMinmaxEXT(args);
# define glResetHistogramEXT(args...)	std::cerr << GL_DEBUG << "glResetHistogramEXT("#args")\n"; glResetHistogramEXT(args);
# define glResetMinmaxEXT(args...)	std::cerr << GL_DEBUG << "glResetMinmaxEXT("#args")\n"; glResetMinmaxEXT(args);
# define glConvolutionFilter1DEXT(args...)	std::cerr << GL_DEBUG << "glConvolutionFilter1DEXT("#args")\n"; glConvolutionFilter1DEXT(args);
# define glConvolutionFilter2DEXT(args...)	std::cerr << GL_DEBUG << "glConvolutionFilter2DEXT("#args")\n"; glConvolutionFilter2DEXT(args);
# define glConvolutionParameterfEXT(args...)	std::cerr << GL_DEBUG << "glConvolutionParameterfEXT("#args")\n"; glConvolutionParameterfEXT(args);
# define glConvolutionParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glConvolutionParameterfvEXT("#args")\n"; glConvolutionParameterfvEXT(args);
# define glConvolutionParameteriEXT(args...)	std::cerr << GL_DEBUG << "glConvolutionParameteriEXT("#args")\n"; glConvolutionParameteriEXT(args);
# define glConvolutionParameterivEXT(args...)	std::cerr << GL_DEBUG << "glConvolutionParameterivEXT("#args")\n"; glConvolutionParameterivEXT(args);
# define glCopyConvolutionFilter1DEXT(args...)	std::cerr << GL_DEBUG << "glCopyConvolutionFilter1DEXT("#args")\n"; glCopyConvolutionFilter1DEXT(args);
# define glCopyConvolutionFilter2DEXT(args...)	std::cerr << GL_DEBUG << "glCopyConvolutionFilter2DEXT("#args")\n"; glCopyConvolutionFilter2DEXT(args);
# define glGetConvolutionFilterEXT(args...)	std::cerr << GL_DEBUG << "glGetConvolutionFilterEXT("#args")\n"; glGetConvolutionFilterEXT(args);
# define glGetConvolutionParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glGetConvolutionParameterfvEXT("#args")\n"; glGetConvolutionParameterfvEXT(args);
# define glGetConvolutionParameterivEXT(args...)	std::cerr << GL_DEBUG << "glGetConvolutionParameterivEXT("#args")\n"; glGetConvolutionParameterivEXT(args);
# define glGetSeparableFilterEXT(args...)	std::cerr << GL_DEBUG << "glGetSeparableFilterEXT("#args")\n"; glGetSeparableFilterEXT(args);
# define glSeparableFilter2DEXT(args...)	std::cerr << GL_DEBUG << "glSeparableFilter2DEXT("#args")\n"; glSeparableFilter2DEXT(args);
# define glColorTableSGI(args...)	std::cerr << GL_DEBUG << "glColorTableSGI("#args")\n"; glColorTableSGI(args);
# define glColorTableParameterfvSGI(args...)	std::cerr << GL_DEBUG << "glColorTableParameterfvSGI("#args")\n"; glColorTableParameterfvSGI(args);
# define glColorTableParameterivSGI(args...)	std::cerr << GL_DEBUG << "glColorTableParameterivSGI("#args")\n"; glColorTableParameterivSGI(args);
# define glCopyColorTableSGI(args...)	std::cerr << GL_DEBUG << "glCopyColorTableSGI("#args")\n"; glCopyColorTableSGI(args);
# define glGetColorTableSGI(args...)	std::cerr << GL_DEBUG << "glGetColorTableSGI("#args")\n"; glGetColorTableSGI(args);
# define glGetColorTableParameterfvSGI(args...)	std::cerr << GL_DEBUG << "glGetColorTableParameterfvSGI("#args")\n"; glGetColorTableParameterfvSGI(args);
# define glGetColorTableParameterivSGI(args...)	std::cerr << GL_DEBUG << "glGetColorTableParameterivSGI("#args")\n"; glGetColorTableParameterivSGI(args);
# define glPixelTexGenSGIX(args...)	std::cerr << GL_DEBUG << "glPixelTexGenSGIX("#args")\n"; glPixelTexGenSGIX(args);
# define glPixelTexGenParameteriSGIS(args...)	std::cerr << GL_DEBUG << "glPixelTexGenParameteriSGIS("#args")\n"; glPixelTexGenParameteriSGIS(args);
# define glPixelTexGenParameterivSGIS(args...)	std::cerr << GL_DEBUG << "glPixelTexGenParameterivSGIS("#args")\n"; glPixelTexGenParameterivSGIS(args);
# define glPixelTexGenParameterfSGIS(args...)	std::cerr << GL_DEBUG << "glPixelTexGenParameterfSGIS("#args")\n"; glPixelTexGenParameterfSGIS(args);
# define glPixelTexGenParameterfvSGIS(args...)	std::cerr << GL_DEBUG << "glPixelTexGenParameterfvSGIS("#args")\n"; glPixelTexGenParameterfvSGIS(args);
# define glGetPixelTexGenParameterivSGIS(args...)	std::cerr << GL_DEBUG << "glGetPixelTexGenParameterivSGIS("#args")\n"; glGetPixelTexGenParameterivSGIS(args);
# define glGetPixelTexGenParameterfvSGIS(args...)	std::cerr << GL_DEBUG << "glGetPixelTexGenParameterfvSGIS("#args")\n"; glGetPixelTexGenParameterfvSGIS(args);
# define glTexImage4DSGIS(args...)	std::cerr << GL_DEBUG << "glTexImage4DSGIS("#args")\n"; glTexImage4DSGIS(args);
# define glTexSubImage4DSGIS(args...)	std::cerr << GL_DEBUG << "glTexSubImage4DSGIS("#args")\n"; glTexSubImage4DSGIS(args);
# define glAreTexturesResidentEXT(args...)	std::cerr << GL_DEBUG << "glAreTexturesResidentEXT("#args")\n"; glAreTexturesResidentEXT(args);
# define glBindTextureEXT(args...)	std::cerr << GL_DEBUG << "glBindTextureEXT("#args")\n"; glBindTextureEXT(args);
# define glDeleteTexturesEXT(args...)	std::cerr << GL_DEBUG << "glDeleteTexturesEXT("#args")\n"; glDeleteTexturesEXT(args);
# define glGenTexturesEXT(args...)	std::cerr << GL_DEBUG << "glGenTexturesEXT("#args")\n"; glGenTexturesEXT(args);
# define glIsTextureEXT(args...)	std::cerr << GL_DEBUG << "glIsTextureEXT("#args")\n"; glIsTextureEXT(args);
# define glPrioritizeTexturesEXT(args...)	std::cerr << GL_DEBUG << "glPrioritizeTexturesEXT("#args")\n"; glPrioritizeTexturesEXT(args);
# define glDetailTexFuncSGIS(args...)	std::cerr << GL_DEBUG << "glDetailTexFuncSGIS("#args")\n"; glDetailTexFuncSGIS(args);
# define glGetDetailTexFuncSGIS(args...)	std::cerr << GL_DEBUG << "glGetDetailTexFuncSGIS("#args")\n"; glGetDetailTexFuncSGIS(args);
# define glSharpenTexFuncSGIS(args...)	std::cerr << GL_DEBUG << "glSharpenTexFuncSGIS("#args")\n"; glSharpenTexFuncSGIS(args);
# define glGetSharpenTexFuncSGIS(args...)	std::cerr << GL_DEBUG << "glGetSharpenTexFuncSGIS("#args")\n"; glGetSharpenTexFuncSGIS(args);
# define glSampleMaskSGIS(args...)	std::cerr << GL_DEBUG << "glSampleMaskSGIS("#args")\n"; glSampleMaskSGIS(args);
# define glSamplePatternSGIS(args...)	std::cerr << GL_DEBUG << "glSamplePatternSGIS("#args")\n"; glSamplePatternSGIS(args);
# define glArrayElementEXT(args...)	std::cerr << GL_DEBUG << "glArrayElementEXT("#args")\n"; glArrayElementEXT(args);
# define glColorPointerEXT(args...)	std::cerr << GL_DEBUG << "glColorPointerEXT("#args")\n"; glColorPointerEXT(args);
# define glDrawArraysEXT(args...)	std::cerr << GL_DEBUG << "glDrawArraysEXT("#args")\n"; glDrawArraysEXT(args);
# define glEdgeFlagPointerEXT(args...)	std::cerr << GL_DEBUG << "glEdgeFlagPointerEXT("#args")\n"; glEdgeFlagPointerEXT(args);
# define glGetPointervEXT(args...)	std::cerr << GL_DEBUG << "glGetPointervEXT("#args")\n"; glGetPointervEXT(args);
# define glIndexPointerEXT(args...)	std::cerr << GL_DEBUG << "glIndexPointerEXT("#args")\n"; glIndexPointerEXT(args);
# define glNormalPointerEXT(args...)	std::cerr << GL_DEBUG << "glNormalPointerEXT("#args")\n"; glNormalPointerEXT(args);
# define glTexCoordPointerEXT(args...)	std::cerr << GL_DEBUG << "glTexCoordPointerEXT("#args")\n"; glTexCoordPointerEXT(args);
# define glVertexPointerEXT(args...)	std::cerr << GL_DEBUG << "glVertexPointerEXT("#args")\n"; glVertexPointerEXT(args);
# define glBlendEquationEXT(args...)	std::cerr << GL_DEBUG << "glBlendEquationEXT("#args")\n"; glBlendEquationEXT(args);
# define glSpriteParameterfSGIX(args...)	std::cerr << GL_DEBUG << "glSpriteParameterfSGIX("#args")\n"; glSpriteParameterfSGIX(args);
# define glSpriteParameterfvSGIX(args...)	std::cerr << GL_DEBUG << "glSpriteParameterfvSGIX("#args")\n"; glSpriteParameterfvSGIX(args);
# define glSpriteParameteriSGIX(args...)	std::cerr << GL_DEBUG << "glSpriteParameteriSGIX("#args")\n"; glSpriteParameteriSGIX(args);
# define glSpriteParameterivSGIX(args...)	std::cerr << GL_DEBUG << "glSpriteParameterivSGIX("#args")\n"; glSpriteParameterivSGIX(args);
# define glPointParameterfEXT(args...)	std::cerr << GL_DEBUG << "glPointParameterfEXT("#args")\n"; glPointParameterfEXT(args);
# define glPointParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glPointParameterfvEXT("#args")\n"; glPointParameterfvEXT(args);
# define glPointParameterfSGIS(args...)	std::cerr << GL_DEBUG << "glPointParameterfSGIS("#args")\n"; glPointParameterfSGIS(args);
# define glPointParameterfvSGIS(args...)	std::cerr << GL_DEBUG << "glPointParameterfvSGIS("#args")\n"; glPointParameterfvSGIS(args);
# define glGetInstrumentsSGIX(args...)	std::cerr << GL_DEBUG << "glGetInstrumentsSGIX("#args")\n"; glGetInstrumentsSGIX(args);
# define glInstrumentsBufferSGIX(args...)	std::cerr << GL_DEBUG << "glInstrumentsBufferSGIX("#args")\n"; glInstrumentsBufferSGIX(args);
# define glPollInstrumentsSGIX(args...)	std::cerr << GL_DEBUG << "glPollInstrumentsSGIX("#args")\n"; glPollInstrumentsSGIX(args);
# define glReadInstrumentsSGIX(args...)	std::cerr << GL_DEBUG << "glReadInstrumentsSGIX("#args")\n"; glReadInstrumentsSGIX(args);
# define glStartInstrumentsSGIX(args...)	std::cerr << GL_DEBUG << "glStartInstrumentsSGIX("#args")\n"; glStartInstrumentsSGIX(args);
# define glStopInstrumentsSGIX(args...)	std::cerr << GL_DEBUG << "glStopInstrumentsSGIX("#args")\n"; glStopInstrumentsSGIX(args);
# define glFrameZoomSGIX(args...)	std::cerr << GL_DEBUG << "glFrameZoomSGIX("#args")\n"; glFrameZoomSGIX(args);
# define glTagSampleBufferSGIX(args...)	std::cerr << GL_DEBUG << "glTagSampleBufferSGIX("#args")\n"; glTagSampleBufferSGIX(args);
# define glDeformationMap3dSGIX(args...)	std::cerr << GL_DEBUG << "glDeformationMap3dSGIX("#args")\n"; glDeformationMap3dSGIX(args);
# define glDeformationMap3fSGIX(args...)	std::cerr << GL_DEBUG << "glDeformationMap3fSGIX("#args")\n"; glDeformationMap3fSGIX(args);
# define glDeformSGIX(args...)	std::cerr << GL_DEBUG << "glDeformSGIX("#args")\n"; glDeformSGIX(args);
# define glLoadIdentityDeformationMapSGIX(args...)	std::cerr << GL_DEBUG << "glLoadIdentityDeformationMapSGIX("#args")\n"; glLoadIdentityDeformationMapSGIX(args);
# define glReferencePlaneSGIX(args...)	std::cerr << GL_DEBUG << "glReferencePlaneSGIX("#args")\n"; glReferencePlaneSGIX(args);
# define glFlushRasterSGIX(args...)	std::cerr << GL_DEBUG << "glFlushRasterSGIX("#args")\n"; glFlushRasterSGIX(args);
# define glFogFuncSGIS(args...)	std::cerr << GL_DEBUG << "glFogFuncSGIS("#args")\n"; glFogFuncSGIS(args);
# define glGetFogFuncSGIS(args...)	std::cerr << GL_DEBUG << "glGetFogFuncSGIS("#args")\n"; glGetFogFuncSGIS(args);
# define glImageTransformParameteriHP(args...)	std::cerr << GL_DEBUG << "glImageTransformParameteriHP("#args")\n"; glImageTransformParameteriHP(args);
# define glImageTransformParameterfHP(args...)	std::cerr << GL_DEBUG << "glImageTransformParameterfHP("#args")\n"; glImageTransformParameterfHP(args);
# define glImageTransformParameterivHP(args...)	std::cerr << GL_DEBUG << "glImageTransformParameterivHP("#args")\n"; glImageTransformParameterivHP(args);
# define glImageTransformParameterfvHP(args...)	std::cerr << GL_DEBUG << "glImageTransformParameterfvHP("#args")\n"; glImageTransformParameterfvHP(args);
# define glGetImageTransformParameterivHP(args...)	std::cerr << GL_DEBUG << "glGetImageTransformParameterivHP("#args")\n"; glGetImageTransformParameterivHP(args);
# define glGetImageTransformParameterfvHP(args...)	std::cerr << GL_DEBUG << "glGetImageTransformParameterfvHP("#args")\n"; glGetImageTransformParameterfvHP(args);
# define glColorSubTableEXT(args...)	std::cerr << GL_DEBUG << "glColorSubTableEXT("#args")\n"; glColorSubTableEXT(args);
# define glCopyColorSubTableEXT(args...)	std::cerr << GL_DEBUG << "glCopyColorSubTableEXT("#args")\n"; glCopyColorSubTableEXT(args);
# define glHintPGI(args...)	std::cerr << GL_DEBUG << "glHintPGI("#args")\n"; glHintPGI(args);
# define glColorTableEXT(args...)	std::cerr << GL_DEBUG << "glColorTableEXT("#args")\n"; glColorTableEXT(args);
# define glGetColorTableEXT(args...)	std::cerr << GL_DEBUG << "glGetColorTableEXT("#args")\n"; glGetColorTableEXT(args);
# define glGetColorTableParameterivEXT(args...)	std::cerr << GL_DEBUG << "glGetColorTableParameterivEXT("#args")\n"; glGetColorTableParameterivEXT(args);
# define glGetColorTableParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glGetColorTableParameterfvEXT("#args")\n"; glGetColorTableParameterfvEXT(args);
# define glGetListParameterfvSGIX(args...)	std::cerr << GL_DEBUG << "glGetListParameterfvSGIX("#args")\n"; glGetListParameterfvSGIX(args);
# define glGetListParameterivSGIX(args...)	std::cerr << GL_DEBUG << "glGetListParameterivSGIX("#args")\n"; glGetListParameterivSGIX(args);
# define glListParameterfSGIX(args...)	std::cerr << GL_DEBUG << "glListParameterfSGIX("#args")\n"; glListParameterfSGIX(args);
# define glListParameterfvSGIX(args...)	std::cerr << GL_DEBUG << "glListParameterfvSGIX("#args")\n"; glListParameterfvSGIX(args);
# define glListParameteriSGIX(args...)	std::cerr << GL_DEBUG << "glListParameteriSGIX("#args")\n"; glListParameteriSGIX(args);
# define glListParameterivSGIX(args...)	std::cerr << GL_DEBUG << "glListParameterivSGIX("#args")\n"; glListParameterivSGIX(args);
# define glIndexMaterialEXT(args...)	std::cerr << GL_DEBUG << "glIndexMaterialEXT("#args")\n"; glIndexMaterialEXT(args);
# define glIndexFuncEXT(args...)	std::cerr << GL_DEBUG << "glIndexFuncEXT("#args")\n"; glIndexFuncEXT(args);
# define glLockArraysEXT(args...)	std::cerr << GL_DEBUG << "glLockArraysEXT("#args")\n"; glLockArraysEXT(args);
# define glUnlockArraysEXT(args...)	std::cerr << GL_DEBUG << "glUnlockArraysEXT("#args")\n"; glUnlockArraysEXT(args);
# define glCullParameterdvEXT(args...)	std::cerr << GL_DEBUG << "glCullParameterdvEXT("#args")\n"; glCullParameterdvEXT(args);
# define glCullParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glCullParameterfvEXT("#args")\n"; glCullParameterfvEXT(args);
# define glFragmentColorMaterialSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentColorMaterialSGIX("#args")\n"; glFragmentColorMaterialSGIX(args);
# define glFragmentLightfSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightfSGIX("#args")\n"; glFragmentLightfSGIX(args);
# define glFragmentLightfvSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightfvSGIX("#args")\n"; glFragmentLightfvSGIX(args);
# define glFragmentLightiSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightiSGIX("#args")\n"; glFragmentLightiSGIX(args);
# define glFragmentLightivSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightivSGIX("#args")\n"; glFragmentLightivSGIX(args);
# define glFragmentLightModelfSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightModelfSGIX("#args")\n"; glFragmentLightModelfSGIX(args);
# define glFragmentLightModelfvSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightModelfvSGIX("#args")\n"; glFragmentLightModelfvSGIX(args);
# define glFragmentLightModeliSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightModeliSGIX("#args")\n"; glFragmentLightModeliSGIX(args);
# define glFragmentLightModelivSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentLightModelivSGIX("#args")\n"; glFragmentLightModelivSGIX(args);
# define glFragmentMaterialfSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentMaterialfSGIX("#args")\n"; glFragmentMaterialfSGIX(args);
# define glFragmentMaterialfvSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentMaterialfvSGIX("#args")\n"; glFragmentMaterialfvSGIX(args);
# define glFragmentMaterialiSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentMaterialiSGIX("#args")\n"; glFragmentMaterialiSGIX(args);
# define glFragmentMaterialivSGIX(args...)	std::cerr << GL_DEBUG << "glFragmentMaterialivSGIX("#args")\n"; glFragmentMaterialivSGIX(args);
# define glGetFragmentLightfvSGIX(args...)	std::cerr << GL_DEBUG << "glGetFragmentLightfvSGIX("#args")\n"; glGetFragmentLightfvSGIX(args);
# define glGetFragmentLightivSGIX(args...)	std::cerr << GL_DEBUG << "glGetFragmentLightivSGIX("#args")\n"; glGetFragmentLightivSGIX(args);
# define glGetFragmentMaterialfvSGIX(args...)	std::cerr << GL_DEBUG << "glGetFragmentMaterialfvSGIX("#args")\n"; glGetFragmentMaterialfvSGIX(args);
# define glGetFragmentMaterialivSGIX(args...)	std::cerr << GL_DEBUG << "glGetFragmentMaterialivSGIX("#args")\n"; glGetFragmentMaterialivSGIX(args);
# define glLightEnviSGIX(args...)	std::cerr << GL_DEBUG << "glLightEnviSGIX("#args")\n"; glLightEnviSGIX(args);
# define glDrawRangeElementsEXT(args...)	std::cerr << GL_DEBUG << "glDrawRangeElementsEXT("#args")\n"; glDrawRangeElementsEXT(args);
# define glApplyTextureEXT(args...)	std::cerr << GL_DEBUG << "glApplyTextureEXT("#args")\n"; glApplyTextureEXT(args);
# define glTextureLightEXT(args...)	std::cerr << GL_DEBUG << "glTextureLightEXT("#args")\n"; glTextureLightEXT(args);
# define glTextureMaterialEXT(args...)	std::cerr << GL_DEBUG << "glTextureMaterialEXT("#args")\n"; glTextureMaterialEXT(args);
# define glAsyncMarkerSGIX(args...)	std::cerr << GL_DEBUG << "glAsyncMarkerSGIX("#args")\n"; glAsyncMarkerSGIX(args);
# define glFinishAsyncSGIX(args...)	std::cerr << GL_DEBUG << "glFinishAsyncSGIX("#args")\n"; glFinishAsyncSGIX(args);
# define glPollAsyncSGIX(args...)	std::cerr << GL_DEBUG << "glPollAsyncSGIX("#args")\n"; glPollAsyncSGIX(args);
# define glGenAsyncMarkersSGIX(args...)	std::cerr << GL_DEBUG << "glGenAsyncMarkersSGIX("#args")\n"; glGenAsyncMarkersSGIX(args);
# define glDeleteAsyncMarkersSGIX(args...)	std::cerr << GL_DEBUG << "glDeleteAsyncMarkersSGIX("#args")\n"; glDeleteAsyncMarkersSGIX(args);
# define glIsAsyncMarkerSGIX(args...)	std::cerr << GL_DEBUG << "glIsAsyncMarkerSGIX("#args")\n"; glIsAsyncMarkerSGIX(args);
# define glVertexPointervINTEL(args...)	std::cerr << GL_DEBUG << "glVertexPointervINTEL("#args")\n"; glVertexPointervINTEL(args);
# define glNormalPointervINTEL(args...)	std::cerr << GL_DEBUG << "glNormalPointervINTEL("#args")\n"; glNormalPointervINTEL(args);
# define glColorPointervINTEL(args...)	std::cerr << GL_DEBUG << "glColorPointervINTEL("#args")\n"; glColorPointervINTEL(args);
# define glTexCoordPointervINTEL(args...)	std::cerr << GL_DEBUG << "glTexCoordPointervINTEL("#args")\n"; glTexCoordPointervINTEL(args);
# define glPixelTransformParameteriEXT(args...)	std::cerr << GL_DEBUG << "glPixelTransformParameteriEXT("#args")\n"; glPixelTransformParameteriEXT(args);
# define glPixelTransformParameterfEXT(args...)	std::cerr << GL_DEBUG << "glPixelTransformParameterfEXT("#args")\n"; glPixelTransformParameterfEXT(args);
# define glPixelTransformParameterivEXT(args...)	std::cerr << GL_DEBUG << "glPixelTransformParameterivEXT("#args")\n"; glPixelTransformParameterivEXT(args);
# define glPixelTransformParameterfvEXT(args...)	std::cerr << GL_DEBUG << "glPixelTransformParameterfvEXT("#args")\n"; glPixelTransformParameterfvEXT(args);
# define glSecondaryColor3bEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3bEXT("#args")\n"; glSecondaryColor3bEXT(args);
# define glSecondaryColor3bvEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3bvEXT("#args")\n"; glSecondaryColor3bvEXT(args);
# define glSecondaryColor3dEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3dEXT("#args")\n"; glSecondaryColor3dEXT(args);
# define glSecondaryColor3dvEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3dvEXT("#args")\n"; glSecondaryColor3dvEXT(args);
# define glSecondaryColor3fEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3fEXT("#args")\n"; glSecondaryColor3fEXT(args);
# define glSecondaryColor3fvEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3fvEXT("#args")\n"; glSecondaryColor3fvEXT(args);
# define glSecondaryColor3iEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3iEXT("#args")\n"; glSecondaryColor3iEXT(args);
# define glSecondaryColor3ivEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3ivEXT("#args")\n"; glSecondaryColor3ivEXT(args);
# define glSecondaryColor3sEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3sEXT("#args")\n"; glSecondaryColor3sEXT(args);
# define glSecondaryColor3svEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3svEXT("#args")\n"; glSecondaryColor3svEXT(args);
# define glSecondaryColor3ubEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3ubEXT("#args")\n"; glSecondaryColor3ubEXT(args);
# define glSecondaryColor3ubvEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3ubvEXT("#args")\n"; glSecondaryColor3ubvEXT(args);
# define glSecondaryColor3uiEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3uiEXT("#args")\n"; glSecondaryColor3uiEXT(args);
# define glSecondaryColor3uivEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3uivEXT("#args")\n"; glSecondaryColor3uivEXT(args);
# define glSecondaryColor3usEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3usEXT("#args")\n"; glSecondaryColor3usEXT(args);
# define glSecondaryColor3usvEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3usvEXT("#args")\n"; glSecondaryColor3usvEXT(args);
# define glSecondaryColorPointerEXT(args...)	std::cerr << GL_DEBUG << "glSecondaryColorPointerEXT("#args")\n"; glSecondaryColorPointerEXT(args);
# define glTextureNormalEXT(args...)	std::cerr << GL_DEBUG << "glTextureNormalEXT("#args")\n"; glTextureNormalEXT(args);
# define glMultiDrawArraysEXT(args...)	std::cerr << GL_DEBUG << "glMultiDrawArraysEXT("#args")\n"; glMultiDrawArraysEXT(args);
# define glMultiDrawElementsEXT(args...)	std::cerr << GL_DEBUG << "glMultiDrawElementsEXT("#args")\n"; glMultiDrawElementsEXT(args);
# define glFogCoordfEXT(args...)	std::cerr << GL_DEBUG << "glFogCoordfEXT("#args")\n"; glFogCoordfEXT(args);
# define glFogCoordfvEXT(args...)	std::cerr << GL_DEBUG << "glFogCoordfvEXT("#args")\n"; glFogCoordfvEXT(args);
# define glFogCoorddEXT(args...)	std::cerr << GL_DEBUG << "glFogCoorddEXT("#args")\n"; glFogCoorddEXT(args);
# define glFogCoorddvEXT(args...)	std::cerr << GL_DEBUG << "glFogCoorddvEXT("#args")\n"; glFogCoorddvEXT(args);
# define glFogCoordPointerEXT(args...)	std::cerr << GL_DEBUG << "glFogCoordPointerEXT("#args")\n"; glFogCoordPointerEXT(args);
# define glTangent3bEXT(args...)	std::cerr << GL_DEBUG << "glTangent3bEXT("#args")\n"; glTangent3bEXT(args);
# define glTangent3bvEXT(args...)	std::cerr << GL_DEBUG << "glTangent3bvEXT("#args")\n"; glTangent3bvEXT(args);
# define glTangent3dEXT(args...)	std::cerr << GL_DEBUG << "glTangent3dEXT("#args")\n"; glTangent3dEXT(args);
# define glTangent3dvEXT(args...)	std::cerr << GL_DEBUG << "glTangent3dvEXT("#args")\n"; glTangent3dvEXT(args);
# define glTangent3fEXT(args...)	std::cerr << GL_DEBUG << "glTangent3fEXT("#args")\n"; glTangent3fEXT(args);
# define glTangent3fvEXT(args...)	std::cerr << GL_DEBUG << "glTangent3fvEXT("#args")\n"; glTangent3fvEXT(args);
# define glTangent3iEXT(args...)	std::cerr << GL_DEBUG << "glTangent3iEXT("#args")\n"; glTangent3iEXT(args);
# define glTangent3ivEXT(args...)	std::cerr << GL_DEBUG << "glTangent3ivEXT("#args")\n"; glTangent3ivEXT(args);
# define glTangent3sEXT(args...)	std::cerr << GL_DEBUG << "glTangent3sEXT("#args")\n"; glTangent3sEXT(args);
# define glTangent3svEXT(args...)	std::cerr << GL_DEBUG << "glTangent3svEXT("#args")\n"; glTangent3svEXT(args);
# define glBinormal3bEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3bEXT("#args")\n"; glBinormal3bEXT(args);
# define glBinormal3bvEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3bvEXT("#args")\n"; glBinormal3bvEXT(args);
# define glBinormal3dEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3dEXT("#args")\n"; glBinormal3dEXT(args);
# define glBinormal3dvEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3dvEXT("#args")\n"; glBinormal3dvEXT(args);
# define glBinormal3fEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3fEXT("#args")\n"; glBinormal3fEXT(args);
# define glBinormal3fvEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3fvEXT("#args")\n"; glBinormal3fvEXT(args);
# define glBinormal3iEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3iEXT("#args")\n"; glBinormal3iEXT(args);
# define glBinormal3ivEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3ivEXT("#args")\n"; glBinormal3ivEXT(args);
# define glBinormal3sEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3sEXT("#args")\n"; glBinormal3sEXT(args);
# define glBinormal3svEXT(args...)	std::cerr << GL_DEBUG << "glBinormal3svEXT("#args")\n"; glBinormal3svEXT(args);
# define glTangentPointerEXT(args...)	std::cerr << GL_DEBUG << "glTangentPointerEXT("#args")\n"; glTangentPointerEXT(args);
# define glBinormalPointerEXT(args...)	std::cerr << GL_DEBUG << "glBinormalPointerEXT("#args")\n"; glBinormalPointerEXT(args);
# define glFinishTextureSUNX(args...)	std::cerr << GL_DEBUG << "glFinishTextureSUNX("#args")\n"; glFinishTextureSUNX(args);
# define glGlobalAlphaFactorbSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactorbSUN("#args")\n"; glGlobalAlphaFactorbSUN(args);
# define glGlobalAlphaFactorsSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactorsSUN("#args")\n"; glGlobalAlphaFactorsSUN(args);
# define glGlobalAlphaFactoriSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactoriSUN("#args")\n"; glGlobalAlphaFactoriSUN(args);
# define glGlobalAlphaFactorfSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactorfSUN("#args")\n"; glGlobalAlphaFactorfSUN(args);
# define glGlobalAlphaFactordSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactordSUN("#args")\n"; glGlobalAlphaFactordSUN(args);
# define glGlobalAlphaFactorubSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactorubSUN("#args")\n"; glGlobalAlphaFactorubSUN(args);
# define glGlobalAlphaFactorusSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactorusSUN("#args")\n"; glGlobalAlphaFactorusSUN(args);
# define glGlobalAlphaFactoruiSUN(args...)	std::cerr << GL_DEBUG << "glGlobalAlphaFactoruiSUN("#args")\n"; glGlobalAlphaFactoruiSUN(args);
# define glReplacementCodeuiSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiSUN("#args")\n"; glReplacementCodeuiSUN(args);
# define glReplacementCodeusSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeusSUN("#args")\n"; glReplacementCodeusSUN(args);
# define glReplacementCodeubSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeubSUN("#args")\n"; glReplacementCodeubSUN(args);
# define glReplacementCodeuivSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuivSUN("#args")\n"; glReplacementCodeuivSUN(args);
# define glReplacementCodeusvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeusvSUN("#args")\n"; glReplacementCodeusvSUN(args);
# define glReplacementCodeubvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeubvSUN("#args")\n"; glReplacementCodeubvSUN(args);
# define glReplacementCodePointerSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodePointerSUN("#args")\n"; glReplacementCodePointerSUN(args);
# define glColor4ubVertex2fSUN(args...)	std::cerr << GL_DEBUG << "glColor4ubVertex2fSUN("#args")\n"; glColor4ubVertex2fSUN(args);
# define glColor4ubVertex2fvSUN(args...)	std::cerr << GL_DEBUG << "glColor4ubVertex2fvSUN("#args")\n"; glColor4ubVertex2fvSUN(args);
# define glColor4ubVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glColor4ubVertex3fSUN("#args")\n"; glColor4ubVertex3fSUN(args);
# define glColor4ubVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glColor4ubVertex3fvSUN("#args")\n"; glColor4ubVertex3fvSUN(args);
# define glColor3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glColor3fVertex3fSUN("#args")\n"; glColor3fVertex3fSUN(args);
# define glColor3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glColor3fVertex3fvSUN("#args")\n"; glColor3fVertex3fvSUN(args);
# define glNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glNormal3fVertex3fSUN("#args")\n"; glNormal3fVertex3fSUN(args);
# define glNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glNormal3fVertex3fvSUN("#args")\n"; glNormal3fVertex3fvSUN(args);
# define glColor4fNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glColor4fNormal3fVertex3fSUN("#args")\n"; glColor4fNormal3fVertex3fSUN(args);
# define glColor4fNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glColor4fNormal3fVertex3fvSUN("#args")\n"; glColor4fNormal3fVertex3fvSUN(args);
# define glTexCoord2fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fVertex3fSUN("#args")\n"; glTexCoord2fVertex3fSUN(args);
# define glTexCoord2fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fVertex3fvSUN("#args")\n"; glTexCoord2fVertex3fvSUN(args);
# define glTexCoord4fVertex4fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord4fVertex4fSUN("#args")\n"; glTexCoord4fVertex4fSUN(args);
# define glTexCoord4fVertex4fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord4fVertex4fvSUN("#args")\n"; glTexCoord4fVertex4fvSUN(args);
# define glTexCoord2fColor4ubVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fColor4ubVertex3fSUN("#args")\n"; glTexCoord2fColor4ubVertex3fSUN(args);
# define glTexCoord2fColor4ubVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fColor4ubVertex3fvSUN("#args")\n"; glTexCoord2fColor4ubVertex3fvSUN(args);
# define glTexCoord2fColor3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fColor3fVertex3fSUN("#args")\n"; glTexCoord2fColor3fVertex3fSUN(args);
# define glTexCoord2fColor3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fColor3fVertex3fvSUN("#args")\n"; glTexCoord2fColor3fVertex3fvSUN(args);
# define glTexCoord2fNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fNormal3fVertex3fSUN("#args")\n"; glTexCoord2fNormal3fVertex3fSUN(args);
# define glTexCoord2fNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fNormal3fVertex3fvSUN("#args")\n"; glTexCoord2fNormal3fVertex3fvSUN(args);
# define glTexCoord2fColor4fNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fColor4fNormal3fVertex3fSUN("#args")\n"; glTexCoord2fColor4fNormal3fVertex3fSUN(args);
# define glTexCoord2fColor4fNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord2fColor4fNormal3fVertex3fvSUN("#args")\n"; glTexCoord2fColor4fNormal3fVertex3fvSUN(args);
# define glTexCoord4fColor4fNormal3fVertex4fSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord4fColor4fNormal3fVertex4fSUN("#args")\n"; glTexCoord4fColor4fNormal3fVertex4fSUN(args);
# define glTexCoord4fColor4fNormal3fVertex4fvSUN(args...)	std::cerr << GL_DEBUG << "glTexCoord4fColor4fNormal3fVertex4fvSUN("#args")\n"; glTexCoord4fColor4fNormal3fVertex4fvSUN(args);
# define glReplacementCodeuiVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiVertex3fSUN("#args")\n"; glReplacementCodeuiVertex3fSUN(args);
# define glReplacementCodeuiVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiVertex3fvSUN("#args")\n"; glReplacementCodeuiVertex3fvSUN(args);
# define glReplacementCodeuiColor4ubVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiColor4ubVertex3fSUN("#args")\n"; glReplacementCodeuiColor4ubVertex3fSUN(args);
# define glReplacementCodeuiColor4ubVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiColor4ubVertex3fvSUN("#args")\n"; glReplacementCodeuiColor4ubVertex3fvSUN(args);
# define glReplacementCodeuiColor3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiColor3fVertex3fSUN("#args")\n"; glReplacementCodeuiColor3fVertex3fSUN(args);
# define glReplacementCodeuiColor3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiColor3fVertex3fvSUN("#args")\n"; glReplacementCodeuiColor3fVertex3fvSUN(args);
# define glReplacementCodeuiNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiNormal3fVertex3fSUN("#args")\n"; glReplacementCodeuiNormal3fVertex3fSUN(args);
# define glReplacementCodeuiNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiNormal3fVertex3fvSUN("#args")\n"; glReplacementCodeuiNormal3fVertex3fvSUN(args);
# define glReplacementCodeuiColor4fNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiColor4fNormal3fVertex3fSUN("#args")\n"; glReplacementCodeuiColor4fNormal3fVertex3fSUN(args);
# define glReplacementCodeuiColor4fNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiColor4fNormal3fVertex3fvSUN("#args")\n"; glReplacementCodeuiColor4fNormal3fVertex3fvSUN(args);
# define glReplacementCodeuiTexCoord2fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiTexCoord2fVertex3fSUN("#args")\n"; glReplacementCodeuiTexCoord2fVertex3fSUN(args);
# define glReplacementCodeuiTexCoord2fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiTexCoord2fVertex3fvSUN("#args")\n"; glReplacementCodeuiTexCoord2fVertex3fvSUN(args);
# define glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN("#args")\n"; glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN(args);
# define glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN("#args")\n"; glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN(args);
# define glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN("#args")\n"; glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN(args);
# define glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN(args...)	std::cerr << GL_DEBUG << "glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN("#args")\n"; glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN(args);
# define glBlendFuncSeparateEXT(args...)	std::cerr << GL_DEBUG << "glBlendFuncSeparateEXT("#args")\n"; glBlendFuncSeparateEXT(args);
# define glBlendFuncSeparateINGR(args...)	std::cerr << GL_DEBUG << "glBlendFuncSeparateINGR("#args")\n"; glBlendFuncSeparateINGR(args);
# define glVertexWeightfEXT(args...)	std::cerr << GL_DEBUG << "glVertexWeightfEXT("#args")\n"; glVertexWeightfEXT(args);
# define glVertexWeightfvEXT(args...)	std::cerr << GL_DEBUG << "glVertexWeightfvEXT("#args")\n"; glVertexWeightfvEXT(args);
# define glVertexWeightPointerEXT(args...)	std::cerr << GL_DEBUG << "glVertexWeightPointerEXT("#args")\n"; glVertexWeightPointerEXT(args);
# define glFlushVertexArrayRangeNV(args...)	std::cerr << GL_DEBUG << "glFlushVertexArrayRangeNV("#args")\n"; glFlushVertexArrayRangeNV(args);
# define glVertexArrayRangeNV(args...)	std::cerr << GL_DEBUG << "glVertexArrayRangeNV("#args")\n"; glVertexArrayRangeNV(args);
# define glCombinerParameterfvNV(args...)	std::cerr << GL_DEBUG << "glCombinerParameterfvNV("#args")\n"; glCombinerParameterfvNV(args);
# define glCombinerParameterfNV(args...)	std::cerr << GL_DEBUG << "glCombinerParameterfNV("#args")\n"; glCombinerParameterfNV(args);
# define glCombinerParameterivNV(args...)	std::cerr << GL_DEBUG << "glCombinerParameterivNV("#args")\n"; glCombinerParameterivNV(args);
# define glCombinerParameteriNV(args...)	std::cerr << GL_DEBUG << "glCombinerParameteriNV("#args")\n"; glCombinerParameteriNV(args);
# define glCombinerInputNV(args...)	std::cerr << GL_DEBUG << "glCombinerInputNV("#args")\n"; glCombinerInputNV(args);
# define glCombinerOutputNV(args...)	std::cerr << GL_DEBUG << "glCombinerOutputNV("#args")\n"; glCombinerOutputNV(args);
# define glFinalCombinerInputNV(args...)	std::cerr << GL_DEBUG << "glFinalCombinerInputNV("#args")\n"; glFinalCombinerInputNV(args);
# define glGetCombinerInputParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetCombinerInputParameterfvNV("#args")\n"; glGetCombinerInputParameterfvNV(args);
# define glGetCombinerInputParameterivNV(args...)	std::cerr << GL_DEBUG << "glGetCombinerInputParameterivNV("#args")\n"; glGetCombinerInputParameterivNV(args);
# define glGetCombinerOutputParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetCombinerOutputParameterfvNV("#args")\n"; glGetCombinerOutputParameterfvNV(args);
# define glGetCombinerOutputParameterivNV(args...)	std::cerr << GL_DEBUG << "glGetCombinerOutputParameterivNV("#args")\n"; glGetCombinerOutputParameterivNV(args);
# define glGetFinalCombinerInputParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetFinalCombinerInputParameterfvNV("#args")\n"; glGetFinalCombinerInputParameterfvNV(args);
# define glGetFinalCombinerInputParameterivNV(args...)	std::cerr << GL_DEBUG << "glGetFinalCombinerInputParameterivNV("#args")\n"; glGetFinalCombinerInputParameterivNV(args);
# define glResizeBuffersMESA(args...)	std::cerr << GL_DEBUG << "glResizeBuffersMESA("#args")\n"; glResizeBuffersMESA(args);
# define glWindowPos2dMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2dMESA("#args")\n"; glWindowPos2dMESA(args);
# define glWindowPos2dvMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2dvMESA("#args")\n"; glWindowPos2dvMESA(args);
# define glWindowPos2fMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2fMESA("#args")\n"; glWindowPos2fMESA(args);
# define glWindowPos2fvMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2fvMESA("#args")\n"; glWindowPos2fvMESA(args);
# define glWindowPos2iMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2iMESA("#args")\n"; glWindowPos2iMESA(args);
# define glWindowPos2ivMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2ivMESA("#args")\n"; glWindowPos2ivMESA(args);
# define glWindowPos2sMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2sMESA("#args")\n"; glWindowPos2sMESA(args);
# define glWindowPos2svMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos2svMESA("#args")\n"; glWindowPos2svMESA(args);
# define glWindowPos3dMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3dMESA("#args")\n"; glWindowPos3dMESA(args);
# define glWindowPos3dvMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3dvMESA("#args")\n"; glWindowPos3dvMESA(args);
# define glWindowPos3fMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3fMESA("#args")\n"; glWindowPos3fMESA(args);
# define glWindowPos3fvMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3fvMESA("#args")\n"; glWindowPos3fvMESA(args);
# define glWindowPos3iMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3iMESA("#args")\n"; glWindowPos3iMESA(args);
# define glWindowPos3ivMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3ivMESA("#args")\n"; glWindowPos3ivMESA(args);
# define glWindowPos3sMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3sMESA("#args")\n"; glWindowPos3sMESA(args);
# define glWindowPos3svMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos3svMESA("#args")\n"; glWindowPos3svMESA(args);
# define glWindowPos4dMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4dMESA("#args")\n"; glWindowPos4dMESA(args);
# define glWindowPos4dvMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4dvMESA("#args")\n"; glWindowPos4dvMESA(args);
# define glWindowPos4fMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4fMESA("#args")\n"; glWindowPos4fMESA(args);
# define glWindowPos4fvMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4fvMESA("#args")\n"; glWindowPos4fvMESA(args);
# define glWindowPos4iMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4iMESA("#args")\n"; glWindowPos4iMESA(args);
# define glWindowPos4ivMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4ivMESA("#args")\n"; glWindowPos4ivMESA(args);
# define glWindowPos4sMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4sMESA("#args")\n"; glWindowPos4sMESA(args);
# define glWindowPos4svMESA(args...)	std::cerr << GL_DEBUG << "glWindowPos4svMESA("#args")\n"; glWindowPos4svMESA(args);
# define glMultiModeDrawArraysIBM(args...)	std::cerr << GL_DEBUG << "glMultiModeDrawArraysIBM("#args")\n"; glMultiModeDrawArraysIBM(args);
# define glMultiModeDrawElementsIBM(args...)	std::cerr << GL_DEBUG << "glMultiModeDrawElementsIBM("#args")\n"; glMultiModeDrawElementsIBM(args);
# define glColorPointerListIBM(args...)	std::cerr << GL_DEBUG << "glColorPointerListIBM("#args")\n"; glColorPointerListIBM(args);
# define glSecondaryColorPointerListIBM(args...)	std::cerr << GL_DEBUG << "glSecondaryColorPointerListIBM("#args")\n"; glSecondaryColorPointerListIBM(args);
# define glEdgeFlagPointerListIBM(args...)	std::cerr << GL_DEBUG << "glEdgeFlagPointerListIBM("#args")\n"; glEdgeFlagPointerListIBM(args);
# define glFogCoordPointerListIBM(args...)	std::cerr << GL_DEBUG << "glFogCoordPointerListIBM("#args")\n"; glFogCoordPointerListIBM(args);
# define glIndexPointerListIBM(args...)	std::cerr << GL_DEBUG << "glIndexPointerListIBM("#args")\n"; glIndexPointerListIBM(args);
# define glNormalPointerListIBM(args...)	std::cerr << GL_DEBUG << "glNormalPointerListIBM("#args")\n"; glNormalPointerListIBM(args);
# define glTexCoordPointerListIBM(args...)	std::cerr << GL_DEBUG << "glTexCoordPointerListIBM("#args")\n"; glTexCoordPointerListIBM(args);
# define glVertexPointerListIBM(args...)	std::cerr << GL_DEBUG << "glVertexPointerListIBM("#args")\n"; glVertexPointerListIBM(args);
# define glTbufferMask3DFX(args...)	std::cerr << GL_DEBUG << "glTbufferMask3DFX("#args")\n"; glTbufferMask3DFX(args);
# define glSampleMaskEXT(args...)	std::cerr << GL_DEBUG << "glSampleMaskEXT("#args")\n"; glSampleMaskEXT(args);
# define glSamplePatternEXT(args...)	std::cerr << GL_DEBUG << "glSamplePatternEXT("#args")\n"; glSamplePatternEXT(args);
# define glTextureColorMaskSGIS(args...)	std::cerr << GL_DEBUG << "glTextureColorMaskSGIS("#args")\n"; glTextureColorMaskSGIS(args);
# define glIglooInterfaceSGIX(args...)	std::cerr << GL_DEBUG << "glIglooInterfaceSGIX("#args")\n"; glIglooInterfaceSGIX(args);
# define glDeleteFencesNV(args...)	std::cerr << GL_DEBUG << "glDeleteFencesNV("#args")\n"; glDeleteFencesNV(args);
# define glGenFencesNV(args...)	std::cerr << GL_DEBUG << "glGenFencesNV("#args")\n"; glGenFencesNV(args);
# define glIsFenceNV(args...)	std::cerr << GL_DEBUG << "glIsFenceNV("#args")\n"; glIsFenceNV(args);
# define glTestFenceNV(args...)	std::cerr << GL_DEBUG << "glTestFenceNV("#args")\n"; glTestFenceNV(args);
# define glGetFenceivNV(args...)	std::cerr << GL_DEBUG << "glGetFenceivNV("#args")\n"; glGetFenceivNV(args);
# define glFinishFenceNV(args...)	std::cerr << GL_DEBUG << "glFinishFenceNV("#args")\n"; glFinishFenceNV(args);
# define glSetFenceNV(args...)	std::cerr << GL_DEBUG << "glSetFenceNV("#args")\n"; glSetFenceNV(args);
# define glMapControlPointsNV(args...)	std::cerr << GL_DEBUG << "glMapControlPointsNV("#args")\n"; glMapControlPointsNV(args);
# define glMapParameterivNV(args...)	std::cerr << GL_DEBUG << "glMapParameterivNV("#args")\n"; glMapParameterivNV(args);
# define glMapParameterfvNV(args...)	std::cerr << GL_DEBUG << "glMapParameterfvNV("#args")\n"; glMapParameterfvNV(args);
# define glGetMapControlPointsNV(args...)	std::cerr << GL_DEBUG << "glGetMapControlPointsNV("#args")\n"; glGetMapControlPointsNV(args);
# define glGetMapParameterivNV(args...)	std::cerr << GL_DEBUG << "glGetMapParameterivNV("#args")\n"; glGetMapParameterivNV(args);
# define glGetMapParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetMapParameterfvNV("#args")\n"; glGetMapParameterfvNV(args);
# define glGetMapAttribParameterivNV(args...)	std::cerr << GL_DEBUG << "glGetMapAttribParameterivNV("#args")\n"; glGetMapAttribParameterivNV(args);
# define glGetMapAttribParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetMapAttribParameterfvNV("#args")\n"; glGetMapAttribParameterfvNV(args);
# define glEvalMapsNV(args...)	std::cerr << GL_DEBUG << "glEvalMapsNV("#args")\n"; glEvalMapsNV(args);
# define glCombinerStageParameterfvNV(args...)	std::cerr << GL_DEBUG << "glCombinerStageParameterfvNV("#args")\n"; glCombinerStageParameterfvNV(args);
# define glGetCombinerStageParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetCombinerStageParameterfvNV("#args")\n"; glGetCombinerStageParameterfvNV(args);
# define glAreProgramsResidentNV(args...)	std::cerr << GL_DEBUG << "glAreProgramsResidentNV("#args")\n"; glAreProgramsResidentNV(args);
# define glBindProgramNV(args...)	std::cerr << GL_DEBUG << "glBindProgramNV("#args")\n"; glBindProgramNV(args);
# define glDeleteProgramsNV(args...)	std::cerr << GL_DEBUG << "glDeleteProgramsNV("#args")\n"; glDeleteProgramsNV(args);
# define glExecuteProgramNV(args...)	std::cerr << GL_DEBUG << "glExecuteProgramNV("#args")\n"; glExecuteProgramNV(args);
# define glGenProgramsNV(args...)	std::cerr << GL_DEBUG << "glGenProgramsNV("#args")\n"; glGenProgramsNV(args);
# define glGetProgramParameterdvNV(args...)	std::cerr << GL_DEBUG << "glGetProgramParameterdvNV("#args")\n"; glGetProgramParameterdvNV(args);
# define glGetProgramParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetProgramParameterfvNV("#args")\n"; glGetProgramParameterfvNV(args);
# define glGetProgramivNV(args...)	std::cerr << GL_DEBUG << "glGetProgramivNV("#args")\n"; glGetProgramivNV(args);
# define glGetProgramStringNV(args...)	std::cerr << GL_DEBUG << "glGetProgramStringNV("#args")\n"; glGetProgramStringNV(args);
# define glGetTrackMatrixivNV(args...)	std::cerr << GL_DEBUG << "glGetTrackMatrixivNV("#args")\n"; glGetTrackMatrixivNV(args);
# define glGetVertexAttribdvNV(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribdvNV("#args")\n"; glGetVertexAttribdvNV(args);
# define glGetVertexAttribfvNV(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribfvNV("#args")\n"; glGetVertexAttribfvNV(args);
# define glGetVertexAttribivNV(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribivNV("#args")\n"; glGetVertexAttribivNV(args);
# define glGetVertexAttribPointervNV(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribPointervNV("#args")\n"; glGetVertexAttribPointervNV(args);
# define glIsProgramNV(args...)	std::cerr << GL_DEBUG << "glIsProgramNV("#args")\n"; glIsProgramNV(args);
# define glLoadProgramNV(args...)	std::cerr << GL_DEBUG << "glLoadProgramNV("#args")\n"; glLoadProgramNV(args);
# define glProgramParameter4dNV(args...)	std::cerr << GL_DEBUG << "glProgramParameter4dNV("#args")\n"; glProgramParameter4dNV(args);
# define glProgramParameter4dvNV(args...)	std::cerr << GL_DEBUG << "glProgramParameter4dvNV("#args")\n"; glProgramParameter4dvNV(args);
# define glProgramParameter4fNV(args...)	std::cerr << GL_DEBUG << "glProgramParameter4fNV("#args")\n"; glProgramParameter4fNV(args);
# define glProgramParameter4fvNV(args...)	std::cerr << GL_DEBUG << "glProgramParameter4fvNV("#args")\n"; glProgramParameter4fvNV(args);
# define glProgramParameters4dvNV(args...)	std::cerr << GL_DEBUG << "glProgramParameters4dvNV("#args")\n"; glProgramParameters4dvNV(args);
# define glProgramParameters4fvNV(args...)	std::cerr << GL_DEBUG << "glProgramParameters4fvNV("#args")\n"; glProgramParameters4fvNV(args);
# define glRequestResidentProgramsNV(args...)	std::cerr << GL_DEBUG << "glRequestResidentProgramsNV("#args")\n"; glRequestResidentProgramsNV(args);
# define glTrackMatrixNV(args...)	std::cerr << GL_DEBUG << "glTrackMatrixNV("#args")\n"; glTrackMatrixNV(args);
# define glVertexAttribPointerNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribPointerNV("#args")\n"; glVertexAttribPointerNV(args);
# define glVertexAttrib1dNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1dNV("#args")\n"; glVertexAttrib1dNV(args);
# define glVertexAttrib1dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1dvNV("#args")\n"; glVertexAttrib1dvNV(args);
# define glVertexAttrib1fNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1fNV("#args")\n"; glVertexAttrib1fNV(args);
# define glVertexAttrib1fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1fvNV("#args")\n"; glVertexAttrib1fvNV(args);
# define glVertexAttrib1sNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1sNV("#args")\n"; glVertexAttrib1sNV(args);
# define glVertexAttrib1svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1svNV("#args")\n"; glVertexAttrib1svNV(args);
# define glVertexAttrib2dNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2dNV("#args")\n"; glVertexAttrib2dNV(args);
# define glVertexAttrib2dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2dvNV("#args")\n"; glVertexAttrib2dvNV(args);
# define glVertexAttrib2fNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2fNV("#args")\n"; glVertexAttrib2fNV(args);
# define glVertexAttrib2fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2fvNV("#args")\n"; glVertexAttrib2fvNV(args);
# define glVertexAttrib2sNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2sNV("#args")\n"; glVertexAttrib2sNV(args);
# define glVertexAttrib2svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2svNV("#args")\n"; glVertexAttrib2svNV(args);
# define glVertexAttrib3dNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3dNV("#args")\n"; glVertexAttrib3dNV(args);
# define glVertexAttrib3dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3dvNV("#args")\n"; glVertexAttrib3dvNV(args);
# define glVertexAttrib3fNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3fNV("#args")\n"; glVertexAttrib3fNV(args);
# define glVertexAttrib3fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3fvNV("#args")\n"; glVertexAttrib3fvNV(args);
# define glVertexAttrib3sNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3sNV("#args")\n"; glVertexAttrib3sNV(args);
# define glVertexAttrib3svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3svNV("#args")\n"; glVertexAttrib3svNV(args);
# define glVertexAttrib4dNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4dNV("#args")\n"; glVertexAttrib4dNV(args);
# define glVertexAttrib4dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4dvNV("#args")\n"; glVertexAttrib4dvNV(args);
# define glVertexAttrib4fNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4fNV("#args")\n"; glVertexAttrib4fNV(args);
# define glVertexAttrib4fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4fvNV("#args")\n"; glVertexAttrib4fvNV(args);
# define glVertexAttrib4sNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4sNV("#args")\n"; glVertexAttrib4sNV(args);
# define glVertexAttrib4svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4svNV("#args")\n"; glVertexAttrib4svNV(args);
# define glVertexAttrib4ubNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4ubNV("#args")\n"; glVertexAttrib4ubNV(args);
# define glVertexAttrib4ubvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4ubvNV("#args")\n"; glVertexAttrib4ubvNV(args);
# define glVertexAttribs1dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs1dvNV("#args")\n"; glVertexAttribs1dvNV(args);
# define glVertexAttribs1fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs1fvNV("#args")\n"; glVertexAttribs1fvNV(args);
# define glVertexAttribs1svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs1svNV("#args")\n"; glVertexAttribs1svNV(args);
# define glVertexAttribs2dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs2dvNV("#args")\n"; glVertexAttribs2dvNV(args);
# define glVertexAttribs2fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs2fvNV("#args")\n"; glVertexAttribs2fvNV(args);
# define glVertexAttribs2svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs2svNV("#args")\n"; glVertexAttribs2svNV(args);
# define glVertexAttribs3dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs3dvNV("#args")\n"; glVertexAttribs3dvNV(args);
# define glVertexAttribs3fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs3fvNV("#args")\n"; glVertexAttribs3fvNV(args);
# define glVertexAttribs3svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs3svNV("#args")\n"; glVertexAttribs3svNV(args);
# define glVertexAttribs4dvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs4dvNV("#args")\n"; glVertexAttribs4dvNV(args);
# define glVertexAttribs4fvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs4fvNV("#args")\n"; glVertexAttribs4fvNV(args);
# define glVertexAttribs4svNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs4svNV("#args")\n"; glVertexAttribs4svNV(args);
# define glVertexAttribs4ubvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs4ubvNV("#args")\n"; glVertexAttribs4ubvNV(args);
# define glTexBumpParameterivATI(args...)	std::cerr << GL_DEBUG << "glTexBumpParameterivATI("#args")\n"; glTexBumpParameterivATI(args);
# define glTexBumpParameterfvATI(args...)	std::cerr << GL_DEBUG << "glTexBumpParameterfvATI("#args")\n"; glTexBumpParameterfvATI(args);
# define glGetTexBumpParameterivATI(args...)	std::cerr << GL_DEBUG << "glGetTexBumpParameterivATI("#args")\n"; glGetTexBumpParameterivATI(args);
# define glGetTexBumpParameterfvATI(args...)	std::cerr << GL_DEBUG << "glGetTexBumpParameterfvATI("#args")\n"; glGetTexBumpParameterfvATI(args);
# define glGenFragmentShadersATI(args...)	std::cerr << GL_DEBUG << "glGenFragmentShadersATI("#args")\n"; glGenFragmentShadersATI(args);
# define glBindFragmentShaderATI(args...)	std::cerr << GL_DEBUG << "glBindFragmentShaderATI("#args")\n"; glBindFragmentShaderATI(args);
# define glDeleteFragmentShaderATI(args...)	std::cerr << GL_DEBUG << "glDeleteFragmentShaderATI("#args")\n"; glDeleteFragmentShaderATI(args);
# define glBeginFragmentShaderATI(args...)	std::cerr << GL_DEBUG << "glBeginFragmentShaderATI("#args")\n"; glBeginFragmentShaderATI(args);
# define glEndFragmentShaderATI(args...)	std::cerr << GL_DEBUG << "glEndFragmentShaderATI("#args")\n"; glEndFragmentShaderATI(args);
# define glPassTexCoordATI(args...)	std::cerr << GL_DEBUG << "glPassTexCoordATI("#args")\n"; glPassTexCoordATI(args);
# define glSampleMapATI(args...)	std::cerr << GL_DEBUG << "glSampleMapATI("#args")\n"; glSampleMapATI(args);
# define glColorFragmentOp1ATI(args...)	std::cerr << GL_DEBUG << "glColorFragmentOp1ATI("#args")\n"; glColorFragmentOp1ATI(args);
# define glColorFragmentOp2ATI(args...)	std::cerr << GL_DEBUG << "glColorFragmentOp2ATI("#args")\n"; glColorFragmentOp2ATI(args);
# define glColorFragmentOp3ATI(args...)	std::cerr << GL_DEBUG << "glColorFragmentOp3ATI("#args")\n"; glColorFragmentOp3ATI(args);
# define glAlphaFragmentOp1ATI(args...)	std::cerr << GL_DEBUG << "glAlphaFragmentOp1ATI("#args")\n"; glAlphaFragmentOp1ATI(args);
# define glAlphaFragmentOp2ATI(args...)	std::cerr << GL_DEBUG << "glAlphaFragmentOp2ATI("#args")\n"; glAlphaFragmentOp2ATI(args);
# define glAlphaFragmentOp3ATI(args...)	std::cerr << GL_DEBUG << "glAlphaFragmentOp3ATI("#args")\n"; glAlphaFragmentOp3ATI(args);
# define glSetFragmentShaderConstantATI(args...)	std::cerr << GL_DEBUG << "glSetFragmentShaderConstantATI("#args")\n"; glSetFragmentShaderConstantATI(args);
# define glPNTrianglesiATI(args...)	std::cerr << GL_DEBUG << "glPNTrianglesiATI("#args")\n"; glPNTrianglesiATI(args);
# define glPNTrianglesfATI(args...)	std::cerr << GL_DEBUG << "glPNTrianglesfATI("#args")\n"; glPNTrianglesfATI(args);
# define glNewObjectBufferATI(args...)	std::cerr << GL_DEBUG << "glNewObjectBufferATI("#args")\n"; glNewObjectBufferATI(args);
# define glIsObjectBufferATI(args...)	std::cerr << GL_DEBUG << "glIsObjectBufferATI("#args")\n"; glIsObjectBufferATI(args);
# define glUpdateObjectBufferATI(args...)	std::cerr << GL_DEBUG << "glUpdateObjectBufferATI("#args")\n"; glUpdateObjectBufferATI(args);
# define glGetObjectBufferfvATI(args...)	std::cerr << GL_DEBUG << "glGetObjectBufferfvATI("#args")\n"; glGetObjectBufferfvATI(args);
# define glGetObjectBufferivATI(args...)	std::cerr << GL_DEBUG << "glGetObjectBufferivATI("#args")\n"; glGetObjectBufferivATI(args);
# define glFreeObjectBufferATI(args...)	std::cerr << GL_DEBUG << "glFreeObjectBufferATI("#args")\n"; glFreeObjectBufferATI(args);
# define glArrayObjectATI(args...)	std::cerr << GL_DEBUG << "glArrayObjectATI("#args")\n"; glArrayObjectATI(args);
# define glGetArrayObjectfvATI(args...)	std::cerr << GL_DEBUG << "glGetArrayObjectfvATI("#args")\n"; glGetArrayObjectfvATI(args);
# define glGetArrayObjectivATI(args...)	std::cerr << GL_DEBUG << "glGetArrayObjectivATI("#args")\n"; glGetArrayObjectivATI(args);
# define glVariantArrayObjectATI(args...)	std::cerr << GL_DEBUG << "glVariantArrayObjectATI("#args")\n"; glVariantArrayObjectATI(args);
# define glGetVariantArrayObjectfvATI(args...)	std::cerr << GL_DEBUG << "glGetVariantArrayObjectfvATI("#args")\n"; glGetVariantArrayObjectfvATI(args);
# define glGetVariantArrayObjectivATI(args...)	std::cerr << GL_DEBUG << "glGetVariantArrayObjectivATI("#args")\n"; glGetVariantArrayObjectivATI(args);
# define glBeginVertexShaderEXT(args...)	std::cerr << GL_DEBUG << "glBeginVertexShaderEXT("#args")\n"; glBeginVertexShaderEXT(args);
# define glEndVertexShaderEXT(args...)	std::cerr << GL_DEBUG << "glEndVertexShaderEXT("#args")\n"; glEndVertexShaderEXT(args);
# define glBindVertexShaderEXT(args...)	std::cerr << GL_DEBUG << "glBindVertexShaderEXT("#args")\n"; glBindVertexShaderEXT(args);
# define glGenVertexShadersEXT(args...)	std::cerr << GL_DEBUG << "glGenVertexShadersEXT("#args")\n"; glGenVertexShadersEXT(args);
# define glDeleteVertexShaderEXT(args...)	std::cerr << GL_DEBUG << "glDeleteVertexShaderEXT("#args")\n"; glDeleteVertexShaderEXT(args);
# define glShaderOp1EXT(args...)	std::cerr << GL_DEBUG << "glShaderOp1EXT("#args")\n"; glShaderOp1EXT(args);
# define glShaderOp2EXT(args...)	std::cerr << GL_DEBUG << "glShaderOp2EXT("#args")\n"; glShaderOp2EXT(args);
# define glShaderOp3EXT(args...)	std::cerr << GL_DEBUG << "glShaderOp3EXT("#args")\n"; glShaderOp3EXT(args);
# define glSwizzleEXT(args...)	std::cerr << GL_DEBUG << "glSwizzleEXT("#args")\n"; glSwizzleEXT(args);
# define glWriteMaskEXT(args...)	std::cerr << GL_DEBUG << "glWriteMaskEXT("#args")\n"; glWriteMaskEXT(args);
# define glInsertComponentEXT(args...)	std::cerr << GL_DEBUG << "glInsertComponentEXT("#args")\n"; glInsertComponentEXT(args);
# define glExtractComponentEXT(args...)	std::cerr << GL_DEBUG << "glExtractComponentEXT("#args")\n"; glExtractComponentEXT(args);
# define glGenSymbolsEXT(args...)	std::cerr << GL_DEBUG << "glGenSymbolsEXT("#args")\n"; glGenSymbolsEXT(args);
# define glSetInvariantEXT(args...)	std::cerr << GL_DEBUG << "glSetInvariantEXT("#args")\n"; glSetInvariantEXT(args);
# define glSetLocalConstantEXT(args...)	std::cerr << GL_DEBUG << "glSetLocalConstantEXT("#args")\n"; glSetLocalConstantEXT(args);
# define glVariantbvEXT(args...)	std::cerr << GL_DEBUG << "glVariantbvEXT("#args")\n"; glVariantbvEXT(args);
# define glVariantsvEXT(args...)	std::cerr << GL_DEBUG << "glVariantsvEXT("#args")\n"; glVariantsvEXT(args);
# define glVariantivEXT(args...)	std::cerr << GL_DEBUG << "glVariantivEXT("#args")\n"; glVariantivEXT(args);
# define glVariantfvEXT(args...)	std::cerr << GL_DEBUG << "glVariantfvEXT("#args")\n"; glVariantfvEXT(args);
# define glVariantdvEXT(args...)	std::cerr << GL_DEBUG << "glVariantdvEXT("#args")\n"; glVariantdvEXT(args);
# define glVariantubvEXT(args...)	std::cerr << GL_DEBUG << "glVariantubvEXT("#args")\n"; glVariantubvEXT(args);
# define glVariantusvEXT(args...)	std::cerr << GL_DEBUG << "glVariantusvEXT("#args")\n"; glVariantusvEXT(args);
# define glVariantuivEXT(args...)	std::cerr << GL_DEBUG << "glVariantuivEXT("#args")\n"; glVariantuivEXT(args);
# define glVariantPointerEXT(args...)	std::cerr << GL_DEBUG << "glVariantPointerEXT("#args")\n"; glVariantPointerEXT(args);
# define glEnableVariantClientStateEXT(args...)	std::cerr << GL_DEBUG << "glEnableVariantClientStateEXT("#args")\n"; glEnableVariantClientStateEXT(args);
# define glDisableVariantClientStateEXT(args...)	std::cerr << GL_DEBUG << "glDisableVariantClientStateEXT("#args")\n"; glDisableVariantClientStateEXT(args);
# define glBindLightParameterEXT(args...)	std::cerr << GL_DEBUG << "glBindLightParameterEXT("#args")\n"; glBindLightParameterEXT(args);
# define glBindMaterialParameterEXT(args...)	std::cerr << GL_DEBUG << "glBindMaterialParameterEXT("#args")\n"; glBindMaterialParameterEXT(args);
# define glBindTexGenParameterEXT(args...)	std::cerr << GL_DEBUG << "glBindTexGenParameterEXT("#args")\n"; glBindTexGenParameterEXT(args);
# define glBindTextureUnitParameterEXT(args...)	std::cerr << GL_DEBUG << "glBindTextureUnitParameterEXT("#args")\n"; glBindTextureUnitParameterEXT(args);
# define glBindParameterEXT(args...)	std::cerr << GL_DEBUG << "glBindParameterEXT("#args")\n"; glBindParameterEXT(args);
# define glIsVariantEnabledEXT(args...)	std::cerr << GL_DEBUG << "glIsVariantEnabledEXT("#args")\n"; glIsVariantEnabledEXT(args);
# define glGetVariantBooleanvEXT(args...)	std::cerr << GL_DEBUG << "glGetVariantBooleanvEXT("#args")\n"; glGetVariantBooleanvEXT(args);
# define glGetVariantIntegervEXT(args...)	std::cerr << GL_DEBUG << "glGetVariantIntegervEXT("#args")\n"; glGetVariantIntegervEXT(args);
# define glGetVariantFloatvEXT(args...)	std::cerr << GL_DEBUG << "glGetVariantFloatvEXT("#args")\n"; glGetVariantFloatvEXT(args);
# define glGetVariantPointervEXT(args...)	std::cerr << GL_DEBUG << "glGetVariantPointervEXT("#args")\n"; glGetVariantPointervEXT(args);
# define glGetInvariantBooleanvEXT(args...)	std::cerr << GL_DEBUG << "glGetInvariantBooleanvEXT("#args")\n"; glGetInvariantBooleanvEXT(args);
# define glGetInvariantIntegervEXT(args...)	std::cerr << GL_DEBUG << "glGetInvariantIntegervEXT("#args")\n"; glGetInvariantIntegervEXT(args);
# define glGetInvariantFloatvEXT(args...)	std::cerr << GL_DEBUG << "glGetInvariantFloatvEXT("#args")\n"; glGetInvariantFloatvEXT(args);
# define glGetLocalConstantBooleanvEXT(args...)	std::cerr << GL_DEBUG << "glGetLocalConstantBooleanvEXT("#args")\n"; glGetLocalConstantBooleanvEXT(args);
# define glGetLocalConstantIntegervEXT(args...)	std::cerr << GL_DEBUG << "glGetLocalConstantIntegervEXT("#args")\n"; glGetLocalConstantIntegervEXT(args);
# define glGetLocalConstantFloatvEXT(args...)	std::cerr << GL_DEBUG << "glGetLocalConstantFloatvEXT("#args")\n"; glGetLocalConstantFloatvEXT(args);
# define glVertexStream1sATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1sATI("#args")\n"; glVertexStream1sATI(args);
# define glVertexStream1svATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1svATI("#args")\n"; glVertexStream1svATI(args);
# define glVertexStream1iATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1iATI("#args")\n"; glVertexStream1iATI(args);
# define glVertexStream1ivATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1ivATI("#args")\n"; glVertexStream1ivATI(args);
# define glVertexStream1fATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1fATI("#args")\n"; glVertexStream1fATI(args);
# define glVertexStream1fvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1fvATI("#args")\n"; glVertexStream1fvATI(args);
# define glVertexStream1dATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1dATI("#args")\n"; glVertexStream1dATI(args);
# define glVertexStream1dvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream1dvATI("#args")\n"; glVertexStream1dvATI(args);
# define glVertexStream2sATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2sATI("#args")\n"; glVertexStream2sATI(args);
# define glVertexStream2svATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2svATI("#args")\n"; glVertexStream2svATI(args);
# define glVertexStream2iATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2iATI("#args")\n"; glVertexStream2iATI(args);
# define glVertexStream2ivATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2ivATI("#args")\n"; glVertexStream2ivATI(args);
# define glVertexStream2fATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2fATI("#args")\n"; glVertexStream2fATI(args);
# define glVertexStream2fvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2fvATI("#args")\n"; glVertexStream2fvATI(args);
# define glVertexStream2dATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2dATI("#args")\n"; glVertexStream2dATI(args);
# define glVertexStream2dvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream2dvATI("#args")\n"; glVertexStream2dvATI(args);
# define glVertexStream3sATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3sATI("#args")\n"; glVertexStream3sATI(args);
# define glVertexStream3svATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3svATI("#args")\n"; glVertexStream3svATI(args);
# define glVertexStream3iATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3iATI("#args")\n"; glVertexStream3iATI(args);
# define glVertexStream3ivATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3ivATI("#args")\n"; glVertexStream3ivATI(args);
# define glVertexStream3fATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3fATI("#args")\n"; glVertexStream3fATI(args);
# define glVertexStream3fvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3fvATI("#args")\n"; glVertexStream3fvATI(args);
# define glVertexStream3dATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3dATI("#args")\n"; glVertexStream3dATI(args);
# define glVertexStream3dvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream3dvATI("#args")\n"; glVertexStream3dvATI(args);
# define glVertexStream4sATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4sATI("#args")\n"; glVertexStream4sATI(args);
# define glVertexStream4svATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4svATI("#args")\n"; glVertexStream4svATI(args);
# define glVertexStream4iATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4iATI("#args")\n"; glVertexStream4iATI(args);
# define glVertexStream4ivATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4ivATI("#args")\n"; glVertexStream4ivATI(args);
# define glVertexStream4fATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4fATI("#args")\n"; glVertexStream4fATI(args);
# define glVertexStream4fvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4fvATI("#args")\n"; glVertexStream4fvATI(args);
# define glVertexStream4dATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4dATI("#args")\n"; glVertexStream4dATI(args);
# define glVertexStream4dvATI(args...)	std::cerr << GL_DEBUG << "glVertexStream4dvATI("#args")\n"; glVertexStream4dvATI(args);
# define glNormalStream3bATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3bATI("#args")\n"; glNormalStream3bATI(args);
# define glNormalStream3bvATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3bvATI("#args")\n"; glNormalStream3bvATI(args);
# define glNormalStream3sATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3sATI("#args")\n"; glNormalStream3sATI(args);
# define glNormalStream3svATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3svATI("#args")\n"; glNormalStream3svATI(args);
# define glNormalStream3iATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3iATI("#args")\n"; glNormalStream3iATI(args);
# define glNormalStream3ivATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3ivATI("#args")\n"; glNormalStream3ivATI(args);
# define glNormalStream3fATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3fATI("#args")\n"; glNormalStream3fATI(args);
# define glNormalStream3fvATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3fvATI("#args")\n"; glNormalStream3fvATI(args);
# define glNormalStream3dATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3dATI("#args")\n"; glNormalStream3dATI(args);
# define glNormalStream3dvATI(args...)	std::cerr << GL_DEBUG << "glNormalStream3dvATI("#args")\n"; glNormalStream3dvATI(args);
# define glClientActiveVertexStreamATI(args...)	std::cerr << GL_DEBUG << "glClientActiveVertexStreamATI("#args")\n"; glClientActiveVertexStreamATI(args);
# define glVertexBlendEnviATI(args...)	std::cerr << GL_DEBUG << "glVertexBlendEnviATI("#args")\n"; glVertexBlendEnviATI(args);
# define glVertexBlendEnvfATI(args...)	std::cerr << GL_DEBUG << "glVertexBlendEnvfATI("#args")\n"; glVertexBlendEnvfATI(args);
# define glElementPointerATI(args...)	std::cerr << GL_DEBUG << "glElementPointerATI("#args")\n"; glElementPointerATI(args);
# define glDrawElementArrayATI(args...)	std::cerr << GL_DEBUG << "glDrawElementArrayATI("#args")\n"; glDrawElementArrayATI(args);
# define glDrawRangeElementArrayATI(args...)	std::cerr << GL_DEBUG << "glDrawRangeElementArrayATI("#args")\n"; glDrawRangeElementArrayATI(args);
# define glDrawMeshArraysSUN(args...)	std::cerr << GL_DEBUG << "glDrawMeshArraysSUN("#args")\n"; glDrawMeshArraysSUN(args);
# define glGenOcclusionQueriesNV(args...)	std::cerr << GL_DEBUG << "glGenOcclusionQueriesNV("#args")\n"; glGenOcclusionQueriesNV(args);
# define glDeleteOcclusionQueriesNV(args...)	std::cerr << GL_DEBUG << "glDeleteOcclusionQueriesNV("#args")\n"; glDeleteOcclusionQueriesNV(args);
# define glIsOcclusionQueryNV(args...)	std::cerr << GL_DEBUG << "glIsOcclusionQueryNV("#args")\n"; glIsOcclusionQueryNV(args);
# define glBeginOcclusionQueryNV(args...)	std::cerr << GL_DEBUG << "glBeginOcclusionQueryNV("#args")\n"; glBeginOcclusionQueryNV(args);
# define glEndOcclusionQueryNV(args...)	std::cerr << GL_DEBUG << "glEndOcclusionQueryNV("#args")\n"; glEndOcclusionQueryNV(args);
# define glGetOcclusionQueryivNV(args...)	std::cerr << GL_DEBUG << "glGetOcclusionQueryivNV("#args")\n"; glGetOcclusionQueryivNV(args);
# define glGetOcclusionQueryuivNV(args...)	std::cerr << GL_DEBUG << "glGetOcclusionQueryuivNV("#args")\n"; glGetOcclusionQueryuivNV(args);
# define glPointParameteriNV(args...)	std::cerr << GL_DEBUG << "glPointParameteriNV("#args")\n"; glPointParameteriNV(args);
# define glPointParameterivNV(args...)	std::cerr << GL_DEBUG << "glPointParameterivNV("#args")\n"; glPointParameterivNV(args);
# define glActiveStencilFaceEXT(args...)	std::cerr << GL_DEBUG << "glActiveStencilFaceEXT("#args")\n"; glActiveStencilFaceEXT(args);
# define glElementPointerAPPLE(args...)	std::cerr << GL_DEBUG << "glElementPointerAPPLE("#args")\n"; glElementPointerAPPLE(args);
# define glDrawElementArrayAPPLE(args...)	std::cerr << GL_DEBUG << "glDrawElementArrayAPPLE("#args")\n"; glDrawElementArrayAPPLE(args);
# define glDrawRangeElementArrayAPPLE(args...)	std::cerr << GL_DEBUG << "glDrawRangeElementArrayAPPLE("#args")\n"; glDrawRangeElementArrayAPPLE(args);
# define glMultiDrawElementArrayAPPLE(args...)	std::cerr << GL_DEBUG << "glMultiDrawElementArrayAPPLE("#args")\n"; glMultiDrawElementArrayAPPLE(args);
# define glMultiDrawRangeElementArrayAPPLE(args...)	std::cerr << GL_DEBUG << "glMultiDrawRangeElementArrayAPPLE("#args")\n"; glMultiDrawRangeElementArrayAPPLE(args);
# define glGenFencesAPPLE(args...)	std::cerr << GL_DEBUG << "glGenFencesAPPLE("#args")\n"; glGenFencesAPPLE(args);
# define glDeleteFencesAPPLE(args...)	std::cerr << GL_DEBUG << "glDeleteFencesAPPLE("#args")\n"; glDeleteFencesAPPLE(args);
# define glSetFenceAPPLE(args...)	std::cerr << GL_DEBUG << "glSetFenceAPPLE("#args")\n"; glSetFenceAPPLE(args);
# define glIsFenceAPPLE(args...)	std::cerr << GL_DEBUG << "glIsFenceAPPLE("#args")\n"; glIsFenceAPPLE(args);
# define glTestFenceAPPLE(args...)	std::cerr << GL_DEBUG << "glTestFenceAPPLE("#args")\n"; glTestFenceAPPLE(args);
# define glFinishFenceAPPLE(args...)	std::cerr << GL_DEBUG << "glFinishFenceAPPLE("#args")\n"; glFinishFenceAPPLE(args);
# define glTestObjectAPPLE(args...)	std::cerr << GL_DEBUG << "glTestObjectAPPLE("#args")\n"; glTestObjectAPPLE(args);
# define glFinishObjectAPPLE(args...)	std::cerr << GL_DEBUG << "glFinishObjectAPPLE("#args")\n"; glFinishObjectAPPLE(args);
# define glBindVertexArrayAPPLE(args...)	std::cerr << GL_DEBUG << "glBindVertexArrayAPPLE("#args")\n"; glBindVertexArrayAPPLE(args);
# define glDeleteVertexArraysAPPLE(args...)	std::cerr << GL_DEBUG << "glDeleteVertexArraysAPPLE("#args")\n"; glDeleteVertexArraysAPPLE(args);
# define glGenVertexArraysAPPLE(args...)	std::cerr << GL_DEBUG << "glGenVertexArraysAPPLE("#args")\n"; glGenVertexArraysAPPLE(args);
# define glIsVertexArrayAPPLE(args...)	std::cerr << GL_DEBUG << "glIsVertexArrayAPPLE("#args")\n"; glIsVertexArrayAPPLE(args);
# define glVertexArrayRangeAPPLE(args...)	std::cerr << GL_DEBUG << "glVertexArrayRangeAPPLE("#args")\n"; glVertexArrayRangeAPPLE(args);
# define glFlushVertexArrayRangeAPPLE(args...)	std::cerr << GL_DEBUG << "glFlushVertexArrayRangeAPPLE("#args")\n"; glFlushVertexArrayRangeAPPLE(args);
# define glVertexArrayParameteriAPPLE(args...)	std::cerr << GL_DEBUG << "glVertexArrayParameteriAPPLE("#args")\n"; glVertexArrayParameteriAPPLE(args);
# define glDrawBuffersATI(args...)	std::cerr << GL_DEBUG << "glDrawBuffersATI("#args")\n"; glDrawBuffersATI(args);
# define glProgramNamedParameter4fNV(args...)	std::cerr << GL_DEBUG << "glProgramNamedParameter4fNV("#args")\n"; glProgramNamedParameter4fNV(args);
# define glProgramNamedParameter4dNV(args...)	std::cerr << GL_DEBUG << "glProgramNamedParameter4dNV("#args")\n"; glProgramNamedParameter4dNV(args);
# define glProgramNamedParameter4fvNV(args...)	std::cerr << GL_DEBUG << "glProgramNamedParameter4fvNV("#args")\n"; glProgramNamedParameter4fvNV(args);
# define glProgramNamedParameter4dvNV(args...)	std::cerr << GL_DEBUG << "glProgramNamedParameter4dvNV("#args")\n"; glProgramNamedParameter4dvNV(args);
# define glGetProgramNamedParameterfvNV(args...)	std::cerr << GL_DEBUG << "glGetProgramNamedParameterfvNV("#args")\n"; glGetProgramNamedParameterfvNV(args);
# define glGetProgramNamedParameterdvNV(args...)	std::cerr << GL_DEBUG << "glGetProgramNamedParameterdvNV("#args")\n"; glGetProgramNamedParameterdvNV(args);
# define glVertex2hNV(args...)	std::cerr << GL_DEBUG << "glVertex2hNV("#args")\n"; glVertex2hNV(args);
# define glVertex2hvNV(args...)	std::cerr << GL_DEBUG << "glVertex2hvNV("#args")\n"; glVertex2hvNV(args);
# define glVertex3hNV(args...)	std::cerr << GL_DEBUG << "glVertex3hNV("#args")\n"; glVertex3hNV(args);
# define glVertex3hvNV(args...)	std::cerr << GL_DEBUG << "glVertex3hvNV("#args")\n"; glVertex3hvNV(args);
# define glVertex4hNV(args...)	std::cerr << GL_DEBUG << "glVertex4hNV("#args")\n"; glVertex4hNV(args);
# define glVertex4hvNV(args...)	std::cerr << GL_DEBUG << "glVertex4hvNV("#args")\n"; glVertex4hvNV(args);
# define glNormal3hNV(args...)	std::cerr << GL_DEBUG << "glNormal3hNV("#args")\n"; glNormal3hNV(args);
# define glNormal3hvNV(args...)	std::cerr << GL_DEBUG << "glNormal3hvNV("#args")\n"; glNormal3hvNV(args);
# define glColor3hNV(args...)	std::cerr << GL_DEBUG << "glColor3hNV("#args")\n"; glColor3hNV(args);
# define glColor3hvNV(args...)	std::cerr << GL_DEBUG << "glColor3hvNV("#args")\n"; glColor3hvNV(args);
# define glColor4hNV(args...)	std::cerr << GL_DEBUG << "glColor4hNV("#args")\n"; glColor4hNV(args);
# define glColor4hvNV(args...)	std::cerr << GL_DEBUG << "glColor4hvNV("#args")\n"; glColor4hvNV(args);
# define glTexCoord1hNV(args...)	std::cerr << GL_DEBUG << "glTexCoord1hNV("#args")\n"; glTexCoord1hNV(args);
# define glTexCoord1hvNV(args...)	std::cerr << GL_DEBUG << "glTexCoord1hvNV("#args")\n"; glTexCoord1hvNV(args);
# define glTexCoord2hNV(args...)	std::cerr << GL_DEBUG << "glTexCoord2hNV("#args")\n"; glTexCoord2hNV(args);
# define glTexCoord2hvNV(args...)	std::cerr << GL_DEBUG << "glTexCoord2hvNV("#args")\n"; glTexCoord2hvNV(args);
# define glTexCoord3hNV(args...)	std::cerr << GL_DEBUG << "glTexCoord3hNV("#args")\n"; glTexCoord3hNV(args);
# define glTexCoord3hvNV(args...)	std::cerr << GL_DEBUG << "glTexCoord3hvNV("#args")\n"; glTexCoord3hvNV(args);
# define glTexCoord4hNV(args...)	std::cerr << GL_DEBUG << "glTexCoord4hNV("#args")\n"; glTexCoord4hNV(args);
# define glTexCoord4hvNV(args...)	std::cerr << GL_DEBUG << "glTexCoord4hvNV("#args")\n"; glTexCoord4hvNV(args);
# define glMultiTexCoord1hNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1hNV("#args")\n"; glMultiTexCoord1hNV(args);
# define glMultiTexCoord1hvNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord1hvNV("#args")\n"; glMultiTexCoord1hvNV(args);
# define glMultiTexCoord2hNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2hNV("#args")\n"; glMultiTexCoord2hNV(args);
# define glMultiTexCoord2hvNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord2hvNV("#args")\n"; glMultiTexCoord2hvNV(args);
# define glMultiTexCoord3hNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3hNV("#args")\n"; glMultiTexCoord3hNV(args);
# define glMultiTexCoord3hvNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord3hvNV("#args")\n"; glMultiTexCoord3hvNV(args);
# define glMultiTexCoord4hNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4hNV("#args")\n"; glMultiTexCoord4hNV(args);
# define glMultiTexCoord4hvNV(args...)	std::cerr << GL_DEBUG << "glMultiTexCoord4hvNV("#args")\n"; glMultiTexCoord4hvNV(args);
# define glFogCoordhNV(args...)	std::cerr << GL_DEBUG << "glFogCoordhNV("#args")\n"; glFogCoordhNV(args);
# define glFogCoordhvNV(args...)	std::cerr << GL_DEBUG << "glFogCoordhvNV("#args")\n"; glFogCoordhvNV(args);
# define glSecondaryColor3hNV(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3hNV("#args")\n"; glSecondaryColor3hNV(args);
# define glSecondaryColor3hvNV(args...)	std::cerr << GL_DEBUG << "glSecondaryColor3hvNV("#args")\n"; glSecondaryColor3hvNV(args);
# define glVertexWeighthNV(args...)	std::cerr << GL_DEBUG << "glVertexWeighthNV("#args")\n"; glVertexWeighthNV(args);
# define glVertexWeighthvNV(args...)	std::cerr << GL_DEBUG << "glVertexWeighthvNV("#args")\n"; glVertexWeighthvNV(args);
# define glVertexAttrib1hNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1hNV("#args")\n"; glVertexAttrib1hNV(args);
# define glVertexAttrib1hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib1hvNV("#args")\n"; glVertexAttrib1hvNV(args);
# define glVertexAttrib2hNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2hNV("#args")\n"; glVertexAttrib2hNV(args);
# define glVertexAttrib2hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib2hvNV("#args")\n"; glVertexAttrib2hvNV(args);
# define glVertexAttrib3hNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3hNV("#args")\n"; glVertexAttrib3hNV(args);
# define glVertexAttrib3hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib3hvNV("#args")\n"; glVertexAttrib3hvNV(args);
# define glVertexAttrib4hNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4hNV("#args")\n"; glVertexAttrib4hNV(args);
# define glVertexAttrib4hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttrib4hvNV("#args")\n"; glVertexAttrib4hvNV(args);
# define glVertexAttribs1hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs1hvNV("#args")\n"; glVertexAttribs1hvNV(args);
# define glVertexAttribs2hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs2hvNV("#args")\n"; glVertexAttribs2hvNV(args);
# define glVertexAttribs3hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs3hvNV("#args")\n"; glVertexAttribs3hvNV(args);
# define glVertexAttribs4hvNV(args...)	std::cerr << GL_DEBUG << "glVertexAttribs4hvNV("#args")\n"; glVertexAttribs4hvNV(args);
# define glPixelDataRangeNV(args...)	std::cerr << GL_DEBUG << "glPixelDataRangeNV("#args")\n"; glPixelDataRangeNV(args);
# define glFlushPixelDataRangeNV(args...)	std::cerr << GL_DEBUG << "glFlushPixelDataRangeNV("#args")\n"; glFlushPixelDataRangeNV(args);
# define glPrimitiveRestartNV(args...)	std::cerr << GL_DEBUG << "glPrimitiveRestartNV("#args")\n"; glPrimitiveRestartNV(args);
# define glPrimitiveRestartIndexNV(args...)	std::cerr << GL_DEBUG << "glPrimitiveRestartIndexNV("#args")\n"; glPrimitiveRestartIndexNV(args);
# define glMapObjectBufferATI(args...)	std::cerr << GL_DEBUG << "glMapObjectBufferATI("#args")\n"; glMapObjectBufferATI(args);
# define glUnmapObjectBufferATI(args...)	std::cerr << GL_DEBUG << "glUnmapObjectBufferATI("#args")\n"; glUnmapObjectBufferATI(args);
# define glStencilOpSeparateATI(args...)	std::cerr << GL_DEBUG << "glStencilOpSeparateATI("#args")\n"; glStencilOpSeparateATI(args);
# define glStencilFuncSeparateATI(args...)	std::cerr << GL_DEBUG << "glStencilFuncSeparateATI("#args")\n"; glStencilFuncSeparateATI(args);
# define glVertexAttribArrayObjectATI(args...)	std::cerr << GL_DEBUG << "glVertexAttribArrayObjectATI("#args")\n"; glVertexAttribArrayObjectATI(args);
# define glGetVertexAttribArrayObjectfvATI(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribArrayObjectfvATI("#args")\n"; glGetVertexAttribArrayObjectfvATI(args);
# define glGetVertexAttribArrayObjectivATI(args...)	std::cerr << GL_DEBUG << "glGetVertexAttribArrayObjectivATI("#args")\n"; glGetVertexAttribArrayObjectivATI(args);
# define glDepthBoundsEXT(args...)	std::cerr << GL_DEBUG << "glDepthBoundsEXT("#args")\n"; glDepthBoundsEXT(args);
# define glBlendEquationSeparateEXT(args...)	std::cerr << GL_DEBUG << "glBlendEquationSeparateEXT("#args")\n"; glBlendEquationSeparateEXT(args);
//	}}} glext-Functions

//	{{{ glu-Functions
# define gluBeginCurve(args...)	std::cerr << GL_DEBUG << "gluBeginCurve("#args")\n"; gluBeginCurve(args);
# define gluBeginPolygon(args...)	std::cerr << GL_DEBUG << "gluBeginPolygon("#args")\n"; gluBeginPolygon(args);
# define gluBeginSurface(args...)	std::cerr << GL_DEBUG << "gluBeginSurface("#args")\n"; gluBeginSurface(args);
# define gluBeginTrim(args...)	std::cerr << GL_DEBUG << "gluBeginTrim("#args")\n"; gluBeginTrim(args);
# define gluBuild1DMipmapLevels(args...)	std::cerr << GL_DEBUG << "gluBuild1DMipmapLevels("#args")\n"; gluBuild1DMipmapLevels(args);
# define gluBuild1DMipmaps(args...)	std::cerr << GL_DEBUG << "gluBuild1DMipmaps("#args")\n"; gluBuild1DMipmaps(args);
# define gluBuild2DMipmapLevels(args...)	std::cerr << GL_DEBUG << "gluBuild2DMipmapLevels("#args")\n"; gluBuild2DMipmapLevels(args);
# define gluBuild2DMipmaps(args...)	std::cerr << GL_DEBUG << "gluBuild2DMipmaps("#args")\n"; gluBuild2DMipmaps(args);
# define gluBuild3DMipmapLevels(args...)	std::cerr << GL_DEBUG << "gluBuild3DMipmapLevels("#args")\n"; gluBuild3DMipmapLevels(args);
# define gluBuild3DMipmaps(args...)	std::cerr << GL_DEBUG << "gluBuild3DMipmaps("#args")\n"; gluBuild3DMipmaps(args);
# define gluCheckExtension(args...)	std::cerr << GL_DEBUG << "gluCheckExtension("#args")\n"; gluCheckExtension(args);
# define gluCylinder(args...)	std::cerr << GL_DEBUG << "gluCylinder("#args")\n"; gluCylinder(args);
# define gluDeleteNurbsRenderer(args...)	std::cerr << GL_DEBUG << "gluDeleteNurbsRenderer("#args")\n"; gluDeleteNurbsRenderer(args);
# define gluDeleteQuadric(args...)	std::cerr << GL_DEBUG << "gluDeleteQuadric("#args")\n"; gluDeleteQuadric(args);
# define gluDeleteTess(args...)	std::cerr << GL_DEBUG << "gluDeleteTess("#args")\n"; gluDeleteTess(args);
# define gluDisk(args...)	std::cerr << GL_DEBUG << "gluDisk("#args")\n"; gluDisk(args);
# define gluEndCurve(args...)	std::cerr << GL_DEBUG << "gluEndCurve("#args")\n"; gluEndCurve(args);
# define gluEndPolygon(args...)	std::cerr << GL_DEBUG << "gluEndPolygon("#args")\n"; gluEndPolygon(args);
# define gluEndSurface(args...)	std::cerr << GL_DEBUG << "gluEndSurface("#args")\n"; gluEndSurface(args);
# define gluEndTrim(args...)	std::cerr << GL_DEBUG << "gluEndTrim("#args")\n"; gluEndTrim(args);
# define gluErrorString(args...)	std::cerr << GL_DEBUG << "gluErrorString("#args")\n"; gluErrorString(args);
# define gluGetNurbsProperty(args...)	std::cerr << GL_DEBUG << "gluGetNurbsProperty("#args")\n"; gluGetNurbsProperty(args);
# define gluGetString(args...)	std::cerr << GL_DEBUG << "gluGetString("#args")\n"; gluGetString(args);
# define gluGetTessProperty(args...)	std::cerr << GL_DEBUG << "gluGetTessProperty("#args")\n"; gluGetTessProperty(args);
# define gluLoadSamplingMatrices(args...)	std::cerr << GL_DEBUG << "gluLoadSamplingMatrices("#args")\n"; gluLoadSamplingMatrices(args);
# define gluLookAt(args...)	std::cerr << GL_DEBUG << "gluLookAt("#args")\n"; gluLookAt(args);
# define gluNewNurbsRenderer(args...)	std::cerr << GL_DEBUG << "gluNewNurbsRenderer("#args")\n"; gluNewNurbsRenderer(args);
//# define gluNewQuadric(args...)	std::cerr << GL_DEBUG << "gluNewQuadric("#args")\n"; gluNewQuadric(args);
# define gluNewTess(args...)	std::cerr << GL_DEBUG << "gluNewTess("#args")\n"; gluNewTess(args);
# define gluNextContour(args...)	std::cerr << GL_DEBUG << "gluNextContour("#args")\n"; gluNextContour(args);
# define gluNurbsCallback(args...)	std::cerr << GL_DEBUG << "gluNurbsCallback("#args")\n"; gluNurbsCallback(args);
# define gluNurbsCallbackData(args...)	std::cerr << GL_DEBUG << "gluNurbsCallbackData("#args")\n"; gluNurbsCallbackData(args);
# define gluNurbsCallbackDataEXT(args...)	std::cerr << GL_DEBUG << "gluNurbsCallbackDataEXT("#args")\n"; gluNurbsCallbackDataEXT(args);
# define gluNurbsCurve(args...)	std::cerr << GL_DEBUG << "gluNurbsCurve("#args")\n"; gluNurbsCurve(args);
# define gluNurbsProperty(args...)	std::cerr << GL_DEBUG << "gluNurbsProperty("#args")\n"; gluNurbsProperty(args);
# define gluNurbsSurface(args...)	std::cerr << GL_DEBUG << "gluNurbsSurface("#args")\n"; gluNurbsSurface(args);
# define gluOrtho2D(args...)	std::cerr << GL_DEBUG << "gluOrtho2D("#args")\n"; gluOrtho2D(args);
# define gluPartialDisk(args...)	std::cerr << GL_DEBUG << "gluPartialDisk("#args")\n"; gluPartialDisk(args);
# define gluPerspective(args...)	std::cerr << GL_DEBUG << "gluPerspective("#args")\n"; gluPerspective(args);
# define gluPickMatrix(args...)	std::cerr << GL_DEBUG << "gluPickMatrix("#args")\n"; gluPickMatrix(args);
# define gluProject(args...)	std::cerr << GL_DEBUG << "gluProject("#args")\n"; gluProject(args);
# define gluPwlCurve(args...)	std::cerr << GL_DEBUG << "gluPwlCurve("#args")\n"; gluPwlCurve(args);
# define gluQuadricCallback(args...)	std::cerr << GL_DEBUG << "gluQuadricCallback("#args")\n"; gluQuadricCallback(args);
# define gluQuadricDrawStyle(args...)	std::cerr << GL_DEBUG << "gluQuadricDrawStyle("#args")\n"; gluQuadricDrawStyle(args);
# define gluQuadricNormals(args...)	std::cerr << GL_DEBUG << "gluQuadricNormals("#args")\n"; gluQuadricNormals(args);
# define gluQuadricOrientation(args...)	std::cerr << GL_DEBUG << "gluQuadricOrientation("#args")\n"; gluQuadricOrientation(args);
# define gluQuadricTexture(args...)	std::cerr << GL_DEBUG << "gluQuadricTexture("#args")\n"; gluQuadricTexture(args);
# define gluScaleImage(args...)	std::cerr << GL_DEBUG << "gluScaleImage("#args")\n"; gluScaleImage(args);
# define gluSphere(args...)	std::cerr << GL_DEBUG << "gluSphere("#args")\n"; gluSphere(args);
# define gluTessBeginContour(args...)	std::cerr << GL_DEBUG << "gluTessBeginContour("#args")\n"; gluTessBeginContour(args);
# define gluTessBeginPolygon(args...)	std::cerr << GL_DEBUG << "gluTessBeginPolygon("#args")\n"; gluTessBeginPolygon(args);
# define gluTessCallback(args...)	std::cerr << GL_DEBUG << "gluTessCallback("#args")\n"; gluTessCallback(args);
# define gluTessEndContour(args...)	std::cerr << GL_DEBUG << "gluTessEndContour("#args")\n"; gluTessEndContour(args);
# define gluTessEndPolygon(args...)	std::cerr << GL_DEBUG << "gluTessEndPolygon("#args")\n"; gluTessEndPolygon(args);
# define gluTessNormal(args...)	std::cerr << GL_DEBUG << "gluTessNormal("#args")\n"; gluTessNormal(args);
# define gluTessProperty(args...)	std::cerr << GL_DEBUG << "gluTessProperty("#args")\n"; gluTessProperty(args);
# define gluTessVertex(args...)	std::cerr << GL_DEBUG << "gluTessVertex("#args")\n"; gluTessVertex(args);
# define gluUnProject(args...)	std::cerr << GL_DEBUG << "gluUnProject("#args")\n"; gluUnProject(args);
# define gluUnProject4(args...)	std::cerr << GL_DEBUG << "gluUnProject4("#args")\n"; gluUnProject4(args);
//	}}} glu-Functions

//	{{{ glut-Functions
# define glutInit(args...)	std::cerr << GL_DEBUG << "glutInit("#args")\n"; glutInit(args);
# define glutInit_ATEXIT_HACK(args...)	std::cerr << GL_DEBUG << "glutInit_ATEXIT_HACK("#args")\n"; glutInit_ATEXIT_HACK(args);
# define glutInitDisplayMode(args...)	std::cerr << GL_DEBUG << "glutInitDisplayMode("#args")\n"; glutInitDisplayMode(args);
# define glutInitDisplayString(args...)	std::cerr << GL_DEBUG << "glutInitDisplayString("#args")\n"; glutInitDisplayString(args);
# define glutInitWindowPosition(args...)	std::cerr << GL_DEBUG << "glutInitWindowPosition("#args")\n"; glutInitWindowPosition(args);
# define glutInitWindowSize(args...)	std::cerr << GL_DEBUG << "glutInitWindowSize("#args")\n"; glutInitWindowSize(args);
# define glutMainLoop(args...)	std::cerr << GL_DEBUG << "glutMainLoop("#args")\n"; glutMainLoop(args);
# define glutCreateWindow(args...)	std::cerr << GL_DEBUG << "glutCreateWindow("#args")\n"; glutCreateWindow(args);
# define glutCreateWindow_ATEXIT_HACK(args...)	std::cerr << GL_DEBUG << "glutCreateWindow_ATEXIT_HACK("#args")\n"; glutCreateWindow_ATEXIT_HACK(args);
# define glutCreateSubWindow(args...)	std::cerr << GL_DEBUG << "glutCreateSubWindow("#args")\n"; glutCreateSubWindow(args);
# define glutDestroyWindow(args...)	std::cerr << GL_DEBUG << "glutDestroyWindow("#args")\n"; glutDestroyWindow(args);
# define glutPostRedisplay(args...)	std::cerr << GL_DEBUG << "glutPostRedisplay("#args")\n"; glutPostRedisplay(args);
# define glutPostWindowRedisplay(args...)	std::cerr << GL_DEBUG << "glutPostWindowRedisplay("#args")\n"; glutPostWindowRedisplay(args);
# define glutSwapBuffers(args...)	std::cerr << GL_DEBUG << "glutSwapBuffers("#args")\n"; glutSwapBuffers(args);
# define glutGetWindow(args...)	std::cerr << GL_DEBUG << "glutGetWindow("#args")\n"; glutGetWindow(args);
# define glutSetWindow(args...)	std::cerr << GL_DEBUG << "glutSetWindow("#args")\n"; glutSetWindow(args);
# define glutSetWindowTitle(args...)	std::cerr << GL_DEBUG << "glutSetWindowTitle("#args")\n"; glutSetWindowTitle(args);
# define glutSetIconTitle(args...)	std::cerr << GL_DEBUG << "glutSetIconTitle("#args")\n"; glutSetIconTitle(args);
# define glutPositionWindow(args...)	std::cerr << GL_DEBUG << "glutPositionWindow("#args")\n"; glutPositionWindow(args);
# define glutReshapeWindow(args...)	std::cerr << GL_DEBUG << "glutReshapeWindow("#args")\n"; glutReshapeWindow(args);
# define glutPopWindow(args...)	std::cerr << GL_DEBUG << "glutPopWindow("#args")\n"; glutPopWindow(args);
# define glutPushWindow(args...)	std::cerr << GL_DEBUG << "glutPushWindow("#args")\n"; glutPushWindow(args);
# define glutIconifyWindow(args...)	std::cerr << GL_DEBUG << "glutIconifyWindow("#args")\n"; glutIconifyWindow(args);
# define glutShowWindow(args...)	std::cerr << GL_DEBUG << "glutShowWindow("#args")\n"; glutShowWindow(args);
# define glutHideWindow(args...)	std::cerr << GL_DEBUG << "glutHideWindow("#args")\n"; glutHideWindow(args);
# define glutFullScreen(args...)	std::cerr << GL_DEBUG << "glutFullScreen("#args")\n"; glutFullScreen(args);
# define glutSetCursor(args...)	std::cerr << GL_DEBUG << "glutSetCursor("#args")\n"; glutSetCursor(args);
# define glutWarpPointer(args...)	std::cerr << GL_DEBUG << "glutWarpPointer("#args")\n"; glutWarpPointer(args);
# define glutEstablishOverlay(args...)	std::cerr << GL_DEBUG << "glutEstablishOverlay("#args")\n"; glutEstablishOverlay(args);
# define glutRemoveOverlay(args...)	std::cerr << GL_DEBUG << "glutRemoveOverlay("#args")\n"; glutRemoveOverlay(args);
# define glutUseLayer(args...)	std::cerr << GL_DEBUG << "glutUseLayer("#args")\n"; glutUseLayer(args);
# define glutPostOverlayRedisplay(args...)	std::cerr << GL_DEBUG << "glutPostOverlayRedisplay("#args")\n"; glutPostOverlayRedisplay(args);
# define glutPostWindowOverlayRedisplay(args...)	std::cerr << GL_DEBUG << "glutPostWindowOverlayRedisplay("#args")\n"; glutPostWindowOverlayRedisplay(args);
# define glutShowOverlay(args...)	std::cerr << GL_DEBUG << "glutShowOverlay("#args")\n"; glutShowOverlay(args);
# define glutHideOverlay(args...)	std::cerr << GL_DEBUG << "glutHideOverlay("#args")\n"; glutHideOverlay(args);
# define glutCreateMenu(args...)	std::cerr << GL_DEBUG << "glutCreateMenu("#args")\n"; glutCreateMenu(args);
# define glutCreateMenu_ATEXIT_HACK(args...)	std::cerr << GL_DEBUG << "glutCreateMenu_ATEXIT_HACK("#args")\n"; glutCreateMenu_ATEXIT_HACK(args);
# define glutDestroyMenu(args...)	std::cerr << GL_DEBUG << "glutDestroyMenu("#args")\n"; glutDestroyMenu(args);
# define glutGetMenu(args...)	std::cerr << GL_DEBUG << "glutGetMenu("#args")\n"; glutGetMenu(args);
# define glutSetMenu(args...)	std::cerr << GL_DEBUG << "glutSetMenu("#args")\n"; glutSetMenu(args);
# define glutAddMenuEntry(args...)	std::cerr << GL_DEBUG << "glutAddMenuEntry("#args")\n"; glutAddMenuEntry(args);
# define glutAddSubMenu(args...)	std::cerr << GL_DEBUG << "glutAddSubMenu("#args")\n"; glutAddSubMenu(args);
# define glutChangeToMenuEntry(args...)	std::cerr << GL_DEBUG << "glutChangeToMenuEntry("#args")\n"; glutChangeToMenuEntry(args);
# define glutChangeToSubMenu(args...)	std::cerr << GL_DEBUG << "glutChangeToSubMenu("#args")\n"; glutChangeToSubMenu(args);
# define glutRemoveMenuItem(args...)	std::cerr << GL_DEBUG << "glutRemoveMenuItem("#args")\n"; glutRemoveMenuItem(args);
# define glutAttachMenu(args...)	std::cerr << GL_DEBUG << "glutAttachMenu("#args")\n"; glutAttachMenu(args);
# define glutDetachMenu(args...)	std::cerr << GL_DEBUG << "glutDetachMenu("#args")\n"; glutDetachMenu(args);
# define glutDisplayFunc(args...)	std::cerr << GL_DEBUG << "glutDisplayFunc("#args")\n"; glutDisplayFunc(args);
# define glutReshapeFunc(args...)	std::cerr << GL_DEBUG << "glutReshapeFunc("#args")\n"; glutReshapeFunc(args);
# define glutKeyboardFunc(args...)	std::cerr << GL_DEBUG << "glutKeyboardFunc("#args")\n"; glutKeyboardFunc(args);
# define glutMouseFunc(args...)	std::cerr << GL_DEBUG << "glutMouseFunc("#args")\n"; glutMouseFunc(args);
# define glutMotionFunc(args...)	std::cerr << GL_DEBUG << "glutMotionFunc("#args")\n"; glutMotionFunc(args);
# define glutPassiveMotionFunc(args...)	std::cerr << GL_DEBUG << "glutPassiveMotionFunc("#args")\n"; glutPassiveMotionFunc(args);
# define glutEntryFunc(args...)	std::cerr << GL_DEBUG << "glutEntryFunc("#args")\n"; glutEntryFunc(args);
# define glutVisibilityFunc(args...)	std::cerr << GL_DEBUG << "glutVisibilityFunc("#args")\n"; glutVisibilityFunc(args);
# define glutIdleFunc(args...)	std::cerr << GL_DEBUG << "glutIdleFunc("#args")\n"; glutIdleFunc(args);
# define glutTimerFunc(args...)	std::cerr << GL_DEBUG << "glutTimerFunc("#args")\n"; glutTimerFunc(args);
# define glutMenuStateFunc(args...)	std::cerr << GL_DEBUG << "glutMenuStateFunc("#args")\n"; glutMenuStateFunc(args);
# define glutSpecialFunc(args...)	std::cerr << GL_DEBUG << "glutSpecialFunc("#args")\n"; glutSpecialFunc(args);
# define glutSpaceballMotionFunc(args...)	std::cerr << GL_DEBUG << "glutSpaceballMotionFunc("#args")\n"; glutSpaceballMotionFunc(args);
# define glutSpaceballRotateFunc(args...)	std::cerr << GL_DEBUG << "glutSpaceballRotateFunc("#args")\n"; glutSpaceballRotateFunc(args);
# define glutSpaceballButtonFunc(args...)	std::cerr << GL_DEBUG << "glutSpaceballButtonFunc("#args")\n"; glutSpaceballButtonFunc(args);
# define glutButtonBoxFunc(args...)	std::cerr << GL_DEBUG << "glutButtonBoxFunc("#args")\n"; glutButtonBoxFunc(args);
# define glutDialsFunc(args...)	std::cerr << GL_DEBUG << "glutDialsFunc("#args")\n"; glutDialsFunc(args);
# define glutTabletMotionFunc(args...)	std::cerr << GL_DEBUG << "glutTabletMotionFunc("#args")\n"; glutTabletMotionFunc(args);
# define glutTabletButtonFunc(args...)	std::cerr << GL_DEBUG << "glutTabletButtonFunc("#args")\n"; glutTabletButtonFunc(args);
# define glutMenuStatusFunc(args...)	std::cerr << GL_DEBUG << "glutMenuStatusFunc("#args")\n"; glutMenuStatusFunc(args);
# define glutOverlayDisplayFunc(args...)	std::cerr << GL_DEBUG << "glutOverlayDisplayFunc("#args")\n"; glutOverlayDisplayFunc(args);
# define glutWindowStatusFunc(args...)	std::cerr << GL_DEBUG << "glutWindowStatusFunc("#args")\n"; glutWindowStatusFunc(args);
# define glutKeyboardUpFunc(args...)	std::cerr << GL_DEBUG << "glutKeyboardUpFunc("#args")\n"; glutKeyboardUpFunc(args);
# define glutSpecialUpFunc(args...)	std::cerr << GL_DEBUG << "glutSpecialUpFunc("#args")\n"; glutSpecialUpFunc(args);
# define glutJoystickFunc(args...)	std::cerr << GL_DEBUG << "glutJoystickFunc("#args")\n"; glutJoystickFunc(args);
# define glutSetColor(args...)	std::cerr << GL_DEBUG << "glutSetColor("#args")\n"; glutSetColor(args);
# define glutGetColor(args...)	std::cerr << GL_DEBUG << "glutGetColor("#args")\n"; glutGetColor(args);
# define glutCopyColormap(args...)	std::cerr << GL_DEBUG << "glutCopyColormap("#args")\n"; glutCopyColormap(args);
# define glutGet(args...)	std::cerr << GL_DEBUG << "glutGet("#args")\n"; glutGet(args);
# define glutDeviceGet(args...)	std::cerr << GL_DEBUG << "glutDeviceGet("#args")\n"; glutDeviceGet(args);
# define glutExtensionSupported(args...)	std::cerr << GL_DEBUG << "glutExtensionSupported("#args")\n"; glutExtensionSupported(args);
# define glutGetModifiers(args...)	std::cerr << GL_DEBUG << "glutGetModifiers("#args")\n"; glutGetModifiers(args);
# define glutLayerGet(args...)	std::cerr << GL_DEBUG << "glutLayerGet("#args")\n"; glutLayerGet(args);
# define glutGetProcAddress(args...)	std::cerr << GL_DEBUG << "glutGetProcAddress("#args")\n"; glutGetProcAddress(args);
# define glutBitmapCharacter(args...)	std::cerr << GL_DEBUG << "glutBitmapCharacter("#args")\n"; glutBitmapCharacter(args);
# define glutBitmapWidth(args...)	std::cerr << GL_DEBUG << "glutBitmapWidth("#args")\n"; glutBitmapWidth(args);
# define glutStrokeCharacter(args...)	std::cerr << GL_DEBUG << "glutStrokeCharacter("#args")\n"; glutStrokeCharacter(args);
# define glutStrokeWidth(args...)	std::cerr << GL_DEBUG << "glutStrokeWidth("#args")\n"; glutStrokeWidth(args);
# define glutBitmapLength(args...)	std::cerr << GL_DEBUG << "glutBitmapLength("#args")\n"; glutBitmapLength(args);
# define glutStrokeLength(args...)	std::cerr << GL_DEBUG << "glutStrokeLength("#args")\n"; glutStrokeLength(args);
# define glutWireSphere(args...)	std::cerr << GL_DEBUG << "glutWireSphere("#args")\n"; glutWireSphere(args);
# define glutSolidSphere(args...)	std::cerr << GL_DEBUG << "glutSolidSphere("#args")\n"; glutSolidSphere(args);
# define glutWireCone(args...)	std::cerr << GL_DEBUG << "glutWireCone("#args")\n"; glutWireCone(args);
# define glutSolidCone(args...)	std::cerr << GL_DEBUG << "glutSolidCone("#args")\n"; glutSolidCone(args);
# define glutWireCube(args...)	std::cerr << GL_DEBUG << "glutWireCube("#args")\n"; glutWireCube(args);
# define glutSolidCube(args...)	std::cerr << GL_DEBUG << "glutSolidCube("#args")\n"; glutSolidCube(args);
# define glutWireTorus(args...)	std::cerr << GL_DEBUG << "glutWireTorus("#args")\n"; glutWireTorus(args);
# define glutSolidTorus(args...)	std::cerr << GL_DEBUG << "glutSolidTorus("#args")\n"; glutSolidTorus(args);
# define glutWireDodecahedron(args...)	std::cerr << GL_DEBUG << "glutWireDodecahedron("#args")\n"; glutWireDodecahedron(args);
# define glutSolidDodecahedron(args...)	std::cerr << GL_DEBUG << "glutSolidDodecahedron("#args")\n"; glutSolidDodecahedron(args);
# define glutWireTeapot(args...)	std::cerr << GL_DEBUG << "glutWireTeapot("#args")\n"; glutWireTeapot(args);
# define glutSolidTeapot(args...)	std::cerr << GL_DEBUG << "glutSolidTeapot("#args")\n"; glutSolidTeapot(args);
# define glutWireOctahedron(args...)	std::cerr << GL_DEBUG << "glutWireOctahedron("#args")\n"; glutWireOctahedron(args);
# define glutSolidOctahedron(args...)	std::cerr << GL_DEBUG << "glutSolidOctahedron("#args")\n"; glutSolidOctahedron(args);
# define glutWireTetrahedron(args...)	std::cerr << GL_DEBUG << "glutWireTetrahedron("#args")\n"; glutWireTetrahedron(args);
# define glutSolidTetrahedron(args...)	std::cerr << GL_DEBUG << "glutSolidTetrahedron("#args")\n"; glutSolidTetrahedron(args);
# define glutWireIcosahedron(args...)	std::cerr << GL_DEBUG << "glutWireIcosahedron("#args")\n"; glutWireIcosahedron(args);
# define glutSolidIcosahedron(args...)	std::cerr << GL_DEBUG << "glutSolidIcosahedron("#args")\n"; glutSolidIcosahedron(args);
# define glutVideoResizeGet(args...)	std::cerr << GL_DEBUG << "glutVideoResizeGet("#args")\n"; glutVideoResizeGet(args);
# define glutSetupVideoResizing(args...)	std::cerr << GL_DEBUG << "glutSetupVideoResizing("#args")\n"; glutSetupVideoResizing(args);
# define glutStopVideoResizing(args...)	std::cerr << GL_DEBUG << "glutStopVideoResizing("#args")\n"; glutStopVideoResizing(args);
# define glutVideoResize(args...)	std::cerr << GL_DEBUG << "glutVideoResize("#args")\n"; glutVideoResize(args);
# define glutVideoPan(args...)	std::cerr << GL_DEBUG << "glutVideoPan("#args")\n"; glutVideoPan(args);
# define glutReportErrors(args...)	std::cerr << GL_DEBUG << "glutReportErrors("#args")\n"; glutReportErrors(args);
# define glutIgnoreKeyRepeat(args...)	std::cerr << GL_DEBUG << "glutIgnoreKeyRepeat("#args")\n"; glutIgnoreKeyRepeat(args);
# define glutSetKeyRepeat(args...)	std::cerr << GL_DEBUG << "glutSetKeyRepeat("#args")\n"; glutSetKeyRepeat(args);
# define glutForceJoystickFunc(args...)	std::cerr << GL_DEBUG << "glutForceJoystickFunc("#args")\n"; glutForceJoystickFunc(args);
# define glutGameModeString(args...)	std::cerr << GL_DEBUG << "glutGameModeString("#args")\n"; glutGameModeString(args);
# define glutEnterGameMode(args...)	std::cerr << GL_DEBUG << "glutEnterGameMode("#args")\n"; glutEnterGameMode(args);
# define glutLeaveGameMode(args...)	std::cerr << GL_DEBUG << "glutLeaveGameMode("#args")\n"; glutLeaveGameMode(args);
# define glutGameModeGet(args...)	std::cerr << GL_DEBUG << "glutGameModeGet("#args")\n"; glutGameModeGet(args);
//	}}} glut-Functions

//	{{{ glx-Functions
# define glXChooseVisual(args...)	std::cerr << GL_DEBUG << "glXChooseVisual("#args")\n"; glXChooseVisual(args);
# define glXCopyContext(args...)	std::cerr << GL_DEBUG << "glXCopyContext("#args")\n"; glXCopyContext(args);
# define glXCreateContext(args...)	std::cerr << GL_DEBUG << "glXCreateContext("#args")\n"; glXCreateContext(args);
# define glXCreateGLXPixmap(args...)	std::cerr << GL_DEBUG << "glXCreateGLXPixmap("#args")\n"; glXCreateGLXPixmap(args);
# define glXDestroyContext(args...)	std::cerr << GL_DEBUG << "glXDestroyContext("#args")\n"; glXDestroyContext(args);
# define glXDestroyGLXPixmap(args...)	std::cerr << GL_DEBUG << "glXDestroyGLXPixmap("#args")\n"; glXDestroyGLXPixmap(args);
# define glXGetConfig(args...)	std::cerr << GL_DEBUG << "glXGetConfig("#args")\n"; glXGetConfig(args);
# define glXGetCurrentContext(args...)	std::cerr << GL_DEBUG << "glXGetCurrentContext("#args")\n"; glXGetCurrentContext(args);
# define glXGetCurrentDrawable(args...)	std::cerr << GL_DEBUG << "glXGetCurrentDrawable("#args")\n"; glXGetCurrentDrawable(args);
# define glXIsDirect(args...)	std::cerr << GL_DEBUG << "glXIsDirect("#args")\n"; glXIsDirect(args);
# define glXMakeCurrent(args...)	std::cerr << GL_DEBUG << "glXMakeCurrent("#args")\n"; glXMakeCurrent(args);
# define glXQueryExtension(args...)	std::cerr << GL_DEBUG << "glXQueryExtension("#args")\n"; glXQueryExtension(args);
# define glXQueryVersion(args...)	std::cerr << GL_DEBUG << "glXQueryVersion("#args")\n"; glXQueryVersion(args);
# define glXSwapBuffers(args...)	std::cerr << GL_DEBUG << "glXSwapBuffers("#args")\n"; glXSwapBuffers(args);
# define glXUseXFont(args...)	std::cerr << GL_DEBUG << "glXUseXFont("#args")\n"; glXUseXFont(args);
# define glXWaitGL(args...)	std::cerr << GL_DEBUG << "glXWaitGL("#args")\n"; glXWaitGL(args);
# define glXWaitX(args...)	std::cerr << GL_DEBUG << "glXWaitX("#args")\n"; glXWaitX(args);
# define glXGetClientString(args...)	std::cerr << GL_DEBUG << "glXGetClientString("#args")\n"; glXGetClientString(args);
# define glXQueryServerString(args...)	std::cerr << GL_DEBUG << "glXQueryServerString("#args")\n"; glXQueryServerString(args);
# define glXQueryExtensionsString(args...)	std::cerr << GL_DEBUG << "glXQueryExtensionsString("#args")\n"; glXQueryExtensionsString(args);
# define glXGetCurrentDisplay(args...)	std::cerr << GL_DEBUG << "glXGetCurrentDisplay("#args")\n"; glXGetCurrentDisplay(args);
# define glXChooseFBConfig(args...)	std::cerr << GL_DEBUG << "glXChooseFBConfig("#args")\n"; glXChooseFBConfig(args);
# define glXCreateNewContext(args...)	std::cerr << GL_DEBUG << "glXCreateNewContext("#args")\n"; glXCreateNewContext(args);
# define glXCreatePbuffer(args...)	std::cerr << GL_DEBUG << "glXCreatePbuffer("#args")\n"; glXCreatePbuffer(args);
# define glXCreatePixmap(args...)	std::cerr << GL_DEBUG << "glXCreatePixmap("#args")\n"; glXCreatePixmap(args);
# define glXCreateWindow(args...)	std::cerr << GL_DEBUG << "glXCreateWindow("#args")\n"; glXCreateWindow(args);
# define glXDestroyPbuffer(args...)	std::cerr << GL_DEBUG << "glXDestroyPbuffer("#args")\n"; glXDestroyPbuffer(args);
# define glXDestroyPixmap(args...)	std::cerr << GL_DEBUG << "glXDestroyPixmap("#args")\n"; glXDestroyPixmap(args);
# define glXDestroyWindow(args...)	std::cerr << GL_DEBUG << "glXDestroyWindow("#args")\n"; glXDestroyWindow(args);
# define glXGetCurrentReadDrawable(args...)	std::cerr << GL_DEBUG << "glXGetCurrentReadDrawable("#args")\n"; glXGetCurrentReadDrawable(args);
# define glXGetFBConfigAttrib(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigAttrib("#args")\n"; glXGetFBConfigAttrib(args);
# define glXGetFBConfigs(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigs("#args")\n"; glXGetFBConfigs(args);
# define glXGetSelectedEvent(args...)	std::cerr << GL_DEBUG << "glXGetSelectedEvent("#args")\n"; glXGetSelectedEvent(args);
# define glXGetVisualFromFBConfig(args...)	std::cerr << GL_DEBUG << "glXGetVisualFromFBConfig("#args")\n"; glXGetVisualFromFBConfig(args);
# define glXMakeContextCurrent(args...)	std::cerr << GL_DEBUG << "glXMakeContextCurrent("#args")\n"; glXMakeContextCurrent(args);
# define glXQueryContext(args...)	std::cerr << GL_DEBUG << "glXQueryContext("#args")\n"; glXQueryContext(args);
# define glXQueryDrawable(args...)	std::cerr << GL_DEBUG << "glXQueryDrawable("#args")\n"; glXQueryDrawable(args);
# define glXSelectEvent(args...)	std::cerr << GL_DEBUG << "glXSelectEvent("#args")\n"; glXSelectEvent(args);
# define glXGetProcAddressARB(args...)	std::cerr << GL_DEBUG << "glXGetProcAddressARB("#args")\n"; glXGetProcAddressARB(args);
# define glXFreeContextEXT(args...)	std::cerr << GL_DEBUG << "glXFreeContextEXT("#args")\n"; glXFreeContextEXT(args);
# define glXGetContextIDEXT(args...)	std::cerr << GL_DEBUG << "glXGetContextIDEXT("#args")\n"; glXGetContextIDEXT(args);
# define glXGetCurrentDrawableEXT(args...)	std::cerr << GL_DEBUG << "glXGetCurrentDrawableEXT("#args")\n"; glXGetCurrentDrawableEXT(args);
# define glXImportContextEXT(args...)	std::cerr << GL_DEBUG << "glXImportContextEXT("#args")\n"; glXImportContextEXT(args);
# define glXQueryContextInfoEXT(args...)	std::cerr << GL_DEBUG << "glXQueryContextInfoEXT("#args")\n"; glXQueryContextInfoEXT(args);
# define glXAllocateMemoryNV(args...)	std::cerr << GL_DEBUG << "glXAllocateMemoryNV("#args")\n"; glXAllocateMemoryNV(args);
# define glXFreeMemoryNV(args...)	std::cerr << GL_DEBUG << "glXFreeMemoryNV("#args")\n"; glXFreeMemoryNV(args);
# define glXGetVideoSyncSGI(args...)	std::cerr << GL_DEBUG << "glXGetVideoSyncSGI("#args")\n"; glXGetVideoSyncSGI(args);
# define glXWaitVideoSyncSGI(args...)	std::cerr << GL_DEBUG << "glXWaitVideoSyncSGI("#args")\n"; glXWaitVideoSyncSGI(args);
# define glXGetRefreshRateSGI(args...)	std::cerr << GL_DEBUG << "glXGetRefreshRateSGI("#args")\n"; glXGetRefreshRateSGI(args);
# define glXSwapIntervalSGI(args...)	std::cerr << GL_DEBUG << "glXSwapIntervalSGI("#args")\n"; glXSwapIntervalSGI(args);
# define glXJoinSwapGroupNV(args...)	std::cerr << GL_DEBUG << "glXJoinSwapGroupNV("#args")\n"; glXJoinSwapGroupNV(args);
# define glXBindSwapBarrierNV(args...)	std::cerr << GL_DEBUG << "glXBindSwapBarrierNV("#args")\n"; glXBindSwapBarrierNV(args);
# define glXQuerySwapGroupNV(args...)	std::cerr << GL_DEBUG << "glXQuerySwapGroupNV("#args")\n"; glXQuerySwapGroupNV(args);
# define glXQueryMaxSwapGroupsNV(args...)	std::cerr << GL_DEBUG << "glXQueryMaxSwapGroupsNV("#args")\n"; glXQueryMaxSwapGroupsNV(args);
# define glXQueryFrameCountNV(args...)	std::cerr << GL_DEBUG << "glXQueryFrameCountNV("#args")\n"; glXQueryFrameCountNV(args);
# define glXResetFrameCountNV(args...)	std::cerr << GL_DEBUG << "glXResetFrameCountNV("#args")\n"; glXResetFrameCountNV(args);
# define glXGetFBConfigAttribSGIX(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigAttribSGIX("#args")\n"; glXGetFBConfigAttribSGIX(args);
# define glXChooseFBConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXChooseFBConfigSGIX("#args")\n"; glXChooseFBConfigSGIX(args);
# define glXCreateGLXPixmapWithConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateGLXPixmapWithConfigSGIX("#args")\n"; glXCreateGLXPixmapWithConfigSGIX(args);
# define glXCreateContextWithConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateContextWithConfigSGIX("#args")\n"; glXCreateContextWithConfigSGIX(args);
# define glXGetVisualFromFBConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXGetVisualFromFBConfigSGIX("#args")\n"; glXGetVisualFromFBConfigSGIX(args);
# define glXGetFBConfigFromVisualSGIX(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigFromVisualSGIX("#args")\n"; glXGetFBConfigFromVisualSGIX(args);
# define glXCreateGLXPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateGLXPbufferSGIX("#args")\n"; glXCreateGLXPbufferSGIX(args);
# define glXDestroyGLXPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXDestroyGLXPbufferSGIX("#args")\n"; glXDestroyGLXPbufferSGIX(args);
# define glXQueryGLXPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryGLXPbufferSGIX("#args")\n"; glXQueryGLXPbufferSGIX(args);
# define glXSelectEventSGIX(args...)	std::cerr << GL_DEBUG << "glXSelectEventSGIX("#args")\n"; glXSelectEventSGIX(args);
# define glXGetSelectedEventSGIX(args...)	std::cerr << GL_DEBUG << "glXGetSelectedEventSGIX("#args")\n"; glXGetSelectedEventSGIX(args);
//	}}} glx-Functions

//	{{{ glxext-Functions
# define glXGetFBConfigs(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigs("#args")\n"; glXGetFBConfigs(args);
# define glXChooseFBConfig(args...)	std::cerr << GL_DEBUG << "glXChooseFBConfig("#args")\n"; glXChooseFBConfig(args);
# define glXGetFBConfigAttrib(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigAttrib("#args")\n"; glXGetFBConfigAttrib(args);
# define glXGetVisualFromFBConfig(args...)	std::cerr << GL_DEBUG << "glXGetVisualFromFBConfig("#args")\n"; glXGetVisualFromFBConfig(args);
# define glXCreateWindow(args...)	std::cerr << GL_DEBUG << "glXCreateWindow("#args")\n"; glXCreateWindow(args);
# define glXDestroyWindow(args...)	std::cerr << GL_DEBUG << "glXDestroyWindow("#args")\n"; glXDestroyWindow(args);
# define glXCreatePixmap(args...)	std::cerr << GL_DEBUG << "glXCreatePixmap("#args")\n"; glXCreatePixmap(args);
# define glXDestroyPixmap(args...)	std::cerr << GL_DEBUG << "glXDestroyPixmap("#args")\n"; glXDestroyPixmap(args);
# define glXCreatePbuffer(args...)	std::cerr << GL_DEBUG << "glXCreatePbuffer("#args")\n"; glXCreatePbuffer(args);
# define glXDestroyPbuffer(args...)	std::cerr << GL_DEBUG << "glXDestroyPbuffer("#args")\n"; glXDestroyPbuffer(args);
# define glXQueryDrawable(args...)	std::cerr << GL_DEBUG << "glXQueryDrawable("#args")\n"; glXQueryDrawable(args);
# define glXCreateNewContext(args...)	std::cerr << GL_DEBUG << "glXCreateNewContext("#args")\n"; glXCreateNewContext(args);
# define glXMakeContextCurrent(args...)	std::cerr << GL_DEBUG << "glXMakeContextCurrent("#args")\n"; glXMakeContextCurrent(args);
# define glXGetCurrentReadDrawable(args...)	std::cerr << GL_DEBUG << "glXGetCurrentReadDrawable("#args")\n"; glXGetCurrentReadDrawable(args);
# define glXGetCurrentDisplay(args...)	std::cerr << GL_DEBUG << "glXGetCurrentDisplay("#args")\n"; glXGetCurrentDisplay(args);
# define glXQueryContext(args...)	std::cerr << GL_DEBUG << "glXQueryContext("#args")\n"; glXQueryContext(args);
# define glXSelectEvent(args...)	std::cerr << GL_DEBUG << "glXSelectEvent("#args")\n"; glXSelectEvent(args);
# define glXGetSelectedEvent(args...)	std::cerr << GL_DEBUG << "glXGetSelectedEvent("#args")\n"; glXGetSelectedEvent(args);
# define glXGetProcAddress(args...)	std::cerr << GL_DEBUG << "glXGetProcAddress("#args")\n"; glXGetProcAddress(args);
# define glXGetProcAddressARB(args...)	std::cerr << GL_DEBUG << "glXGetProcAddressARB("#args")\n"; glXGetProcAddressARB(args);
# define glXSwapIntervalSGI(args...)	std::cerr << GL_DEBUG << "glXSwapIntervalSGI("#args")\n"; glXSwapIntervalSGI(args);
# define glXGetVideoSyncSGI(args...)	std::cerr << GL_DEBUG << "glXGetVideoSyncSGI("#args")\n"; glXGetVideoSyncSGI(args);
# define glXWaitVideoSyncSGI(args...)	std::cerr << GL_DEBUG << "glXWaitVideoSyncSGI("#args")\n"; glXWaitVideoSyncSGI(args);
# define glXMakeCurrentReadSGI(args...)	std::cerr << GL_DEBUG << "glXMakeCurrentReadSGI("#args")\n"; glXMakeCurrentReadSGI(args);
# define glXGetCurrentReadDrawableSGI(args...)	std::cerr << GL_DEBUG << "glXGetCurrentReadDrawableSGI("#args")\n"; glXGetCurrentReadDrawableSGI(args);
# define glXCreateGLXVideoSourceSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateGLXVideoSourceSGIX("#args")\n"; glXCreateGLXVideoSourceSGIX(args);
# define glXDestroyGLXVideoSourceSGIX(args...)	std::cerr << GL_DEBUG << "glXDestroyGLXVideoSourceSGIX("#args")\n"; glXDestroyGLXVideoSourceSGIX(args);
# define glXGetCurrentDisplayEXT(args...)	std::cerr << GL_DEBUG << "glXGetCurrentDisplayEXT("#args")\n"; glXGetCurrentDisplayEXT(args);
# define glXQueryContextInfoEXT(args...)	std::cerr << GL_DEBUG << "glXQueryContextInfoEXT("#args")\n"; glXQueryContextInfoEXT(args);
# define glXGetContextIDEXT(args...)	std::cerr << GL_DEBUG << "glXGetContextIDEXT("#args")\n"; glXGetContextIDEXT(args);
# define glXImportContextEXT(args...)	std::cerr << GL_DEBUG << "glXImportContextEXT("#args")\n"; glXImportContextEXT(args);
# define glXFreeContextEXT(args...)	std::cerr << GL_DEBUG << "glXFreeContextEXT("#args")\n"; glXFreeContextEXT(args);
# define glXGetFBConfigAttribSGIX(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigAttribSGIX("#args")\n"; glXGetFBConfigAttribSGIX(args);
# define glXChooseFBConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXChooseFBConfigSGIX("#args")\n"; glXChooseFBConfigSGIX(args);
# define glXCreateGLXPixmapWithConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateGLXPixmapWithConfigSGIX("#args")\n"; glXCreateGLXPixmapWithConfigSGIX(args);
# define glXCreateContextWithConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateContextWithConfigSGIX("#args")\n"; glXCreateContextWithConfigSGIX(args);
# define glXGetVisualFromFBConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXGetVisualFromFBConfigSGIX("#args")\n"; glXGetVisualFromFBConfigSGIX(args);
# define glXGetFBConfigFromVisualSGIX(args...)	std::cerr << GL_DEBUG << "glXGetFBConfigFromVisualSGIX("#args")\n"; glXGetFBConfigFromVisualSGIX(args);
# define glXCreateGLXPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXCreateGLXPbufferSGIX("#args")\n"; glXCreateGLXPbufferSGIX(args);
# define glXDestroyGLXPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXDestroyGLXPbufferSGIX("#args")\n"; glXDestroyGLXPbufferSGIX(args);
# define glXQueryGLXPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryGLXPbufferSGIX("#args")\n"; glXQueryGLXPbufferSGIX(args);
# define glXSelectEventSGIX(args...)	std::cerr << GL_DEBUG << "glXSelectEventSGIX("#args")\n"; glXSelectEventSGIX(args);
# define glXGetSelectedEventSGIX(args...)	std::cerr << GL_DEBUG << "glXGetSelectedEventSGIX("#args")\n"; glXGetSelectedEventSGIX(args);
# define glXCushionSGI(args...)	std::cerr << GL_DEBUG << "glXCushionSGI("#args")\n"; glXCushionSGI(args);
# define glXBindChannelToWindowSGIX(args...)	std::cerr << GL_DEBUG << "glXBindChannelToWindowSGIX("#args")\n"; glXBindChannelToWindowSGIX(args);
# define glXChannelRectSGIX(args...)	std::cerr << GL_DEBUG << "glXChannelRectSGIX("#args")\n"; glXChannelRectSGIX(args);
# define glXQueryChannelRectSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryChannelRectSGIX("#args")\n"; glXQueryChannelRectSGIX(args);
# define glXQueryChannelDeltasSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryChannelDeltasSGIX("#args")\n"; glXQueryChannelDeltasSGIX(args);
# define glXChannelRectSyncSGIX(args...)	std::cerr << GL_DEBUG << "glXChannelRectSyncSGIX("#args")\n"; glXChannelRectSyncSGIX(args);
# define glXAssociateDMPbufferSGIX(args...)	std::cerr << GL_DEBUG << "glXAssociateDMPbufferSGIX("#args")\n"; glXAssociateDMPbufferSGIX(args);
# define glXJoinSwapGroupSGIX(args...)	std::cerr << GL_DEBUG << "glXJoinSwapGroupSGIX("#args")\n"; glXJoinSwapGroupSGIX(args);
# define glXBindSwapBarrierSGIX(args...)	std::cerr << GL_DEBUG << "glXBindSwapBarrierSGIX("#args")\n"; glXBindSwapBarrierSGIX(args);
# define glXQueryMaxSwapBarriersSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryMaxSwapBarriersSGIX("#args")\n"; glXQueryMaxSwapBarriersSGIX(args);
# define glXGetTransparentIndexSUN(args...)	std::cerr << GL_DEBUG << "glXGetTransparentIndexSUN("#args")\n"; glXGetTransparentIndexSUN(args);
# define glXCopySubBufferMESA(args...)	std::cerr << GL_DEBUG << "glXCopySubBufferMESA("#args")\n"; glXCopySubBufferMESA(args);
# define glXCreateGLXPixmapMESA(args...)	std::cerr << GL_DEBUG << "glXCreateGLXPixmapMESA("#args")\n"; glXCreateGLXPixmapMESA(args);
# define glXReleaseBuffersMESA(args...)	std::cerr << GL_DEBUG << "glXReleaseBuffersMESA("#args")\n"; glXReleaseBuffersMESA(args);
# define glXSet3DfxModeMESA(args...)	std::cerr << GL_DEBUG << "glXSet3DfxModeMESA("#args")\n"; glXSet3DfxModeMESA(args);
# define glXGetSyncValuesOML(args...)	std::cerr << GL_DEBUG << "glXGetSyncValuesOML("#args")\n"; glXGetSyncValuesOML(args);
# define glXGetMscRateOML(args...)	std::cerr << GL_DEBUG << "glXGetMscRateOML("#args")\n"; glXGetMscRateOML(args);
# define glXSwapBuffersMscOML(args...)	std::cerr << GL_DEBUG << "glXSwapBuffersMscOML("#args")\n"; glXSwapBuffersMscOML(args);
# define glXWaitForMscOML(args...)	std::cerr << GL_DEBUG << "glXWaitForMscOML("#args")\n"; glXWaitForMscOML(args);
# define glXWaitForSbcOML(args...)	std::cerr << GL_DEBUG << "glXWaitForSbcOML("#args")\n"; glXWaitForSbcOML(args);
# define glXQueryHyperpipeNetworkSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryHyperpipeNetworkSGIX("#args")\n"; glXQueryHyperpipeNetworkSGIX(args);
# define glXHyperpipeConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXHyperpipeConfigSGIX("#args")\n"; glXHyperpipeConfigSGIX(args);
# define glXQueryHyperpipeConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryHyperpipeConfigSGIX("#args")\n"; glXQueryHyperpipeConfigSGIX(args);
# define glXDestroyHyperpipeConfigSGIX(args...)	std::cerr << GL_DEBUG << "glXDestroyHyperpipeConfigSGIX("#args")\n"; glXDestroyHyperpipeConfigSGIX(args);
# define glXBindHyperpipeSGIX(args...)	std::cerr << GL_DEBUG << "glXBindHyperpipeSGIX("#args")\n"; glXBindHyperpipeSGIX(args);
# define glXQueryHyperpipeBestAttribSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryHyperpipeBestAttribSGIX("#args")\n"; glXQueryHyperpipeBestAttribSGIX(args);
# define glXHyperpipeAttribSGIX(args...)	std::cerr << GL_DEBUG << "glXHyperpipeAttribSGIX("#args")\n"; glXHyperpipeAttribSGIX(args);
# define glXQueryHyperpipeAttribSGIX(args...)	std::cerr << GL_DEBUG << "glXQueryHyperpipeAttribSGIX("#args")\n"; glXQueryHyperpipeAttribSGIX(args);
# define glXGetAGPOffsetMESA(args...)	std::cerr << GL_DEBUG << "glXGetAGPOffsetMESA("#args")\n"; glXGetAGPOffsetMESA(args);
//	}}} glxext-Functions

#else

# define DEBUGGL(arg)

#endif
// }}}

// {{{ DEBUG_MODE <= 99
#if DEBUG_MODE <= 99
# define TODO(msg)	std::cerr << "TODO!!|" << DEBUG_LOCATION << msg
#else
# define TODO(msg)
#endif
// }}}


#endif /* }}} __DEBUG_H__ */


// vim:fdm=marker:fdc=3:tw=100
