/**
 * @file	MarchingCubesData.h
 *
 * Data structures and classes for data management.
 *
 * This file collects all classes and structures to manage the marching cubes engines data. For more
 * information see \ref MCoverview.
 *
 * @author	Markus Trenkwalder
 * @date	06. Sep 2004, 17:31:24
 * @version	0.0.1
 *
 * $Id: MarchingCubesData.h,v 1.17 2004/11/24 14:29:55 maxx Exp $
 *
 */


#ifndef __MARCHINGCUBESDATA_H__	// {{{
#define __MARCHINGCUBESDATA_H__


#include <GL/gl.h>
#include <list>
#include <map>
#include <vector>

#include "vuVector.h"
#include "MarchingCubesGeneral.h"
#include "MCGlobalData.h"
#include "MCBProgressCallback.h"


// {{{ Page: Marching Cubes Data
/**
 * \page MCdata Marching Cubes Data
 *
 * <h1>Overview</h1>
 *
 * The purpose of the data structure classes is to first provides different views of the data for
 * the application itself and OpenGL on the other side. It also provides additional information
 * about the vertex connectivity such as vertex neighbourhood and triangles. The data structure is
 * designed to be accessed one way which means that elements can be added to the structure but not
 * removed. This makes the memory management simpler and the structure faster. The following image
 * tries to give an impression of the data structures and it's application.
 *
 * \image html data.png "Data Classes Overview"
 *
 * The image also shows the class hierarchy (which is the same as in \ref MCoverview) for easyer
 * understanding. It shows an extract of a mesh and the corresponding data. What you can see is how
 * nodes and triangles are related. Every element of the data structure can be accessed only through
 * \c mesh. This is neccessary because the marching cubes algorithm must look up vertices in a
 * special way. The dashed connections between the vertices and the data arrray \c _data and between
 * the data array and the index array \c _indices shoud be read as \e is \e related \e to.
 *
 * <h1>Adding Data</h1>
 *
 * As told earlier data in the data structure can only be accessed via MCMesh. This class provides
 * all necessary functions to add and access the data. The normal way to add a triangle of vertices
 * to the structure is to first add a new triangle to the structure. The function to accomplish this
 * returns the index of the triangle in an array.
 *
 * The next step in data creation is to add vertices. In this step the triangle index is needed.
 * Also the position of the corresponding cube and the number of the edge on which the vertex lies
 * are needed to add the vertex to the structure. The structure itself takes care of the
 * connectivity information (see below).
 *
 * <h1>Drawing the mesh</h1>
 *
 * The idea behind all this data classes is to provide the data so that it could be used by \ref
 * glDrawElements. This function uses several data arrays (i.e. vertex data, normal data, ...) and
 * takes a index array that specifies in which order the function should use the vertex in the
 * arrays. In this way all the data can put onto the graphics card at once and OpenGL than accesses
 * these data much faster. This goal is achiefed by splitting the data creation into two phases. The
 * first phase is the above described mesh creation. The second phase compiles the spezial arrays
 * from the mesh.
 *
 * <h1>Thoughts on connectivity</h1>
 *
 * One kind of connectivity information has to be provided by the user of the data structure: the
 * triangle information. The other kind of connectivity information, the vertex neighbours, will be
 * created by the data structure.
 *
 * This information can be accomplished trough two facts. First all vertices of a triangle are
 * neighbours and second neighbouring triangles share the same vertices. With these two informations
 * all the vertex neighbours information can be compiled. Imagine the first triangle in the
 * structure. Every vertex added to the triangle is automatically provided with the neighbourhood
 * information in the triangle (i.e. each other vertex in the triangle is added to the vertexe's
 * neighbours list). When all triangles are added the neigbourhood information is complete for this
 * triangle.
 *
 * The data structure is designed to easily reuse already existing vertices. And this is the point
 * where overall connectivity can be provided. Because a vertex will be used for all neighbouring
 * triangles the triangle connectivitiy information of all triangles will be collected in this
 * single vertex and a complete mesh arrises.
 *
 * With these connectivity informations also the triangle neighbourhood could be calculated, but
 * this is complecated and not implemented in this first version.
 *
 * <h1>Internal data representation</h1>
 *
 * As described below, \ref glDrawElements needs the data array along with a index array. The data
 * array holds all vertices in the mesh, but each vertex is contained only once. This is the reason
 * why an index is needed. In order to make it possible, to color each type of cube different, this
 * index array in turn is split into 255 index arrays, each representing on type of cube.
 *
 * <h2>Generating the arrays</h2>
 *
 * The vertex and index array is created and filled when needed, i.e. when drawing starts. At that
 * point the mesh tree is walked through and the relevant information is compiled into the arrays.
 *
 * This step is also split into two phases because of the above described structure of the index
 * array. At the start of compilation, MCMesh does not know, how big each of this index arrays will
 * be, and maybe some will even be empty. Therefor the first step only compiles the data into the
 * vertex array (along with the normals) but also counts the occurence of the different types. This
 * is possible because the algorithm has to walk through the mesh in the same way as it should be
 * drawn because it must calculate the correct normals (this is described below in \ref
 * calcNormals).
 *
 * The second step than can reserve exactly the necessary memory for each index array and enter the
 * data into it.
 *
 * An important issues is the fact, that only complete triangles are compiled into the arrays.
 *
 * <h1>glDrawElements \anchor glDrawElements</h1>
 *
 * glDrawElements uses arrays defined before the function is called. To define an array OpenGL must
 * first be pointed to the array data. For each type of array another function has to be used.
 * Secondly the array has to be activated through the glEnableClientState function with a function
 * parameter also depending on the type of the array. The mapping between array types, functions,
 * and function parameters is shown in th following table.
 *
 * \code
 * data type                | definition        | glEnableClientState
 *                          | function          | parameter
 * -------------------------+-------------------+-----------------------
 * vertex array             | glVertexPointer   | GL_VERTEX_ARRAY
 * normal array             | glNormalPointer   | GL_NORMAL_ARRAY
 * color array              | glColorPointer    | GL_COLOR_ARRAY
 * texture coordinate array | glTexCoordPointer | GL_TEXTURE_COORD_ARRAY
 * \endcode
 *
 * After the declaration and the enabling of the arrays glDrawElements can make use of them. For
 * this reason the function takes an input parameter representing an index array which in turn
 * specifies in which order the data in the arrays should be used.
 *
 * <h1>Calculating Normals \anchor calcNormals</h1>
 *
 * MCMesh does not take the triangle normals as it would be the easiest way, but calculates the
 * average normal of each neighbouring triangle of a vertex. This is done by adding the normal of
 * the new triangle to the existing one and make the normal a unit vector again afterwards. In this
 * way the survace can be smoothed a little.
 *
 * <h1>Coloring</h1>
 *
 * As mentioned above the application is able to color the triangles of each cube type differently.
 * This is done by creating 256 index arrays and drawing the mesh 256 times with different arrays.
 * For each drawing step another color can be used.
 *
 * The actual application supports only one different color because the coloring is connected with
 * the cube browser which by deffinition can show only one type of cube.
 */
// }}}


/**
 * The complete mesh. This class represents the entry point to the geometry data. For more
 * informations see \ref MCData.
 *
 * <b>Destroying objects</b>
 *
 * Many of the objects are not only refereces in the base class but also in many classes in the
 * hierarchy. Such objects must not be destroyed in this classes but in the base class.
 *
 * \bug		When drawing the normals is enabled during cube browsing and the user than switches
 *		over to object drawing, the nromals are only if drawing the normals is turned of and
 *		on again.
 * \todo	Do vertex buffer objects.
 */
class MCMesh // {{{
{

	public:

		/**
		 * Type of the index in \ref _vertices. If you want to use the results of AddVertex
		 * or GetVertex you should use this type.
		 */
		typedef unsigned int index_t;
		/**
		 * This defines the return value, in case of an error inside this classes.
		 * \attention	If \ref index_t is changed, this value may also be changed.
		 */
#define MCD_ERROR		(0xffffffff)

	private:

//! This value is the default for MCData::_extendSize.
#define MCM_DATA_EXTEND_DEFAULT	1000

//! Sets a bit in an integer. This macro should be used to set flags in a flag variable.
#define MCM_SET_BIT(var, bit)	var |= (1<<(bit))

//! Tests if a bit is set. The result should be a boolean.
#define MCM_BIT_SET(var, bit)	( ((var)&(1<<(bit))) == 1 )

//! Unsets a bit in an integer.
#define MCM_UNSET_BIT(var, bit)	var &= ( 0xffffffff ^ (1<<bit) );

		/**
		 * Specifies the data structure accapted by glInterleavedArrays. The structure
		 * corresponds with the type GL_N3F_V3F.
		 */
		struct MCData { // {{{
			float normal[3];
			float vertex[3];

			inline void copyNormalFrom(const float *n);
			inline void copyVertexFrom(const float *n);
		}; // }}}
		/**
		 * This simple struct is intended to make life easier in the normals array.
		 */
		struct MCNormal {
			float normal[3];
		};

		/**
		 * Represents one vertex in the net.
		 */
		class MCVertex // {{{
		{

			private:

				/**
				 * The index of the vertex data in the vertex array.
				 * \attention	This class assumes that the index is the same for
				 *		_vertex and _normal!
				 */
				index_t _idx;		
				//! True if a index is assigned with the vertex.
				bool _indexed;
				/**
				 * Reference to the vertex data in MCData. Must not be destroyed
				 * because it only references an array.
				 */
				vuVector _vertex;
				/**
				 * List of all neighbouring vertices. The elements in this list must
				 * not be destroyed, because they only reference elements saved in
				 * the base class if the data structure.
				 */
				std::list<MCVertex *> _neighbours;

				//! Default constructor should not be used.
				MCVertex() {}

			public:

				//! This is the only way to create an object of this type.
				MCVertex(int idx);
				/**
				 * Destructor does not destroy _neighbours because this is done by
				 * MCMesh.
				 */
				virtual ~MCVertex();

				//! Adds the vertex \a v to the \ref _neighbours list.
				void AddNeigbour(MCVertex *v);
				//! Sets the vertex position to \a v.
				void Vertex(vuVector &v);
				//! Return the vertex as vuVector.
				vuVector& Vertex();

				//! Sets the index of the vertex in the data array.
				void Idx(index_t idx);
				//! Returns the index of the vertex in the data array.
				index_t Idx();
				//! Returns true, if a index has been assigned with the vertex.
				bool Indexed();
				//! Restets the indexing.
				void resetIndexing();

		}; // }}} MCVertex

		/**
		 * A single triangle in the mesh.
		 */
		class MCTriangle // {{{
		{

			private:

				/**
				 * Indicates the type of cube the triangle belongs to.
				 */
				int _type;
#define MCT_COMPETE_FLAG 1
				/**
				 * The member vertices of the triangle. A triangle has only 3
				 * vertices, therefore a fixed size array is the best choice. The
				 * elements in the array must not be destroyed, because they are
				 * only references.
				 */
				MCVertex *_vertices[3];
				/**
				 * Fill level of the _vertices array. This counter will also be
				 * used by the Complete function.
				 */
				unsigned short _vertCnt;
				/**
				 * A list of all neighbours. The neighbours need not be accessed
				 * directly but sequentially, this is why I use STLs single linked
				 * list.
				 */
				std::list<MCTriangle *> _neighbours;

			public:

				//! Default constructor
				MCTriangle(int type);
				//! Destructor
				virtual ~MCTriangle();

				/**
				 * Adds the vertex \a v to its list of member vertices. If this list
				 * is already complete, the function returns -1. If the addition was
				 * successfull the function returns the index of the vertex in the
				 * member list (i.e. 0-2).
				 *
				 * \attention	The neighbourhood information will only be written
				 *		when all vertices are added to the triangle. In
				 *		other words in an incomplete triangle there is on
				 *		neighbourhood informations for this triangle.
				 *
				 * \param v	The vertex to add to the triangle.
				 *
				 * \return	If the vertex could be added, the index in the
				 *		internal list and otherwise -1 is returned.
				 */
				int AddVertex(MCVertex *v);
				/**
				 * Returns the \a n -th vertex in the triangle.
				 */
				MCVertex* GetVertex(int n);
				/**
				 * Calculates the normal for the n-th vertex in the triangle.
				 */
				vuVector GetNormal();
				/**
				 * Indication the completion of the triangle.
				 */
				bool Complete();

				MCMesh::index_t GetVertexIdx(int v);

				/**
				 * Returns the type of this triangle.
				 */
				int Type();

		}; // }}} MCTriangle

	// {{{ Private Member Variables:
	private:

		/**
		 * A spezial edge mapping table for this data structure. Because this class only
		 * needs to traverse through all possible neighbour cubes this is a simple 12/3/4
		 * list. The data for each edge is ordered counter clockwise if you look in the
		 * positive direction of the coordiante axis.
		 */
		static const int		_edgeMapping[12][3][4];
		/**
		 * This array holds the data passed to the graphics card.
		 */
		MCData				*_data;
		//! The size of the data array.
		index_t				_dataSize;
		/**
		 * This indexes will be passed to \ref glDrawElements.
		 */
		index_t				*_indices[256];
		//! The size of the index array.
		index_t				_indexSize[256];
		//MCData				*_normalsArray;
		float				*_normalsArray;
		index_t				_normalsArraySize;
		/**
		 * Each vertex in the mesh is referenced here four times. One for each neighbour
		 * cube of the edge. The hash for this map corresponds with the index of the edge in
		 * the volume. The index is composed of a 4 bit number for the edge and a number of
		 * 9 bits for each direction. This leads to a 31 bit number. The ordering of the
		 * numbers in the index is as follows (bit display):
		 *
		 * \code
		 *	0zzzzzzz zzyyyyyy yyyxxxxx xxxxeeee B
		 * \endcode
		 *
		 * Where \c z stands for the \e z-direction, \c y for the \e y-direction, \c x for
		 * the \e x-direction and \c e for the \e edge.
		 */
		std::map<index_t, MCVertex *>	_vertices;
		/**
		 * For easyer access to all real vertices in \ref _vertices this list also holds
		 * the vertices (see also \ref MCdata).
		 */
		std::list<MCVertex *>		_vertexList;
		//! Holds all triangles in the mesh. 
		std::vector<MCTriangle *>	_triangles;
		//! Indicates that the normals are calcualted.
		bool				_normalsPrepared;
		//! The number of bytes occupied by a MCVertex object in memory.
		index_t				_vertexSize;
		//! The number of bytes occupied by a MCTriangle object in memory.
		index_t				_triangleSize;
		//! Data structure holding data used by several classes.
		MCGlobalData			*_gd;
		//! A callback to update a progress bar in the GUI.
		MCBProgressCallback		*_progCB;

	// }}} Private Member Variables

 	// {{{ Private Member Functions:
	private:

#define MCD_X_SHIFT		(4)
#define MCD_Y_SHIFT		(9+4)
#define MCD_Z_SHIFT		(9+9+4)
#define MCD_EDGE_MASK		(0xF)
				// 1111 binary
#define MCD_DIR_MASK		(0x1FF)
				// 111111111 binary
#define MCD_X_MASK		(MCD_DIR_MASK<<MCD_X_SHIFT)
#define MCD_Y_MASK		(MCD_DIR_MASK<<MCD_Y_SHIFT)
#define MCD_Z_MASK		(MCD_DIR_MASK<<MCD_Z_SHIFT)
#define MCD_GET_EDGE(idx)	((idx)&MCD_EDGE_MASK)
#define	MCD_GET_X(idx)		(((idx)&MCD_X_MASK)>>MCD_X_SHIFT)
#define	MCD_GET_Y(idx)		(((idx)&MCD_Y_MASK)>>MCD_Y_SHIFT)
#define	MCD_GET_Z(idx)		(((idx)&MCD_Z_MASK)>>MCD_Z_SHIFT)

		/**
		 * Calculates the index of a vertex in \ref _vertices. For informations on the
		 * composition of the index see \ref _vertices.
		 *
		 * \param x		x-position of the cube in the volume.
		 * \param y		y-position of the cube in the volume.
		 * \param z		z-position of the cube in the volume.
		 * \param edge		Number of the edge, at which the vertex lies.
		 *
		 * \return		The index of the vertex in \ref _vertices.
		 */
		inline index_t _getIndex(int x, int y, int z, int edge);
		/**
		 * Takes the data from the mesh (MCTriangle and MCVertex) and compiles an array
		 * which is passed to the graphics card.
		 *
		 * \bug	Some normals are wrong, but which. Maybe we should use more asserts.
		 */
		void _compileData();
		/**
		 * Generates a vertex array and an index array which holds data for drawing the
		 * normals.
		 */
		void _compileNormalsArray();

	// }}} Private Member Functions

	// {{{ Public Member Functions:
	public:	

		/**
		 * Default constructor. The constructor also disables OpenGL array data.
		 * \param ves	Vertices extend size (see MCData::_extendSize).
		 * \param ies	Indices extend size (see MCData::_extendSize).
		 */
		MCMesh(MCGlobalData *gd, MCBProgressCallback *cb = 0);
		//! Destructor
		virtual ~MCMesh();

		/**
		 * Add a vertex at the specified position. With \a x, \a y, \a z and \a edge the
		 * position of a vertex in the grid is completly defined. With these parameters a
		 * uniqe index in the map \ref _vertices is calculated. This function also adds the
		 * newly created MCVertex to the three corresonding positions in the map (see also
		 * \ref MCedgeMapping).
		 *
		 * <b>Mode of Operation</b>
		 *
		 * The function first looks, if the vertex is already added to \ref _vertices by
		 * looking it up at the index, given by the parameters. If it exists a neighbouring
		 * cube of the given edge has added the vertex. In this case the vertex is only
		 * added to the triangle given by \a triangle. Otherwise the vertex is added to all
		 * four positions in \ref _vertices according to \ref MCedgeMapping and after this
		 * added to the triangle.
		 * 
		 * \param x		x-position of the cube in the volume.
		 * \param y		y-position of the cube in the volume.
		 * \param z		z-position of the cube in the volume.
		 * \param edge		Number of the edge, at which the vertex lies.
		 * \param triangle	Adds the vertex to the triangle with this index.
		 * \param type		The type of the cube, in which the vertex lies.
		 * \param v		Holds the coordinates of the vertex. These will be added to
		 *			MCData.
		 *
		 * \return		The index of the added vertex or \ref MCD_ERROR otherwise.
		 */
		index_t AddVertex(int x, int y, int z, int edge, int triangle, vuVector& v);

		/**
		 * Adds a already existing vertex to the triangle given by \a triangle. In contrast
		 * to AddVertex(int, int, int, int, int, vuVector &) this function only adds a
		 * already existing vertex to a triangle. If the vertex does not exist, the function
		 * fails and returns \ref MCD_ERROR.
		 *
		 * \param idx		The index of the vertex in \ref _vertices.
		 * \param triangle	The triangle index in \ref _triangles.
		 *
		 * \return		The index of the vertex or \ref MCD_ERROR if the vertex does
		 *			not exist.
		 */
		index_t AddVertex(int idx, int triangle);

		/**
		 * Determines if the vertex already exists in \ref _vertices.
		 *
		 * \param x		x-position of the cube in the volume.
		 * \param y		y-position of the cube in the volume.
		 * \param z		z-position of the cube in the volume.
		 * \param edge		Number of the edge, at which the vertex lies.
		 *
		 * \return		If the vertex is found, the index of the vertex in
		 * 			\ref _vertices is returned. Otherwise the function returns
		 * 			\ref MCD_ERROR.
		 */
		index_t GetVertex(int x, int y, int z, int edge);

		/**
		 * Adds a new triangle to \ref _triangles and returns the index in the array.
		 *
		 * \return	The index of the new triangle in \ref _triangles.
		 */
		int AddTriangle(int type);

		/**
		 * Prints the actual triangle buffer. This function is for debug use only. If
		 * debugging is turned off, this function does print nothing.
		 */
		void DebugPrint(void);

		/**
		 * Does last preparations on the data before the rendering can start. These
		 * preparations are:
		 * \li compiling the vertex-, normals- and color-array and
		 * \li compiling the array for drawing the normals.
		 * 
		 * This function may be called after the data in MCMesh is complete. If it is not
		 * called, graphics output may not be what you want because normals will not be
		 * calcualted.
		 */
		void prepare(void);

		/**
		 * Renders the triangles. This function actually calls \ref glDrawElements.
		 * Unfortunatly this is another drawing function (the fourth) on the stack but as
		 * far as this class holds the data it is teh best way to do the rendering.
		 */
		void render();

		/**
		 * Turn on and off the normals rendering. If \a mode is set to \b true, there will
		 * be drawn a line for each normal using _drawNormalsSize as their size. The start
		 * point of such a line is the position of the corresponding vertex, and the end
		 * point is \c normal*_drawNormalsSize away from this point.
		 */
		void drawNormals(bool mode);

		/**
		 * Returns the actual value of the rendering mode for the normals lines (see
		 * drawNormals(bool mode) for more information).
		 */
		bool drawNormals();

		/**
		 * Sets the size of the line drawn for each normal (see drawNormals(bool mode) for
		 * more information).
		 */
		void drawNormalsSize(float size);

		/**
		 * Returns the size of the line drawn for each normal (see drawNormals(bool mode)
		 * for more information).
		 */
		float drawNormalsSize();

		//! Returns the actual number of triangles in the data structure.
		index_t numTriangles();
		//! Returns the number of vertices in the data structure.
		index_t numVertices();
		//! Returns the number of elements in the indices array.
		index_t numIndices();

	// }}} Public Member Functions:


}; // }}} MCMesh


#endif /* }}} __MARCHINGCUBESDATA_H__ */


// vim:fdm=marker:fdc=4:tw=100
