/**
 * @file	MarchingCubesData.cpp
 *
 * Holds the implementations of the marching cubes data classes.
 *
 * @author	Markus Trenkwalder
 * @date	14. Sep 2004, 18:38:02
 * @version	0.0.1
 *
 * $id: $
 *
 */


#include "MarchingCubesData.h"

// Actual Level: 6
#define DEBUG_MODE 6
//#define DEBUG_GL
//#define DEBUG_CONST
//#define DEBUG_DEST
#include "General/vuMarchingCubes/debug.h"

#include "MarchingCubesGeneral.h"


// {{{ struct MCMech::MCData

inline void MCMesh::MCData::copyNormalFrom(const float *n) // {{{
{
	DEBUG0("inline void MCMesh::MCData::copyNormalFrom(float *n)\n");
	memcpy((void*)normal, (void*)n, sizeof(float)*3);
} // }}} inline void MCMesh::MCData::copyNormalFrom(float *n)

inline void MCMesh::MCData::copyVertexFrom(const float *v) // {{{
{
	DEBUG0("inline void MCMesh::MCData::copyVertexFrom(float *v)\n");
	memcpy((void*)vertex, (void*)v, sizeof(float)*3);
} // }}} inline void MCMesh::MCData::copyVertexFrom(float *v)

// }}} MCMech::MCData


// {{{ class MCMesh::MCVertex

MCMesh::MCVertex::MCVertex(int idx) // {{{
: _idx(idx),
  _indexed(false)
{
	DEBUGC("MCMesh::MCVertex::MCVertex(int idx="<<idx<<")\n");
} // }}} MCMesh::MCVertex::MCVertex(int idx)

MCMesh::MCVertex::~MCVertex() // {{{
{
	DEBUGD("virtual MCMesh::MCVertex::~MCVertex()\n");
} // }}} MCMesh::MCVertex::~MCVertex()

/// \todo Should this function check the existence of the vertex in the array?
void MCMesh::MCVertex::AddNeigbour(MCVertex *v) // {{{
{
	DEBUG0("void MCMesh::MCVertex::AddNeigbour(MCVertex *v)\n");
	assert(v);	// We do not want to push back a NULL pointer :)
	_neighbours.push_back(v);
} // }}} void MCMesh::MCVertex::AddNeigbour(MCVertex *v)

void MCMesh::MCVertex::Vertex(vuVector &v) // {{{
{
	DEBUG0("void MCMesh::MCVertex::Vertex(vuVector &v)\n");
	_vertex = v;
} // }}} void MCMesh::MCVertex::Vertex(vuVector &v)

vuVector& MCMesh::MCVertex::Vertex() // {{{
{
	DEBUG0("vuVector MCMesh::MCVertex::Vertex()\n");
	return _vertex;
} // }}} vuVector MCMesh::MCVertex::Vertex()

void MCMesh::MCVertex::Idx(MCMesh::index_t idx) // {{{
{
	DEBUG0("void MCMesh::MCVertex::Idx(MCMesh::index_t idx)\n");
	_idx = idx;
	_indexed = true;
} // }}} void MCMesh::MCVertex::Idx(MCMesh::index_t idx)

MCMesh::index_t MCMesh::MCVertex::Idx() // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::MCVertex::Idx()\n");
	return _idx;
} // }}} MCMesh::index_t MCMesh::MCVertex::Idx()

bool MCMesh::MCVertex::Indexed() // {{{
{
	DEBUG0("bool MCMesh::MCVertex::Indexed()\n");
	return _indexed;
} // }}} bool MCMesh::MCVertex::Indexed()

void MCMesh::MCVertex::resetIndexing() // {{{
{
	DEBUG0("void MCMesh::MCVertex::resetIndexing()\n");
	_indexed = false;
} // }}} void MCMesh::MCVertex::resetIndexing()

// }}} MCMesh::MCVertex


// {{{ class MCMesh::MCTriangle

MCMesh::MCTriangle::MCTriangle(int type) // {{{
: _type(type),
  _vertCnt(0)
{
	DEBUGC("MCMesh::MCTriangle::MCTriangle()\n");
	_vertices[0] = _vertices[1] = _vertices[2] = 0;
} // }}} MCMesh::MCTriangle::MCTriangle()

MCMesh::MCTriangle::~MCTriangle() // {{{
{
	DEBUGD("virtual MCMesh::MCTriangle::~MCTriangle()\n");
} // }}} MCMesh::MCTriangle::~MCTriangle()

int MCMesh::MCTriangle::AddVertex(MCVertex *v) // {{{
{
	DEBUG0("int MCMesh::MCTriangle::AddVertex(MCVertex *v)\n");
	if (3==_vertCnt) return -1;
	_vertices[_vertCnt] = v;
	//DEBUG5("Added vertex "<<i<<"\n");
	if (2 == _vertCnt) {
		//! The triangle is complete so it is time to adjust the neighbourhood.
		_vertices[0]->AddNeigbour(_vertices[1]);
		_vertices[0]->AddNeigbour(_vertices[2]);
		_vertices[1]->AddNeigbour(_vertices[0]);
		_vertices[1]->AddNeigbour(_vertices[2]);
		_vertices[2]->AddNeigbour(_vertices[0]);
		_vertices[2]->AddNeigbour(_vertices[1]);
	}
	return _vertCnt++;
} // }}} int MCMesh::MCTriangle::AddVertex(MCVertex *v)

MCMesh::MCVertex* MCMesh::MCTriangle::GetVertex(int n) // {{{
{
	DEBUG0("MCMesh::MCVertex* MCMesh::MCTriangle::GetVertex(int n)\n");
	assert(n >= 0 && n < 3);
	return _vertices[n];
} // }}} MCMesh::MCVertex* MCMesh::MCTriangle::GetVertex(int n)

//! \todo This may be subject to massive optimisation.
vuVector MCMesh::MCTriangle::GetNormal() // {{{
{
	DEBUG0("vuVector MCMesh::MCTriangle::GetNormal(int n)\n");
	assert(Complete());
	vuVector v0 = _vertices[1]->Vertex() - _vertices[0]->Vertex();
	vuVector v1 = _vertices[1]->Vertex() - _vertices[2]->Vertex();
	v0.makeUnit(); v1.makeUnit();
	//return v0.cross(v1).makeUnit();
	return v1.cross(v0).makeUnit();
} // }}} vuVector MCMesh::MCTriangle::GetNormal()

bool MCMesh::MCTriangle::Complete() // {{{
{
	DEBUG0("bool MCMesh::MCTriangle::Complete()\n");
	return _vertCnt == 3;
} // }}} bool MCMesh::MCTriangle::Complete()

MCMesh::index_t MCMesh::MCTriangle::GetVertexIdx(int v) // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::MCTriangle::GetVertexIdx(int v)\n");
	assert(v < 3);
	assert(_vertices[v] != 0);
	return _vertices[v]->Idx();
} // }}} MCMesh::index_t MCMesh::MCTriangle::GetVertexIdx(int v)

int MCMesh::MCTriangle::Type() // {{{
{
	DEBUG0("int MCMesh::MCTriangle::Type()\n");
	return _type;
} // }}} int MCMesh::MCTriangle::Type()

// }}} MCMesh::MCTriangle


// {{{ class MCMesh

const int MCMesh::_edgeMapping[12][3][4] = { // {{{
	{ //  0:   x   y   z  edge	   {{{
		{  0,  0, -1,  4 },
		{  0, -1, -1,  6 },
		{  0, -1,  0,  2 }
	},				// }}}
	{ //  1:   x   y   z  edge	   {{{
		{  0,  0, -1,  5 },
		{  1,  0, -1,  7 },
		{  1,  0,  0,  3 }
	},				// }}}
	{ //  2:   x   y   z  edge	   {{{
		{  0,  1,  0,  0 },
		{  0,  1, -1,  4 },
		{  0,  0, -1,  6 }
	},				// }}}
	{ //  3:   x   y   z  edge	   {{{
		{ -1,  0,  0,  1 },
		{ -1,  0, -1,  5 },
		{  0,  0, -1,  7 }
	},				// }}}
	{ //  4:   x   y   z  edge	   {{{
		{  0, -1,  0,  6 },
		{  0, -1,  1,  2 },
		{  0,  0,  1,  0 }
	},				// }}}
	{ //  5:   x   y   z  edge	   {{{
		{  1,  0,  0,  7 },
		{  1,  0,  1,  3 },
		{  0,  0,  1,  1 }
	},				// }}}
	{ //  6:   x   y   z  edge	   {{{
		{  0,  0,  1,  2 },
		{  0,  1,  1,  0 },
		{  0,  1,  0,  4 }
	},				// }}}
	{ //  7:   x   y   z  edge	   {{{
		{  0,  0,  1,  3 },
		{ -1,  0,  1,  1 },
		{ -1,  0,  0,  5 }
	},				// }}}
	{ //  8:   x   y   z  edge	   {{{
		{  0, -1,  0, 11 },
		{ -1, -1,  0, 10 },
		{ -1,  0,  0,  9 }
	},				// }}}
	{ //  9:   x   y   z  edge	   {{{
		{  1,  0,  0,  8 },
		{  1, -1,  0, 11 },
		{  0, -1,  0, 10 }
	},				// }}}
	{ // 10:   x   y   z  edge	   {{{
		{  0,  1,  0,  9 },
		{  1,  1,  0,  8 },
		{  1,  0,  0, 11 }
	},				// }}}
	{ // 11:   x   y   z  edge	   {{{
		{ -1,  0,  0, 10 },
		{ -1,  1,  0,  9 },
		{  0,  1,  0,  8 }
	}				// }}}
}; // }}}

MCMesh::MCMesh(MCGlobalData *gd, MCBProgressCallback *cb) // {{{
: _data(0),
  _dataSize(0),
  _normalsArray(0),
  _normalsArraySize(0),
  _normalsPrepared(false),
  _gd(gd),
  _progCB(cb)
{
	DEBUGC("MCMesh::MCMesh(MCGlobalData *gd, MCBProgressCallback *cb)\n");
	assert(gd != NULL);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	_gd->compileNormals = true;
	for(int i = 0; i < 256; ++i) {
		_indices[i] = 0;
		_indexSize[i] = 0;
	}
	_vertexSize = sizeof(MCVertex) + 2*sizeof(MCVertex*);
	_triangleSize = sizeof(MCTriangle) + sizeof(MCTriangle*);
	_gd->meshSize = 0;
} // }}} MCMesh::MCMesh(MCGlobalData *gd, MCBProgressCallback *cb)

MCMesh::~MCMesh() // {{{
{
	DEBUGD("virtual MCMesh::~MCMesh()\n");
	for(std::list<MCVertex *>::iterator i = _vertexList.begin(); i!=_vertexList.end(); ++i)
			delete *i;
	for(unsigned int i = 0; i < _triangles.size(); ++i) delete _triangles[i];
	if (_data) delete _data;
	for(int i = 0; i < 256; ++i) {
		if (_indices[i]) delete _indices[i];
	}
	if (_normalsArray) delete _normalsArray;
} // }}} MCMesh::~MCMesh()

MCMesh::index_t MCMesh::_getIndex(int x, int y, int z, int edge) // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::_getIndex(int x="<<x<<", int y="
	       <<y<<", int z="<<z<<", int edge="<<edge<<")\n");
	return (x<<MCD_X_SHIFT) + (y<<MCD_Y_SHIFT) + (z<<MCD_Z_SHIFT) + edge;
} // }}} MCMesh::index_t MCMesh::_getIndex(int x, int y, int z, int edge)

void MCMesh::_compileData() // {{{
{
	DEBUG0("void MCMesh::_compileData()\n");
	_dataSize = numVertices();
	if (_progCB) {
		_progCB->SetText("Compiling scene data:");
		_progCB->Start();
	}
	//! First of all we need to clean up the results from a prviouse compilation if there are
	//! some.
	try {
		if (_data) delete _data;
		for(int i = 0; i < 256; ++i) {
			if (_indices[i]) delete _indices[i];
			_indices[i] = 0;
			_indexSize[i] = 0;
		}
		for(std::list<MCVertex *>::iterator i = _vertexList.begin();
		    i!=_vertexList.end(); ++i)
			(*i)->resetIndexing();
	} catch(...) {
		MC_FAIL_EXIT("Error while cleaning up during the data compilation.", -1);
	}
	//! Next we need to create the data array.
	index_t maxIndexSize = numTriangles()*3;	// The extreme value.
	DEBUG5("_dataSize = "<<_dataSize<<", _indexSize = "<<_indexSize<<"\n");
	try {
		_data = new MCData[_dataSize];
		if (_gd->doMeshStatistics) _gd->meshSize += sizeof(MCData)*_dataSize;
		DEBUG5("Allocated "<<_dataSize*sizeof(MCData)
		       <<" Bytes (MCData: "<<sizeof(MCData)<<")\n");
	} catch (...) {
		MC_FAIL_EXIT("could not allocate "<<_dataSize<<" data elements ("
			     <<_dataSize*sizeof(MCData)<<")", -1);
	}
	//! The rest happens in two steps.
	//! The first step is to add the vertices to the array and to assign the indices to the
	//! vertex objects. This is also the position where smoothing should happen.
	int update = _triangles.size()/50;
	update = (update>0) ? update : 1;
	DEBUG5("update = "<<update<<"\n");
	index_t triSize = _triangles.size();
	index_t vertCnt = 0;
	DEBUG5("cubeSize: "<<_gd->cubeSize[0]<<", "<<_gd->cubeSize[1]<<", "<<_gd->cubeSize[1]<<"\n");
	for(unsigned int t = 0; t < triSize; ++t) {
		if (!_triangles[t]->Complete()) continue;
		DEBUG5("Adding triangle "<<t<<"\n");
		vuVector n = _triangles[t]->GetNormal();
		int type = _triangles[t]->Type();
		for(int v = 0; v < 3; ++v) {
			MCVertex* vert = _triangles[t]->GetVertex(v);
			vuVector theNormal;
			index_t theIdx;
			if (!vert->Indexed()) {
				_data[vertCnt].vertex[0] = vert->Vertex()[0]*_gd->cubeSize[0];
				_data[vertCnt].vertex[1] = vert->Vertex()[1]*_gd->cubeSize[1];
				_data[vertCnt].vertex[2] = vert->Vertex()[2]*_gd->cubeSize[2];
				//_data[vertCnt].copyVertexFrom(vert->Vertex().getData());
				DEBUG5("vertex: "<<_data[vertCnt].vertex[0]
				       <<", "<<_data[vertCnt].vertex[1]
				       <<", "<<_data[vertCnt].vertex[3]<<"\n");
				theNormal = vuVector(0.0, 0.0, 0.0);
				vert->Idx(vertCnt);
				theIdx = vertCnt;
				++vertCnt;		// <-- vertCnt
			} else {
				theIdx = vert->Idx();
				theNormal = vuVector(_data[theIdx].normal[0],
						     _data[theIdx].normal[1],
						     _data[theIdx].normal[2]);
			}
			theNormal += n;
			theNormal.makeUnit();
			/*
			_data[theIdx].normal[0] = theNormal[0];
			_data[theIdx].normal[1] = theNormal[1];
			_data[theIdx].normal[2] = theNormal[2];
			*/
			_data[theIdx].copyNormalFrom(theNormal.getData());
			++_indexSize[type];		// <-- indexSize
			assert(vertCnt <= _dataSize);
			DEBUG4("_indexSize[type]: "<<_indexSize[type]
			       <<", maxIndexSize: "<<maxIndexSize<<"\n");
			assert(_indexSize[type] <= maxIndexSize);
		}
		DEBUG4("_progCB: "<<(void*)_progCB<<"\n");
		if (!(t%update) && _progCB) {
			DEBUG4("progress: "<<((t*100/triSize)>>1)<<"\n");
			_progCB->Update((t*100/triSize)>>1);
			if (_gd->termProgress)
				std::cerr << ((float)t/(float)triSize*100.0)
					<< "% added to array.\n";
		}
	}
	if (_progCB) _progCB->Update(50);
	index_t idxCnt[255];
	for(unsigned int t = 0; t < triSize; ++t) {
		if (!_triangles[t]->Complete()) continue;
		int type = _triangles[t]->Type();
		DEBUG4("calc indices: t = "<<t<<", type = "<<type<<"\n");
		for(int v = 0; v < 3; ++v) {
			if (!_indices[type]) {
				idxCnt[type] = 0;
				try {
					DEBUG5("Creating index array for type "<<type
					       <<", size: "<<_indexSize[type]<<"\n");
					_indices[type] = new index_t[_indexSize[type]];
					if (_gd->doMeshStatistics)
						_gd->meshSize += sizeof(index_t)*_indexSize[type];
				} catch (...) {
					MC_FAIL_EXIT("could not allocate "<<maxIndexSize
						     <<" index elements for type "<<type<<" ("
						     <<_dataSize*sizeof(MCData)<<")", -1);
				}
			}
			_indices[type][idxCnt[type]++] =
				_triangles[t]->GetVertex(v)->Idx();	// <-- indexSize
			DEBUG4("idxCnt["<<type<<"] = "<<idxCnt[type]<<"\n");
			assert(idxCnt[type] <= _indexSize[type]);
		}
		if (!(t%update) && _progCB) {
			DEBUG5("progress: "<<(50+((t*100/triSize)>>1))<<"\n");
			_progCB->Update(50+((t*100/triSize)>>1));
			if (_gd->termProgress)
				std::cerr << ((float)t/(float)triSize*100.0) << "% indexed.\n";
		}
	}
	if (_progCB) _progCB->End();
	DEBUG5("meshSize = "<<_gd->meshSize<<"\n");
	_gd->compileData = false;
} // }}} void MCMesh::_compileData()

void MCMesh::_compileNormalsArray() // {{{
{
	DEBUG0("void MCMesh::_compileNormalsArray()\n");
	if (_normalsArray) delete _normalsArray;
	_normalsArraySize = _dataSize*2;
	try {
		DEBUG3("normals array size: "<<_normalsArraySize*3*sizeof(float)<<"\n");
		_normalsArray = new float[_normalsArraySize*3];
	} catch (...) {
		MC_FAIL_EXIT("could not allocate "<<_normalsArraySize
			     <<" elements for the normals array ("
			     <<_normalsArraySize*3*sizeof(float)<<")", -1);
	}
	index_t cnt = 0;
	for(index_t i = 0; i < _dataSize; ++i) {
		vuVector v = vuVector(_data[i].vertex);
		vuVector n = vuVector(_data[i].normal);
		_normalsArray[cnt*3]   = _data[i].vertex[0];
		_normalsArray[cnt*3+1] = _data[i].vertex[1];
		_normalsArray[cnt*3+2] = _data[i].vertex[2];
		++cnt;
		DEBUG1("normalsLength = "<<_gd->normalsLength<<"\n");
		vuVector v2 = v+n*_gd->normalsLength;
		_normalsArray[cnt*3]   = v2[0];
		_normalsArray[cnt*3+1] = v2[1];
		_normalsArray[cnt*3+2] = v2[2];
		++cnt;
		assert(cnt <= _normalsArraySize);
	}
	_gd->compileNormals = false;
	DEBUG3("Normals are compiled!!!\n");
} // }}} void MCMesh::_compileNormalsArray()

MCMesh::index_t MCMesh::AddVertex(int x, int y, int z, int edge, int triangle, vuVector& v) // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::AddVertex(int x, int y, int z, "
	       <<"int edge, int triangle, vuVector& v)\n");
	index_t idx = _getIndex(x, y, z, edge);
	if (_vertices.find(idx) == _vertices.end()) {
		MCVertex *theVert = new MCVertex(-1);
		if (_gd->doMeshStatistics) _gd->meshSize += _vertexSize;
		_vertexList.push_back(theVert);
		_vertices[idx] = theVert;
		theVert->Vertex(v);
		DEBUG1(">>> added index: "<<idx<<"\n");
		//! After that we add it to the three corresponding position.
		for(int i = 0; i < 3; ++i) {
			//! We retreave this position from _edgeMapping.
			int otherX = x+_edgeMapping[edge][i][0];
			int otherY = y+_edgeMapping[edge][i][1];
			int otherZ = z+_edgeMapping[edge][i][2];
			int otherEdge = _edgeMapping[edge][i][3];
			DEBUG1("other: "<<otherX<<", "<<otherY<<", "<<otherZ<<", "<<otherEdge<<"\n");
			if (otherX<0 || otherY<0 || otherZ<0) continue;	// there is no index < 0.
			index_t otherIdx = _getIndex(otherX, otherY, otherZ, otherEdge);
			/**
			 * \todo	Remove the following assert if you are sure that all is
			 *		okay. I assume this to be very slow.
			 */
			assert(_vertices.find(otherIdx) == _vertices.end());
			//! And finally assign the vertex to this position.
			_vertices[otherIdx] = theVert;
			DEBUG1(">>>> added index: "<<otherIdx<<".\n");
		}
	}
	//! Finally we have to add the vertex to the triangle.
	return AddVertex(idx, triangle);
} // }}} MCMesh::index_t MCMesh::AddVertex(int x, int y, int z, int edge, int triangle, vuVector& v)

MCMesh::index_t MCMesh::AddVertex(int idx, int triangle) // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::AddVertex(int idx, int triangle)\n");
	_normalsPrepared = false;
	assert(triangle != -1);		// This is realy bad :)
	if (_vertices.find(idx) == _vertices.end()) return MCD_ERROR;	// Vertex does not exist.
	if (_triangles[triangle]->AddVertex(_vertices[idx]) == -1) return MCD_ERROR; // Error while
									// adding vertex to triangle.
	return idx;
} // }}} MCMesh::index_t MCMesh::AddVertex(int idx, int triangle)

MCMesh::index_t MCMesh::GetVertex(int x, int y, int z, int edge) // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::GetVertex(int x, int y, int z, int edge)\n");
	int idx = _getIndex(x, y, z, edge);
	if (_vertices.find(idx) != _vertices.end())
		return idx;
	return MCD_ERROR;
} // }}} MCMesh::index_t MCMesh::GetVertex(int x, int y, int z, int edge)

//! \todo Does the last index of a element in _triangles correspond with the size of the vector?
int MCMesh::AddTriangle(int type) // {{{
{
	DEBUG0("int MCMesh::AddTriangle(int type)\n");
	_normalsPrepared = false;
	_triangles.push_back(new MCTriangle(type));
	if (_gd->doMeshStatistics) _gd->meshSize += _triangleSize;
	return _triangles.size() - 1;
} // }}} int MCMesh::AddTriangle(int type)

void MCMesh::DebugPrint(void) // {{{
{
	DEBUG0("void MCMesh::DebugPrint(void)\n");
#if DEBUG_MODE < 99
	/*
	for(int i=0; i<_indices.GetSize(); ++i) {
		if (i%3 == 0) {
			cerr << DEBUG_LOCATION << "Triange:\n";
		}
		int idx = *_indices[i];
		cerr << DEBUG_LOCATION << "\t(" << _vertexData[idx][0] << ", "
			<< _vertexData[idx][1] << ", " << _vertexData[idx][2] << ")\n";
	}
	*/
#endif
} // }}} void MCMesh::DebugPrint(void)

void MCMesh::prepare(void) // {{{
{
	DEBUG0("void MCMesh::prepare(void)\n");
	if (!_data || _gd->compileData) _compileData();
	DEBUG3("drawNormals = "<<_gd->drawNormals<<", compileNormals = "<<_gd->compileNormals<<"\n");
	if (_gd->drawNormals && _gd->compileNormals) _compileNormalsArray();
} // }}} void MCMesh::prepare(void)

void MCMesh::render() // {{{
{
	DEBUG0("void MCMesh::render()\n");
	prepare();
	GLboolean oldCullMode;
	glGetBooleanv(GL_CULL_FACE, &oldCullMode);
	GLint oldFaceMode;
	glGetIntegerv(GL_CULL_FACE_MODE, &oldFaceMode);

	glInterleavedArrays(GL_N3F_V3F, 0, (void*)_data);
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);

	for(int i = 1; i < 255; ++i) {
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, _gd->objectColor[i].getData());
		glDrawElements(GL_TRIANGLES, _indexSize[i], GL_UNSIGNED_INT, (void*)_indices[i]);
	}

	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, _gd->lineColor.getData());
	if (_gd->linesMode) {
		GLboolean enabled;
		glGetBooleanv(GL_CULL_FACE, &enabled);
		GLint mode;
		glGetIntegerv(GL_CULL_FACE_MODE, &mode);
		if (_gd->linesMode == GL_FRONT_AND_BACK) {
			glDisable(GL_CULL_FACE);
		} else if (_gd->linesMode == GL_BACK) {
			glCullFace(GL_FRONT);
			glEnable(GL_CULL_FACE);
		} else if (_gd->linesMode == GL_FRONT) {
			glCullFace(GL_BACK);
			glEnable(GL_CULL_FACE);
		}
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, _gd->lineColor.getData());
		glPolygonMode(_gd->linesMode, GL_LINE);
		for(int i = 1; i < 255; ++i) {
			glDrawElements(GL_TRIANGLES, _indexSize[i],
				       GL_UNSIGNED_INT, (void*)_indices[i]);
		}
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		if (!mode) {
			glDisable(GL_CULL_FACE);
		} else {
			glEnable(GL_CULL_FACE);
		}
		glCullFace(mode);
	}
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);

	if (_gd->drawNormals) {
		DEBUG3("Rendering Normals ...\n");
		glVertexPointer(3, GL_FLOAT, 0, (void*)_normalsArray);
		glEnableClientState(GL_VERTEX_ARRAY);
		glDrawArrays(GL_LINES, 0, _normalsArraySize);
		glDisableClientState(GL_VERTEX_ARRAY);
		DEBUG3("rendering done!!!\n");
	}

	if (oldCullMode) {
		glDisable(GL_CULL_FACE);
	} else {
		glEnable(GL_CULL_FACE);
	}
	glCullFace(oldFaceMode);
} // }}} void MCMesh::render()

MCMesh::index_t MCMesh::numTriangles() // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::numTriangles()\n");
	return _triangles.size();
} // }}} MCMesh::index_t MCMesh::numTriangles()

MCMesh::index_t MCMesh::numVertices() // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::numVertices()\n");
	//return _vertices.size();
	return _vertexList.size();
} // }}} MCMesh::index_t MCMesh::numVertices()

MCMesh::index_t MCMesh::numIndices() // {{{
{
	DEBUG0("MCMesh::index_t MCMesh::numIndices()\n");
	index_t cnt = 0;
	for(int i = 0; i < 255; ++i) {
		cnt += _indexSize[i];
	}
	return cnt;
} // }}} MCMesh::index_t MCMesh::numIndices()

// }}} MCMesh


// vim:fdm=marker:fdc=3:tw=100
