/**
 * @file	MarchingCubesBase.h
 *
 * Declaration of the MarchingCubesBase class.
 *
 * @author	Markus Trenkwalder
 * @date	26. Jun 2004, 11:05:12
 * @version	0.0.1
 *
 * $Id: MarchingCubesBase.h,v 1.20 2004/11/23 04:15:27 maxx Exp $
 *
 */


#ifndef __MARCHINGCUBESBASE_H__	// {{{ #endif
#define __MARCHINGCUBESBASE_H__


#include <GL/gl.h>
#include <GL/glu.h>
#define GLUT_DISABLE_ATEXIT_HACK
#include <GL/glut.h>

#include "General/vuLinAlg/vuVector.h"
#include "MarchingCubesData.h"
#include "MCBProgressCallback.h"


/**
 * Number of different cubes the MC algorithm knows of.
 */
#define MC_NUMCUBES	256


/**
 * When reading data from files, use this value as buffer size.
 */
#define MC_DEFAULT_BUFFER_SIZE 1000


/**
 * Base class of the Marching Cubes Model (see vuMarchingCubes). This class does the actual scanning
 * of the data.
 *
 * <strong>Geometry</strong>
 *
 * The variable \ref _vertOffset defines the cube geometry which (according to the actual values in
 * \ref _vertOffset) is:
 *
 * \code
 *        (3)-----2-----(2)
 *        /|            /|
 *      11 |          10 |
 *      /  |          /  |
 *    (7)-----6-----(6)  1
 *     |   3         |   |
 *     |   |         |   |
 *     |   |         |   |
 *     7  (0)-----0--|--(1)    y
 *     |  /          5  /      |
 *     | 8           | 9       o-- x
 *     |/            |/       /
 *    (4)-----4-----(5)      z
 * \endcode
 *
 * The above picture also shows the edge numbering. All code in this class referes to this geometry.
 *
 * <strong>Cube Type</strong>
 *
 * The type of a cube is calculated by testing every edge vertex of the cube against the threshold
 * saved in \ref _thres. A value that is higher means the vertex is inside. Each edge is associated
 * with a bit in the type integer where the position of the bit corresponds with the nomber of the
 * edge in the above given geometry.
 */
class MarchingCubesBase // {{{
{

	private:

		int _size[3];		//!< Data size.
		float _origin1;		//!< Origin in first direction
		float _origin2;		//!< Origin in second direction
		float _origin3;		//!< Origin in third direction

#define MCB_IDX(x, y, z)	( ((z)*_size[0]*_size[1]) + ((y)*_size[0]) + (x) )

	protected:

		int* _data;		/**<
					 * Hold's the volume data. This member
					 * is protected, because it should be
					 * accessable from derived classes.
					 */

	private:

		MCGlobalData *_gd;	/**<
					 * A reference to the global data structure. This need not
					 * be destroyed.
					 */
		bool _valueRangeAdjusted;
					/**<
					 * Range calculation state machine. \c False in the
					 * beginning and \c true after calculation.
					 */

		//! As described in the class documentation, this table defines the cube geometry.
		static const GLfloat _vertOffset[8][3];
		/**
		 * This is the marching cubes table. It connects the cube type with a number of
		 * edges. Each edge on it's part represents a vertex of a triangle. Also this table
		 * assumes, that a set bit in the type means, that the corrseponding edge is outside
		 * the volume.
		 */
		static const int     _marchingCubesTable[256][16];
		//! Describes the connection between the edges and it's end points in the cube.
		static const int     _edgeConnect[12][2];
		//! Defines which \ref MCGlobalData::cubeSize value is relevant for each edge.
		static const int     _edgeSizeConnect[12];
		/**
		 * This mapping table connectes edges of neighbouring cubes (see \ref MCoverview
		 * and \ref MCedgeMapping).
		 */
		static const int     _edgeMapping[12][3][3][3];
		static const GLfloat _edgeDir[12][3];
		static const int     _edgeFlags[256];

		MCBProgressCallback  *_progCB;
		MCBHistoCallback     *_histoCB;

		MCMesh               *_mesh;

		int                  _theCube;
		MCMesh               *_cube;

	public:

		//! Default constructor
		MarchingCubesBase();
		//! Destructor
		virtual ~MarchingCubesBase();

		//! This has to be done first!
		void setGlobalData(MCGlobalData *gd) { _gd = gd; }

		/**
		 * Renders the frame that borders the volume.
		 * \param width line width of the frame.
		 */
		void glRenderFrame();
		//! This function executes the code to render the marching
		//! cubes volume.
		void glRender();

		/**
		 * Size setter. This is the point where the \ref _data array is created. This
		 * function also does some other settings. In other words, when this function has
		 * not been called, the class will not work correctly.
		 */
		void Size(int sz1, int sz2, int sz3);
		//! Size 1 getter.
		int Size1() { return _size[0]; }
		//! Size 2 getter.
		int Size2() { return _size[1]; }
		//! Size 3 getter.
		int Size3() { return _size[2]; }
		//! Returns the bigges size.
		int maxSize();
		//! Origin setter.
		void Origin(float or1, float or2, float or3);
		float Origin1() { return _origin1; }
		float Origin2() { return _origin2; }
		float Origin3() { return _origin3; }

		/**
		 * This function calculates the triangle mesh by marching trough the cubes. As
		 * described in the class documentation the function visits every cube, calculates
		 * it's type and then determines the part of the mesh introduced by this cube. For
		 * more information see \ref MCoverview.
		 */
		void marchCubes();

		long long numVertices();
		long long numTriangles();
		long long numIndices();

		/**
		 * The function determines the type of the cube. According to the values in
		 * \ref _vertOffset each vertex in the cube gets the value of true if it is inside
		 * the volume defined by the threshold. Otherwise it is false. Each vertex is
		 * associated with a bit in the returned integer and this integer represents the
		 * cube type.
		 */
		int getCubeType(int x, int y, int z);

		/**
		 * Takes a point inside the mesh and returns the value at that point. The
		 * calculation of the value is done using
		 * <a href="http://astronomy.swin.edu.au/~pbourke/analysis/trilinear/">trilinear
		 * interpolation</a>.
		 */
		float valueAt(float px, float py, float pz);

	private:

		//! Draws a horizontal cylinder.
		void drawHorCylinder(int pos, float angle, float r, float l);
		//! Draw a vertical cylinder.
		void drawVertCylinder(int pos, float r, float l);
		//! Draws a ball at the coordinate p.
		void drawCorner(int pos, float r);

		void drawSpot(int pos);
		void drawTriangles();

		/**
		 * Calculates the vertex position from the cube position (\a x, \a y, \a z), the
		 * edge number (\a edge) and the threshold (\ref _thres).
		 */
		inline vuVector calcVertex(int x, int y, int z, int edge);

	public:

		void SetProgressCallback(MCBProgressCallback* cb);
		void SetHistoCallback(MCBHistoCallback* cb);

	protected:

		void SetProgressText(const char* txt);
		void StartProgress();
		void UpdateProgress(int v);
		void EndProgress();

}; /* }}} MarchingCubesBase */


#endif /* }}} __MARCHINGCUBESBASE_H__ */


// vim:fdm=marker:fdc=3:fdl=99:tw=100
