/* Written by Steven Kilthau */

#ifndef _VECTOR_H_
#define _VECTOR_H_

#include <stdlib.h>
#include <string.h>
#include <iostream.h>

#include "vuSimpleTypes.h"
#include "vuString.h"

class vuVector; //forward declaration.
typedef vuVector* vuVectorPointer;
typedef vuVectorPointer* vuVectorPointerArray;

class vuMatrix;

//!A class for representing 3d vectors.
/*!
  The vuVector class is a 3D homogeneous vector
  that works in conjunction with the vuMatrix
  class (a 4x4 matrix).  The elements of the
  vuVector are all single precision floating-
  point numbers.
*/

class vuVector
{
friend class vuMatrix;
public:
    //!Default constructor making a zero vector.
    vuVector();
    //!Copy constructor.
    vuVector(const vuVector& v);
    //!Constructor initializing to a 3d vector in homogenous space.
    vuVector(float v1, float v2, float v3);
    //!Constructor initializing to a homogenous 3d vector.
    vuVector(float v1, float v2, float v3, float v4);
    //!Constructor initializing the x,y,z coordinates to v.
    vuVector(float v);
    //!Constructor initializing the vector from an array of floats.
    /*!\param v An array of 3 floating point values corresponding to x,y,z.
    */
    vuVector(const float* v);
    //!The destructor.
    ~vuVector();

    //!The assignment operator.
    vuVector& operator=(const vuVector& v);
    //!Assigns the value v to the x,y,z elements of the vector.
    vuVector& operator=(float v);
    //!Assigns a 3d float array to the vector.
    vuVector& operator=(const float* v);

    //!Computes the 2-norm on the vector (lenght operator).
    float norm(void) const;
    //!Computes squared 2-norm (scalar product with itself).
    float norm2(void) const;

    //!Normalizes the vector to a length of 1.
    /*!This makes a unit vector which points in the same direction as the instance.
    */
    vuVector& makeUnit(void);
    //!Normalizes the vector in homogenous space.
    /*!Divides the whole vector by the 4th coordinate to make it 1.
    */
    vuVector& normalize(void);

    //!Returns the entry-wise inverse of the vector.
    /*!Entry wise meaning that each of the x, y, z coordinates is treated separately,
       leaving the fourth w coordinate untouched.
    */
    vuVector inv(void) const;
    //!Inverts the vector entry-wise.
    vuVector& invEq(void);
    //!Returns the entry-wise product of the vector with another vector.
    vuVector mul(const vuVector& rhs) const;
    //!Multiplies the vector, entry-wise, by another vector.
    vuVector& mulEq(const vuVector& rhs);
    //!Returns the entry-wise division of the vector by another vector.
    vuVector div(const vuVector& rhs) const;
    //!Divides the vector, entry-wise, by another vector.
    vuVector& divEq(const vuVector& rhs);

    //!The access operator.
    float& operator[](dword index);
    //!The const access operator.
    const float& operator[](dword index) const;

    //!Returns the data pointer.
    /*!The data is stored as a four dimensional float array.
    	The getData() function can both retrieve and modify; it
     	is similar to the [] operator. To warrant the return of
      a const pointer (for use in other const member functions)
      explicitly cast the pointer returned to (const).
    */
    float* getData(void);
    //! const version of getData()
    float const* getData(void) const;
    
    //!Returns the dot product of two vectors.
    friend float dot(const vuVector& v1,const vuVector& v2);
    //!Returns the cross product of two vectors.
    friend vuVector cross(const vuVector& v1,const vuVector& v2);
    //!Returns the dot product of the vector with another vector.
    float dot(const vuVector& v) const;
    //!Returns the cross product of the vector with another vector.
    vuVector cross(const vuVector& v) const;

    //!Returns the sum of the vector with another vector.
    vuVector operator+(const vuVector& v) const;
    //!Returns the difference of the vector from another vector.
    vuVector operator-(const vuVector& v) const;
    //!Returns the product of the vector and a matrix.
    vuVector operator*(const vuMatrix& m) const;
    //!Returns the product of the vector with another vector.
    vuMatrix operator*(const vuVector& v) const;
    //!Returns the product of the vector with a scalar.
    vuVector operator*(float s) const;
    //!Returns the fraction of the vector with a scalar
    vuVector vuVector::operator/(float s) const;
    //!Returns the product of a scalar with a vector.
    friend const vuVector operator*(float s, const vuVector& v);
    //!Adds a vector to this vector.
    vuVector& operator+=(const vuVector& v);
    //!Subtracts a vector from this vector.
    vuVector& operator-=(const vuVector& v);
    //!Multiplies the vector by a matrix.
    vuVector& operator*=(const vuMatrix& m);
    //!Multiplies the vector by a scalar.
    vuVector& operator*=(float s);

    //!An equality operator, comparing only the x,y,z coordinates.
    bool operator==(const vuVector& v) const;
    //!An inequality operator, comparing only the x,y,z coordinates.
    bool operator!=(const vuVector& v) const;

    //!Returns index of component with highest absolute
    int getDominantAxis() const;

    //!prints values to the stdout
    void print();

    //! this will load the values referenced by load_from
    /*! the values will come in as ascii values and will have to be converted
    	to floats.  they will be received in the format;
		x, y, z, w \0
	the last position will be returned.
	*/
    int load (char* load_from);
    //! this will set read_to to be the four ascii values in x, y, z, w
    /*! this will use the same format as load */
    void save (char* save_to);

    //! writes the foure elements to a stream; separated by whitespaces
    friend ostream& operator<<(ostream& out,const vuVector& v);
    //! reads a vector from a stream
    friend istream& operator>>(istream& in, vuVector& v);

    //!returns a c-string representation without '\n'
    vuString getString();
    
private:
    //!The coordinates of the vector stored as a floating point array.
    float val[4];
};

//! This will return the position of the first comma in the string
int get_next_comma (char* temp);
//! this will return the position of the first ' ' or '\t' in the string
int get_next_blank (char* temp);
//! this will return the position of the first non ' ' and non '\t' character in the string
int clear_blanks (char* temp);
//! This will return the position of the first return in te string.
int get_next_return (char* temp);


#endif
