#include <math.h>
#include <stdio.h>
#include "vuVector.h"
#include "vuMatrix.h"

// Default constructor - set initial values
vuVector::vuVector()
{
    val[0] = val[1] = val[2] = 0.0f;
    val[3] = 1.0f;
}

// Copy constructor
vuVector::vuVector(const vuVector& v)
{
    val[0] = v.val[0];
    val[1] = v.val[1];
    val[2] = v.val[2];
    val[3] = v.val[3];
}

// Constructor - set to <v1,v2,v3,1.0>
vuVector::vuVector(float v1, float v2, float v3)
{
    val[0] = v1;
    val[1] = v2;
    val[2] = v3;
    val[3] = 1.0f;
}

// Constructor - set to <v1,v2,v3,v4>
vuVector::vuVector(float v1, float v2, float v3, float v4)
{
    val[0] = v1;
    val[1] = v2;
    val[2] = v3;
    val[3] = v4;
}

// Constructor - set all values to v
vuVector::vuVector(float v)
{
    val[0] = val[1] = val[2] = v;
    val[3] = 1.0f;
}

// Constroctor - get values from an array
vuVector::vuVector(const float* v)
{
    val[0] = v[0];
    val[1] = v[1];
    val[2] = v[2];
    val[3] = 1.0f;
}

// Destructor
vuVector::~vuVector()
{
}

// Assignment operator
vuVector& vuVector::operator=(const vuVector& v)
{
    if (this != &v)
    {
        val[0] = v.val[0];
        val[1] = v.val[1];
        val[2] = v.val[2];
        val[3] = v.val[3];
    }
    return *this;
}

// Assignment operator - set all values to v
vuVector& vuVector::operator=(float v)
{
    val[0] = val[1] = val[2] = v;
    val[3] = 1.0f;
    return *this;
}

// Assignment operator - get values from an array
vuVector& vuVector::operator=(const float* v)
{
    val[0] = v[0];
    val[1] = v[1];
    val[2] = v[2];
    val[3] = 1.0f;
    return *this;
}

// Access operator - return the value at the given index
float& vuVector::operator[](dword index)
{
    return val[index];
}

// Const Access operator - return the value at the given index
const float& vuVector::operator[](dword index) const
{
    return val[index];
}


/*!The data is stored as a four dimensional float array.
  The getData() function can both retrieve and modify; it
  is similar to the [] operator. To warrant the return of
  a const pointer (for use in other const member functions)
  explicitly cast the pointer returned to (const).
*/
float* vuVector::getData(void)
{ return val; }

float const* vuVector::getData(void) const
{ return val; }

// Compute the norm of the vector  - |v|
float vuVector::norm(void) const
{
    return (float)sqrt((double)(val[0]*val[0]+val[1]*val[1]+val[2]*val[2]));
}

// Compute the square norm of the vector
float vuVector::norm2(void) const
{
    return val[0]*val[0]+val[1]*val[1]+val[2]*val[2];
}

// Normalize the vector - set its length to 1
vuVector& vuVector::makeUnit(void)
{
    float d = (float)sqrt((double)(val[0]*val[0]+val[1]*val[1]+val[2]*val[2]));
    if (d)
    {
        val[0] /= d;
        val[1] /= d;
        val[2] /= d;
    }
    else
    {
        val[0] = val[1] = val[2] = 0.0f;
    }
    return *this;
}

vuVector& vuVector::normalize(void)
{
    float d;
    d = (val[3])?(1.0f/val[3]):0.0f;
    val[0] *= d;
    val[1] *= d;
    val[2] *= d;
    val[3] = 1.0f;
    return *this;
}

// Invert the vector - divide all entries into 1
vuVector vuVector::inv(void) const
{
    vuVector r;
    r[0] = (val[0])?(1.0f/val[0]):0.0f;
    r[1] = (val[1])?(1.0f/val[1]):0.0f;
    r[2] = (val[2])?(1.0f/val[2]):0.0f;
    return r;
}

// Invert and assign - divide all entries into 1
// for current vector
vuVector& vuVector::invEq(void)
{
    val[0] = (val[0])?(1.0f/val[0]):0.0f;
    val[1] = (val[1])?(1.0f/val[1]):0.0f;
    val[2] = (val[2])?(1.0f/val[2]):0.0f;
    return *this;
}

// Entry-wise multiplication with another vector
vuVector vuVector::mul(const vuVector& rhs) const
{
    vuVector r;
    r[0] = val[0] * rhs.val[0];
    r[1] = val[1] * rhs.val[1];
    r[2] = val[2] * rhs.val[2];
    return r;
}

// Entry-wise multiplication with another vector, then
// assign result to the current vector.
vuVector& vuVector::mulEq(const vuVector& rhs)
{
    val[0] *= rhs.val[0];
    val[1] *= rhs.val[1];
    val[2] *= rhs.val[2];
    return *this;
}

// Entry-wise division with another vector
vuVector vuVector::div(const vuVector& rhs) const
{
    vuVector r;
    r[0] = (rhs.val[0])?(val[0]/rhs.val[0]):0.0f;
    r[1] = (rhs.val[1])?(val[1]/rhs.val[1]):0.0f;
    r[2] = (rhs.val[2])?(val[2]/rhs.val[2]):0.0f;
    return r;
}

// Entry-wise division with another vector, then
// assign result to the current vector.
vuVector& vuVector::divEq(const vuVector& rhs)
{
    val[0] = (rhs.val[0])?(val[0]/rhs.val[0]):val[0]=0.0f;
    val[1] = (rhs.val[1])?(val[1]/rhs.val[1]):val[1]=0.0f;
    val[2] = (rhs.val[2])?(val[2]/rhs.val[2]):val[2]=0.0f;
    return *this;
}

// Dot product friend function
float dot(const vuVector& v1,const vuVector& v2)
{
    return v1.val[0]*v2.val[0]+v1.val[1]*v2.val[1]+v1.val[2]*v2.val[2];
}

// Cross product friend function
vuVector cross(const vuVector& v1,const vuVector& v2)
{
    vuVector v;
    v.val[0] = (v1.val[1]*v2.val[2] - v1.val[2]*v2.val[1]);
    v.val[1] = (v1.val[2]*v2.val[0] - v1.val[0]*v2.val[2]);
    v.val[2] = (v1.val[0]*v2.val[1] - v1.val[1]*v2.val[0]);
    v.val[3] = 1.0f;
    return v;
}

// Dot product member function
float vuVector::dot(const vuVector& v) const
{
    return val[0]*v.val[0]+val[1]*v.val[1]+val[2]*v.val[2];
}

// Cross product member function
vuVector vuVector::cross(const vuVector& v) const
{
    vuVector r;
    r.val[0] = (val[1]*v.val[2] - val[2]*v.val[1]);
    r.val[1] = (val[2]*v.val[0] - val[0]*v.val[2]);
    r.val[2] = (val[0]*v.val[1] - val[1]*v.val[0]);
    r.val[3] = 1.0f;

    return r;
}

// Addition operator
vuVector vuVector::operator+(const vuVector& v) const
{
    vuVector r;
    r.val[0] = val[0] + v.val[0];
    r.val[1] = val[1] + v.val[1];
    r.val[2] = val[2] + v.val[2];
    r.val[3] = 1.0f;
    return r;
}

// Subtraction operator
vuVector vuVector::operator-(const vuVector& v) const
{
    vuVector r;
    r.val[0] = val[0] - v.val[0];
    r.val[1] = val[1] - v.val[1];
    r.val[2] = val[2] - v.val[2];
    r.val[3] = 1.0f;
    return r;
}

// Multiplication operator - vector*matrix
vuVector vuVector::operator*(const vuMatrix& m) const
{
    vuVector r;
    dword i, j;
    r[3] = 0.0f;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[i] += m.val[(i<<2)+j]*val[j];
    return r;
}

// Multiplication operator - vector*vector -> matrix
vuMatrix vuVector::operator*(const vuVector& v) const
{
    vuMatrix r;
    dword i, j;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[(j<<2)+i] = val[i]*v.val[j];
    return r;
}

// Multiplication operator - vector*scalar
vuVector vuVector::operator*(float s) const
{
    vuVector r;
    r.val[0] = val[0] * s;
    r.val[1] = val[1] * s;
    r.val[2] = val[2] * s;
    r.val[3] = 1.0f;
    return r;
}

// Division operator - vector/scalar
vuVector vuVector::operator/(float s) const
{
    vuVector r;
    r.val[0] = val[0] / s;
    r.val[1] = val[1] / s;
    r.val[2] = val[2] / s;
    r.val[3] = 1.0f;
    return r;
}

// Multiplication operator - scalar*vector
const vuVector operator*(float s,const vuVector& v)
{
    vuVector r;
    r.val[0] = v.val[0] * s;
    r.val[1] = v.val[1] * s;
    r.val[2] = v.val[2] * s;
    r.val[3] = 1.0f;
    return r;
}

// Add then assign operator
vuVector& vuVector::operator+=(const vuVector& v)
{
    val[0] += v.val[0];
    val[1] += v.val[1];
    val[2] += v.val[2];
    val[3] = 1.0f;
    return *this;
}

// Subtract then assign operator
vuVector& vuVector::operator-=(const vuVector& v)
{
    val[0] -= v.val[0];
    val[1] -= v.val[1];
    val[2] -= v.val[2];
    val[3] = 1.0f;
    return *this;
}

/*
// Multiply then assign operator - vector*matrix
vuVector& vuVector::operator*=(const vuMatrix& m)
{
    vuVector r;
    dword i, j;
    r[3] = 0.0f;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[i] += m.val[(i<<2)+j]*val[j];
    return (*this=r);
}
*/

//WARNING: you should NOT use this function if you can,
//becuase it is unclear weather it does vector*matrix
//or matrix*vector when someone is simply reading through
//your code.

//Multiply then assign operator - matrix*vector
//Old code (see above) does vector*matrix, which
//is not the same. Since openGL uses column major
//ordering and postmultiplies (matrix is left
//of vector in multiplication), the above function
//is not easily useable. Since I (Tai) seem to be the only one
//using this function, if you change it back, please
//make sure you modify CellProjector so that its
//matrix multiplications work properly. Feel free to email me
//at tmeng@sfu.ca if you have any questions.
vuVector& vuVector::operator*=(const vuMatrix& m)
{
    vuVector r;
    dword i, j;
    r[3] = 0.0f; //r starts out being a vector of all 0's.
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
	        //j<<2 is left shift; same as j * 4
      	  //column major ordering
          //1   5   9   13
          //2   6   10  14
          //3... etc
	        //it used to be m.val[(i<<2 + j)] which was wrong
	        r.val[i] += m.val[(j<<2)+i]*val[j];
    return (*this=r);
}

// Multiply then assign operator - vector*scalar
vuVector& vuVector::operator*=(float s)
{
    val[0] *= s;
    val[1] *= s;
    val[2] *= s;
    val[3] = 1.0f;
    return *this;
}

// Equality operator
bool vuVector::operator==(const vuVector& v) const
{
    return ((val[0]==v.val[0])&&(val[1]==v.val[1])&&(val[2]==v.val[2]));
}

// Inequality operator
bool vuVector::operator!=(const vuVector& v) const
{
    return !(operator==(v));
}

int vuVector::getDominantAxis() const
{
  float a0 = fabs(val[0]);
  float a1 = fabs(val[1]);
  float a2 = fabs(val[2]);
  if(a0<a1) {
    if(a1<a2) return 2;
    else return 1;
  } else 
    if(a0<a2) return 2;
    else return 0;
}

void vuVector::print()
{
  printf("( %4.4f,  %4.4f,  %4.4f,  %4.4f )\n",
	  val[0], val[1], val[2], val[3]);
}

int strip_white_space (char* temp)

{
	int position = 0;

	while ((temp [position] == ' ') || (temp [position] == '\t'))
		position++;

	return position;
}

int get_next_comma (char* temp)

{
	int position = 0;

	while ((temp [position] != ',') && (temp [position] != '\0'))
		position++;

	return position;
}

int get_next_return (char* temp)

{
	int position = 0;

	while ((temp [position] != '\n') && (temp [position] != '\0'))
		position++;

	return position;
}

int get_next_blank (char* temp)

{
	int position = 0;

	while ((temp [position] != ' ') && (temp [position] != '\t') && (temp [position] != '\0'))
		position++;

	return position;
}

int clear_blanks (char* temp)

{
	int position = 0;

	while (((temp [position] == ' ') || (temp [position] == '\t')) && (temp [position] != '\0'))
		position++;

	return position;
}

int vuVector::load (char* load_from)

{
	int position = 0;

	for (int i = 0; i < 3; i++)

	{
		position += strip_white_space (&(load_from [position]));

		val [i] = atof (&(load_from [position]));

		position += get_next_comma (&(load_from [position]));
		position++;
	}

	position += strip_white_space (&(load_from [position]));
	val [3] = atof (&(load_from [position]));

	position += get_next_blank (&(load_from [position]));

	return position;
}

void vuVector::save (char* save_to)

{
	int position = 0;
	save_to [0] = '\0';

	char temp [64];

	for (int i = 0; i < 4; i++)

	{
		gcvt (val [i], 14, temp);

		strcat (save_to, temp);

		position += strlen (temp);

		if (i != 3)
			strcat (save_to, ", ");
	}
}

ostream& operator<<(ostream& out,const vuVector& v)
{
    //this one is from Chris, but it works better without comma,
    //hope this doesn't mess up things...
    //out << v[0] <<", "<< v[1] <<", "<< v[2] <<", "<< v[3];
    out<<v.val[0]<<" "<<v.val[1]<<" "<<v.val[2]<<" "<<v.val[3]<<endl;
    return out;
}

istream& operator>>(istream& in, vuVector& v)
{
	in>>v.val[0];
	in>>v.val[1];
	in>>v.val[2];
	in>>v.val[3];
	return in;
}

vuString vuVector::getString()
{
  vuString str;
  float tmp[3];
  char  buf[64];

  if (val[3] == 0)
    tmp[0] = tmp[1] = tmp[2] = 0.0f;
  else {
    tmp[0] = val[0] / val[3];
    tmp[1] = val[1] / val[3];
    tmp[2] = val[2] / val[3];
  }

  sprintf(buf,"[%4.4f,  %4.4f,  %4.4f]", tmp[0], tmp[1], tmp[2]);

  str = buf;
  return str;
}
