//! Written by Martin Spindler Jan. 2003

#ifndef _SPHERICAL_H_
#define _SPHERICAL_H_

#include <stdlib.h>
#include <string.h>
#include <iostream.h>

#include "vuVector.h"
#include "vuSimpleTypes.h"

//! A class for representing 3d spherical coordinates
/*! 
  A spherical coordinate consists of two angles (longitude, latitude) and a
  length (radius).

  The formal notation is: vuSpherical = [longitude, latitude, radius]

  longitude: angle in x-y-plane, starting with x-axis (counterclockwise)
   latitude: angle in y-z-plane, starting with y-axis (+z=>+PI/2, -z=>-PI/2)
     radius: length of vector

    0 <= longitude <  2*PI
  -PI <= latitude  <=  +PI
    0 <= radius    < infinity

  Definition: if (radius==0), then longitude=0 and latitude=0

*/

class vuSpherical
{
public:
    //!Default constructor making a zero spherical.
    vuSpherical();
    //!Copy constructor.
    vuSpherical(const vuSpherical& v);
    //!Constructor initializing to a 3d spherical.
    vuSpherical(float longitude, float latitude, float radius);
    //!Constructor initializing the spherical from an array of floats.
    /*!\param v An array of 3 floating point values corresponding to
        longitude, latitude, radius.
    */
    vuSpherical(const float* v);
    /*!Constructor initializing the spherical from an vuVector, by transforming
       the vector into a spherical.
    */
    vuSpherical(vuVector &vector);
    //!The destructor.
    ~vuSpherical();

    //!The assignment operator.
    vuSpherical& operator=(const vuSpherical& v);
    //!Assigns a 3d float array (longitude,latitude,radius) to the spherical.
    vuSpherical& operator=(const float* v);

    //!Getting the longitude
    float getLongitude() { return val[0]; }
    //!Getting the latitude
    float getLatitude()  { return val[1]; }
    //!Getting the radius
    float getRadius()    { return val[2]; }


    //!Getting the longitude
    void setLongitude(float _val) { val[0] = _val; }
    //!Getting the latitude
    void setLatitude(float _val)  { val[1] = _val; }
    //!Getting the radius
    void setRadius(float _val)    { val[2] = _val; }

    //!Performs the 1-norm (manhattan norm) on the spherical. 
    float norm(void) const;
    //!Performs the 2-norm on the vector.
    /*!This is the same as the length operator or radius.
    */
    float norm2(void) const;

    //!Normalizes the vector to a length of 1.
    /*!This simply sets the radius to 1.*/
    vuSpherical& makeUnit(void);

    //!The access operator.
    float& operator[](dword index);
    //!The const access operator.
    const float& operator[](dword index) const;

    //!Returns the data pointer.
    /*!The data is stored as a four dimensional float array.
    	The getData() function can both retrieve and modify; it
     	is similar to the [] operator. To warrant the return of
      a const pointer (for use in other const member functions)
      explicitly cast the pointer returned to (const).
    */
    float* getData(void);
    //! const version of getData()
    float const* getData(void) const;
    
    //!An equality operator.
    bool operator==(const vuSpherical& v) const;
    //!An inequality operator.
    bool operator!=(const vuSpherical& v) const;

    //! writes the three elements to a stream; separated by whitespaces
    friend ostream& operator<<(ostream& out,const vuSpherical& v);
    //! reads a spherical from a stream
    friend istream& operator>>(istream& in, vuSpherical& v);

    //!prints values to the stdout
    void print();    

    //! returns a cartesian vector
    vuVector getVector();

private:
    /*!The coordinates of the vector stored as a floating point array:
      longitude, latitude, radius
     */
    float val[3];

    inline void ensure(float &longitude, float &latitude, float &radius);
};

#endif /* _SPHERICAL_H_ */
