#include <math.h>
#include <stdio.h>
#include "vuSpherical.h"
#include "vuVector.h"


// Default constructor - set initial values
vuSpherical::vuSpherical()
{
  val[0] = val[1] = val[2] = 0.0f;
}

// Copy constructor
vuSpherical::vuSpherical(const vuSpherical& v)
{
  val[0] = v.val[0];
  val[1] = v.val[1];
  val[2] = v.val[2];
}

// Constructor - set to <longitude,latitude,radius>
vuSpherical::vuSpherical(float longitude, float latitude, float radius)
{
  val[0] = longitude;
  val[1] = latitude;
  val[2] = radius;
}

vuSpherical::vuSpherical(const float* v)
{
  val[0] = v[0];
  val[1] = v[1];
  val[2] = v[2];
}

vuSpherical::vuSpherical(vuVector &vector)
{
  float longitude = 0.0f;
  float latitude  = 0.0f;
  float radius    = vector.norm();
  float xx        = vector[0];
  float yy        = vector[1];
  float zz        = vector[2];

  if ((xx == 0.0f) && (yy == 0.0f)) {
    longitude = 0.0;
    latitude  = (zz == 0) ? 0 : (zz > 0) ? M_PI : -M_PI;
  }
  else {
    latitude = acos(zz/radius); // radius > 0, since xx!=0 || yy!=0
    if (xx == 0.0f) {
      if (yy > 0)
	longitude = M_PI_2;
      else if (yy < 0)
	longitude = M_PI_2 + M_PI;
      else // xx==0 && yy==0
	throw "this case is not possible"; 
    }
    else if (xx > 0.0f) {
      if (yy >= 0)
	longitude = atan(yy/xx);
      else
	longitude = 2*M_PI + atan(yy/xx);
    }
    else { // xx < 0.0f
      longitude = M_PI + atan(yy/xx);
    }
  }
  val[0] = longitude;
  val[1] = latitude;
  val[2] = radius;
}

// Destructor
vuSpherical::~vuSpherical()
{
}

// Assignment operator
vuSpherical& vuSpherical::operator=(const vuSpherical& v)
{
  if (this != &v)
    {
      val[0] = v.val[0];
      val[1] = v.val[1];
      val[2] = v.val[2];
    }
  return *this;
}

// Const Access operator - return the value at the given index
const float& vuSpherical::operator[](dword index) const
{
  return val[index];
}

float const* vuSpherical::getData(void) const
{ return val; }

// Compute the norm of the vector  - |v|
float vuSpherical::norm(void) const
{
  return val[2]; // radius
}

// Compute the square norm of the vector
float vuSpherical::norm2(void) const
{
  return val[2]*val[2];
}

// Normalize the vector - set its length (radius) to 1
vuSpherical& vuSpherical::makeUnit(void)
{
  val[2] = 1.0f; // radius = 1.0f
  return *this;
}

// Equality operator
bool vuSpherical::operator==(const vuSpherical& v) const
{
  return ((val[0]==v.val[0])&&(val[1]==v.val[1])&&(val[2]==v.val[2]));
}

// Inequality operator
bool vuSpherical::operator!=(const vuSpherical& v) const
{
  return !(operator==(v));
}

void vuSpherical::print()
{
  printf("( %4.4f,  %4.4f,  %4.4f )\n", val[0], val[1], val[2]);
}

ostream& operator<<(ostream& out,const vuSpherical& v)
{
  out<<v.val[0]<<" "<<v.val[1]<<" "<<v.val[2]<<endl;
  return out;
}

istream& operator>>(istream& in, vuSpherical& v)
{
  in>>v.val[0];
  in>>v.val[1];
  in>>v.val[2];
  return in;
}

vuVector vuSpherical::getVector()
{
  float v[3];
  float sinLong = sin(v[0]);
  v[0] = val[2]*cos(val[1])*sinLong;
  v[1] = val[2]*sin(val[1])*sinLong;
  v[2] = val[2]*cos(val[0]);

  return vuVector(v);
}

/* ------------------------------------------------------------------------- */
/* --- private methods ----------------------------------------------------- */
/* ------------------------------------------------------------------------- */

inline void vuSpherical::ensure(float &longitude, 
				float &latitude,
				float &radius)
{
  if (longitude >= 2*M_PI) {
    while (longitude >= 2*M_PI) longitude -= 2*M_PI;
  }
  else if (longitude < 0.0f) {
    while (longitude < 0.0f) longitude += 2*M_PI;
  }
}
