/*	Written by Steven Kilthau */

#ifndef _MATRIX_H_
#define _MATRIX_H_

#include "vuSimpleTypes.h"

class vuVector;

/**
 * The vuMatrix class is a 4x4 matrix that
 * works in conjunction with the vuVector class
 * (a 3D homogeneous vector).  The elements of
 * the vuMatrix are all single precision
 * floating-point numbers, stored in column-major
 * order to be compatible with OpenGL.
*/ 

class vuMatrix
{
friend class vuVector;
public:
    //!Default constructor that sets all values to 0.0f.
    vuMatrix();
    //!Copy constructor
    vuMatrix(const vuMatrix& m);
    //!Constructor that sets all values to v.
    vuMatrix(float v);
    //!Constructor that initializes the vector from a float array.
    /*!\param v An array of 16 floating point values.
    */
    vuMatrix(const float* v);
    //!Destructor
    ~vuMatrix();

    //!Assigns a matrix to the instance.
    vuMatrix& operator=(const vuMatrix& m);
    //!Assigns a floating point value to each entry of the matrix.
    vuMatrix& operator=(float v);
    //!Assigns an array of floating point values to the matrix.
    vuMatrix& operator=(const float* v);

    //
    // Generate various special matrices by
    // changing values in the current instance.
    // All transformation matrices are pre-
    // transforms, meaning they should be left
    // multiplied against a vuVector.
    //
    //!Makes the matrix an identity matrix and returns a reference.
    vuMatrix& makeIdentity(void);
    //!Makes a rotation matrix for a rotation of a degrees around the axis vector.
    /*!\param axis The axis of rotation.
       \param a The angle of rotation, in degrees.
    */
    vuMatrix& makeRotate(const vuVector& axis, float a);
    //!Makes this matrix a rotation matrix for a rotation of a degrees about the x axis.
    vuMatrix& makeRotateX(float a);
    //!Makes this matrix a rotation matrix for a rotation of a degrees about the y axis.
    vuMatrix& makeRotateY(float a);
    //!Makes this matrix a rotation matrix for a rotation of a degrees about the z axis.
    vuMatrix& makeRotateZ(float a);
    //!Makes this matrix a translation matrix corresponding to the given displacement.
    vuMatrix& makeTranslate(float x, float y, float z);
    //!Makes this matrix a scaling matrix.
    vuMatrix& makeScale(float x, float y, float z);
    //!Makes this matrix a shearing matrix in the XY plane.
    vuMatrix& makeShearXY(float s);
    //!Makes this matrix a shearing matrix in the XZ plane.
    vuMatrix& makeShearXZ(float s);
    //!Makes this matrix a shearing matrix in the YX plane.
    vuMatrix& makeShearYX(float s);
    //!Makes this matrix a shearing matrix in the YZ plane.
    vuMatrix& makeShearYZ(float s);
    //!Makes this matrix a shearing matrix in the ZX plane.
    vuMatrix& makeShearZX(float s);
    //!Makes this matrix a shearing matrix in the ZY plane.
    vuMatrix& makeShearZY(float s);
    //!Makes this matrix a reflection matrix about the x axis.
    vuMatrix& makeReflectX(void);
    //!Makes this matrix a reflection matrix about the y axis.
    vuMatrix& makeReflectY(void);
    //!Makes this matrix a reflection matrix about the z axis.
    vuMatrix& makeReflectZ(void);
    //!Makes this matrix a perspective matrix for the distance d.
    vuMatrix& makePerspective(float d);
    //!Makes this matrix a perspective matrix for the distance d, keeping the z value.
    vuMatrix& makePerspectiveKeepZ(float d);

    //!The access operator.
    float* operator[](unsigned int index);
    //!The const access operator.
    const float* operator[](unsigned int index) const;
    //!Returns the data pointer.
    /*!Same as calling operator[0].*/
    float* getData(void);
    //!returns const data pointer
    float const* getData(void) const;

    //!Returns the transposed 3x3 matrix with negative translation
    vuMatrix invOrtho() const;
    //!Returns the addition of the two matrices..
    vuMatrix operator+(const vuMatrix& m) const;
    //!Returns the subtraction a matrix from the instance.
    vuMatrix operator-(const vuMatrix& m) const;
    //!Returns the product of the two matrices.
    vuMatrix operator*(const vuMatrix& m) const;
    //!Returns the product of the instance with a vector.
    vuVector operator*(const vuVector& v) const;
    //!Returns the product of the instance with a scalar.
    vuMatrix operator*(float s) const;
    //!Multiplies a scalar by a matrix.
    friend vuMatrix operator*(float s,const vuMatrix& m);
    //!Adds a matrix to the instance.
    vuMatrix& operator+=(const vuMatrix& m);
    //!Subtracts a matrix from the instance.
    vuMatrix& operator-=(const vuMatrix& m);
    //!Multiplies the instance by a matrix.
    vuMatrix& operator*=(const vuMatrix& m);
    //!Multiplies the instance by a scalar.
    vuMatrix& operator*=(float s);

    //!An equality operator.
    bool operator==(const vuMatrix& m) const;
    //!An inequality operator.
    bool operator!=(const vuMatrix& m) const;

    //!Returns the inverse of the matrix.
    vuMatrix inverse(void) ;

    //!Inverts a rotation matrix
    /*!
      The inverse of a rotation matrix is simply its transpose, which takes 3 swaps
    */
    void invertRotationMatrix();

private:
    float val[16]; //!< this is the buffer that stores the entries of the matrix
};

#endif
