#include <math.h>
#include "vuMatrix.h"
#include "vuVector.h"

#define PI_OVER_180 0.01745329251994

// Default constructor - set initial values
vuMatrix::vuMatrix()
{
    for(int i=0;i<16;++i)
        val[i] = 0.0f;
}

// Copy constructor
vuMatrix::vuMatrix(const vuMatrix& m)
{
    for(int i=0;i<16;++i)
        val[i] = m.val[i];
}

// Constructor - set all values to v
vuMatrix::vuMatrix(float v)
{
    for(int i=0;i<16;++i)
        val[i] = v;
}

// Constructor - get values from an array
vuMatrix::vuMatrix(const float* v)
{
    for(int i=0;i<16;++i)
        val[i] = v[i];
}

// Destructor
vuMatrix::~vuMatrix()
{
}

// Assignment operator
vuMatrix& vuMatrix::operator=(const vuMatrix& m)
{
    if (this != &m)
    {
        for(int i=0;i<16;++i)
            val[i] = m.val[i];
    }
    return *this;
}

// Assignment operator - set all values to v
vuMatrix& vuMatrix::operator=(float v)
{
    for(int i=0;i<16;++i)
        val[i] = v;
    return *this;
}

// Assignment operator - get values from an array
vuMatrix& vuMatrix::operator=(const float* v)
{
    for(int i=0;i<16;++i)
        val[i] = v[i];
    return *this;
}

// make the current matrix the identity matrix
vuMatrix& vuMatrix::makeIdentity(void)
{
    for(int i=1;i<15;++i)
        val[i] = 0.0f;
    val[0] = val[5] = val[10] = val[15] = 1.0f;
    return *this;
}

vuMatrix& vuMatrix::makeRotate(const vuVector& axis, float a)
{
	vuVector axis2;
	float s;
	float c;

	s = (float)sin(a*PI_OVER_180);
	c = (float)cos(a*PI_OVER_180);
	axis2 = axis.mul(axis);

	val[0] = axis2[0]+(1.0f-axis2[0])*c;
	val[1] = axis[0]*axis[1]*(1.0f-c)+axis[2]*s;
	val[2] = axis[0]*axis[2]*(1.0f-c)-axis[1]*s;
    val[3] = 0.0f;

	val[4] = axis[0]*axis[1]*(1.0f-c)-axis[2]*s;
	val[5] = axis2[1]+(1.0f-axis2[1])*c;
	val[6] = axis[1]*axis[2]*(1.0f-c)+axis[0]*s;
    val[7] = 0.0f;

	val[8]  = axis[0]*axis[2]*(1.0f-c)+axis[1]*s;
	val[9]  = axis[1]*axis[2]*(1.0f-c)-axis[0]*s;
	val[10] = axis2[2]+(1.0f-axis2[2])*c;
    val[11] = 0.0f;

    val[12] = 0.0f;
    val[13] = 0.0f;
    val[14] = 0.0f;
	val[15] = 1.0f;

    return *this;
}

vuMatrix& vuMatrix::makeRotateX(float a)
{
    makeIdentity();
    val[5] = val[10] = (float)cos((double)a*PI_OVER_180);
    val[9] = -(val[6] = (float)sin((double)a*PI_OVER_180));
    return *this;
}

vuMatrix& vuMatrix::makeRotateY(float a)
{
    makeIdentity();
    val[0] = val[10] = (float)cos((double)a*PI_OVER_180);
    val[2] = -(val[8] = (float)sin((double)a*PI_OVER_180));
    return *this;
}

vuMatrix& vuMatrix::makeRotateZ(float a)
{
    makeIdentity();
    val[0] = val[5] = (float)cos((double)a*PI_OVER_180);
    val[4] = -(val[1] = (float)sin((double)a*PI_OVER_180));
    return *this;
}

vuMatrix& vuMatrix::makeTranslate(float x, float y, float z)
{
    makeIdentity();
    val[12] = x;
    val[13] = y;
    val[14] = z;
    return *this;
}

vuMatrix& vuMatrix::makeScale(float x, float y, float z)
{
    makeIdentity();
    val[0] = x;
    val[5] = y;
    val[10] = z;
    return *this;
}

vuMatrix& vuMatrix::makeShearXY(float s)
{
    makeIdentity();
    val[1] = s;
    return *this;
}

vuMatrix& vuMatrix::makeShearXZ(float s)
{
    makeIdentity();
    val[2] = s;
    return *this;
}

vuMatrix& vuMatrix::makeShearYX(float s)
{
    makeIdentity();
    val[4] = s;
    return *this;
}

vuMatrix& vuMatrix::makeShearYZ(float s)
{
    makeIdentity();
    val[6] = s;
    return *this;
}

vuMatrix& vuMatrix::makeShearZX(float s)
{
    makeIdentity();
    val[8] = s;
    return *this;
}

vuMatrix& vuMatrix::makeShearZY(float s)
{
    makeIdentity();
    val[9] = s;
    return *this;
}

vuMatrix& vuMatrix::makeReflectX(void)
{
    makeIdentity();
    val[0] = -1.0f;
    return *this;
}

vuMatrix& vuMatrix::makeReflectY(void)
{
    makeIdentity();
    val[5] = -1.0f;
    return *this;
}

vuMatrix& vuMatrix::makeReflectZ(void)
{
    makeIdentity();
    val[10] = -1.0f;
    return *this;
}

vuMatrix& vuMatrix::makePerspective(float d)
{
    makeIdentity();
    val[10] = 0.0f;
    val[11] = 1.0f/d;
    return *this;
}

vuMatrix& vuMatrix::makePerspectiveKeepZ(float d)
{
    makeIdentity();
    val[11] = 1.0f/d;
    return *this;
}

// Access operator - get col at the given index
float* vuMatrix::operator[](unsigned int index)
{
    if (index > 3)
        throw "vuMatrix: Index out of range.";
    return &val[index*4];
}

// Const Access operator - get col at the given index
const float* vuMatrix::operator[](unsigned int index) const
{
    if (index > 3)
        throw "vuMatrix: Index out of range.";
    return &val[index*4];
}

// Return the internal data pointer.
float* vuMatrix::getData(void)
{
    return val;
}

float const* vuMatrix::getData(void) const
{ return val; }

//transpose and inverse translation
vuMatrix vuMatrix::invOrtho() const
{
  vuMatrix r;
  float *vp = r.getData();
  vp[0] = val[0]; vp[5] = val[5]; vp[10] = val[10];			//diagonal
  vp[3] = val[3]; vp[7] = val[7]; vp[11] = val[11]; vp[15] = 1/val[15];	//weights
  vp[12] = -val[12]; vp[13] = -val[13]; vp[14] = -val[14];		//-transl
  vp[1] = val[4]; vp[4] = val[1];					//mirror
  vp[2] = val[8]; vp[8] = val[2];
  vp[6] = val[9]; vp[9] = val[6];
  return r;
}

// Addition operator
vuMatrix vuMatrix::operator+(const vuMatrix& m) const
{
    vuMatrix r;
    for(int i=0;i<16;++i)
        r.val[i] = val[i] + m.val[i];
    return r;
}

// Subtraction operator
vuMatrix vuMatrix::operator-(const vuMatrix& m) const
{
    vuMatrix r;
    for(int i=0;i<16;++i)
        r.val[i] = val[i] - m.val[i];
    return r;
}

// Multiplication operator - matrix*matrix
vuMatrix vuMatrix::operator*(const vuMatrix& m) const
{
    vuMatrix r;
    int i, j, k;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            for(k=0;k<4;++k)
                r.val[(j<<2)+i] += val[(k<<2)+i] * m.val[(j<<2)+k];
    return r;
}

// Multiplication operator - matrix*vector -> vector
vuVector vuMatrix::operator*(const vuVector& v) const
{
    vuVector r;
    int i, j;
    r[3] = (float)(0.0);
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[i] += val[(j<<2)+i]*v.val[j];
    return r;
}

// Multiplication operator - matrix*scalar
vuMatrix vuMatrix::operator*(float s) const
{
    vuMatrix r;
    for(int i=0;i<16;++i)
        r.val[i] = val[i] * s;
    return r;
}

// Multiplication operator - scalar*matrix
vuMatrix operator*(float s,const vuMatrix& m)
{
    vuMatrix r;
    for(int i=0;i<16;++i)
        r.val[i] = m.val[i] * s;
    return r;
}

// Add then assign operator
vuMatrix& vuMatrix::operator+=(const vuMatrix& m)
{
    for(int i=0;i<16;++i)
        val[i] += m.val[i];
    return *this;
}

// Subtract the assign operator
vuMatrix& vuMatrix::operator-=(const vuMatrix& m)
{
    for(int i=0;i<16;++i)
        val[i] -= m.val[i];
    return *this;
}

// Multiply then assign operator - matrix*matrix
vuMatrix& vuMatrix::operator*=(const vuMatrix& m)
{
    vuMatrix r;
    int i, j, k;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            for(k=0;k<4;++k)
                 r.val[(j<<2)+i] += val[(k<<2)+i] * m.val[(j<<2)+k];
    return (*this=r);
}

// Multiply then assign operator - matrix*scalar
vuMatrix& vuMatrix::operator*=(float s)
{
    for(int i=0;i<16;++i)
        val[i] *= s;
    return *this;
}

// Equality operator
bool vuMatrix::operator==(const vuMatrix& m) const
{
    for(int i=0;i<16;++i)
        if (val[i] != m.val[i])
            return false;
    return true;
}

// Inequality operator
bool vuMatrix::operator!=(const vuMatrix& m) const
{
    return !(operator==(m));
}


/* the inverse() method is taken from a vuVolume version of the TU Wien,
   and is needed by the ShearWarp implementation.

   Perhaps we can simplyfy this a bit in future days... --martin-2003-04-01
*/   

// Compute the inverse of the matrix
vuMatrix vuMatrix::inverse(void) 
{
    vuMatrix result;

    // compute the determinante of the 4x4 matrix
    float det =        (val[2*4+0] * val[3*4+1] - val[3*4+0] * val[2*4+1]) *
	               (val[0*4+2] * val[1*4+3] - val[1*4+2] * val[0*4+3]) +

	        (-1) * (val[2*4+0] * val[3*4+2] - val[3*4+0] * val[2*4+2]) *
	               (val[0*4+1] * val[1*4+3] - val[1*4+1] * val[0*4+3]) +

	               (val[2*4+1] * val[3*4+2] - val[3*4+1] * val[2*4+2]) *
	               (val[0*4+0] * val[1*4+3] - val[1*4+0] * val[0*4+3]) +

	        (-1) * (val[2*4+1] * val[3*4+3] - val[3*4+1] * val[2*4+3]) *
	               (val[0*4+0] * val[1*4+2] - val[1*4+0] * val[0*4+2]) +

	               (val[2*4+0] * val[3*4+3] - val[3*4+0] * val[2*4+3]) *
	               (val[0*4+1] * val[1*4+2] - val[1*4+1] * val[0*4+2]) +

	               (val[2*4+2] * val[3*4+3] - val[3*4+2] * val[2*4+3]) *
	               (val[0*4+0] * val[1*4+1] - val[1*4+0] * val[0*4+1]);

    result.val[0] = (val[1*4+1] * val[2*4+2] * val[3*4+3] + val[1*4+2] * val[2*4+3] * val[3*4+1] +
                  val[1*4+3] * val[2*4+1] * val[3*4+2] - val[3*4+1] * val[2*4+2] * val[1*4+3] -
	          val[3*4+2] * val[2*4+3] * val[1*4+1] - val[3*4+3] * val[2*4+1] * val[1*4+2]) 
	/ det;
    result.val[1] = -(val[0*4+1] * val[2*4+2] * val[3*4+3] + val[0*4+2] * val[2*4+3] * val[3*4+1] +
                   val[0*4+3] * val[2*4+1] * val[3*4+2] - val[3*4+1] * val[2*4+2] * val[0*4+3] -
	           val[3*4+2] * val[2*4+3] * val[0*4+1] - val[3*4+3] * val[2*4+1] * val[0*4+2]) 
	/ det;
    result.val[2] = (val[0*4+1] * val[1*4+2] * val[3*4+3] + val[0*4+2] * val[1*4+3] * val[3*4+1] +
                  val[0*4+3] * val[1*4+1] * val[3*4+2] - val[3*4+1] * val[1*4+2] * val[0*4+3] -
	          val[3*4+2] * val[1*4+3] * val[0*4+1] - val[3*4+3] * val[1*4+1] * val[0*4+2]) 
	/ det;
    result.val[3] = -(val[0*4+1] * val[1*4+2] * val[2*4+3] + val[0*4+2] * val[1*4+3] * val[2*4+1] +
                   val[0*4+3] * val[1*4+1] * val[2*4+2] - val[2*4+1] * val[1*4+2] * val[0*4+3] -
	           val[2*4+2] * val[1*4+3] * val[0*4+1] - val[2*4+3] * val[1*4+1] * val[0*4+2]) 
	/ det;


    result.val[4] = -(val[1*4+0] * val[2*4+2] * val[3*4+3] + val[1*4+2] * val[2*4+3] * val[3*4+0] +
                   val[1*4+3] * val[2*4+0] * val[3*4+2] - val[3*4+0] * val[2*4+2] * val[1*4+3] -
	           val[3*4+2] * val[2*4+3] * val[1*4+0] - val[3*4+3] * val[2*4+0] * val[1*4+2]) 
	/ det;
    result.val[5] = (val[0*4+0] * val[2*4+2] * val[3*4+3] + val[0*4+2] * val[2*4+3] * val[3*4+0] +
                  val[0*4+3] * val[2*4+0] * val[3*4+2] - val[3*4+0] * val[2*4+2] * val[0*4+3] -
	          val[3*4+2] * val[2*4+3] * val[0*4+0] - val[3*4+3] * val[2*4+0] * val[0*4+2]) 
	/ det;
    result.val[6] = -(val[0*4+0] * val[1*4+2] * val[3*4+3] + val[0*4+2] * val[1*4+3] * val[3*4+0] +
                   val[0*4+3] * val[1*4+0] * val[3*4+2] - val[3*4+0] * val[1*4+2] * val[0*4+3] -
	           val[3*4+2] * val[1*4+3] * val[0*4+0] - val[3*4+3] * val[1*4+0] * val[0*4+2]) 
	/ det;
    result.val[7] = (val[0*4+0] * val[1*4+2] * val[2*4+3] + val[0*4+2] * val[1*4+3] * val[2*4+0] +
                  val[0*4+3] * val[1*4+0] * val[2*4+2] - val[2*4+0] * val[1*4+2] * val[0*4+3] -
	          val[2*4+2] * val[1*4+3] * val[0*4+0] - val[2*4+3] * val[1*4+0] * val[0*4+2]) 
	/ det;



    result.val[8] = (val[1*4+0] * val[2*4+1] * val[3*4+3] + val[1*4+1] * val[2*4+3] * val[3*4+0] +
                  val[1*4+3] * val[2*4+0] * val[3*4+1] - val[3*4+0] * val[2*4+1] * val[1*4+3] -
	          val[3*4+1] * val[2*4+3] * val[1*4+0] - val[3*4+3] * val[2*4+0] * val[1*4+1]) 
	/ det;
    result.val[9] = -(val[0*4+0] * val[2*4+1] * val[3*4+3] + val[0*4+1] * val[2*4+3] * val[3*4+0] +
                   val[0*4+3] * val[2*4+0] * val[3*4+1] - val[3*4+0] * val[2*4+1] * val[0*4+3] -
	           val[3*4+1] * val[2*4+3] * val[0*4+0] - val[3*4+3] * val[2*4+0] * val[0*4+1]) 
	/ det;
    result.val[10] = (val[0*4+0] * val[1*4+1] * val[3*4+3] + val[0*4+1] * val[1*4+3] * val[3*4+0] +
                   val[0*4+3] * val[1*4+0] * val[3*4+1] - val[3*4+0] * val[1*4+1] * val[0*4+3] -
	           val[3*4+1] * val[1*4+3] * val[0*4+0] - val[3*4+3] * val[1*4+0] * val[0*4+1]) 
	/ det;
    result.val[11] = -(val[0*4+0] * val[1*4+1] * val[2*4+3] + val[0*4+1] * val[1*4+3] * val[2*4+0] +
                    val[0*4+3] * val[1*4+0] * val[2*4+1] - val[2*4+0] * val[1*4+1] * val[0*4+3] -
	            val[2*4+1] * val[1*4+3] * val[0*4+0] - val[2*4+3] * val[1*4+0] * val[0*4+1]) 
	/ det;




    result.val[12] =-(val[1*4+0] * val[2*4+1] * val[3*4+2] + val[1*4+1] * val[2*4+2] * val[3*4+0] +
                   val[1*4+2] * val[2*4+0] * val[3*4+1] - val[3*4+0] * val[2*4+1] * val[1*4+2] -
	           val[3*4+1] * val[2*4+2] * val[1*4+0] - val[3*4+2] * val[2*4+0] * val[1*4+1]) 
	/ det;
    result.val[13] = (val[0*4+0] * val[2*4+1] * val[3*4+2] + val[0*4+1] * val[2*4+2] * val[3*4+0] +
                   val[0*4+2] * val[2*4+0] * val[3*4+1] - val[3*4+0] * val[2*4+1] * val[0*4+2] -
	           val[3*4+1] * val[2*4+2] * val[0*4+0] - val[3*4+2] * val[2*4+0] * val[0*4+1]) 
	/ det;
    result.val[14] = -(val[0*4+0] * val[1*4+1] * val[3*4+2] + val[0*4+1] * val[1*4+2] * val[3*4+0] +
                    val[0*4+2] * val[1*4+0] * val[3*4+1] - val[3*4+0] * val[1*4+1] * val[0*4+2] -
	            val[3*4+1] * val[1*4+2] * val[0*4+0] - val[3*4+2] * val[1*4+0] * val[0*4+1]) 
	/ det;
    result.val[15] = (val[0*4+0] * val[1*4+1] * val[2*4+2] + val[0*4+1] * val[1*4+2] * val[2*4+0] +
                   val[0*4+2] * val[1*4+0] * val[2*4+1] - val[2*4+0] * val[1*4+1] * val[0*4+2] -
	           val[2*4+1] * val[1*4+2] * val[0*4+0] - val[2*4+2] * val[1*4+0] * val[0*4+1]) 
	/ det;

    return result;
}


/*============================================================*/

void swap(float& a, float& b)
{
  float temp = a;
  a = b;
  b = temp;
}

void vuMatrix::invertRotationMatrix()
  //vuUtilities::swap(float&, float&) should be written and used
  //tested
{
  swap((val[1]), (val[4]));
  swap((val[2]), (val[8]));
  swap((val[6]), (val[9]));
}
