/** A FixelMap is comparable to an image. It's a 2 dimensional
    datastructure with width and height, but instead of a colour as basic
    element, it supports so called fixels.

    Typical examples are: float    RGB-image           -> vuFixelMap<3,float>
                          byte     RGB-image           -> vuFixelMap<3,byte>
                          float    intensity     image -> vuFixelMap<1,float>
                          byte     intensity     image -> vuFixelMap<1,byte>
                          complex  fourier freq. image -> vuFixelMap<2,float>
                          real     fourier freq. image -> vuFixelMap<1,float>
		          real     spectral      image -> vuFixel<7,float>
                                                          vuFixel<31,float>
*/

#ifndef __VU_FIXELMAP_H__
#define __VU_FIXELMAP_H__

#include <stddef.h>
#include <fstream.h>
#include "vuSimpleTypes.h"
#include "vuFixel.h" 
#include <GL/gl.h>

class vuFixelMap_ST
{
 public:
  virtual const void* getBuffer() const = 0;
  virtual void* getBuffer() = 0;
  virtual dword getWidth()  const = 0;
  virtual dword getHeight() const = 0;
};

template <int SIZE, class TYPE>
class vuFixelMap : public vuFixelMap_ST{
 private:
  TYPE *m_buffer;
  bool m_isBufferNewed;
  dword m_width;
  dword m_height;
  
 public:
  // constructor
  vuFixelMap();

  vuFixelMap(dword width, dword height);

  // destructor
  virtual ~vuFixelMap();

  // copy constructor
  vuFixelMap(const vuFixelMap &other);

  // width and height
  void setWidthAndHeight(const dword sizx, const dword sizy);

  dword getWidth() const;

  dword getHeight() const;

  // values on (x,y)
  void setFixel(dword x, dword y, const vuFixel<SIZE,TYPE> &fixel);

  vuFixel<SIZE,TYPE> getFixel(dword x, dword y);

  // buffer
  const TYPE* getBuffer() const;

  TYPE* getBuffer();

  // find the lowest and highest value
  void getMinAndMaxValue(TYPE &minValue, TYPE &maxValue);
  // find the lowest and highest value in one channel
  void getMinAndMaxValue(TYPE &minValue, TYPE &maxValue, word channel);

  void scaleAndBias(float scale, TYPE bias = 0);

  void copyMapToChannel(vuFixelMap<1,TYPE> *map, word channel=0);

  //! copies one channel to map, if map==NULL, a new map instance will created
  void getChannel(vuFixelMap<1,TYPE>* &map, word channel);

  void clear(vuFixel<SIZE,TYPE> clearColour);

  bool hasSameDimensions(vuFixelMap_ST *other);  
/* ----------------------------------------------------------------------- */
/* --- some Open Gl related things --------------------------------------- */
/* ----------------------------------------------------------------------- */

  //!Initialize openGL
  void initOpenGL(void);

  //!OpenGl Resize callback
  void glResize(dword width, dword height);

  //! Renders the Map
  void glRender();



  // operator overloading
  vuFixelMap &operator=(const vuFixelMap &other); // deep copy;

  vuFixelMap &operator=(const vuFixel<SIZE,TYPE> &fixel); // clears fixelMap


  vuFixelMap &operator+=(vuFixelMap &other);   // add two FixelMaps

  vuFixelMap &operator-=(vuFixelMap &other);   // subtract two FixelMaps

  vuFixelMap &operator+=(float bias);          // add a float

  vuFixelMap &operator-=(float bias);          // substract a float

  vuFixelMap &operator*=(float scale);         // multiply image and float

  vuFixelMap &operator/=(float scale);         // divide image by float

  bool writeToFileStream(ostream *out);

  bool readFromFileStream(istream *in, dword width, dword height);

  void assignBuffer(TYPE *buffer, dword width, dword height);

  void writeBufferToFile(FILE *file);

  // transforming fixelmaps

  void _shearX(float shear, vuFixelMap<SIZE,TYPE>* inMap,
	       vuFixelMap<SIZE,TYPE>* &outMap);
  void _shearY(float shear, vuFixelMap<SIZE,TYPE>* inMap,
	       vuFixelMap<SIZE,TYPE>* &outMap);

  void rotate90();
  void rotate180();
  void rotate270();

  void rotate(float angle);
  
  inline dword index(dword width, dword height)
  { 
    return  (height * m_width + width) * SIZE;
  }
  
  
// ------------------------------------------------------------------------
// --- STREAM I/O ---------------------------------------------------------
// ------------------------------------------------------------------------


  ostream &write(ostream& os);

  istream &read(istream& is);

 private:
  void _ensureBuffer(dword sizx, dword sizy, const TYPE* source=NULL);
};

template class vuFixelMap<1,byte>;
template class vuFixelMap<2,byte>;
template class vuFixelMap<3,byte>;

template class vuFixelMap<1,float>;
template class vuFixelMap<2,float>;
template class vuFixelMap<3,float>;


typedef vuFixelMap<1,byte> vuFixelMap1B;
typedef vuFixelMap<2,byte> vuFixelMap2B;
typedef vuFixelMap<3,byte> vuFixelMap3B;

typedef vuFixelMap<1,float> vuFixelMap1F;
typedef vuFixelMap<2,float> vuFixelMap2F;
typedef vuFixelMap<3,float> vuFixelMap3F;

#endif /* __VU_FIXELMAP_H__ */
