#include "vuFixelMap.h"
#include <stddef.h>
#include <fstream.h>
#include <GL/gl.h>
#include "vuFixelType.h"
#include "math.h"

template <int S, class T>
vuFixelMap<S,T>::vuFixelMap()
{
  m_buffer        = NULL;
  m_width         = 0;
  m_height        = 0;
  m_isBufferNewed = true;
}

template <int S, class T>
vuFixelMap<S,T>::vuFixelMap(dword width, dword height)
{
  m_buffer        = NULL;
  m_width         = 0;
  m_height        = 0;
  m_isBufferNewed = true;
  _ensureBuffer(width, height);
}

template <int S, class T>
vuFixelMap<S,T>::~vuFixelMap()
{
  if((m_buffer != NULL) && m_isBufferNewed) delete m_buffer;
  m_isBufferNewed = true;
  m_buffer        = NULL;
}

template <int S, class T>
vuFixelMap<S,T>::vuFixelMap(const vuFixelMap &other)
{
  // cerr << "+++ fixelMap copy constructor" << endl;
  m_buffer        = NULL;
  m_width         = 0;
  m_height        = 0;
  m_isBufferNewed = true;

  _ensureBuffer(other.m_width, other.m_height, other.getBuffer());
}

template <int S, class T>
void vuFixelMap<S,T>::setWidthAndHeight(const dword sizx, const dword sizy)
{
  _ensureBuffer(sizx, sizy);
}

template <int S, class T>
dword vuFixelMap<S,T>::getWidth() const
{
  return m_width;
}

template <int S, class T>
dword vuFixelMap<S,T>::getHeight() const
{
  return m_height;
}

template <int S, class T>
void vuFixelMap<S,T>::setFixel(dword x, dword y, const vuFixel<S,T> &fixel)
{
  T *pix = &m_buffer[(m_width*y+x)*S];
  for (dword i=0; i<S; i++) pix[i] = fixel[i];
}

template <int S, class T>
vuFixel<S,T> vuFixelMap<S,T>::getFixel(dword x, dword y)
{
  T *ptr = m_buffer+((m_width*y+x)*S);
  return vuFixel<S,T>(ptr);
}

template <int S, class T>
const T* vuFixelMap<S,T>::getBuffer() const
{
  return m_buffer;
}

template <int S, class T>
T* vuFixelMap<S,T>::getBuffer()
{
  return m_buffer;
}

/* ----------------------------------------------------------------------- */
/* --- some Open Gl related things --------------------------------------- */
/* ----------------------------------------------------------------------- */

template <int S, class T>
void vuFixelMap<S,T>::initOpenGL(void)
{
  glDisable(GL_LIGHTING);
}

template <int S, class T>
void vuFixelMap<S,T>::glResize(dword width, dword height)
{
  if (m_width == 0 || m_height == 0)
    cerr << "WARNING: vuFixelMap.glResize(): width or height is 0." << endl;
  glPixelZoom((float)width/m_width,(float)height/m_height);
}

template <int S, class T>
void vuFixelMap<S,T>::glRender()
{
  if (vuFixelType<T>::isByte()) {
    if (S == 1) {
      glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
      glPixelStorei(GL_UNPACK_ROW_LENGTH, m_width);
      //glRasterPos2i(0,0);
      glDrawPixels(m_width,m_height,GL_LUMINANCE,GL_UNSIGNED_BYTE,m_buffer);
    }
    else if (S == 2) {
      glDrawPixels(m_width, m_height, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, m_buffer);
    }
    else if (S == 3) {
      glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
      glPixelStorei(GL_UNPACK_ROW_LENGTH, m_width);
      glDrawPixels(m_width, m_height, GL_RGB, GL_UNSIGNED_BYTE, m_buffer);
    }
    else {
      cerr << "WARNING: vuFixelMap.glRender():: case not implemented yet";
      cerr << " (S=" << S << ")" << endl;
    }
  }
  else if (vuFixelType<T>::isFloat()) {
    if (S == 1) {
      glDrawPixels(m_width, m_height, GL_LUMINANCE, GL_FLOAT, m_buffer);
    }
    else if (S == 2) {
      glDrawPixels(m_width, m_height, GL_LUMINANCE_ALPHA, GL_FLOAT, m_buffer);
    }
    else if (S == 3) {
      glDrawPixels(m_width, m_height, GL_RGB, GL_FLOAT, m_buffer);
    }
    else {
      cerr << "WARNING: vuFixelMap.glRender():: case not implemented yet";
      cerr << " (S=" << S << ")" << endl;
    }
  }
  else {
    cerr << "WARNING: vuFixelMap.glRender():: case not implemented yet";
    cerr << " (T unknown)" << endl;
  }
}

template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator=(const vuFixelMap<S,T> &other)
{
  if (this == &other) return *this;
  _ensureBuffer(other.m_width, other.m_height, other.getBuffer());
  return *this;
}

template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator=(const vuFixel<S,T> &fixel)
{
  dword   len = getWidth() * getHeight();
  T *ptr = getBuffer();

  for (;len>0;len--) {
    for (dword i=0; i<S; i++,ptr++) *ptr = fixel[i];
  }

  return *this;
}

template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator+=(vuFixelMap<S,T> &other)
{
  if (m_width != other.getWidth() || m_height != other.getHeight()) {
    cerr << "Warning: vuFixelMap::operator+= width or height are not equal!";
    cerr << endl;
    return *this;
  }

  dword   len  = getWidth() * getHeight() * S;
  T *dest = getBuffer();
  T *src  = other.getBuffer();

  for (;len>0;len--) *(dest++) += *(src++);

  return *this;
}

template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator+=(float bias)
{
  dword   len  = getWidth() * getHeight() * S;
  T *dest = getBuffer();

  float minValue = vuFixelType<T>::getMinValue();
  float maxValue = vuFixelType<T>::getMaxValue();

  for (;len>0;len--) {
    register float tmp = (*dest + bias);
    
    if (tmp > maxValue) 
      tmp = maxValue;
    else if (tmp < minValue)
      tmp = minValue;

    *dest = (T)tmp;
  }

  return *this;
}

template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator-=(float bias)
{
  dword   len  = getWidth() * getHeight() * S;
  T *dest = getBuffer();

  float minValue = vuFixelType<T>::getMinValue();
  float maxValue = vuFixelType<T>::getMaxValue();

  for (;len>0;len--) {
    register float tmp = (*dest - bias);
    
    if (tmp > maxValue) 
      tmp = maxValue;
    else if (tmp < minValue)
      tmp = minValue;

    *(dest++) = (T)tmp;
  }

  return *this;
}



template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator-=(vuFixelMap<S,T> &other)
{
  if (m_width != other.getWidth() || m_height != other.getHeight()) {
    cerr << "Warning: vuFixelMap::operator-= width or height are not equal!";
    cerr << endl;
    return *this;
  }

  dword   len  = getWidth() * getHeight() * S;
  T *dest = getBuffer();
  T *src  = other.getBuffer();

  for (;len>0;len--) *(dest++) -= *(src++);

  return *this;
}


template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator*=(float scale)
{
  T  *ptr = getBuffer();
  dword  len = getWidth() * getHeight() * S;

  float minValue = vuFixelType<T>::getMinValue();
  float maxValue = vuFixelType<T>::getMaxValue();

  for (;len>0;len--,ptr++) {
    register float tmp = (*ptr * scale);
    
    if (tmp > maxValue) 
      tmp = maxValue;
    else if (tmp < minValue)
      tmp = minValue;

    *ptr = (T)tmp;
  }

  return *this;
}

template <int S, class T>
vuFixelMap<S,T> &vuFixelMap<S,T>::operator/=(float scale)
{
  T  *ptr = getBuffer();
  dword  len = getWidth() * getHeight() * S;

  float minValue = vuFixelType<T>::getMinValue();
  float maxValue = vuFixelType<T>::getMaxValue();

  for (;len>0;len--,ptr++) {
    register float tmp = (*ptr / scale);
    
    if (tmp > maxValue) 
      tmp = maxValue;
    else if (tmp < minValue)
      tmp = minValue;

    *ptr = (T)tmp;
  }

  return *this;
}

template <int S, class T>
bool vuFixelMap<S,T>::writeToFileStream(ostream *out)
{
  dword size = m_width * m_height * S;

  if (out && (size > 0) && m_buffer) {
    out->write((const char *)m_buffer, size * sizeof(T));
    return !out->bad();
  }
  return false;
}

template <int S, class T>
bool vuFixelMap<S,T>::readFromFileStream(istream *in, dword width, dword height)
{
  dword size = width * height * S;

  if (in && (size > 0)) {
    if (m_isBufferNewed) CHECKNDELETE (m_buffer);

    T *buffer = new T[size];
    in->read((char *)buffer, size * sizeof(T));
    if (in->bad()) return false;
    _ensureBuffer(width, height, buffer);
    return true;
  }
  return false;
}

template <int S, class T>
void vuFixelMap<S,T>::assignBuffer(T *buffer, dword width, dword height)
{
  if (buffer != NULL) {
    if ((m_buffer != NULL) && m_isBufferNewed) delete m_buffer;
    m_buffer = buffer; // this is NOT "new"ed!!!
    m_isBufferNewed = false;
    m_width  = width;
    m_height = height;
  }
}

template <int S, class T>
void vuFixelMap<S,T>::writeBufferToFile(FILE *file)
{
  fwrite(m_buffer, sizeof(T), S * m_width * m_height, file);
}

// ------------------------------------------------------------------------
// --- finding min and max value ------------------------------------------
// ------------------------------------------------------------------------

template <int S, class T>
void vuFixelMap<S,T>::getMinAndMaxValue(T &minValue, T &maxValue)
{
  T minVal, maxVal;

  minValue = vuFixelType<T>::getPosInfinity();
  maxValue = vuFixelType<T>::getNegInfinity();

  for (int i=0; i<S; i++) {

    getMinAndMaxValue(minVal, maxVal, i);

    if (minValue > minVal) minValue = minVal;
    if (maxValue < maxVal) maxValue = maxVal;
  }
}

template <int S, class T>
void vuFixelMap<S,T>::getMinAndMaxValue(T &minValue, T &maxValue, word channel)
{
  T     *ptr = getBuffer();
  dword cnt  = getWidth() * getHeight();

  // ??? WARNING: might NOT work for unsigned types... ???
  minValue = vuFixelType<T>::getPosInfinity();
  maxValue = vuFixelType<T>::getNegInfinity();

  for (dword i=0; i<cnt; i++) {
    register T tmp = ptr[channel];
    
    if (minValue > tmp) minValue = tmp;
    if (maxValue < tmp) maxValue = tmp;
    ptr += S;
  }
}

// ------------------------------------------------------------------------
// --- scale and bias -----------------------------------------------------
// ------------------------------------------------------------------------

template <int S, class T>
void vuFixelMap<S,T>::scaleAndBias(float scale, T bias)
{
  T  *ptr = getBuffer();
  dword  len = getWidth() * getHeight() * S;
  float minValue = vuFixelType<T>::getMinValue();
  float maxValue = vuFixelType<T>::getMaxValue();

  for (;len>0;len--,ptr++) {
    register float tmp = (float)(*ptr * scale) + (float)bias;
    if (tmp < minValue)
      tmp = (T)minValue;
    else if (tmp > maxValue)
      tmp = (T)maxValue;
    *ptr = (T)tmp;
  }
}

template <int S, class T>
void vuFixelMap<S,T>::copyMapToChannel(vuFixelMap<1,T> *map, word channel)
{
  if (channel>=S) {
    cerr << "Warning: vuFixelMap::copyMapToChannel: channel>= SIZE" << endl;
    return;
  }
  T     *dest = getBuffer();
  T     *src  = map->getBuffer();
  dword cnt   = getWidth() * getHeight();

  for (dword i=0; i<cnt; i++) {
    dest[channel] = *(src++);
    dest += S;
  }
}

template <int S, class T>
void vuFixelMap<S,T>::getChannel(vuFixelMap<1,T>* &map, word channel)
{
  if (channel>=S) {
    cerr << "Warning: vuFixelMap::getChannel: channel>= SIZE" << endl;
    return;
  }

  if (map == NULL) map = new vuFixelMap<1,T>(getWidth(), getHeight());

  T     *dest = map->getBuffer();
  T     *src  = getBuffer();
  dword cnt   = getWidth() * getHeight();

  for (dword i=0; i<cnt; i++) {
    *(dest++) = src[channel];
    src += S;
  }
}

template <int S, class T>
void vuFixelMap<S,T>::clear(vuFixel<S,T> clearColour)
{
  T     *ptr = getBuffer();
  dword  len = getWidth() * getHeight();

  for (;len>0;len--,ptr++) {
    for (int i=0; i<S; i++) {
      ptr[i] = clearColour[i];
    }
  }
}

template <int S, class T>
bool vuFixelMap<S,T>::hasSameDimensions(vuFixelMap_ST *other)
{
  if (other == NULL)
    return false;
  else if (this->getWidth() != other->getWidth())
    return false;
  else if (this->getHeight() != other->getHeight())
    return false;

  return true;
}



// ------------------------------------------------------------------------
// --- STREAM I/O ---------------------------------------------------------
// ------------------------------------------------------------------------

template <int S, class T>
ostream &vuFixelMap<S,T>::write(ostream& os)
{
  T *ptr = getBuffer();
  dword   len = getWidth() * getHeight() * S;

  for (;len>0;len--,ptr++) {
    os << (*ptr) << " ";
  }
  return os;
}

template <int S, class T>
istream &vuFixelMap<S,T>::read(istream& is)
{
  T *ptr = getBuffer();
  dword   len = getWidth() * getHeight() * S;

  for (;len>0;len--,ptr++) {
    is >> *ptr;
  }
  return is;
}

template <int S, class T>
void vuFixelMap<S,T>::_ensureBuffer(dword sizx, dword sizy, const T* source)
{
  if (m_buffer == NULL) m_isBufferNewed = true;
    
  if (!m_isBufferNewed && (sizx != m_width || sizy != m_height)) {
    cerr << "ERROR: vuFixelMap._ensureBuffer(): can't change buffer size";
    cerr << " , since an external buffer is used" << endl;
    throw "vuFixelMap._ensureBuffer(): can't change buffer size, since"
      "an external buffer is used";
    return;
  }

  if ((sizx != m_width) || (sizy != m_height)) {
    m_width=sizx;
    m_height=sizy;
    if ((m_buffer != NULL) && m_isBufferNewed) {
      delete m_buffer;
      m_buffer = NULL;
    }
  }

  if ((m_width == 0) || (m_height == 0)) return;

  if(m_buffer == NULL) {
    m_buffer        = new T[m_width*m_height*S];
    m_isBufferNewed = true;
  }

  if (source == NULL) {
    T *ptr = m_buffer;
    dword  len = m_width*m_height*S;

    for(;len>0;len--,ptr++) *ptr = 0;
  }
  else {
    T        *dest = m_buffer;
    const  T *src  = source;
    dword    len   = m_width*m_height*S;

    for(;len>0;len--,dest++,src++) *dest= *src;
  }
}


/* ************************************************************************* */
/* *** rotation ************************************************************ */
/* ************************************************************************* */

template <int S,class T>
void vuFixelMap<S,T>::_shearX(float shear, vuFixelMap<S,T>* inMap,
			      vuFixelMap<S,T>* &outMap)
{
  if (inMap == NULL) return;
  
  dword inHeight  = inMap->getHeight();
  dword inWidth   = inMap->getWidth();
  dword outHeight = inHeight;
  dword outWidth  = (dword)(inHeight * fabs(shear)) + inWidth;

  if (outMap == NULL) {
    outMap = new vuFixelMap<S,T>(outWidth, outHeight);
  }
  else if (outMap->getWidth()!=outWidth || outMap->getHeight()!=outHeight) {
    CHECKNDELETE(outMap);
    outMap = new vuFixelMap<S,T>(outWidth, outHeight);
  }
  *outMap = (T)0;

  T left[S];
  T oldLeft[S];
  T *inBuffer  = inMap->getBuffer();
  
  for (dword j=0; j<inHeight; j++) { // inHeight
    float skew    = shear * j;
    int   skewi   = (int)(skew);   // floor
    float skewf   = skew - skewi;  // fract
    
    for (int k=0; k<S; k++) oldLeft[k] = 0;

    dword idx        = (shear >= 0) ? skewi : outWidth + skewi - inWidth;
    T     *outBuffer = outMap->getBuffer();

    outBuffer += (j*outWidth + idx) * S;
    for (dword i=0; i<inWidth; i++) { //inWidth
      for (int k=0; k<S; k++) {
	register T     fixel = *(inBuffer++);
	register float tmp   = fixel * skewf;

	if (tmp < 0.0f)        tmp = 0.0f;
	else if (tmp > 255.0f) tmp = 255.0f;
	
	left[k]        = (T)tmp;
	fixel          = fixel - left[k] + oldLeft[k];
	*(outBuffer++) = fixel;
	oldLeft[k]     = left[k];
      }
    }
  }
}

template <int S, class T>
void vuFixelMap<S,T>::_shearY(float shear, vuFixelMap<S,T>* inMap,
			      vuFixelMap<S,T>* &outMap)
{
  if (inMap == NULL) return;
  
  dword inHeight  = inMap->getHeight();
  dword inWidth   = inMap->getWidth();
  dword outHeight = (dword)(inWidth * fabs(shear)) + inHeight;
  dword outWidth  = inWidth;

  if (outMap == NULL) {
    outMap = new vuFixelMap<S,T>(outWidth, outHeight);
  }
  else if (outMap->getWidth() != outWidth || outMap->getHeight() != outHeight){
    CHECKNDELETE(outMap);
    outMap = new vuFixelMap<S,T>(outWidth, outHeight);
  }
  *outMap = (T)0;

  T old[S];
  T *inBuffer  = inMap->getBuffer();
  T *outBuffer = outMap->getBuffer();
  
  for (dword j=0; j<inWidth; j++) { // j<inWidth
    float skew    = shear * j;
    int   skewi   = (int)(skew);   // floor
    float skewf   = skew - skewi;  // fract

    for (int k=0; k<S; k++) old[k] = 0;

    dword idx = (shear >= 0) ? skewi : outHeight + skewi - inHeight;
    for (dword i=0; i<inHeight; i++) { // i<inHeight
      dword idx1 = (j+(i*inWidth)) * S;
      dword idx2 = (j+(idx+i)*outWidth) * S;

      for (int k=0; k<S; k++) {	
	register T     fixel = inBuffer[idx1];
	register float valF  = fixel * skewf;

	if (valF < 0.0f)        valF = 0.0f;
	else if (valF > 255.0f) valF = 255.0f;

	register T     valT  = (T)valF;

	fixel           = fixel - valT + old[k];
	outBuffer[idx2] = fixel;
	old[k]          = valT;
	idx1++;
	idx2++;
      }
    }
  }
}

template <int S, class T>
void vuFixelMap<S,T>::rotate90() 
{
  if (m_width != m_height) {
    cerr << "rotate90(): width != height, case not implemented!!!" << endl;
    throw "rotate90(): width != height, case not implemented!!!";
  }
  
  dword y_step  = m_width * S;
  T     *buffer = getBuffer();

  vuFixelMap<S,T> *myMap = new vuFixelMap<S,T>(m_width, m_height);
  *myMap = *this; // copy image to myMap
  T *ptr = myMap->getBuffer();
  *this  = (T)0; // clear image;

  for (dword j=0; j<m_height; j++) {
    T *s_ptr = ptr;
    T *d_ptr = buffer + (m_width - j) * S;
    for (dword i=0; i<m_width; i++) {
      for (int k=0; k<S; k++) {
	*(d_ptr++) = *(s_ptr++);
      }
      d_ptr += y_step - S;
    }
    ptr +=  y_step;
  }
  CHECKNDELETE(myMap);
}

template <int S, class T>
void vuFixelMap<S,T>::rotate180() 
{
  dword count  = m_width * m_height;
  T     *d_ptr = getBuffer();
  T     *s_ptr = getBuffer();
  d_ptr += (count - 1) * S;

  count = count / 2;
  
  for (dword i=0; i<count; i++) {
    for (dword j=0; j<S; j++) {
      T tmp = *s_ptr;
      *(s_ptr++) = *d_ptr;
      *(d_ptr++) = tmp;
    }
    d_ptr -= S+S;
  }
}

template <int S, class T>
void vuFixelMap<S,T>::rotate270()
{
  rotate180();
  rotate90();
}


template <int S, class T>
void vuFixelMap<S,T>::rotate(float angle)
{
  if (angle == 0) return;
  
  vuFixelMap<S,T> *map1 = NULL;
  vuFixelMap<S,T> *map2 = NULL;

  while (angle > 360) angle -= 360;
  while (angle < 0)   angle += 360;

  if (angle >= 315)
    angle -= 360;
  else if (angle >= 225) {
    angle -= 270;
    rotate270();
  }
  else if (angle >= 135) {
    angle -= 180;
    rotate180();
  }
  else if (angle >= 45) {
    angle -= 90;
    rotate90();
  }

  angle *= M_PI / 180;
  
  float alpha = -tan(angle/2);
  float beta  = sin(angle);
    
  // do the three shear steps
  _shearX(alpha, this, map1);
  _shearY(beta,  map1, map2);
  CHECKNDELETE(map1);
  _shearX(alpha, map2, map1);
  CHECKNDELETE(map2);

  // reduce resulting image to original image size
  dword outWidth  = map1->getWidth();
  dword outHeight = map1->getHeight();
  dword deltaX    = (outWidth  - m_width)  / 2;
  dword deltaY    = (outHeight - m_height) / 2;
  dword y_step    = outWidth * S;

  T *s_ptr = map1->getBuffer() + y_step * deltaY + deltaX * S;
  T *d_ptr = getBuffer();

  for (dword j=0; j<m_height; j++) {
    T *ptr = s_ptr;
    for (dword i=0; i<m_width; i++) {
      for (int k=0; k<S; k++) {
	*(d_ptr++) = *(ptr++);
      }
    }
    s_ptr += y_step;
  }
  CHECKNDELETE(map1);
}
