/** vuFixel defines a basic template class for a fixel.
    
    A fixel is an arbitrary FIeld ELement, comparable to a colour, but does
    not support alpha.

    It's a one dimensional vector of length SIZE and basic type TYPE.
*/

#ifndef _VU_FIXEL_H_
#define _VU_FIXEL_H_

#include <iostream.h>
#include <stdio.h>
#include "vuSimpleTypes.h"
#include "vuFixelType.h"


template <int SIZE, class TYPE>
class vuFixel
{
public:
    /** default constructor. Sets all values to zero. */
    vuFixel();

    /** copy constructor */
    vuFixel(const vuFixel<SIZE,TYPE>& inst);

    /** initialize all components with values from array */
    vuFixel(const TYPE *array);

    /** initialize all values with value */
    vuFixel(const TYPE value);

    /** destructor */
    ~vuFixel();

    /** provides direct random access to the components */
    TYPE& operator[](unsigned int index);

    /** provides direct reading access to the components */
    TYPE operator[](unsigned int index) const;

    /** returns a pointer to the TYPE array containing the data */
    TYPE *getBuffer();

    /** const version of  getBuffer() */
    TYPE const *getBuffer() const;

    //!assignment from const TYPE*
    vuFixel<SIZE,TYPE>& operator=(const TYPE* rhs);

    //!assignment from TYPE*
    vuFixel<SIZE,TYPE>& operator=(TYPE* rhs);

    /** assignment operator */
    vuFixel<SIZE,TYPE>& operator=(const vuFixel<SIZE,TYPE>& rhs);

    /** operator ==,
        This returns true if the two fixels are equivalent. */
    bool operator==(const vuFixel<SIZE,TYPE>& rhs) const;

    /** addition operator,
        This performs a component wise addition of each component of the 2
	fixels and returns the result. */
    vuFixel<SIZE,TYPE> operator+(const vuFixel<SIZE,TYPE>& rhs) const;

    /** addition operator,
    	This will add a to the all components */
    vuFixel<SIZE,TYPE> operator+(TYPE a) const;

    /** minus operator,
	This will perform a component wise subtraction on all components*/
    vuFixel<SIZE,TYPE> operator-(const vuFixel<SIZE,TYPE>& rhs) const;

    /** multiply operator,
	This will perform a component wise multiplication on all components */
    vuFixel<SIZE,TYPE> operator*(const vuFixel<SIZE,TYPE>& rhs) const;

    /** divide operator,
    	This will perform a component wise division on all components */
    vuFixel<SIZE,TYPE> operator/(const vuFixel<SIZE,TYPE>& rhs) const;

    /** operator multiplication with scalar
        In comparision to *= this operator creates a copy where the result
        written to. */
    vuFixel<SIZE,TYPE> operator*(TYPE rhs) const;

    /** operator for component wise addition */
    vuFixel<SIZE,TYPE>& operator+=(const vuFixel<SIZE,TYPE>& rhs);

    /** operator for component wise subtraction */
    vuFixel<SIZE,TYPE>& operator-=(const vuFixel<SIZE,TYPE>& rhs);

    /** operator for component wise multiplication */
    vuFixel<SIZE,TYPE>& operator*=(const vuFixel<SIZE,TYPE>& rhs);

    /** operator for component wise division */
    vuFixel<SIZE,TYPE>& operator/=(const vuFixel<SIZE,TYPE>& rhs);
    
     /** operator for multiplication with scalar */
    vuFixel<SIZE,TYPE>& operator*=(TYPE rhs);

    /** operator for division by scalar */
    vuFixel<SIZE,TYPE>& operator/=(TYPE rhs);

    /** returns the number of components.
        This depends on the SIZE given during creation of the template. */
    word size() const;

    /** returns value of maximal component */
    TYPE maxComponent() const;

    /** returns value of minimal component */
    TYPE minComponent() const;

#if 0
    /** operator for commutative multiplication with a TYPE.
        In comparision to *= this operator creates a copy where the result
        written to. 
    */
    friend vuFixel<SIZE,TYPE>
      operator*(TYPE lhs, const vuFixel<SIZE,TYPE>& rhs);

    //!a friend, write fixel to stream
    friend ostream& operator<<(ostream& os, const vuFixel<SIZE,TYPE>& A);

    //! a friend, read fixel from stream
    friend istream& operator>>(istream& is, vuFixel<SIZE,TYPE>& A);
#endif

    //! a friend, read fixel from stream
    istream& read (istream& is);

    //! a friend, write fixel to stream
    ostream& write(ostream& os) const;

protected:
    TYPE m_buffer[SIZE]; // !<these are the fixel component values
};

template class vuFixel<1,byte>;
template class vuFixel<2,byte>;
template class vuFixel<3,byte>;

template class vuFixel<1,float>;
template class vuFixel<2,float>;
template class vuFixel<3,float>;


typedef vuFixel<1,byte> vuFixel1B;
typedef vuFixel<2,byte> vuFixel2B;
typedef vuFixel<3,byte> vuFixel3B;

typedef vuFixel<1,float> vuFixel1F;
typedef vuFixel<2,float> vuFixel2F;
typedef vuFixel<3,float> vuFixel3F;

#endif /* _VU_FIXEL_H_ */
