#include "vuFourierCluster.h"
#include <math.h>
#include <stddef.h>

#define MAX_FILTER_SIZE 6 // max filter size

#define ODD(x) ((x)&1)

template <int S, class T>
vuFourierCluster<S,T>::vuFourierCluster()
{
  m_PlanWidth                = 0;
  m_PlanHeight               = 0;
  m_IsPreprocessed           = false;
  m_IsPreparedForInteractive = false;
  m_WeightVolume             = NULL;
  m_SliceFilter              = NULL;
  m_CacheVolume              = NULL;
}

template <int S, class T>
vuFourierCluster<S,T>::~vuFourierCluster()
{
  destroyPlan();
  CHECKNDELETE(m_WeightVolume);
  CHECKNDELETE(m_CacheVolume);
  // m_SliceFilter is an outside living object
}

template <int S, class T>
bool vuFourierCluster<S,T>::isPreprocessed()
{
  return m_IsPreprocessed;
}

template <int S, class T>
bool vuFourierCluster<S,T>::isPreparedForInteractive()
{
  return m_IsPreparedForInteractive;
}

template <int S, class T>
void vuFourierCluster<S,T>::setNoInteractiveMode()
{
  m_IsPreparedForInteractive = false;
  CHECKNDELETE(m_CacheVolume);
  CHECKNDELETE(m_WeightVolume);
  // ToDo: delete in case of true
}

template <int S, class T>
void vuFourierCluster<S,T>::prepareForInteractive(dword width, dword height)
{
  initializeVolume(width, height);

  CHECKNDELETE(m_CacheVolume);
  
  // initialize interactive cache
  {
    dword cnt     = m_XSize * m_YSize * m_ZSize * 2 *S;
    m_CacheVolume = new float[cnt];
  
    float *ptr = m_CacheVolume;

    for (dword i=0; i<cnt; i++) *(ptr++) = 0.0f;
  }
  
  wrapAndInitialize(1.0);
  
  m_IsPreparedForInteractive = true;
  m_IsPreprocessed           = true;
}

template <int S, class T>
void vuFourierCluster<S,T>::addView(vuSphericView<S,T> *view)
{
  if (m_SliceFilter == NULL) {
    cerr << "vuFourierCluster::addView() no slice filter set" << endl;
    return;
  }

  if (!m_IsPreparedForInteractive) {
    cerr << "vuFourierCluster::addView() is not prepared for interactive";
    cerr << " reconstruction!" << endl;
    return;
  }
 
  // add slice to m_CacheVolume
  {  
    float *tmpVolume = m_Volume;

    // use cache volume for slicing...
    m_Volume = m_CacheVolume;
  
    addViewToVolume(view);

    m_Volume = tmpVolume;
  }
  
  //weight m_CacheVolume with m_WeightVolume...
  {    
    dword sliceWidth, sliceHeight;
    calcSliceDimensions(view, sliceWidth, sliceHeight);  
    handleSlice(NULL, sliceWidth, sliceHeight, view);
  }
}

template <int S, class T>
void vuFourierCluster<S,T>::preprocess(dword numOfViews,
				       vuSphericView<S,T> **views,
				       vuProgressHandler *handler)
{
  if (numOfViews == 0) return;  

  dword width  = views[0]->getWidth();
  dword height = views[0]->getHeight();

  // progress handler variables...
  dword constFact    = 10;
  dword range        = 0;
  float progress     = 0.0f;
  float delta        = 0.0f;

  if (handler) {
    range = handler->getRange();
    delta = (float)range / (float)(numOfViews+2*constFact);    
  }

  if (handler) handler->update((int)progress, "Initialize Volume");
  initializeVolume(width, height);
  progress += delta*constFact;

  if (m_SliceFilter != NULL) {
    for (dword i=0; i<numOfViews; i++) {

      if (handler) {
	vuString msg("add View: ");
	msg += i;
	progress += delta;
	if (!handler->update((int)progress, msg)) break;
      }
      else
	cerr << "add slice " << i << "...";

      addViewToVolume(views[i]);

      if (!handler) cerr << "done." << endl;
    }
    if (handler) handler->update((int)progress, "Normalize Volume");

    normalizeVolume();

    if (handler) {
      progress += delta * constFact;
      handler->update((int)progress, "Initialize FFT");
    }
  }
  else {    
    cerr << "vuFourierCluster::preprocess(): m_SliceFilter is not set!"<<endl;
  }

  wrapAndInitialize(1.0);
  
  m_IsPreprocessed           = true;
  m_IsPreparedForInteractive = false;// not prepared for interactive reconstr.
}

template <int S, class T>
void vuFourierCluster<S,T>::normalizeVolume()
{
  dword size   = m_XSize * m_YSize * m_ZSize;
  float *v_ptr = m_Volume;
  float *w_ptr = m_WeightVolume;
  int   step   = S * 2;
  
  for (dword i=0; i<size; i++) {
    for (int j=0; j<step; j++) {
      if (*w_ptr == 0)
	*(v_ptr++) = 0.0f;
      else 
	*(v_ptr++) /= *w_ptr;
    }
    w_ptr++;
  }
  
  CHECKNDELETE(m_WeightVolume);
}

template <int S, class T>
void vuFourierCluster<S,T>::setSliceFilter(vuSliceFilter *sliceFilter)
{
  m_SliceFilter = sliceFilter;
}

template <int S, class T>
vuSliceFilter *vuFourierCluster<S,T>::getSliceFilter()
{
  return m_SliceFilter;
}


/* ************************************************************************* */
/* *** private functions *************************************************** */
/* ************************************************************************* */

template <int S, class T>
void vuFourierCluster<S,T>::initializeVolume(dword width, dword height)
{
  // don't change the next line, it would have strong effect on the algorithm! 
  dword dim = computeDimensions(width,height,height, M_SQRT2,MAX_FILTER_SIZE);
  dword cnt = m_XSize * m_YSize * m_ZSize;

  if (m_Volume == NULL || m_XSize != dim || m_YSize != dim || m_ZSize != dim) {
    m_XSize = m_YSize = m_ZSize = dim;
    cnt     = m_XSize * m_YSize * m_ZSize;

    CHECKNDELETE(m_Volume);    
    m_Volume       = new float[2*cnt*S];
  }

  CHECKNDELETE(m_WeightVolume);
  CHECKNDELETE(m_CacheVolume); // just in case ;-)

  m_WeightVolume = new float[cnt];

  float *v_ptr = m_Volume;
  float *w_ptr = m_WeightVolume;

  for (dword i=0; i<cnt; i++) *(w_ptr++) = 0.0f;
  cnt = cnt * S * 2;
  for (dword i=0; i<cnt; i++) *(v_ptr++) = 0.0f;
}

template <int S, class T>
void vuFourierCluster<S,T>::transformSlice(float *slice, 
					   dword width, dword height)
{
  fftwnd(m_Plan, S, (fftw_complex*)slice, S, 1, NULL, 0, 0);
  dword count  = width * height * 2 * S;
  float size_f = (float) width * height;
  float *ptr   = slice;
  for (dword i = 0; i < count; ++i) *(ptr++) /= size_f;
}

template <int S, class T>
void vuFourierCluster<S,T>::ensurePlan(dword width, dword height)
{
  if ((m_PlanWidth != width) || (m_PlanHeight != height)) {
    destroyPlan();
      
    m_Plan  = fftw2d_create_plan(width, height,
				 FFTW_FORWARD, FFTW_MEASURE |
				 FFTW_IN_PLACE | FFTW_USE_WISDOM);
    m_PlanWidth  = width;
    m_PlanHeight = height;
  }
}

template <int S, class T>
void vuFourierCluster<S,T>::destroyPlan()
{
  if ((m_PlanWidth) != 0 || (m_PlanHeight != 0)) {
    fftwnd_destroy_plan(m_Plan);
    m_PlanWidth = m_PlanHeight = 0;
  }
}

template <int S, class T>
void vuFourierCluster<S,T>::calcSliceDimensions(vuSphericView<S,T> *view,
						dword &sliceWidth,
						dword &sliceHeight)
{
  sliceWidth  = (dword)((m_XSize - MAX_FILTER_SIZE) / M_SQRT2);
  sliceHeight = (dword)((m_YSize - MAX_FILTER_SIZE) / M_SQRT2);
  int deltaWidth  = view->getWidth()  - sliceWidth;
  int deltaHeight = view->getHeight() - sliceHeight;
  
  if (deltaWidth  == 0) deltaWidth  = 1;
  if (deltaHeight == 0) deltaHeight = 1;

  if (abs(deltaWidth)  > 1) {
    cerr << "vuFourierCluster::addViewToVolume: sliceWidth does not fit"<<endl;
    return;
  }
  
  if (abs(deltaHeight) > 1) {
    cerr <<"vuFourierCluster::addViewToVolume: sliceHeight does not fit"<<endl;
    return;
  }
  
  if (ODD(sliceWidth))  sliceWidth  += deltaWidth;
  if (ODD(sliceHeight)) sliceHeight += deltaHeight;

  if ((sliceWidth != view->getWidth()) || (sliceHeight != view->getHeight())) {
    cerr << "vuFourierCluster::calcSliceDimensions:slice and view have not the"
            " same size" << endl;
    throw("vuFourierCluster::calcSliceDimensions: slice and view have not the"
	  " same size");
  }
}

template <int S, class T>
void vuFourierCluster<S,T>::addViewToVolume(vuSphericView<S,T> *view)
{
  dword sliceWidth, sliceHeight;

  // ToDo: simplify calcSliceDimension, e.g. sliceWidth = view->getWidth()
  calcSliceDimensions(view, sliceWidth, sliceHeight);
  
  float *slice = new float [sliceWidth * sliceHeight * 2 * S];

  ensurePlan(sliceWidth, sliceHeight);

  // copy view to slice
  T     *v_ptr = view->getMap()->getBuffer();
  float *s_ptr = slice;
  
  for (dword j=0; j< sliceHeight; j++) {
    for (dword i = 0; i < sliceWidth; ++i) {
      for (int k=0; k<S; k++) {
	*(s_ptr++) = (float)*(v_ptr++);
	*(s_ptr++) = 0.0f;
      }
    }
  }

  shift2D(slice, sliceWidth, sliceHeight);
  transformSlice(slice, sliceWidth, sliceHeight);
  shift2D(slice, sliceWidth, sliceHeight);

  //premultiplySlice(slice);
  handleSlice(slice, sliceWidth, sliceHeight, view); // add slice to volume

  CHECKNDELETE(slice);
}

template <int S, class T>
void vuFourierCluster<S,T>::handleSlice(float *slice,
					dword sliceWidth,
					dword sliceHeight,
					vuSphericView<S,T> *view)
{
  vuVector lookAt = view->getLookFrom() * -1;
  vuVector yStep  = view->getUp();
  vuVector xStep  = vuVector(1,0,0);

  calcViewVectors(lookAt, yStep, xStep);

  xStep.makeUnit();
  yStep.makeUnit();

  dword innerXSize = sliceWidth;
  dword innerYSize = sliceHeight;

  vuVector origin(m_XSize, m_YSize, m_ZSize);
  origin *= 0.5;

  origin += xStep*(innerXSize*-0.5f);
  origin += yStep*(innerYSize*-0.5f);

  dword XStart = 0;
  dword YStart = 0;
  dword XStop  = innerXSize;
  dword YStop  = innerYSize;

  vuVector pos = origin;

  if (m_SliceFilter->getLowPassFactor() < 1.0) {
    float lowPass = m_SliceFilter->getLowPassFactor();
    lowPass = 1.0 - sqrt(lowPass);

    dword cutOffX = innerXSize / 2;
    dword cutOffY = innerYSize / 2;

    cutOffX = dword(cutOffX * lowPass);
    cutOffY = dword(cutOffY * lowPass);

    XStart += cutOffX;
    YStart += cutOffY;
    XStop  -= cutOffX;
    YStop  -= cutOffY;

    pos += xStep * cutOffX;
    pos += yStep * cutOffY;
  }

  dword width   = m_SliceFilter->getWidth();
  mc_HalfWidth  = (float)width / 2;          // float
  mc_High       = (width/2);                 // int
  mc_Low        = -mc_High + (1- width % 2); // int
  mc_IsWidthOdd = (ODD(width));              // bool

  //mc_HalfWidth *= 1.5;

  KernelCallback kernelCB;

  if (slice == NULL)
    kernelCB = &vuFourierCluster<S,T>::doWeighting;
  else if (m_SliceFilter->getKind() == 0)
    kernelCB = &vuFourierCluster<S,T>::doFilteringSeparable;
  else
    kernelCB = &vuFourierCluster<S,T>::doFilteringSpheric;

  for (dword j = YStart; j < YStop; ++j) {
    vuVector p = pos;
    for (dword i = XStart; i < XStop; ++i) {
      (this->*kernelCB)(p, &slice[(j*innerXSize+i) * 2 * S]);
      p += xStep;
    }
    pos += yStep;
  }
}

template <int S, class T>
inline void vuFourierCluster<S,T>::doFilteringSeparable(vuVector& pos, float *value)
{
  dword center[3]; // center in Volume (grid coordinate)
  float POS[3];    // 

  if (mc_IsWidthOdd) {
    center[0] = (dword)(pos[0]+0.5f);
    center[1] = (dword)(pos[1]+0.5f);
    center[2] = (dword)(pos[2]+0.5f);
  }
  else {
    center[0] = (dword)pos[0];
    center[1] = (dword)pos[1];
    center[2] = (dword)pos[2];
  }

  POS[0] = pos[0] - center[0];
  POS[1] = pos[1] - center[1];
  POS[2] = pos[2] - center[2];

  for (int k=mc_Low; k<=mc_High; k++) {
    register float facZ = (POS[2]-k) / mc_HalfWidth;

    facZ = m_SliceFilter->getWeight(facZ);

    for (int l=mc_Low; l<=mc_High; l++) {
      register float facYZ = (POS[1]-l) / mc_HalfWidth;

      facYZ = facZ *m_SliceFilter->getWeight(facYZ);

      for (int m=mc_Low; m<=mc_High; m++) {
	register dword idx   = wcoord(center[0]+m,center[1]+l,center[2]+k);
	register float fac   = (POS[0]-m)/mc_HalfWidth;

	fac = facYZ * m_SliceFilter->getWeight(fac);

	if (fac < 0.0) cerr << fac << " "; // this should never happen

	m_WeightVolume[idx] += fac;
	idx *= (S+S);

	float *v_ptr = &m_Volume[idx];
	float *s_ptr = value;
	for (int i=0; i<S; i++) {
	  *(v_ptr++) += fac * *(s_ptr++);
	  *(v_ptr++) += fac * *(s_ptr++);
	}
      }
    }
  }
}

template <int S, class T>
inline void vuFourierCluster<S,T>::doFilteringSpheric(vuVector& pos, float *value)
{
  dword center[3]; // center in Volume (grid coordinate)
  float POS[3];    // 

  if (mc_IsWidthOdd) {
    center[0] = (dword)(pos[0]+0.5f);
    center[1] = (dword)(pos[1]+0.5f);
    center[2] = (dword)(pos[2]+0.5f);
  }
  else {
    center[0] = (dword)pos[0];
    center[1] = (dword)pos[1];
    center[2] = (dword)pos[2];
  }

  POS[0] = pos[0] - center[0];
  POS[1] = pos[1] - center[1];
  POS[2] = pos[2] - center[2];

  for (int k=mc_Low; k<=mc_High; k++) {
    for (int l=mc_Low; l<=mc_High; l++) {
      for (int m=mc_Low; m<=mc_High; m++) {
	register float xx    = POS[0] - m;
	register float yy    = POS[1] - l;
	register float zz    = POS[2] - k;
	register float fac   = sqrt(xx*xx+yy*yy+zz*zz) / mc_HalfWidth;
	register dword idx   = wcoord(center[0]+m,center[1]+l,center[2]+k);

        fac = m_SliceFilter->getWeight(fac);

	if (fac < 0.0) cerr << fac << " "; // this should never happen

	m_WeightVolume[idx] += fac;
	idx *= (S+S);

	float *v_ptr = &m_Volume[idx];
	float *s_ptr = value;
	for (int i=0; i<S; i++) {
	  *(v_ptr++) += fac * *(s_ptr++);
	  *(v_ptr++) += fac * *(s_ptr++);
	}
      }
    }
  }
}

// --------------------------------------------------------------------------
// --- private - interactive reconstruction - weighting - functions ---------
// --------------------------------------------------------------------------

// ?_? to be done:  (channels)
template <int S, class T>
void vuFourierCluster<S,T>::weightView(vuSphericView<S,T> *view)
{
  vuVector lookAt = view->getLookFrom();
  vuVector up     = view->getUp();
  vuVector right  = vuVector(1,0,0);

  calcViewVectors(lookAt, up, right);

  // ----------------------------------------------------------------------

  vuVector xStep  = right.makeUnit(); //  * (float)(m_XSize/m_XSize);
  vuVector yStep  = up.makeUnit();    //  * (float)(m_YSize/m_YSize);

  vuVector origin(m_XSize, m_YSize, m_ZSize);
  origin *= 0.5;
  
  dword innerXSize = (dword)((m_XSize - MAX_FILTER_SIZE) / M_SQRT2);
  dword innerYSize = (dword)((m_YSize - MAX_FILTER_SIZE) / M_SQRT2);

  if (ODD(innerXSize)) innerXSize += 1;
  if (ODD(innerYSize)) innerYSize += 1;

  origin += xStep*(innerXSize*-0.5f);
  origin += yStep*(innerYSize*-0.5f);

  dword XStart = (m_XSize - innerXSize) / 2;
  dword YStart = (m_YSize - innerYSize) / 2;
  dword XStop  = XStart + innerXSize;
  dword YStop  = YStart + innerYSize;

  vuVector pos = origin;

  dword width       = m_SliceFilter->getWidth();
  mc_High           = (width/2);                 // int
  mc_Low            = -mc_High + (1- width % 2); // int
  mc_IsWidthOdd     = (ODD(width));              // bool

  for (dword j = YStart; j < YStop; ++j) {
    vuVector p = pos;
    for (dword i = XStart; i < XStop; ++i) {
      doWeighting(p, NULL);
      p += xStep;
    }
    pos += yStep;
  }
}

// ?_?: to be done (channels)
template <int S, class T>
inline void vuFourierCluster<S,T>::doWeighting(vuVector& pos, float *value)
{
  dword center[3]; // center in Volume (grid coordinate)

  if (mc_IsWidthOdd) {
    center[0] = (dword)(pos[0]+0.5f);
    center[1] = (dword)(pos[1]+0.5f);
    center[2] = (dword)(pos[2]+0.5f);
  }
  else {
    center[0] = (dword)pos[0];
    center[1] = (dword)pos[1];
    center[2] = (dword)pos[2];
  }

  for (int k=mc_Low; k<=mc_High; k++) {
    for (int l=mc_Low; l<=mc_High; l++) {
      for (int m=mc_Low; m<=mc_High; m++) {
	dword idx = vcoord(center[0]+m,center[1]+l,center[2]+k);

	m_Volume[idx]   = m_CacheVolume[idx]   / m_WeightVolume[idx/2];
	m_Volume[idx+1] = m_CacheVolume[idx+1] / m_WeightVolume[idx/2];
      }
    }
  }
}

// --------------------------------------------------------------------------
// --- private - premultiplication ------------------------------------------
// --------------------------------------------------------------------------

// ?_? to be done: (channels)
template <int S, class T>
void vuFourierCluster<S,T>::premultiplySlice(vuFixelMap2F *slice)
{
  dword width      = slice->getWidth();
  dword height     = slice->getHeight();
  float halfWidth  = (float)width  / 2;
  float halfHeight = (float)height / 2;
  float *ptr       = slice->getBuffer();

  for (dword j=0; j<height; j++) {
    float zFac = m_SliceFilter->getTransformedWeight(j-halfHeight);

    zFac *= m_SliceFilter->getWidth();
    
    for (dword i=0; i<width; i++) {
      float fac = zFac * m_SliceFilter->getTransformedWeight(i-halfWidth);
	
      fac *= m_SliceFilter->getWidth();
      
      // cerr << "fac=" << fac << endl;
      
      ptr[(j*width+i)*2] /= fac;
    }
  }
}
