#include "vuSphericLightfieldFile.h"

template <int S, class T>
vuSphericLightfieldFile<S,T>::vuSphericLightfieldFile() : vuLightfieldFile()
{
}

template <int S, class T>
vuSphericLightfieldFile<S,T>::vuSphericLightfieldFile(const char *fileName)  :
  vuLightfieldFile(fileName, false)
{
  m_Width         = 0;                  // unknown
  m_Height        = 0;                  // unknown
  m_NumberOfViews = 0;                  // unknown
}

template <int S, class T>
vuSphericLightfieldFile<S,T>::vuSphericLightfieldFile(dword width, dword height, dword views, const char *fileName) : vuLightfieldFile(fileName, true)
{
  m_Width         = width;
  m_Height        = height;
  m_NumberOfViews = views;
  m_DataSize      = vuSphericView<S,T>::getSizeInByte(width, height);
  m_DataSize     *= m_NumberOfViews;
}

template <int S, class T>
vuSphericLightfieldFile<S,T>::~vuSphericLightfieldFile()
{
}

template <int S, class T>
vuSphericLightfieldFile<S,T>::vuSphericLightfieldFile(const vuSphericLightfieldFile<S,T> &other)
{
  throw "vuSphericLightfieldFile:: The copy constructor is not implemented";
}

template <int S, class T>
bool vuSphericLightfieldFile<S,T>::readHeader()
{
  if (!vuLightfieldFile::readHeader()) return false;

  char str[64];

  m_FileStream->get(str, 8);
  if (strcmp(str, "SPHERIC") != 0)
    return _setErrorMessage("No spheric lightfield file");

  *m_FileStream >> m_Width;
  *m_FileStream >> m_Height;
  *m_FileStream >> m_NumberOfViews;

  if (m_Width         == 0) return _setErrorMessage("No width set");
  if (m_Height        == 0) return _setErrorMessage("No height set");
  if (m_NumberOfViews == 0) return _setErrorMessage("No views set");

  m_FileStream->getline(str,32); // get newline
  m_FileStream->get(str, 10);
  if (strcmp(str, "DATA_SIZE") != 0)
    return _setErrorMessage("No DATA_SIZE specified");

  *m_FileStream >> m_DataSize;
  if (!_isDataSizeValid(m_DataSize))
    return _setErrorMessage("Data size is not valid");

  m_FileStream->getline(str,32); // get newline
  m_FileStream->get(str,8);
  if (strcmp(str, "FIELDS ") != 0)
    return _setErrorMessage("No FIELDS specified");

  m_FileStream->get(str, 64, ' ');
  m_DataName = vuString(str);
  if (m_DataName.isEmpty())
    return _setErrorMessage("No data name set");

  int size;
  *m_FileStream >> size;
  if (size != S)
    return _setErrorMessage("Wrong data type size");
  m_FileStream->get(); // read space
  m_FileStream->get(str,32);
  if ((strcmp(str, "byte") == 0) && (sizeof(T) == 1));       // T=byte
  else if ((strcmp(str, "float") == 0) && (sizeof(T) == 4)); // T=float
  else return _setErrorMessage("Unsupported data type");        // unknown

  m_FileStream->getline(str, 32); // read newline
  m_FileStream->getline(str,32);
  if (strcmp(str, "LOOKUP_TABLE default") != 0)
    return _setErrorMessage("Not default LOOKUP_TABLE used");

  m_IsHeaderDone = true;
  return true;
}

template <int S, class T>
bool vuSphericLightfieldFile<S,T>::writeHeader()
{
  if (!vuLightfieldFile::writeHeader()) return false;

  if (m_Width         == 0)  return _setErrorMessage("No width set");
  if (m_Height        == 0)  return _setErrorMessage("No height set");
  if (m_NumberOfViews == 0)  return _setErrorMessage("No views set");
  if (!_isDataSizeValid(m_DataSize))
    return _setErrorMessage("Data size is not valid");
    
  *m_FileStream << "SPHERIC " << m_Width << " " << m_Height;
  *m_FileStream << " " << m_NumberOfViews << endl;
    
  *m_FileStream << "DATA_SIZE " << m_DataSize << endl;
  *m_FileStream << "FIELDS "    <<  m_DataName << " " << S << " ";
  if (sizeof(T) == 1)
    *m_FileStream << "byte";
  else if (sizeof(T) == 4)
    *m_FileStream << "float";
  else
    return _setErrorMessage("Unsupported data type");
    
  *m_FileStream << endl << "LOOKUP_TABLE default" << endl;

  m_IsHeaderDone = true;
  return true;
}

template <int S, class T>
bool vuSphericLightfieldFile<S,T>::readView(vuSphericView<S,T> *view)
{
  if (!_isReadyForReading())
    return false;
  else if (!m_IsHeaderDone)
    return _setErrorMessage("Header is not read yet");
  else if (view == NULL)
    return _setErrorMessage("Can't read into NULL pointer");

  if (!view->readFromFileStream(m_FileStream, m_Width, m_Height))
    return _setErrorMessage("Could not read view");
  else
    return true;
}

template <int S, class T>
bool vuSphericLightfieldFile<S,T>::writeView(vuSphericView<S,T> *view)
{
  if (!_isReadyForWriting()) return false;
  else if (!m_IsHeaderDone)
    return _setErrorMessage("Header is not written yet");
  else if (view == NULL)
    return _setErrorMessage("Can't write NULL pointer");

  if (!view->writeToFileStream(m_FileStream))
    return _setErrorMessage("Could not write view");
  else {
    m_FileStream->sync();
    return true;
  }
}

/* ------------------------------------------------------------------------ */
/* --- accessors ---------------------------------------------------------- */
/* ------------------------------------------------------------------------ */

template <int S, class T>
dword vuSphericLightfieldFile<S,T>::getNumberOfViews()
{
  return m_NumberOfViews;
}

template <int S, class T>
dword vuSphericLightfieldFile<S,T>::getWidth()
{
  return m_Width;
}

template <int S, class T>
dword vuSphericLightfieldFile<S,T>::getHeight()
{
  return m_Height;
}


/* ------------------------------------------------------------------------ */
/* --- private methods ---------------------------------------------------- */
/* ------------------------------------------------------------------------ */

template <int S, class T>
bool vuSphericLightfieldFile<S,T>::_isDataSizeValid(dword dataSize)
{
  dword size = vuSphericView<S,T>::getSizeInByte(m_Width, m_Height);
  return (dataSize == (size * m_NumberOfViews));
}
