#ifndef _FVR_vuFourierVolume_H_
#define _FVR_vuFourierVolume_H_

#include <fstream.h>
#include "vuSimpleTypes.h"
#include "vuLinAlg/vuVector.h"
#include "vuFourierFilter/vuFourierFilter.h"
#include <fftw.h>
#include "vuCamera/vuCamera.h"
#include <math.h>
#include "vuLightfield/vuFixelMap.h"
#include "vuMisc/vuProgressHandler.h"

template <int SIZE>
class vuFourierVolume
{
 public:
  // Default constructor, copy constructor,
  // and destructor.
  vuFourierVolume<SIZE>();
  vuFourierVolume<SIZE>(vuFourierVolume<SIZE>& inst);
  ~vuFourierVolume<SIZE>();

  // Assignment operator for the volume class.
  vuFourierVolume<SIZE>& operator=(vuFourierVolume<SIZE>& rhs);

  // Set the interpolation filter.
  void setFilter(vuFourierFilter* filter);
  vuFourierFilter *getFilter();

  void setCamera(const vuCamera *camera);
  void setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right);

static void calcViewVectors(vuVector& lookAt, vuVector& up, vuVector& right);

  // Compute the slice for the current angle and
  // return the rgb's in a byte*.
  // Stop is middle section, pass is bandwidth
  void clearSlice();
  void computeSlice();
  void transformSlice();

  void computeSlice(dword x_stop, dword y_stop, dword x_pass, dword y_pass);

  // Get the dimensions of the current image (for
  // the current version, the same values are returned
  // for all angles.)
  dword getImageWidth() const;
  dword getImageHeight() const;

  dword getSliceWidth() const;
  dword getSliceHeight() const;

  dword getXSize();
  dword getYSize();
  dword getZSize();

  float *getSliceData() const;

  // set oversampling -- 2 uses twice the number of samples
  void setOversampling(float s);

  dword computeDimensions(dword XSize, dword YSize, dword ZSize,
			  float mult_pad, dword add_pad);

  // Set the width of the wrap in the frequency
  // domain.  Should be half the filter size.
  void setWrap(dword wrap);
  dword getWrap();

  // Interpolate the slice, using the currently selected filter.
  void interpolateSlice();
  void interpolateSlice(dword x_stop, dword y_stop, dword x_pass, dword y_pass);

  bool isInitialized();

  //! converts the frequency volume back into the spatial domain representation
  //! and writes the result to fileName
  bool writeSpatialVolume(const char *fileName,
			  vuProgressHandler *handler=NULL) const;

  // --- rendering ---------------------------------------------------------

  void setRenderMethod(dword renderMethod);
  dword getRenderMethod();

  void setScale(float scale);
  float getScale();
  void fitScale();

  void setBias(float bias);
  float getBias();
  void fitBias();

  bool getIsChannelActive(dword idx);
  void setIsChannelActive(dword idx, bool flag);

  void setIsReRendering(bool flag);
  bool getIsReRendering();

  vuFixelMap<SIZE,float> *getUnscaledImage();
  void computeUnscaledImage(bool doSlicing=true);
  void glResize(dword width, dword height);
  void render();

 protected:
  // Fix the frequency volume so that it wraps its
  // values periodically to half of the filter width.
  void wrapVolume(float *volume);

  // call wrapVolume() and set oversampling
  void wrapAndInitialize(float overSampling);

  inline int vcoord(int x, int y, int z) const
  {
    return ((z * m_YSize + y) * m_XSize + x) * 2 * SIZE;
  }

  inline int scoord(int x, int y) const
  {
    return (y * m_SliceXSize + x) * 2 * SIZE;
  }

 protected:
  // Variables for the volume.
  float* m_Volume;       // Volume frequency data
  dword  m_XSize;        // Width of volume (and slice)
  dword  m_YSize;        // Height of volume (and slice)
  dword  m_ZSize;        // Depth of volume
  dword  m_Wrap;         // Width of the wrap on the volume

  // Variables for the interpolation filter
  vuFourierFilter* m_Filter;

  // Variables for the slice
  float*   m_Slice;        // Slice frequency data

  vuVector m_XStep;        // Xstep for sampling
  vuVector m_YStep;        // Ystep for sampling
  dword    m_SliceXSize;   // Slice x-size
  dword    m_SliceYSize;   // Slice y-size
  vuVector m_XAxis;        // Slice x-axis
  vuVector m_YAxis;        // Slice y-axis
  vuVector m_ZAxis;        // Slice z-axis
  vuVector m_Origin;       // Bottom left corner of slice

  // image vars
  dword  m_ImageXSize;
  dword  m_ImageYSize;
  dword  m_ImageStep;
  float* m_SlicePtr;

  dword m_InnerXSize;
  dword m_InnerYSize;

  fftwnd_plan m_Plan2D;

  // --- rendering --------------------------------------------------------
  dword                   m_RenderMethod;
  float                   m_Scale;
  float                   m_Bias;
  bool                    m_Channels[SIZE];
  bool                    m_IsReRendering;
  vuFixelMap<SIZE,float> *m_UnscaledImage;

 protected:
  void loadWisdom();
  void saveWisdom();
  void transform2D(float* x);
  void destroyTransform2D();
 public:
  static void shift2D(float* x, dword XSize, dword YSize);
  static void shiftCopy2D(float* dest, float* src, dword XSize, dword YSize);

 public:
  /* method: 0 ... spatial image
             1 ... frequency amplitude
             2 ... frequency phase
             3 ... frequency real part
             4 ... frequency imaginary part
  */ 
  void computeUnscaledImage(vuFixelMap<SIZE,float>* &image, word method, bool doSlicing=true);
  void computeUnscaledImage(vuFixelMap<SIZE,float>* &image, float &minVal,
			    float &maxVal, word method, bool doSlicing=true);
 protected:
  void _copyImageAndFindMinMax(vuFixelMap<SIZE,float> *image,
			       float *slice,
			       float &minVal,
			       float &maxVal,
			       word  method);

#if 1
 public:
  // Read the volume
  // bool read(char* in, float s = 1.0, float m_pad = 1.0f, dword a_pad = 0);

  //! reads an already preprocessed fourier volume from filesystem
  //! \param wrap specifies the wrapping (MUST be at least filter.width/2)
  //! \param s specifies the oversampling rate
  bool readFourierFromFile(const char *fileName, dword wrap=4, float s=1.0);

  //! writes the preprocessed fourier volume to a file
  //! can be re-read by readFourierFromFile()
  bool writeFourierToFile(const char *fileName,
			  vuProgressHandler *handler=NULL) const;

  void write_fvr(char* out) const;
  bool read_fvr(ifstream& fin, dword XSize, dword YSize, dword ZSize, dword d_size);

#endif

};

template class vuFourierVolume<1>;
template class vuFourierVolume<2>;
template class vuFourierVolume<3>;

typedef vuFourierVolume<1> vuFourierVolume1;
typedef vuFourierVolume<2> vuFourierVolume2;
typedef vuFourierVolume<3> vuFourierVolume3;

#endif



#if 0
  //! reads the spatial domain volume data with size XSize, YSize, ZSize
  //! and transforms it into an fourier volume
  bool readSpatialVolume(byte *data, dword XSize, dword YSize, dword ZSize,
			 float s=1.0, float mult_pad=M_SQRT2, dword add_pad=0);

  void preprocess();
  static void transform3D(float* x, dword xx, dword yy, dword zz);
  static void shift3D(float* x, dword XSize, dword YSize, dword ZSize);

#endif
