#include "vuFourierVolume.h"

#include <stdio.h>
#include "vuMatrix.h"
#include "math.h"

#define WISDOM "fvr_fftw.wis"

#define ODD(x) ((x)&1)

template <int S>
vuFourierVolume<S>::vuFourierVolume() : m_XStep(1.0f, 0.0f, 0.0f),
					m_YStep(0.0f, 1.0f, 0.0f),
					m_XAxis(1.0f, 0.0f, 0.0f),
					m_YAxis(0.0f, 1.0f, 0.0f),
					m_ZAxis(0.0f, 0.0f, 1.0f)
{
  m_Volume        = NULL;
  m_Slice         = NULL;
  m_Filter        = NULL;

  m_Wrap          = 3; // largest filter.width()/2

  m_ImageStep     = 0;
  m_SlicePtr      = NULL;

  loadWisdom();

  m_SliceXSize    = 0;
  m_SliceYSize    = 0;

  m_RenderMethod  = 0; // spatial image
  m_Scale         = 1.0f;
  m_Bias          = 0.0f;
  m_UnscaledImage = NULL;
  m_IsReRendering = true;
  for (int i=0; i<S; i++) m_Channels[i] = true;
}

template <int S>
vuFourierVolume<S>::vuFourierVolume(vuFourierVolume<S>& inst)
{
  cerr << "vuFourierVolume::copyConstructor is not implemented!" << endl;
  throw "vuFourierVolume::copyConstructor is not implemented!";
}

template <int S>
vuFourierVolume<S>::~vuFourierVolume()
{
  CHECKNDELETE(m_Volume);
  CHECKNDELETE(m_Slice);
  CHECKNDELETE(m_UnscaledImage);
  destroyTransform2D();
  saveWisdom();
}

template <int S>
vuFourierVolume<S>& vuFourierVolume<S>::operator=(vuFourierVolume<S>& rhs)
{
  if (this != &rhs)
  {
  }
  return *this;
}

template <int S>
dword vuFourierVolume<S>::computeDimensions(dword XSize, dword YSize,
					    dword ZSize, float m_pad,
					    dword a_pad)
{
  dword dim = (XSize > YSize) ? XSize : YSize;
  dim = (dim > ZSize) ? dim : ZSize;
  dim = (dword) ceil((float)dim * m_pad) + a_pad;
  dim = (ODD(dim)) ? dim + 1 : dim;

  return dim;
}

template <int S>
void vuFourierVolume<S>::setOversampling(float s) {
  // don't allow undersampling
  if (s < 1.0) s = 1.0;

  dword sliceXSize = (dword) ceil(m_ImageXSize * s);
  dword sliceYSize = (dword) ceil(m_ImageYSize * s);
  sliceXSize = (ODD(sliceXSize)) ? sliceXSize + 1 : sliceXSize;
  sliceYSize = (ODD(sliceYSize)) ? sliceYSize + 1 : sliceYSize;

  // initialize m_Slice and fftw
  if (sliceXSize != m_SliceXSize || sliceYSize != m_SliceYSize || !m_Slice) {
    destroyTransform2D();

    m_SliceXSize = sliceXSize;
    m_SliceYSize = sliceYSize;

    if (m_Slice) delete [] m_Slice;
    m_Slice = new float[m_SliceXSize * m_SliceYSize * 2 * S];
  
    if ((sliceXSize > 0) && (sliceYSize > 0)) {
      cerr << "init plan ";

      m_Plan2D = fftw2d_create_plan_specific(sliceXSize, sliceYSize,
					     FFTW_BACKWARD,
					     FFTW_MEASURE | FFTW_IN_PLACE |
					     FFTW_USE_WISDOM,
					     (fftw_complex*)m_Slice, S,
					     NULL, 0);
      cerr << " done." << endl;
    }
  }

  m_XStep = m_XAxis * m_ImageXSize / m_SliceXSize;
  m_YStep = m_YAxis * m_ImageYSize / m_SliceYSize;

  m_ImageStep = (m_SliceXSize - m_ImageXSize) * 2 * S; // ?_?

  cerr << "Slice size: " << m_SliceXSize << ' ' << m_SliceYSize << endl;

  m_SlicePtr = &m_Slice[scoord( (m_SliceXSize - m_ImageXSize) / 2,
                                (m_SliceYSize - m_ImageYSize) / 2)];

  m_InnerXSize = (dword) floor((float) m_SliceXSize / M_SQRT2);
  m_InnerYSize = (dword) floor((float) m_SliceYSize / M_SQRT2);
  m_InnerXSize = (ODD(m_InnerXSize)) ? m_InnerXSize + 1 : m_InnerXSize;
  m_InnerYSize = (ODD(m_InnerYSize)) ? m_InnerYSize + 1 : m_InnerYSize;
}

template <int S>
void vuFourierVolume<S>::setWrap(dword wrap)
{
  if (m_Filter) {
    if ((m_Filter->getWidth() / 2) > wrap)
      m_Wrap = m_Filter->getWidth()/2;
    else
      m_Wrap = wrap;
  }
  else
    m_Wrap = wrap;
}

template <int S>
dword vuFourierVolume<S>::getWrap()
{
  return m_Wrap;
}

template <int S>
void vuFourierVolume<S>::setFilter(vuFourierFilter* filter)
{
  if (m_Filter != filter) m_IsReRendering = true;
  m_Filter = filter;
}

template <int S>
vuFourierFilter *vuFourierVolume<S>::getFilter()
{
  return m_Filter;
}


template <int S>
void vuFourierVolume<S>::setCamera(const vuCamera *camera)
{
  if (!isInitialized()) return;

  setViewVectors(camera->getLookAtVector(), 
		 camera->getUpVector(),
		 camera->getRightVector());
}

template <int S>
void vuFourierVolume<S>::setViewVectors(const vuVector& view,
					const vuVector& up,
					const vuVector& right)
{
  if (!isInitialized()) return;

  m_XAxis  = right;
  m_YAxis  = up;
  m_ZAxis  = view;

  m_XStep  = right;
  m_YStep  = up;
  
  m_XStep.makeUnit();
  m_YStep.makeUnit();

  m_XStep *= (float)(m_ImageXSize / m_SliceXSize);
  m_YStep *= (float)(m_ImageYSize / m_SliceYSize);

  m_Origin  = m_XAxis*(m_SliceXSize*-0.5f);
  m_Origin += m_YAxis*(m_SliceYSize*-0.5f);
  m_IsReRendering = true;
}

template <int S>
void vuFourierVolume<S>::calcViewVectors(vuVector& lookAt,
					 vuVector& up,
					 vuVector& right)
{
  lookAt.makeUnit();

  if (fabs(lookAt.dot(up)) > 0.999) up = vuVector(0,0,1).makeUnit();

  up = (up - (lookAt.dot(up) * lookAt)).makeUnit();
  right  = up.cross(lookAt).makeUnit() * -1;
}

template <int S>
dword vuFourierVolume<S>::getImageWidth() const
{
  return m_ImageXSize;
}

template <int S>
dword vuFourierVolume<S>::getImageHeight() const
{
  return m_ImageYSize;
}

template <int S>
dword vuFourierVolume<S>::getSliceWidth() const
{
  return m_SliceXSize;
}

template <int S>
dword vuFourierVolume<S>::getSliceHeight() const
{
  return m_SliceYSize;
}

template <int S>
float *vuFourierVolume<S>::getSliceData() const
{
  return m_SlicePtr;
}

template <int S>
dword vuFourierVolume<S>::getXSize()
{
  return m_XSize;
}

template <int S>
dword vuFourierVolume<S>::getYSize()
{
  return m_YSize;
}

template <int S>
dword vuFourierVolume<S>::getZSize()
{
  return m_ZSize;
}

template <int S>
void vuFourierVolume<S>::clearSlice()
{
  float* s_ptr = m_Slice;
  dword  count = m_SliceXSize * m_SliceYSize * 2 * S;
  
  for (dword i = 0; i < count; i++) *(s_ptr++) = 0.0f;
}

#define CONVOLVE_NO_MOD                                                   \
{                                                                         \
  int x = (int) p[0];                                                     \
  int y = (int) p[1];                                                     \
  int z = (int) p[2];                                                     \
                                                                          \
  t[0] = p[0] - x;                                                        \
  t[1] = p[1] - y;                                                        \
  t[2] = p[2] - z;                                                        \
                                                                          \
  x = x - XSize + m_Wrap;                                                 \
  y = y - YSize + m_Wrap;                                                 \
  z = z - ZSize + m_Wrap;                                                 \
                                                                          \
  register float* fweight = m_Filter->getWeights(t);                      \
  register float* v_ptr   = &m_Volume[vcoord(x + f2, y + f2, z + f2)];    \
                                                                          \
  for (dword l = 0; l < f_len; l++) {                                     \
    for (dword m = 0; m < f_len; m++) {                                   \
      for (dword n = 0; n < f_len; n++) {                                 \
        register float tmp     = *(fweight++);                            \
        register float *ss_ptr = s_ptr;                                   \
        register float *vv_ptr = v_ptr;                                   \
        for (int k=0; k<S; k++) {                                         \
          *(ss_ptr++) += *(vv_ptr++) * tmp;                               \
	  *(ss_ptr++) += *(vv_ptr++) * tmp;                               \
        }                                                                 \
        v_ptr -= (S+S);                                                   \
      }                                                                   \
      v_ptr += fXStep;                                                    \
    }                                                                     \
    v_ptr += fYStep;                                                      \
  }                                                                       \
                                                                          \
  s_ptr += (S+S);                                                         \
  p     += m_XStep;                                                       \
}

#define CONVOLVE                                                          \
{                                                                         \
  int x = (int) p[0];                                                     \
  int y = (int) p[1];                                                     \
  int z = (int) p[2];                                                     \
                                                                          \
  t[0] = p[0] - x;                                                        \
  t[1] = p[1] - y;                                                        \
  t[2] = p[2] - z;                                                        \
                                                                          \
  x = x % XSize + m_Wrap;                                                 \
  y = y % YSize + m_Wrap;                                                 \
  z = z % ZSize + m_Wrap;                                                 \
                                                                          \
  register float* fweight = m_Filter->getWeights(t);                      \
  register float* v_ptr   = &m_Volume[vcoord(x + f2, y + f2, z + f2)];    \
                                                                          \
  for (dword l = 0; l < f_len; l++) {                                     \
    for (dword m = 0; m < f_len; m++) {                                   \
      for (dword n = 0; n < f_len; n++) {                                 \
        register float tmp     = *(fweight++);                            \
        register float *ss_ptr = s_ptr;                                   \
        register float *vv_ptr = v_ptr;                                   \
        for (int k=0; k<S; k++) {                                         \
          *(ss_ptr++) += *(vv_ptr++) * tmp;                               \
	  *(ss_ptr++) += *(vv_ptr++) * tmp;                               \
        }                                                                 \
        v_ptr -= (S+S);                                                   \
      }                                                                   \
      v_ptr += fXStep;                                                    \
    }                                                                     \
    v_ptr += fYStep;                                                      \
  }                                                                       \
                                                                          \
  s_ptr += (S+S);                                                         \
  p     += m_XStep;                                                       \
}

template <int S>
void vuFourierVolume<S>::interpolateSlice(dword x_stop, dword y_stop,
					  dword x_pass, dword y_pass)
{
  if (m_Filter == NULL) {
    cerr << "Warning: interpolateSlice(refine): no filter set!" << endl;
    return; 
  }

  vuVector t;
  dword f_len = m_Filter->getWidth();
  int f2 = f_len / 2;

  x_stop *= 2;
  y_stop *= 2;
  dword stop_l = (m_SliceXSize - x_stop - 2 * x_pass) / 2;
  dword stop_r =  m_SliceXSize - x_stop - 2 * x_pass - stop_l;
  dword stop_b = (m_SliceYSize - y_stop - 2 * y_pass) / 2;
  dword x_hi = x_stop + 2 * x_pass;
  dword s_step = stop_l + stop_r;

  dword XSize = m_XSize - m_Wrap * 2;
  dword YSize = m_YSize - m_Wrap * 2;
  dword ZSize = m_ZSize - m_Wrap * 2;

  dword fXStep = (f_len - m_XSize) * 2 * S;
  dword fYStep = (f_len - m_YSize) * m_XSize * 2 * S;

  vuVector q = m_Origin;
  q[0] += XSize / 2 + XSize + stop_l * m_XStep[0] + stop_b * m_YStep[0];
  q[1] += YSize / 2 + YSize + stop_l * m_XStep[1] + stop_b * m_YStep[1];
  q[2] += ZSize / 2 + ZSize + stop_l * m_XStep[2] + stop_b * m_YStep[2];
  
  clearSlice();
  
  float* s_ptr = &m_Slice[scoord(stop_l, stop_b)];

  for (dword j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (dword i = 0; i < x_hi; ++i) CONVOLVE;
    s_ptr += s_step * 2 * S;
    q += m_YStep;
  }

  for (dword j = 0; j < y_stop; ++j) {
    vuVector p = q;
    for (dword i = 0; i < x_pass; ++i) CONVOLVE;

    s_ptr += x_stop * 2 * S;
    p[0] += x_stop * m_XStep[0];
    p[1] += x_stop * m_XStep[1];
    p[2] += x_stop * m_XStep[2];

    for (dword i = 0; i < x_pass; ++i) CONVOLVE;
    s_ptr += s_step * 2 * S;
    q += m_YStep;
  }

  for (dword j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (dword i = 0; i < x_hi; ++i) CONVOLVE;
    s_ptr += s_step * 2 * S;
    q += m_YStep;
  }
}

template <int S>
void vuFourierVolume<S>::interpolateSlice()
{
  if (m_Filter == NULL) {
    cerr << "Warning: interpolateSlice(): no filter set!" << endl;
    return; 
  }

  dword i, j;
  vuVector t;
  dword f_len = m_Filter->getWidth();
  int f2 = f_len / 2;

  dword x_pass = (m_SliceXSize - m_InnerXSize) / 2;
  dword y_pass = (m_SliceYSize - m_InnerYSize) / 2;

  dword XSize = m_XSize - m_Wrap * 2;
  dword YSize = m_YSize - m_Wrap * 2;
  dword ZSize = m_ZSize - m_Wrap * 2;

  dword fXStep = (f_len - m_XSize) * 2 * S;
  dword fYStep = (f_len - m_YSize) * m_XSize * 2 * S;

  vuVector q = m_Origin;
  q[0] += XSize / 2 + XSize;
  q[1] += YSize / 2 + YSize;
  q[2] += ZSize / 2 + ZSize;

  clearSlice();

  float* s_ptr = m_Slice;

  for (j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (i = 0; i < m_SliceXSize; ++i) CONVOLVE;
    q += m_YStep;
  }

  for (j = 0; j < m_InnerYSize; ++j) {
    vuVector p = q;
    for (i = 0; i < x_pass; ++i) CONVOLVE;
    for (i = 0; i < m_InnerXSize; ++i) CONVOLVE_NO_MOD;
    for (i = 0; i < x_pass; ++i) CONVOLVE;
    q += m_YStep;
  }

  for (j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (i = 0; i < m_SliceXSize; ++i) CONVOLVE;
    q += m_YStep;
  }
}

template <int S>
void vuFourierVolume<S>::computeSlice()
{
  interpolateSlice();
  transformSlice();
}

template <int S>
void vuFourierVolume<S>::transformSlice()
{
  shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
  transform2D(m_Slice);
  shift2D(m_Slice, m_SliceXSize, m_SliceYSize);  
}


template <int S>
void vuFourierVolume<S>::computeSlice(dword x_stop, dword y_stop,
				      dword x_pass, dword y_pass)
{
  interpolateSlice(x_stop, y_stop, x_pass, y_pass);
  
  shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
  transform2D(m_Slice);
  shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
}

template <int S>
bool vuFourierVolume<S>::isInitialized()
{
  return (m_SliceXSize > 0 || m_SliceYSize > 0);
}

/* ************************************************************************* */
/* *** private functions *************************************************** */
/* ************************************************************************* */

template <int S>
void vuFourierVolume<S>::wrapAndInitialize(float overSampling)
{
  wrapVolume(m_Volume);

  dword XSize = m_XSize - m_Wrap * 2;
  dword YSize = m_YSize - m_Wrap * 2;

  m_Origin[0] = (float)(XSize / 2) * -1.0f;
  m_Origin[1] = (float)(YSize / 2) * -1.0f;
  m_Origin[2] = 0.0f;

  m_ImageXSize = XSize;
  m_ImageYSize = YSize;
  CHECKNDELETE(m_UnscaledImage);
  m_UnscaledImage = new vuFixelMap<S,float>(m_ImageXSize, m_ImageYSize);

  setOversampling(overSampling);
}

// copy corresponding data to the zeroed wrap-arounds
template <int S>
void vuFourierVolume<S>::wrapVolume(float *volume)
{
  // ?_?: to be done: depend wrapping on SIZE (number of channels)
  if (S==1) {
    dword index;
    dword ii, jj ,kk;

    dword hiX = m_XSize - m_Wrap - 1;
    dword hiY = m_YSize - m_Wrap - 1;
    dword hiZ = m_ZSize - m_Wrap - 1;

    dword XSize = m_XSize - m_Wrap * 2;
    dword YSize = m_YSize - m_Wrap * 2;
    dword ZSize = m_ZSize - m_Wrap * 2;

    index = 0;
    for(dword k = 0; k < m_ZSize; ++k) {
      for(dword j = 0; j < m_YSize; ++j) {
	for(dword i = 0; i < m_XSize; ++i) {
	  if (i < m_Wrap)
	    ii = i + XSize;
	  else if (i > hiX)
	    ii = i - XSize;
	  else {
	    index += 2;
	    continue;
	  }

	  if (j < m_Wrap)
	    jj = j + YSize;
	  else if (j > hiY)
	    jj = j - YSize;
	  else {
	    index += 2;
	    continue;
	  }

	  if (k < m_Wrap)
	    kk = k + ZSize;
	  else if (k > hiZ)
	    kk = k - ZSize;
	  else {
	    index += 2;
	    continue;
	  }

	  ii = vcoord(ii, jj, kk);

	  volume[index++] = volume[ii];
	  volume[index++] = volume[ii+1];
	}
      }
    }
  }
}

/* ************************************************************************* */
/* *** private fourier transformation functions **************************** */
/* ************************************************************************* */

template <int S>
void vuFourierVolume<S>::loadWisdom()
{
  cerr << "loading fftw wisdom..." << endl;
  FILE* wisdom_file = fopen(WISDOM, "a+");
  rewind(wisdom_file);
  fftw_import_wisdom_from_file(wisdom_file);
  fclose(wisdom_file);
}

template <int S>
void vuFourierVolume<S>::saveWisdom()
{
  cerr << "saving fftw wisdom..." << endl;
  FILE* wisdom_file = fopen(WISDOM, "w");
  fftw_export_wisdom_to_file(wisdom_file);
  fclose(wisdom_file);
}

template <int S>
void vuFourierVolume<S>::transform2D(float *data)
{
  fftwnd(m_Plan2D, S, (fftw_complex*)data, S, 1, NULL, 0, 0);
}

template <int S>
void vuFourierVolume<S>::destroyTransform2D()
{
  if (m_SliceXSize > 0 || m_SliceYSize > 0)
    fftwnd_destroy_plan(m_Plan2D);
  m_SliceXSize = 0;
  m_SliceYSize = 0;
}

template <int S>
void vuFourierVolume<S>::shift2D(float* slice, dword XSize, dword YSize)
{
  dword xstep = XSize * 2 * S;
  dword index;
  int start = 0;
  int val = -1;

  index = 0;
  for (dword j=0; j<YSize; j++) {
    for (dword i=start; i<xstep; i+=4*S) {
      float *ptr = &slice[index+i];
      for (int k=0; k<S; k++) {
	*(ptr++) *= -1.0f;
	*(ptr++) *= -1.0f;
      }
    }
    val   *= -1;
    start += val * 2 * S;
    index += xstep;
  }
}

// ?_?
template <int S>
void vuFourierVolume<S>::shiftCopy2D(float* dest, float* src, dword XSize,
				     dword YSize)
{
  dword i, j;
  dword xlast = XSize / 2;
  dword ylast = YSize / 2;

  float* src_ptr = src;
  float* dest_ptr = dest;

  for (j = 0; j < ylast; ++j) {
    for (i = 0; i < xlast; ++i) {
      *(dest_ptr++) = *(src_ptr++) * -1.0f;
      *(dest_ptr++) = *(src_ptr++) * -1.0f;
      *(dest_ptr++) = *(src_ptr++);
      *(dest_ptr++) = *(src_ptr++);
    }
    for (i = 0; i < xlast; ++i) {
      *(dest_ptr++) = *(src_ptr++);
      *(dest_ptr++) = *(src_ptr++);
      *(dest_ptr++) = *(src_ptr++) * -1.0f;
      *(dest_ptr++) = *(src_ptr++) * -1.0f;
    }
  }
}

/* ************************************************************************* */
/* *** rendering *********************************************************** */
/* ************************************************************************* */

template <int S>
void vuFourierVolume<S>::setRenderMethod(dword renderMethod)
{
  if (renderMethod > 4) renderMethod = 0;
  m_RenderMethod = renderMethod;
}
template <int S>
dword vuFourierVolume<S>::getRenderMethod()
{
  return m_RenderMethod;
}

template <int S>
void vuFourierVolume<S>::setScale(float scale)
{
  m_Scale = scale;
}
template <int S>
float vuFourierVolume<S>::getScale()
{
  return m_Scale;
}
template <int S>
void vuFourierVolume<S>::fitScale()
{
  if (m_UnscaledImage == NULL) return;

  float minValue, maxValue;

  m_UnscaledImage->getMinAndMaxValue(minValue, maxValue);

  cerr << "fitScale=" << maxValue << endl;

  m_Scale = 1 / maxValue;
}


template <int S>
void vuFourierVolume<S>::setBias(float bias)
{
  m_Bias = bias;
}
template <int S>
float vuFourierVolume<S>::getBias()
{
  return m_Bias;
}
template <int S>
void vuFourierVolume<S>::fitBias()
{
  if (m_UnscaledImage == NULL) return;

  float minValue, maxValue;

  m_UnscaledImage->getMinAndMaxValue(minValue, maxValue);

  m_Bias = - minValue;
}

template <int S>
bool vuFourierVolume<S>::getIsChannelActive(dword idx)
{
  if (idx < S) 
    return m_Channels[idx];
  else
    return false;
}

template <int S>
void vuFourierVolume<S>::setIsChannelActive(dword idx, bool flag)
{
  if (idx < S)  m_Channels[idx] = flag;
}

template <int S>
void vuFourierVolume<S>::setIsReRendering(bool flag)
{
  m_IsReRendering = flag;
}
template <int S>
bool vuFourierVolume<S>::getIsReRendering()
{
  return m_IsReRendering;
}

template <int S>
void vuFourierVolume<S>::glResize(dword width, dword height)
{
  if (m_UnscaledImage == NULL) return;
  m_UnscaledImage->glResize(width, height);
}

template <int S>
vuFixelMap<S,float> *vuFourierVolume<S>::getUnscaledImage()
{
  return m_UnscaledImage;
}


template <int S>
void vuFourierVolume<S>::computeUnscaledImage(bool doSlicing)
{
  if (m_IsReRendering) {
    computeUnscaledImage(m_UnscaledImage, m_RenderMethod, doSlicing);
    m_IsReRendering = false;
  }
}

// chose the channels from m_Image, apply scale and bias  and display
template <int S>
void vuFourierVolume<S>::render()
{
  computeUnscaledImage();
  // compute unscaled image (m_UnscaledImage)
  if (m_UnscaledImage == NULL) {
    cerr << "vuFourierVolume::displayImage: m_UnscaledImage is NULL" << endl;
    return;
  }
  vuFixelMap<S,float> *img = NULL;

  img = new vuFixelMap<S,float>(m_UnscaledImage->getWidth(),
				m_UnscaledImage->getHeight());

  bool status = true;

  for (int i=0; i<S; i++) status=status && m_Channels[i];

  if (status)
    *img = *m_UnscaledImage;
  else {
    vuFixelMap<1,float> *imag = NULL;

    imag = new vuFixelMap<1,float>(m_UnscaledImage->getWidth(),
				   m_UnscaledImage->getHeight());

    for (int i=0; i<S; i++) {
      if (m_Channels[i]) {
	m_UnscaledImage->getChannel(imag,i);
	img->copyMapToChannel(imag,i);
      }
    }
    CHECKNDELETE(imag);
  }

  *img *= m_Scale;
  *img += m_Bias;
  img->glRender();
  CHECKNDELETE(img);
}

/* ************************************************************************* */
/* *** advanced rendering ************************************************** */
/* ************************************************************************* */

/* method: 0 ... spatial image
           1 ... frequency amplitude
           2 ... frequency phase
           3 ... frequency real part
           4 ... frequency imaginary part
*/

template <int S>
void vuFourierVolume<S>::computeUnscaledImage(vuFixelMap<S,float>* &image,
					      word method, bool doSlicing)
{
  float minValue, maxValue;
  computeUnscaledImage(image, minValue, maxValue, method, doSlicing);
}

template <int S>
void vuFourierVolume<S>::computeUnscaledImage(vuFixelMap<S,float>* &image,
					      float &minVal,
					      float &maxVal,
					      word method, bool doSlicing)
{
  if (!isInitialized()) {
    if (image == NULL) image = new vuFixelMap<S,float>(16,16);
    return;
  }

  if (method > 4) {
    cerr << "vuFourierVolume<S>::computeUnscaledImage():method not supported!";
    cerr << " Chose method=0 instead" << endl;
    method = 0;
  }

  dword width; 
  dword height;
  float *s_ptr;

  if (method == 0) { // spatial method
    if (doSlicing) computeSlice();
    width  = getImageWidth();
    height = getImageHeight();
    s_ptr  = m_SlicePtr;
  }
  else {             // frequency methods...
    if (doSlicing) interpolateSlice();
    width  = getSliceWidth();
    height = getSliceHeight();
    s_ptr  = m_Slice;
  }

  if (image == NULL)
    image = new vuFixelMap<S,float>(width, height);
  else if (width != image->getWidth() || height != image->getHeight()) {
    CHECKNDELETE(image);
    image = new vuFixelMap<S,float>(width, height);
  }

  _copyImageAndFindMinMax(image, s_ptr, minVal, maxVal, method);
}

template <int S>
void vuFourierVolume<S>::_copyImageAndFindMinMax(vuFixelMap<S,float> *image,
						 float *slice,
						 float &minVal,
						 float &maxVal,
						 word  method)
{
  dword width  = image->getWidth();
  dword height = image->getHeight();

  float *s_ptr = slice;
  float *i_ptr = image->getBuffer();

  minVal = + 10000000000.0f;
  maxVal = - 10000000000.0f;

  for (dword j=0; j<height; j++) {
    for (dword i=0; i<width; i++) {
      for (int k=0; k<S; k++) {
	register float value = *(s_ptr++);

	switch (method) {
	case 0: // image
	  s_ptr++;
	  break;
	case 1: // freq. amplitude
	  value = sqrt(value*value + (*s_ptr) * (*s_ptr)); 
	  s_ptr++;
	  break;
	case 2: // freq. phase
	  if ((*s_ptr) == 0)
	    value = (value >= 0) ? 0.0f : M_PI;
	  else
	    value = atan(value/(*s_ptr));
	  s_ptr++;
	  break;
	case 3: // freq. real part
	  s_ptr++;
	  break;
	case 4: // freq. imaginary part
	  value = *(s_ptr++);
	  break;
	}

	*(i_ptr++) = value;
	if (minVal > value) minVal = value;
	if (maxVal < value) maxVal = value;
      }
      // ?_? ...
      if (method == 0) s_ptr += m_ImageStep;
    }
  }
}






// --- ??? kann weg ----------------------------------------------------------
#if 1
#include "vuFourier_IO.h"
// ?_? todo
// write the fourier volume to a file (WITHOUT m_Wrap)!!!
template <int S>
bool vuFourierVolume<S>::writeFourierToFile(const char *fileName,
					    vuProgressHandler *handler) const
{
  ofstream fout;
  fout.open(fileName, ios::out|ios::binary);

  dword XSize = m_XSize; // - (m_Wrap * 2);
  dword YSize = m_YSize; // - (m_Wrap * 2);
  dword ZSize = m_ZSize; // - (m_Wrap * 2);

  //??? ToDO: wrapping does not work!!!
  vuFourier_IO::write_fvr_head(fout, XSize, YSize, ZSize, sizeof(float));
  dword count = XSize*YSize*ZSize*2*sizeof(float);

  if (handler == NULL)
    fout.write((const char*)m_Volume, count);
  else {
#define MEGABYTE 1000000
    int   cnt = count / MEGABYTE; // each 1MB one progress update
    char *ptr = (char *)m_Volume;

    int   oldProg  = handler->getCurrentProgress();
    float progress = (float)oldProg;
    float step     = (float)handler->getRange()/cnt;

    vuString str = "Write file: ";
    str += vuString(fileName).getLastPathComponent();
    handler->update(oldProg, str.c_str());

    for (int i=0; i<cnt; i++) {
      fout.write((const char*)ptr, MEGABYTE);
      ptr += MEGABYTE;
      progress += step;

      if (!handler->update((int)progress)) { // abort if user presses CANCEL
	fout.close();
	return false;
      }
    }
    int len = count % MEGABYTE;
    if (len) fout.write((const char*)ptr, len);
    handler->update(oldProg+handler->getRange());
  }
  fout.close();

  return true;
}

// ?_? todo
// read the fourier volume from a file (data are stored without m_Wrap)!!!
template <int S>
bool vuFourierVolume<S>::readFourierFromFile(const char *fileName, dword wrap,
					     float s)
{
  dword XSize, YSize, ZSize, d_size;

  setWrap(wrap);

  ifstream fin(fileName, ios::in|ios::binary
#ifdef IS_NOCREATE_NEEDED
|ios::nocreate
#endif
);
  int ret = vuFourier_IO::read_head(fin, XSize, YSize, ZSize, d_size);

  if (ret != 2) {
    cerr << "readFourierFromFile(): wrong header type" << endl;
    return false;
  }

  if (d_size != sizeof(float)) {
    cerr << "readFourierFromFile(): wrong data type" << endl;
    return false;
  }

  m_XSize = XSize + m_Wrap * 2;
  m_YSize = YSize + m_Wrap * 2;
  m_ZSize = ZSize + m_Wrap * 2;

  m_Volume = new float[m_XSize * m_YSize * m_ZSize * 2];

  vuFourier_IO::read_raw_r_fast(fin, (byte*)m_Volume,
				m_XSize, m_YSize, m_ZSize,
				XSize, YSize, ZSize, d_size * 2);
  fin.close();

  wrapAndInitialize(s);

  return true;
}

// ?_? todo
template <int S>
bool vuFourierVolume<S>::writeSpatialVolume(const char *fileName,
					    vuProgressHandler *handler) const
{
  return true;
}
#endif
