/* 	This is a ppm reader and writer for vuVolume
	This will save vuImages into ppm and extract them from
	ppm files that are given.

	The info surrounding a ppm is found on the website:

	http://astronomy.swin.edu.au/~pbourke/dataformats/ppm/

	Written by Christopher Steinbach
	March 2002
*/

#ifndef vuVolume_PPM_reader_h__
#define vuVolume_PPM_reader_h__

#include "vuImage.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//using namespace ns_vu1112113;

/**	This class will read/write PPM P6 files into/from a vuImage
	so that the Key Framer (and possibly others as needed)
	will be able to read/write Images to disk.

	This was originally written for the vuKeyFramer so that
	it could precompute the images and play them back later
	for an animation.

	\todo add other file type to save to which allow compression

	\todo add more suitable movie type files.

	\todo add the ability to interpret from preloaded memory buffers (so
		async file reads can occur and other stuff can happen before then without
		multiple threads).

	Note that this has the code to read any P6 file, but the code to read
	any header has been commented out and this uses a much faster version of read
	where it will only read headers of the form

	"P6?XXXX?YYYY?ZZZZ"

	and will exhibit incorrect behaviour on other styles of headers
*/

class vuPPM

{
	public:	// constructors and destructors

		/** default constructor */
		vuPPM ();

		/** destructor */
		virtual ~vuPPM ();

	public: // These functions will load the image from variou sources

		/** This function will write the image stored in img
			to the file named fname (it will also overwrite
			anything in that file, so watch out) */
		bool SaveImage (char* fname, vuImage &img);
		
		bool SaveCompressedImage (char* fname, vuImage &img);

		/** This funcion will read the image stored in file fname
			and store it in img.

			if readtype = 1, then this will read only headers of
			type P6?XXXX?YYYY?ZZZZ which is mush faster than reading
			standard headers.
		*/
		bool ReadImage (char* fname, vuImage &img, int readtype = 0);

		/** To use this function, you simply have to do the following
			load the file into the buffer buf, then call this function,
			and this will set img correctly, basically, it preloads
			to allow for asynchronous loads in the vuKeyFrameBuffer
			if readtype = 1, then this will read only headers of
			type P6?XXXX?YYYY?ZZZZ which is mush faster than reading
			standard headers.
		*/
		bool ReadImageFromBuffer (char* buf, vuImage &img, int readtype = 0);

		/** This will save an image opened through OpenImage (...)
			which you may have edited */
		bool SaveImage (char* fname);

		/** This will read an image opened through open image into the
			image stored here */
		bool ReadImage (char* fname);

		/** This will open the image located at file fname and read it into the
			buffer pointed to by img.  Not working yet since it wasn't
			necessary to my task at hand... */
		bool OpenImage (char* fname, vuImage &img);

		/** This will read a 64KiloByte buffer from the file and update
			img's buffer accordingly... */
		unsigned int Readnext64KB (vuImage &img);

		/** This will close and save the opened image */
		bool CloseImage ();

		/** This will set the maximum x and y values for the image stored by
			this object. */
		bool set_xy (int x, int y);

		/** This will return the size of the current image in the x direction.

		 	It will also return the size in the x-direction of the last
			image header that was successfully read in. */
		int get_x ();

		/** This will return the size of the current image in the y direction

			It will also return the size in the y-direction of the last
			image header that was successfully read in. */
		int get_y ();

	private:	// functions that handle headers.

		/** This function will read a header from the currently opened
			file and will set the vuImage img to have the correct
			dimensions if it is a valid header.

			This will return true if the header is valid, and false
			if the header is not of valid format.

			Note that this contains the code to recognize all P6 files,
			but for speed, that code has been commented out and instead
			this only accepts headers of the format

			P6?XXXX?YYYY?ZZZZ?

			where XXXX is a 4 digit number containing the number of pixels
			in the image in the x direction, YYYY is similar, and ZZZZ
			is any four digits.  Question marks simple denote any character
			(they should be spaces or tabs ideally, but this one doesn't care).
		
			if readtype = 1, then this will read only headers of
			type P6?XXXX?YYYY?ZZZZ which is mush faster than reading
			standard headers.
		*/
		int ReadHeader (vuImage &img, int readtype = 0);

		bool ReadCompressedImage (char* fname, vuImage &img, int readtype);

		/** This will read a header from buf and set the dimensions of
			img corresponding to what it reads.  This was done so that
			the PPM reader could read from a buffer instead of opening
			the file (so that some async stuff could happen earlier).

			if readtype = 1, then this will read only headers of
			type P6?XXXX?YYYY?ZZZZ which is mush faster than reading
			standard headers.
		*/
		int ReadHeader (vuImage &img, char* buf, int readtype = 0);

		/** This will write a header corresponding to a P6 with header
			information corresponding to the information required
			to reconstruct the image img.

			This will return true if the operation was successful, and
			it will return false otehrwise. */
		bool WriteHeader (vuImage &img);

		bool WriteCompressedHeader (vuImage &img);

		int CompressToBuffer (vuImage &img);

	private:
		/** This is just an image... should probably be removed from this... */
		vuImage m_Image;

		/** This is a pointer to the file that the reader has opened and
			is currently reading from, or writing to as the case may be. */
		FILE *fp;

		/** This is the width of the image in pixels. */
		int m_x;

		/** This is the height of the image in pixels. */
		int m_y;

		/** This is the size of the file for C1 compressiones */
		int m_z;
		
		unsigned char* m_buf;
		int m_buf_size;

		/** A stock counter to keep track of the current position this is in
			during a file read. */
		unsigned int counter;
};

#endif




