/* Written by Steven Bergner

        Based on Steve Kilthau's colour classes
	Modified June 2002
	Modified by Chris Steinbach
	Modified to add read and write functions by sbergner so that
		This would work in the CVS.
*/

#ifndef _VUCOLOUR_H_
#define _VUCOLOUR_H_

#include <iostream.h>	// cgs@sfu.ca - April 2002 - modified to add the ".h" so that it would work in IRIX
#include <stdio.h>
#include "vuSimpleTypes.h"

#define USE_GENERAL_BASIS
//using specialized 31to7 and 7toRGBA instead

/** vuColourN is a virtual base class for all colours.
    
    It allows basic access to the data independent from the underlying
    colour model.  The user has to be careful not to read beyond the
    last element given be nComponents().  All vuColour<S> templates
    are derived from this base class and all colour models are derived
    from a specific vuColour<S>. For instance vuColourRGBa is derived
    from vuColour<4>.  It is not possible to access binary operators
    from this base class. These operations have to be performed in the
    specific model.  */
class vuColourN
{
public:
    /** default constructor */
    vuColourN(void) {};

    /** destructor */
    virtual ~vuColourN(void) {};

    /** provides direct random access to the components */
    virtual float& operator[](unsigned int index) { static float n=0; return n;};

    /** provides direct reading access to the components */
    virtual float operator[](unsigned int index) const {return 0;};

    /** returns a pointer to the float array containing the data */
    virtual float *getData() {return 0;};

    /** returns a pointer to the float array containing the data */
    virtual float const* getData() const {return 0;};

    /** returns the number of components.
        This depends on the number given during creation of the template. */
    virtual word nComponents() const {return 0;};

    /** prints the components of the colour */
    virtual void print() const {};
};

/** General colour class.
    The argument S to this template class gives the number of components including the alpha
    value. The indexing via the [] operator starts at 0 with alpha at position S-1.
    Derived classes like vuColourRGBa, vuColourXYZa, vuColour31a... use the protected
    From() and To() functions to provide conversions to other colour models.
*/
template <int S>
class vuColour : public vuColourN
{
public:
    /** default constructor 
     Sets all values to zero and alpha to one. */
    vuColour()
    {
      int i;
      for(i=0;i<S-1;++i)
	m_Data[i] = 0.0f;
      m_Data[S-1]=1;
    }

    /** copy constructor */
    vuColour(const vuColour<S>& inst)
    {
        int i;
        for(i=0;i<S;++i)
            m_Data[i] = inst.m_Data[i];
    }

    /** initialize all components
	(including alpha at N-1) with values from array f */
    vuColour(const float *f)
    {
        int i;
        for(i=0;i<S;++i)
            m_Data[i] = f[i];
    }

    /** initialize all components (from double*)
	(including alpha at N-1) with values from array f */
    vuColour(const double *f)
    {
        int i;
        for(i=0;i<S;++i)
            m_Data[i] = f[i];
    }

    /** initialize all values (including alpha) with f */
    vuColour(const float f)
    {
        int i;
        for(i=0;i<S;++i)
            m_Data[i] = f;
    }

    /** destructor */
    ~vuColour()
    {
    }

    /** get alpha value */
    float getAlpha(void)
    {
      return m_Data[S-1];
    }

    /** set alpha value */
    void setAlpha(const float a)
    {
      m_Data[S-1] = a;
    }

    /** provides direct random access to the components */
    float& operator[](unsigned int index)
    {
      return m_Data[index];
    }

    /** provides direct reading access to the components */
    float operator[](unsigned int index) const
    {
      return m_Data[index];
    }

    /** returns a pointer to the float array containing the data */
    float *getData() { return m_Data; }
    /** const version of  getData() */
    float const *getData() const { return m_Data; }

    //!assignment from double* without alpha
    vuColour<S>& operator=(const double* rhs)
    {
      for(int i=0;i<S-1;++i)
        m_Data[i] = rhs[i];
      return *this;
    }

    //!assignment from double* without alpha
    vuColour<S>& operator=(double* rhs)

    {
      for(int i=0;i<S-1;++i)
        m_Data[i] = rhs[i];
      return *this;
    }

    /** assignment operator */
    vuColour<S>& operator=(const vuColour<S>& rhs)
    {
      if (this != &rhs)
	{
	  for(int i=0;i<S;++i)
            m_Data[i] = rhs.m_Data[i];
	}
      return *this;
    }

    /** assignment operator,
      This will set each compenent to rhs */
    vuColour<S>& operator=(float rhs)
    {
      for(int i=0;i<S;++i)
        m_Data[i] = rhs;
      return *this;
    }

    /** assignment operator,
      This will set the ith colour component to be the ith float pointed to by rhs */
    vuColour<S>& operator=(const float* rhs)
    {
      for(int i=0;i<S-1;++i)
        m_Data[i] = rhs[i];
      return *this;
    }

    /** operator ==,

        This returns true if the two colours are equivalent. */
    bool operator== (const vuColour<S>& rhs) const
	{
	    for(int i=0;i<S;++i)
		if(!m_Data[i] == rhs[i]) return false;
	    return true;
	}

    /** addition operator,

        This performs a component wise addition of each component of the 2 colours
	and returns the result. */
    vuColour<S> operator+(const vuColour<S>& rhs) const
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = m_Data[i] + rhs.m_Data[i];
      return ret;
    }

    /** operator,
    	This will add a to the first S-1 components */
    vuColour<S> operator+(float a) const
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = m_Data[i] + a;
      return ret;
    }

    /** operator,
     This will perform a component wise subtraction on the first S-1 components*/
    vuColour<S> operator-(const vuColour<S>& rhs) const
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = m_Data[i] - rhs.m_Data[i];
      return ret;
    }

    /** operator,
      This will perform a component wise multiplication on the first S-1 components */
    vuColour<S> operator*(const vuColour<S>& rhs) const
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = m_Data[i] * rhs.m_Data[i];
      return ret;
    }

    /** operator,
    	This will perform a component wise division on the first S-1 components */
    vuColour<S> operator/(const vuColour<S>& rhs) const
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = m_Data[i] / rhs.m_Data[i];
      return ret;
    }

    /** operator multiplication with scalar
        In comparision to *= this operator creates a copy where the result
        written to. 
    */
    vuColour<S> operator*(float rhs) const 
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = m_Data[i] * rhs;
      return ret;
    }

    /** operator for commutative multiplication with a float.
        In comparision to *= this operator creates a copy where the result
        written to. 
    */
    friend vuColour<S> operator*(float lhs, const vuColour<S>& rhs)
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = rhs.m_Data[i] * lhs;
      return ret;
    }

    /** operator for component wise addition */
    vuColour<S>& operator+=(const vuColour<S>& rhs)
    {
      for(int i=0;i<S-1;++i)
        m_Data[i] += rhs.m_Data[i];
      return *this;
    }

    /** operator for component wise subtraction */
    vuColour<S>& operator-=(const vuColour<S>& rhs)
    {
      for(int i=0;i<S-1;++i)
        m_Data[i] -= rhs.m_Data[i];
      return *this;
    }

    /** operator for component wise multiplication */
    vuColour<S>& operator*=(const vuColour<S>& rhs)
    {
      for(int i=0;i<S-1;++i)
        m_Data[i] *= rhs.m_Data[i];
      return *this;
    }

    /** operator for component wise division */
    vuColour<S>& operator/=(const vuColour<S>& rhs)
    {
      for(int i=0;i<S-1;++i)
        m_Data[i] /= rhs.m_Data[i];
      return *this;
    }
    
     /** operator for multiplication with scalar */
    vuColour<S>& operator*=(float rhs)
    {
        for(int i=0;i<S-1;++i)
	  m_Data[i] *= rhs;
	return *this;
    }

    /** operator for division by scalar */
    vuColour<S>& operator/=(float rhs)
    {
        for(int i=0;i<S-1;++i)
	  m_Data[i] /= rhs;
	return *this;
    }

    /** Set values above one to one and values below zero to zero. */
    vuColour<S>& clampTo01()
    {
      for(int i=0; i<S; i++)
	if(m_Data[i] < 0) m_Data[i] = 0;
        else if (m_Data[i] > 1) m_Data[i] = 1;
      return *this;
    }

    /** Set values above one to one. */
    vuColour<S>& clampTo1()
    {
      for(int i=0; i<S; i++)
        if (m_Data[i] > 1) m_Data[i] = 1;
      return *this;
    }

    /** return a color with all components set to their distance from one */
    vuColour oneMinus() const
    {
      vuColour<S> ret;
      for(int i=0;i<S-1;++i)
        ret.m_Data[i] = 1-m_Data[i];
      return ret;
    }

    /** returns the number of components.
        This depends on the number given during creation of the template. */
    word nComponents() const
      {
	return S;
      }

    /** returns value of maximal component  
	This is used for a pretty ineffective version of early ray termination. */
    float maxComponent() const
    {
      float maxc = m_Data[0];
      for(int i=1;i<S-1;++i)
        if(maxc<m_Data[i]) maxc=m_Data[i];
      return maxc;
    }

    /** returns value of minimal component  
	This is used for a pretty inefficient version of early ray termination. */
    float minComponent() const 
    {
      float minc = m_Data[0];
      for(int i=1;i<S-1;++i)
        if(minc>m_Data[i]) minc=m_Data[i];
      return minc;
    }

    /** weight components by alpha value
	alpha value (last component) is set to one */
    vuColour<S>& multAlpha()
    {
      if(S>1) {
	for(int i=0; i<S-1; i++)
	  m_Data[i] *= m_Data[S];
	m_Data[S-1] = 1;
      }
      return *this;
    }

    /** this will write each colour component on the screen, in comma separated format */
    void print() const
    {
      printf(" %3.7f", m_Data[0]);
      for(int i=1;i<S;++i)
        printf(", %3.7f", m_Data[i]);
      printf("\n");
    }

    //!a friend, write colour to stream
//template < int T >
friend ostream& operator<< (ostream& os, const vuColour<S>& A)
    {
	 const float *dat = A.m_Data;
	 os << *(dat++);
	 for (dword j = 1; j < S; j++, dat++)
	     os << ", " << (*dat);
	 return os;
    }

    //! a friend, read colour from stream
//template < int T >
friend istream& operator>> (istream& is, vuColour<S>& A)
     {
	 float *dat = A.m_Data;
	 for (dword j = 0; j < S; j++, dat++)
	     is >> (*dat);
	 return is;
     }

//! a friend, read colour from stream
istream& read (istream& is)
     {
	 float *dat = m_Data;
	 for (dword j = 0; j < S; j++, dat++)
		 is >> (*dat);
	 return is;
     }

//! a friend, write colour to stream
	ostream& write(ostream& os) const {
		const float *dat = m_Data;
		for (dword j = 0; j < S; j++, dat++)
			 os << (*dat) << " ";
		return os;
	}



 protected:
 	/** this will assign colour src to this colour, compensating for any format problems */
    template<int T>
    vuColour<S>& fromColourN(const vuColour<T>& src, const float* xfmat)
      {
	int i,j;
	for(i=0;i<S-1;i++) {
	  m_Data[i] = *(xfmat++) * src[0];
	  for(j=1;j<T-1;j++)
	    m_Data[i] += *(xfmat++) * src[j];
	}
	m_Data[S-1] = src[T-1];
	
	return *this;
      }

      /** This will write this colour to the equivalent value in dstcol's format and write the value there */
    template<int T>
    vuColour<T>& toColourN(vuColour<T>& dstcol, const float* xfmat) const
      {
	int i,j;
	float *dst = dstcol.getData();
	for(i=0;i<T-1;i++) {
	  dst[i] = *(xfmat++) * m_Data[0];
	  for(j=1;j<S-1;j++)
	    dst[i] += *(xfmat++) * m_Data[j];
	}
	dst[T-1] = m_Data[S-1];
	
	return dstcol;
      }

    float m_Data[S]; //!< these are the colour component values
};

/*
typedef vuColour<4>  vuColourRGBa;
typedef vuColour<8>  vuColour7a;
typedef vuColour<32> vuColour31a;

#define vuColourRGBa	vuColour<4>
#define vuColour7a	vuColour<8>
#define vuColour31a	vuColour<32>
*/

#endif
