// vuuvolumeadapted by Steven Bergner

// original header:
// File:   Camera.h
// Author: Steve Kilthau
// Date:   August, 1999

/*
 modified February 2002
 File vuPerspectiveCamera.h
 Author Christopher Steinbach
 modified to enable recording and retrieval of camera data.


    Modified Mar 2002
    By Chris Steinbach
    Modified to add the interpolation capabilities that will be needed by
    the Key Frame Animator class

*/
/*!This file defines a camera class.  The camera is specified
   using center-of-projection (COP), view-reference-point (VRP),
   view-up-vector (VUP), field of view (FOV, for x axis), and
   aspect-ratio (width/height ratio).  The camera also depends
   on the width and height in pixels of the final image.
*/

#ifndef _VUPERSPECTIVECAMERA_H_
#define _VUPERSPECTIVECAMERA_H_

#include <stdlib.h>
#include <stdio.h>

#include "../vuLinAlg/vuVector.h"
#include "vuCamera.h"

#include "../vuMisc/vuRay.h"

//!creates a camera for perpective projection
/*! This class extends the standard vuCamera with features for a
   creating vuRay instances for a specific position.
 */
class vuPerspectiveCamera : public vuCamera
{
 public:
  vuCameraType getType(void) {return vuPERSPECTIVE_CAMERA; };

public:
    //!Default constructor - set initial values
    vuPerspectiveCamera();

    /** Copy constructor

    	This will copy all of the data related to the state of the camera

	for safety, if camera c is recording to or has any files loaded, these files will not carry
	into the newly created camera
    */
    vuPerspectiveCamera(const vuPerspectiveCamera& c);

    virtual ~vuPerspectiveCamera();			//!< Destructor

    float getFOV(void);			//!< Get the field of view
    void setFOV(const float fov);		//!< Set the field of view

    float getAspect(void);			//!< Get the aspect ratio
    void setAspect(const float aspect);	//!< Set the aspect ratio

    vuVector getTopLeft (void);
    void setTopLeft (vuVector &temp);	//!< thrown in for consistency, calls init () and ignores temp;

    vuVector getXStep (void);
    void setXStep (vuVector &temp);

    vuVector getYStep (void);
    void setYStep (vuVector &temp);

    //!Initialize the camera for the current position
    /*! This function has to be called everytime the position or
     orientation of the camera changes. We may can integrate
     this into wrapped setPosition... functions.*/
    void init(void);

    //!setup perpective projection for OpenGL according to the settings of the camera
    void glInit();

    //!Get the ray through pixel (xpixel, ypixel)
    vuRay getRay(float xpixel, float ypixel);

    //! this will save the current state of the camera to the open file that is being recorded to
    virtual void TakeSnapShot ();
    /** This will save the current state of the camera to the string to Shot Shot */
    virtual void TakeSnapShot (char* Shot);
    //! this will save a persective snapshot, ie, the members of perspective camera that are not data members of a basic camera.
    void TakeSnapShotPersp (char* Shot);
    //! This will take the snapshot contained in Shot and will restore it to the screen
    virtual int RestoreShot (char* Shot);
	/** this will restore the extra members associated with the perspective camera over the basic camera */
    int RestoreShotPersp (char* Shot);
	/** This Will take the next line from the loaded file and set this camera equal to the state preserved therein

		Please refer to the comments surrounding bool vuCamera::load (char* filename); for a complete description
		Please refer to virtual int vuCamera::RestoreNextShot () for a complete description.
	*/
    virtual int RestoreNextShot ();
	/** This Will take the Previous line from the loaded file and set this camera equal to the state preserved therein

		Please refer to the comments surrounding bool vuCamera::load (char* filename); for a complete description
		Please refer to virtual int vuCamera::RestorePreviousShot () for a complete description.
	*/
    virtual int RestorePreviousShot ();
    	/** This will return a ptr to a copy of the appropriate header for a perspective camera with this type of data to prevent
	situations like a parallel camera attempting to load a perspective camera's data, which inherently cannot work*/
    virtual char* get_id ();
    	/** This will return whether or not id equals get_id ()
	(ie, whether or not id is an appropriate file header for a perspective camera)

	NB, must be overridden in derived classes if any data is added or changed

	\return true if id is a proper file header for this type of data, false otherwise */
    virtual bool verify_id (char* id);

    //!Assignment operator
    virtual vuPerspectiveCamera& operator=(const vuPerspectiveCamera& rhs);

    	//! this will store the data of camera to a string and pass that string to the streeam 'in'
    friend ostream& operator<<(ostream& out, vuPerspectiveCamera &cam);
    	//! This will write restore cam to the state that is passed in by in.
    friend istream& operator>>(istream& in, vuPerspectiveCamera &cam);

    	/** This will create a new perspective camera and return a pointer to it cast
		as a vuCamera pointer.

		This is so that I can use the camera in vuKeyFramerDialog without concern
		as to what type of camera this points to (ie, it's easier to program for
		later). */
    virtual vuCamera* create_new ();

    /** Multiplication Operator,

        This will multiply each component of this camera by t and will create a new
    	camera to store the result and will return a pointer to that camera.
    */
    virtual vuCamera* operator* (float t);

    /** Multiplication/Assignment operator,

        This will multiply each component of this camera by t and return a pointer
    	to this camera. */
    virtual vuCamera* operator*= (float t);

    /** Addition Operator,

        This will add each component of this camera to each component of rhs and
    	store the result in a camera that it creates.  It will then return a pointer
	to the camera which it created in order to store the result. */
    virtual vuCamera* operator+ (vuPerspectiveCamera &rhs);

    /** Addition/Assignment Operator,

        This will add each component of rhs to this camera and will then return a
    	pointer to this camera. */
    virtual vuCamera* operator+= (vuPerspectiveCamera &rhs);

    /** Addition Operator,

        This will add each component of the camera pointed to by rhs with the component
    	of this.  It will then return a pointer to the camera storing the result (cast
	as a vuCamera pointer).

	Note that this method will create the camera which will store the result. */
    virtual vuCamera* operator+ (vuPerspectiveCamera *rhs);

    /** Addition/Assignment Operator,

        This will perform a componentwise addition of the camera pointed to by rhs and
    	return a pointer to this camera. */
    virtual vuCamera* operator+= (vuPerspectiveCamera *rhs);

    /** Assignment operator,

    	This will assign each component of the camera pointed to by rhs to this
	camera. */
    virtual vuPerspectiveCamera* operator= (vuPerspectiveCamera *rhs);

    /** Multiplication operator,

    	This will multiply each component of cam by t and create a new camera to store
	the result in.  It will return a pointer to the camera that stores the result. */
    friend vuCamera* operator*(float t, vuPerspectiveCamera &cam);

    /** This function will restore this camera to it's default value. */
    virtual void set_defaults ();

    virtual vuCamera* set_equal_to_interp (vuCamera* cam1, vuCamera* cam2, float t1, float t2);

protected:
    float  m_FOV;       //!< Field of view
    float  m_Aspect;    //!< Aspect ratio

    vuVector m_TopLeft;   //!< Top left of viewing frustum
    vuVector m_XStep;     //!< Distance between adjacent x pixels
    vuVector m_YStep;     //!< Distance between adjacent y pixels
};

#endif
