#ifdef WIN32
//for proper inclusion of gl.h glu.h
#include <windows.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <iostream.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "vuCamera.h"
//#include "vuMatrix.h"
#include "vuPerspectiveCamera.h"

#ifndef M_PI
#define M_PI 3.141592653589793238462643383279502884197169399375105820974944
	// now pi is defined to an insance level of accuracy, the compiler can make it reasonable
	// depending upon the data type that this is being cast to.
#endif

// Default constructor - set initial values
vuPerspectiveCamera::vuPerspectiveCamera() : m_FOV((float)45), m_Aspect((float)1.333333333)

{
  m_Position = vuVector(0,0,0);
  m_LookAtVector = vuVector(0,0,-1);
  m_UpVector = vuVector(0,1,0);

  fp = NULL;
  line_number = 0;
  m_IsRecording = 0;

  m_IsChanged = true;
}

void vuPerspectiveCamera::set_defaults ()

{
  m_Position = vuVector(0,0,0);
  m_LookAtVector = vuVector(0,0,-1);
  m_UpVector = vuVector(0,1,0);

/*  m_FOV = (float) (45);
  m_Aspect = (float) (1.333333333);
  m_Width = (320.0);
  m_Height = (240.0);*/

  m_IsChanged = true;
}

// Copy constructor
vuPerspectiveCamera::vuPerspectiveCamera(const vuPerspectiveCamera& c)
{
    m_Position = c.m_Position;
    m_LookAtVector = c.m_LookAtVector;
    m_UpVector = c.m_UpVector;
    m_RightVector   = c.m_RightVector;
    m_FOV = c.m_FOV;
    m_IsRecording = 0;
    fp = NULL;
    line_number = 0;

    m_Aspect  = c.m_Aspect;
    m_Width   = c.m_Width;
    m_Height  = c.m_Height;
    m_TopLeft = c.m_TopLeft;
    m_XStep   = c.m_XStep;
    m_YStep   = c.m_YStep;

    m_IsChanged = true;
}

// Destructor
vuPerspectiveCamera::~vuPerspectiveCamera()
{
	clear_lines ();
}

// Get the field of view
float vuPerspectiveCamera::getFOV(void)
{
    return m_FOV;
}

// Set the field of view
void vuPerspectiveCamera::setFOV(float fov)
{
    m_FOV = fov;

    m_IsChanged = true;
}

// Get the aspect ratio
float vuPerspectiveCamera::getAspect(void)
{
    return m_Aspect;
}

// Set the aspect ratio
void vuPerspectiveCamera::setAspect(float aspect)
{
    m_Aspect = aspect;

    m_IsChanged = true;
}

vuVector vuPerspectiveCamera::getTopLeft (void)

{
	return m_TopLeft;
}

void vuPerspectiveCamera::setTopLeft (vuVector &temp)

{
	init ();

	m_IsChanged = true;
}

vuVector vuPerspectiveCamera::getXStep (void)

{
	return m_XStep;
}

void vuPerspectiveCamera::setXStep (vuVector &temp)

{
	init ();

	m_IsChanged = true;
}

vuVector vuPerspectiveCamera::getYStep (void)

{
	return m_YStep;
}

void vuPerspectiveCamera::setYStep (vuVector &temp)

{
	init ();

	m_IsChanged = true;
}

// Compute the following:
//      m_TopLeft - top left coordinate of the view volume
//      m_XStep   - the distance between adjacent x pixels
//      m_YStep   - the distance between adjacent y pixels
void vuPerspectiveCamera::init(void)
{
  vuVector a;
  vuVector b;
  float HalfWidth;
  float HalfHeight;

  if (m_Width<0)
    throw "CAMERA: width of view plane must be 0 or more pixels.";
  if (m_Height<0)
    throw "CAMERA: height of view plane must be 0 or more pixels.";

  // Generate topleft, xstep, and ystep
  // m *vuVector(1,1,-1)
  a = m_LookAtVector;
  vuVector vup = m_UpVector;
  vup.makeUnit();
  b = a.cross(vup);
  b.makeUnit();
  if ((b[0] == 0) && (b[1] == 0) && (b[2] == 0))
    throw "CAMERA: eye-vector and up-vector are coincident.";

  HalfWidth = (float)tan(0.5*M_PI/180.0*m_FOV)*a.norm();
  HalfHeight = HalfWidth/m_Aspect;
  vuVector tmpvec = b*HalfWidth;		// SB: some workaround

  m_TopLeft = (HalfHeight*m_UpVector)-(tmpvec)+m_LookAtVector+m_Position;
  m_XStep = b*(2.0f*HalfWidth/(float)m_Width);
  m_YStep = m_UpVector*(-2.0f*HalfHeight/(float)m_Height);
  m_TopLeft = m_TopLeft + m_XStep;
  tmpvec = m_YStep*0.5f;
  m_TopLeft = m_TopLeft + tmpvec;
  m_TopLeft -= m_Position; // Not really top-left anymore (but who cares!)

  m_IsChanged = true;
}

// Get the ray through pixel (xpixel, ypixel)
// These values can be fractions of a pixel. Ex. (100.5, 45.5)
vuRay vuPerspectiveCamera::getRay(float xpixel, float ypixel)
{
  vuRay r;
  if ((xpixel>m_Width-0.5)  || (xpixel<-0.5) ||
      (ypixel>m_Height-0.5) || (ypixel<-0.5))
    throw "CAMERA: attempt to shoot ray outside view plane.";
  r.m_Position = m_Position;
  r.m_Direction = m_TopLeft + (m_XStep*xpixel) + (m_YStep*ypixel);
  return r;
}

// Assignment operator
vuPerspectiveCamera& vuPerspectiveCamera::operator=(const vuPerspectiveCamera& rhs)
{
  if (this == &rhs)
    return *this;

  cout<< "nice equals" << endl;
  m_Position = rhs.m_Position;
  m_LookAtVector = rhs.m_LookAtVector;
  m_UpVector = rhs.m_UpVector;
  m_RightVector = rhs.m_RightVector;
  m_FOV = rhs.m_FOV;
  m_Aspect = rhs.m_Aspect;
  m_Width = rhs.m_Width;
  m_Height = rhs.m_Height;
  m_TopLeft = rhs.m_TopLeft;
  m_XStep = rhs.m_XStep;
  m_YStep = rhs.m_YStep;

  m_IsChanged = true;

  return *this;
}

void vuPerspectiveCamera::glInit()
{
//#if defined (NVIDIA_EXT)
  ::gluPerspective(m_FOV, m_Aspect, 1.0, 10000.0);
//#endif
}

void vuPerspectiveCamera::TakeSnapShot ()

{
	if (m_IsRecording)

	{
		char Shot [1024];

		TakeSnapShot (Shot);

		fputs (Shot, fp);
	}
}

void vuPerspectiveCamera::TakeSnapShot (char* Shot)

{
	char temp [512];

	TakeSnapShotBasic (Shot);

	TakeSnapShotPersp (temp);

	strcat (Shot, ", ");
	strcat (Shot, temp);
}

void vuPerspectiveCamera::TakeSnapShotPersp (char* Shot)

{
	char temp [256];

	gcvt (m_FOV, 14, Shot);
	strcat (Shot, ", \0");

	gcvt (m_Aspect, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", \0");

//	_itoa (m_Width, &(Shot [position]), 10);
	gcvt (m_Width, 10, temp);
	strcat (Shot, temp);
	strcat (Shot, ", \0");

//	itoa (m_Height, &(Shot [position]), 10);
	gcvt (m_Height, 10, temp);
	strcat (Shot, temp);
	strcat (Shot, ", (\0");

	m_TopLeft.save (temp);
	strcat (Shot, temp);
	strcat (Shot, "), (\0");

	m_XStep.save (temp);
	strcat (Shot, temp);
	strcat (Shot, "), (\0");

	m_YStep.save (temp);
	strcat (Shot, temp);
	strcat (Shot, ")\n\0");
}

int vuPerspectiveCamera::RestoreShot (char* Shot)

{
	int position = RestoreShotBasic (Shot);

	position += get_next_comma (&(Shot [position]));

	return position + RestoreShotPersp (&(Shot [position]));
}

int vuPerspectiveCamera::RestoreShotPersp (char* Shot)

{
	int position = 0;

	m_FOV = atof (&(Shot [position]));

	position = get_next_comma (&(Shot [position])) + 1;
	position += clear_blanks (&(Shot [position]));

	m_Aspect = atof (&(Shot [position]));

	position += get_next_comma (&(Shot [position])) + 1;
	position += clear_blanks (&(Shot [position]));

	m_Width = atoi (&(Shot [position]));

	position += get_next_comma (&(Shot [position])) + 1;
	position += clear_blanks (&(Shot [position]));

	m_Height = atoi (&(Shot [position]));

	position += get_next_comma (&(Shot [position])) + 1;
	position += clear_blanks (&(Shot [position]));

	position += get_next_open (&(Shot [position])) + 1;
	m_TopLeft.load (&(Shot [position]));
	position += get_next_close (&(Shot [position])) + 1;

	position += get_next_open (&(Shot [position])) + 1;
	m_XStep.load (&(Shot [position]));
	position += get_next_close (&(Shot [position])) + 1;

	position += get_next_open (&(Shot [position])) + 1;
	m_YStep.load (&(Shot [position]));
	position += get_next_close (&(Shot [position])) + 1;

	return position;
}

int vuPerspectiveCamera::RestoreNextShot ()

{
//	if (++line_number < int (lines.size ()))

	if (IsNextAvailable ())

	{
	//	cout << "hello: " << lines [line_number + 1] << endl;

		int position = RestoreShotBasic (lines [++line_number]);

		position += get_next_comma (&((lines [line_number]) [position])) + 1;
		return RestoreShotPersp (&((lines [line_number]) [position]));
	}

	return -1;
}

int vuPerspectiveCamera::RestorePreviousShot ()

{
//	if (++line_number < int (lines.size ()))
	if (IsPreviousAvailable ())

	{
		int position = RestoreShotBasic (lines [--line_number]);

		position += get_next_comma (&((lines [line_number]) [position])) + 1;
		return RestoreShotPersp (&((lines [line_number]) [position]));
	}

	return -1;
}

//D1.00 is for the first type of data implemented
char id_string_cgs_xq2 [] = "vuPerspectiveCameraD1.00__\0";

char* vuPerspectiveCamera::get_id ()

{
	return id_string_cgs_xq2;
}

bool vuPerspectiveCamera::verify_id (char* id)

{
	return (strncmp (id, get_id (), strlen (get_id ())) == 0);
}

ostream& operator<<(ostream& out, vuPerspectiveCamera &cam)

{
	char Shot [1024];

	cam.TakeSnapShot (Shot);

	out << Shot;

	cam.m_IsChanged = true;

	return out;
}

istream& operator>>(istream& in, vuPerspectiveCamera &cam)

{
	char Shot [1024];

	in >> Shot;

	cam.RestoreShot (Shot);

	return in;
}

vuCamera* vuPerspectiveCamera::create_new ()

{
	vuPerspectiveCamera* ptr = new vuPerspectiveCamera [1];

	return (vuCamera*) (ptr);
}

vuPerspectiveCamera temp_cam_vpc_cgs__;

vuCamera* vuPerspectiveCamera::operator* (float t)

{
	vuPerspectiveCamera* temp_cam_vpc_cgs__ = (vuPerspectiveCamera *) (create_new ());

		// These four items are the basic camera
		// items...
	temp_cam_vpc_cgs__->setPosition (m_Position * t);
	temp_cam_vpc_cgs__->setUpVector (m_UpVector * t);
	temp_cam_vpc_cgs__->setLookAtVector (m_LookAtVector * t);
//	temp_cam.setRightVector (m_RightVector * t);

	temp_cam_vpc_cgs__->setFOV (m_FOV * t);
	temp_cam_vpc_cgs__->setAspect (m_Aspect * t);

	temp_cam_vpc_cgs__->setWidth((int)(m_Width * t));
	temp_cam_vpc_cgs__->setHeight((int)(m_Height * t));

	temp_cam_vpc_cgs__->init ();

	m_IsChanged = true;

	return (vuCamera *) (temp_cam_vpc_cgs__);
}

vuCamera* vuPerspectiveCamera::operator*= (float t)

{
		// These four items are the basic camera
		// items...

	setPosition (m_Position *= t);
	setUpVector (m_UpVector *= t);
	setLookAtVector (m_LookAtVector *= t);
//	temp_cam.setRightVector (m_RightVector * t);

	setFOV (m_FOV *= t);
	setAspect (m_Aspect *= t);

	m_Width  = (int)(m_Width * t);
	m_Height = (int)(m_Height * t);

	setWidth(m_Width);
	setHeight(m_Height);

	init ();

	m_IsChanged = true;

	return (vuCamera *) (this);
}

vuCamera* operator* (float t, vuPerspectiveCamera &cam)

{
	vuPerspectiveCamera* temp_cam_vpc_cgs__ = (vuPerspectiveCamera *) (cam.create_new ());

		// These four items are the basic camera
		// items...
	temp_cam_vpc_cgs__->setPosition (cam.m_Position * t);
	temp_cam_vpc_cgs__->setUpVector (cam.m_UpVector * t);
	temp_cam_vpc_cgs__->setLookAtVector (cam.m_LookAtVector * t);
//	temp_cam.setRightVector (cam.m_RightVector * t);

	temp_cam_vpc_cgs__->setFOV (cam.m_FOV * t);
	temp_cam_vpc_cgs__->setAspect (cam.m_Aspect * t);

	temp_cam_vpc_cgs__->setWidth((int)(cam.m_Width * t));
	temp_cam_vpc_cgs__->setHeight((int)(cam.m_Height * t));

	temp_cam_vpc_cgs__->init ();

	cam.m_IsChanged = true;

	return (vuCamera *) (temp_cam_vpc_cgs__);
}

vuCamera* vuPerspectiveCamera::operator+ (vuPerspectiveCamera &rhs)

{
	vuPerspectiveCamera* temp_cam_vpc_cgs__ = (vuPerspectiveCamera *) (create_new ());

		// These four items are the basic camera
		// items...
	temp_cam_vpc_cgs__->setPosition (m_Position + rhs.getPosition ());
	temp_cam_vpc_cgs__->setUpVector (m_UpVector + rhs.getUpVector ());
	temp_cam_vpc_cgs__->setLookAtVector (m_LookAtVector + rhs.getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	temp_cam_vpc_cgs__->setFOV (m_FOV + rhs.getFOV ());
	temp_cam_vpc_cgs__->setAspect (m_Aspect + rhs.getAspect ());

	temp_cam_vpc_cgs__->setWidth(m_Width + rhs.getWidth ());
	temp_cam_vpc_cgs__->setHeight(m_Height + rhs.getHeight ());

	temp_cam_vpc_cgs__->init ();

	m_IsChanged = true;

	return (vuCamera *) (temp_cam_vpc_cgs__);
}

vuCamera* vuPerspectiveCamera::operator+= (vuPerspectiveCamera &rhs)

{
		// These four items are the basic camera
		// items...
	setPosition (m_Position += rhs.getPosition ());
	setUpVector (m_UpVector += rhs.getUpVector ());
	setLookAtVector (m_LookAtVector += rhs.getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	setFOV (m_FOV += rhs.getFOV ());
	setAspect (m_Aspect += rhs.getAspect ());

	setWidth(m_Width += rhs.getWidth ());
	setHeight(m_Height += rhs.getHeight ());

	init ();

	m_IsChanged = true;

	return (vuCamera *) (this);
}

vuCamera* vuPerspectiveCamera::operator+ (vuPerspectiveCamera *rhs)

{
	vuPerspectiveCamera* temp_cam_vpc_cgs__ = (vuPerspectiveCamera *) (create_new ());

		// These four items are the basic camera
		// items...
	temp_cam_vpc_cgs__->setPosition (m_Position + rhs->getPosition ());
	temp_cam_vpc_cgs__->setUpVector (m_UpVector + rhs->getUpVector ());
	temp_cam_vpc_cgs__->setLookAtVector (m_LookAtVector + rhs->getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	temp_cam_vpc_cgs__->setFOV (m_FOV + rhs->getFOV ());
	temp_cam_vpc_cgs__->setAspect (m_Aspect + rhs->getAspect ());

	temp_cam_vpc_cgs__->setWidth(m_Width + rhs->getWidth ());
	temp_cam_vpc_cgs__->setHeight(m_Height + rhs->getHeight ());

	temp_cam_vpc_cgs__->init ();

	m_IsChanged = true;

	return (vuCamera *) (temp_cam_vpc_cgs__);
}

vuCamera* vuPerspectiveCamera::operator+= (vuPerspectiveCamera *rhs)

{
		// These four items are the basic camera
		// items...
	setPosition (m_Position += rhs->getPosition ());
	setUpVector (m_UpVector += rhs->getUpVector ());
	setLookAtVector (m_LookAtVector += rhs->getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	setFOV (m_FOV += rhs->getFOV ());
	setAspect (m_Aspect += rhs->getAspect ());

	setWidth(m_Width += rhs->getWidth ());
	setHeight(m_Height += rhs->getHeight ());

	init ();

	m_IsChanged = true;

	return (vuCamera *) (&temp_cam_vpc_cgs__);
}

vuPerspectiveCamera* vuPerspectiveCamera::operator= (vuPerspectiveCamera *rhs)

{
	m_Position = rhs->m_Position;
	m_UpVector = rhs->m_UpVector;
	m_LookAtVector = rhs->m_LookAtVector;
	m_RightVector = rhs->m_RightVector;

	m_FOV = rhs->m_FOV;
	m_Aspect = rhs->m_Aspect;

	m_Width = rhs->m_Width;
	m_Height = rhs->m_Height;

	m_TopLeft = rhs->m_TopLeft;
	m_XStep = rhs->m_XStep;
	m_YStep = rhs->m_YStep;

	m_IsChanged = true;

	return this;
}

vuCamera* vuPerspectiveCamera::set_equal_to_interp (vuCamera* cam1, vuCamera* cam2, float t1, float t2)

{
	vuPerspectiveCamera *pcam1 = (vuPerspectiveCamera *) (cam1);
	vuPerspectiveCamera *pcam2 = (vuPerspectiveCamera *) (cam2);

	m_Position = pcam1->m_Position * t1;
	m_UpVector = pcam1->m_UpVector * t1;
	m_LookAtVector = pcam1->m_LookAtVector * t1;
	m_RightVector = pcam1->m_RightVector * t1;

	m_FOV = pcam1->m_FOV * t1;
	m_Aspect = pcam1->m_Aspect * t1;

	m_Width  = (int)(pcam1->m_Width *t1 + pcam2->m_Width*t2);
	m_Height = (int)(pcam1->m_Height*t1 + pcam2->m_Height*t2);
	m_TopLeft = pcam1->m_TopLeft * t1;
	m_XStep = pcam1->m_XStep * t1;
	m_YStep = pcam1->m_YStep * t1;

	m_Position += pcam2->m_Position * t2;
	m_UpVector += pcam2->m_UpVector * t2;
	m_LookAtVector += pcam2->m_LookAtVector * t2;
	m_RightVector += pcam2->m_RightVector * t2;

	m_FOV += pcam1->m_FOV * t2;
	m_Aspect += pcam1->m_Aspect * t2;
	m_TopLeft += pcam1->m_TopLeft * t2;
	m_XStep += pcam1->m_XStep * t2;
	m_YStep += pcam1->m_YStep * t2;

//	*pcam1 *= t1;
//	*pcam2 *= t2;
//
//	*pcam1 += pcam2;

	cout << "t1:" << t1 << "t2:" << t2 << endl;

	m_IsChanged = true;

	return (vuCamera *) (pcam1);
}

