/** This file defines a parallel camera derived from vuCamera.

    The camera also depends on the width and height in pixels of the
    final image.

    Written by Steven Kilthau

    Modified Feb 2002
    By Chris Steinbach
    Modified to add the capabilities to record and restore states to the camera

    Modified Mar 2002
    By Chris Steinbach
    Modified to add the interpolation capabilities that will be needed by
    the Key Frame Animator class
*/

#ifndef _VUPARALLELCAMERA_H_
#define _VUPARALLELCAMERA_H_

#include "../vuLinAlg/vuVector.h"
#include "vuCamera.h"

#include "../vuMisc/vuRay.h"

/** This class creates a camera class for orthogonal projections.

Derived from vuCamera.
*/
class vuParallelCamera : public vuCamera

{
 public:
  vuCameraType getType(void) {return vuPARALLEL_CAMERA; };

public:
    //!Default constructor - set initial values
    vuParallelCamera();
    /** copy constructor

    	This will only copy the state data, this will not copy loaded files or recording information. */
    vuParallelCamera(const vuParallelCamera& c);
    virtual ~vuParallelCamera();			//!< Destructor

    float getXRange(void);		//!< set X size of the viewing box
    void setXRange(float xrng);		//!< set X size of the viewing box

    float getYRange(void);		//!< set Y size of the viewing box
    void setYRange(float yrng);		//!< set Y size of the viewing box

    void setNear (float near);		/**< this will set the near clipping plane of this
    						camera to be 'near' away from the camera's
						position */
    float getNear ();			/**< This will return the distance to the near
    						clipping plane from the eye of the camera. */

    void setFar (float far);		/**< This will set the far clipping plane of this
    						camera to be 'far' away from the camera's
						position */
    float getFar ();			/**< This will return the distance of the far
    						clipping plane to the eye of the camera. */

    //!Initialize the camera for the current position
    /*! This function has to be called everytime the position or
     orientation of the camera changes. We may can integrate
     this into wrapped setPosition... functions.*/
    void init(void);

    //!setup perpective projection for OpenGL according to the settings of the camera
    void glInit();

    //!Get the ray through pixel (xpixel, ypixel)
    vuRay getRay(float xpixel, float ypixel);

    //!returns the distance of a point from the projection plane
    float getDistance(const vuVector& point) const;

    /** projects a point to the plane
	\return first two components contain x and y component of local camera
	coordinate system. z component is distance to projection plane (not normalized) */
    vuVector& project(vuVector& point) const;


    /** this start recording to the file named "record_to". */
    /** this will record a snapshot to Shot instead of the file that recordings are being written to. */
    virtual void TakeSnapShot (char* Shot);
    /** this will record the current state to the next line of the file that is being recorded to */
    virtual void TakeSnapShot ();
    /** This will take a persp snap shot??? this function is wrong and will be fixed later */
    void TakeSnapShotOrtho (char* Shot);
    /** this will restore the snapshot saved in Shot */
    virtual int RestoreShot (char* Shot);
    /** This will restore the extra members of parallelcamera that are not contained in a basic camera */
    int RestoreShotOrtho (char* Shot);
    /** this will restore the next line of the file that has been loaded into this camera by load

    	Please refer to virtual int RestoreNextShot for a complete description of this function*/
    virtual int RestoreNextShot ();
    /** this will restore the previous line of the file that has been loaded into this camera by load

    	Please refer to virtual int RestorePreviousShot for a complete description of this function*/
    virtual int RestorePreviousShot ();

    	/** This will return a ptr to a string containing a copy of the valid header fromat for this type of camera */
    virtual char* get_id ();
    	/** This will return true if id is an appropriate header, false otherwise */
    virtual bool verify_id (char* id);

    //!Assignment operator
    virtual vuParallelCamera& operator=(const vuParallelCamera& rhs);

    	//! This will get the current state of cam and write it to the stream, "out"
    friend ostream& operator<<(ostream& out, vuParallelCamera &cam);
    	//! This will take the state store in the stream "in" and set cam equal to that state.
    friend istream& operator>>(istream& in, vuParallelCamera &cam);

    /** This will create a new vuParallelCamera and return the pointer to this newly
    	created parallel camera (cast as a vuCamera). */
    virtual vuCamera* create_new ();

    
    virtual vuCamera* set_equal_to_interp (vuCamera* cam1, vuCamera* cam2, float t1, float t2);

    /** Multiplication Operator,

        This will multiply each component of this camera by t and store the result
    	in a new camera (which it will create).  It will then return a pointer to this
	camera (recast as a vucamera pointer). */
    virtual vuCamera* operator* (float t);

    /** Multiplication/Assignment operator,

        This will multiply each component of this camera by t and return a pointer
        to this camera */
    virtual vuCamera* operator*= (float t);

    /** Addition Operator,

        This will add each component of rhs with each component of this, store the result
        in a newly created camera and return a pointer to that camera (recast as a
	vuCamera pointer). */
    virtual vuCamera* operator+ (vuParallelCamera &rhs);

    /** Addition/Assignment operator,

        This will add each component of rhs to this camera and will then return a pointer
        to this camera. */
    virtual vuCamera* operator+= (vuParallelCamera &rhs);

    /** Addition operator,

        This will add each component of the camera pointed to by rhs to this camera and
        store the result in a newly created perspective camera and will return a
	pointer to this newly created perspective camera (recast as a vuCamera). */
    virtual vuCamera* operator+ (vuParallelCamera *rhs);

    /** Addition/Assignment operator,

        This will add each component of the camera pointed to by rhs to this camera and
        will return a pointer to this camera. */
    virtual vuCamera* operator+= (vuParallelCamera *rhs);

    /** Assignment Operator,

    This will assign each component of the camera pointed to by rhs to this camera
    and return a pointer to this camera. */
    virtual vuParallelCamera* operator= (vuParallelCamera *rhs);

    /** Multiplication operator,

    This will multiply each component of cam by t and will store the result in a newly
    created parallel camera, and will return a pointer to that camera */
    friend vuCamera* operator*(float t, vuParallelCamera &cam);

protected:
    //float  m_CullBox[6];   left, right, bottom, top, near, far - clipping box

    float m_XRange;	//!< The
    float m_YRange;
    float m_XScale, m_YScale; //!< The x and y scaling factors
    float m_Near;	//!< The near clipping plane
    float m_Far;	//!< the far clipping plane

    vuVector m_XStep;     //!< Distance between adjacent x pixels
    vuVector m_YStep;     //!< Distance between adjacent y pixels
};

#endif


