#ifdef WIN32
//for proper inclusion of gl.h glu.h
#include <windows.h>
#endif

#include <string.h>
#include <math.h>
#include <iostream.h>
#include <GL/gl.h>

#include "vuCamera.h"
//#include "vuMatrix.h"
#include "vuParallelCamera.h"

// Default constructor - set initial values
vuParallelCamera::vuParallelCamera() :
  m_Near(0), m_Far(1000)
{
  m_Position = vuVector(0,0,0);
  m_LookAtVector = vuVector(0,0,-1);
  m_UpVector = vuVector(0,1,0);
  setXRange(320);
  setYRange(240);

  fp = NULL;
  m_IsRecording = 0;
  line_number = 0;

  m_IsChanged = true;
}

// Copy constructor
vuParallelCamera::vuParallelCamera(const vuParallelCamera& c)
{
    m_Position     = c.m_Position;
    m_LookAtVector = c.m_LookAtVector;
    m_UpVector     = c.m_UpVector;
    m_Width        = c.m_Width;
    m_Height       = c.m_Height;
    m_XRange       = c.m_XRange;
    m_YRange       = c.m_YRange;
    m_XScale       = c.m_XScale;
    m_YScale       = c.m_YScale;
    m_XStep        = c.m_XStep;
    m_YStep        = c.m_YStep;

    fp = NULL;
    m_IsRecording = 0;
    line_number = 0;

    m_IsChanged = true;
}

// Destructor
vuParallelCamera::~vuParallelCamera()
{
	clear_lines ();
}

float vuParallelCamera::getXRange(void)
{
  return m_XRange;
}

void vuParallelCamera::setXRange(float xrng)
{
  if(xrng == 0.0)
    throw "Camera has zero x-range!\n";
  m_XRange = xrng;
  m_XScale = m_Width/m_XRange;

  m_IsChanged = true;
}

    
float vuParallelCamera::getYRange(void)
{
  return m_YRange;
}

void vuParallelCamera::setYRange(float yrng)
{
  if(yrng == 0.0)
    throw "Camera has zero y-range!\n";
  m_YRange = yrng;
  m_YScale = m_Height/m_YRange;

  m_IsChanged = true;
}

void vuParallelCamera::setNear (float near)

{
	m_Near = double (near);	// this double recasting is something stupid.
				// without it, the code won't compile under windows,
				// but with it, it will, so that's why it's here.

	m_IsChanged = true;
}

float vuParallelCamera::getNear ()

{
	return m_Near;
}

void vuParallelCamera::setFar (float far)

{
	m_Far = double (far);	// this double recasting is something stupid.
				// without it, the code won't compile under windows,
				// but with it, it will, so that's why it's here.

	m_IsChanged = true;
}

float vuParallelCamera::getFar ()

{
	return m_Far;
}

// Compute the following:
//      m_XStep   - the distance between adjacent x pixels
//      m_YStep   - the distance between adjacent y pixels
void vuParallelCamera::init(void)
{
  if (m_Width<0)
    throw "CAMERA: width of view plane must be 0 or more pixels.";
  if (m_Height<0)
    throw "CAMERA: height of view plane must be 0 or more pixels.";
  
  // Generate topleft, xstep, and ystep
  // m *vuVector(1,1,-1)

  vuVector right  = m_RightVector;
  vuVector lookAt = m_LookAtVector;
  vuVector     up = lookAt.cross(right);

  right.makeUnit();
  up.makeUnit();
  lookAt.makeUnit();

  m_XStep   = right * (m_XRange / m_Width);
  m_YStep   = up    * (m_YRange / m_Height);

  m_IsChanged = true;
}

// Get the ray through pixel (xpixel, ypixel)
vuRay vuParallelCamera::getRay(float xpixel, float ypixel)
{
  xpixel -= m_Width/2;
  ypixel -= m_Height/2;

  vuVector pos = m_Position + xpixel * m_XStep + ypixel * m_YStep;
  vuRay    r;

  r.m_Position = pos;
  r.m_Direction = m_LookAtVector;
  
  return r;
}

// Assignment operator
vuParallelCamera& vuParallelCamera::operator=(const vuParallelCamera& rhs)
{
  if (this == &rhs)
    return *this;
  m_Position     = rhs.m_Position;
  m_LookAtVector = rhs.m_LookAtVector;
  m_UpVector     = rhs.m_UpVector;

  m_Width    = rhs.m_Width;
  m_Height   = rhs.m_Height;
  m_XStep    = rhs.m_XStep;
  m_YStep    = rhs.m_YStep;

  m_IsChanged = true;

  return *this;
}

void vuParallelCamera::glInit()

{
//#if defined (NVIDIA_EXT)
  ::glOrtho(-m_XRange/2,m_XRange/2,-m_YRange/2,m_YRange/2,m_Near,m_Far);
//#endif
}

float vuParallelCamera::getDistance(const vuVector& point) const
{
  return (point-m_Position).dot(m_LookAtVector);
}

vuVector& vuParallelCamera::project(vuVector& point) const
{
  vuVector a,r,u;
  u = m_RightVector.cross(m_LookAtVector);		// up
  point -= m_Position;
  float dist = point.dot(m_LookAtVector);
  float px = point.dot(m_RightVector)*m_XScale;
  float py = point.dot(u)*m_YScale;
  float *v = point.getData();
  v[0]=px; v[1]=py; v[2]=dist;
  return point;
}

void vuParallelCamera::TakeSnapShot (char* Shot)

{
	TakeSnapShotBasic (Shot);
	char temp [512];

	TakeSnapShotOrtho (temp);
	strcat (Shot, ", ");
	strcat (Shot, temp);
}

void vuParallelCamera::TakeSnapShot ()

{
	if (m_IsRecording)

	{
		char Shot [1024];

		TakeSnapShot (Shot);

		fputs (Shot, fp);
	}
}

void vuParallelCamera::TakeSnapShotOrtho (char* Shot)

{
	char temp [128];

	gcvt (m_Width, 14, Shot);
	strcat (Shot, ", ");

	gcvt (m_Height, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", ");

	gcvt (m_XRange, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", ");

	gcvt (m_YRange, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", ");

	strcat (Shot, ", ");
	gcvt (m_XScale, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", ");

	gcvt (m_YScale, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", ");

	gcvt (m_Near, 14, temp);
	strcat (Shot, temp);
	strcat (Shot, ", ");

	gcvt (m_Far, 14, temp);
	strcat (Shot, temp);

	strcat (Shot, "), (\0");
	m_XStep.save (temp);
	strcat (Shot, temp);
	strcat (Shot, "), (\0");

	m_YStep.save (temp);
	strcat (Shot, temp);
	strcat (Shot, ")\n\0");
}

int vuParallelCamera::RestoreShot (char* Shot)

{
	int position = RestoreShotBasic (Shot);

	position += get_next_comma (&(Shot [position])) + 1;
	position += clear_blanks (&(Shot [position]));

	return position + RestoreShotOrtho (&(Shot [position]));
}

int vuParallelCamera::RestoreShotOrtho (char* Shot)

{
	int position;

	m_Width = int (atof (Shot));
	position = get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_Height = int (atof (&(Shot [position])));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_XRange = atof (&(Shot [position]));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_YRange = atof (&(Shot [position]));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_XScale = atof (&(Shot [position]));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_YScale = atof (&(Shot [position]));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_Near = atof (&(Shot [position]));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	m_Far = atof (&(Shot [position]));
	position += get_next_comma (Shot) + 1;
	position += clear_blanks (&(Shot [position]));

	position += get_next_open (&(Shot [position])) + 1;
	m_XStep.load (&(Shot [position]));
	position += get_next_close (&(Shot [position])) + 1;

	position += get_next_open (&(Shot [position])) + 1;
	m_YStep.load (&(Shot [position]));
	position += get_next_close (&(Shot [position])) + 1;

	return position;
}

int vuParallelCamera::RestoreNextShot ()

{
//	if (++line_number < int (lines.size ()))
	if (IsNextAvailable ())
		return RestoreShot (lines [++line_number]);

	return -1;
}

int vuParallelCamera::RestorePreviousShot ()

{
//	if (++line_number < int (lines.size ()))
	if (IsNextAvailable ())
		return RestoreShot (lines [--line_number]);

	return -1;
}

//the D1.00 is to signify that this is the first type that is written (in case the data changes, you don't want to
// try to pass an old type through into a newer one.
char id_string_cgs_xq3 [] = "vuParallelCameraD1.00__";

char* vuParallelCamera::get_id ()

{
	return id_string_cgs_xq3;
}

bool vuParallelCamera::verify_id (char* id)

{
	return (strncmp (get_id (), id, strlen (get_id ())));
}

ostream& operator<<(ostream& out, vuParallelCamera &cam)

{
	char Shot [1024];

	cam.TakeSnapShot (Shot);

	out << Shot;

	return out;
}

istream& operator>>(istream& in, vuParallelCamera &cam)

{
	char Shot [1024];

	in >> Shot;

	cam.RestoreShot (Shot);

	cam.m_IsChanged = true;

	return in;
}

vuCamera* vuParallelCamera::create_new ()

{
	vuParallelCamera* ptr = new vuParallelCamera [1];

	return (vuCamera *) (ptr);
}

vuParallelCamera temp_cam_vpac_cgs__;

vuCamera* vuParallelCamera::operator* (float t)

{
	vuParallelCamera* temp_cam_vpac_cgs__ = (vuParallelCamera *) (create_new ());

		// These four are the basic things that
		// are in vuCamera.
	temp_cam_vpac_cgs__->setPosition (m_Position * t);
	temp_cam_vpac_cgs__->setUpVector (m_UpVector * t);
	temp_cam_vpac_cgs__->setLookAtVector (m_LookAtVector * t);
//	temp_cam.setRightVector (m_RightVector * t);

	temp_cam_vpac_cgs__->setWidth((int)(m_Width*t));
	temp_cam_vpac_cgs__->setHeight((int)(m_Height*t));

	temp_cam_vpac_cgs__->setXRange (m_XRange * t);
	temp_cam_vpac_cgs__->setYRange (m_YRange * t);

		// already done
//	temp_cam.setXScale (m_XScale * t);
//	temp_cam.setYScale (m_YScale * t);

	temp_cam_vpac_cgs__->setNear (m_Near * t);
	temp_cam_vpac_cgs__->setFar (m_Far * t);

	m_IsChanged = true;

	return temp_cam_vpac_cgs__;
}

vuCamera* vuParallelCamera::operator*= (float t)

{
		// These four are the basic things that
		// are in vuCamera.
	setPosition (m_Position *= t);
	setUpVector (m_UpVector *= t);
	setLookAtVector (m_LookAtVector *= t);
//emp_cam.setRightVector (m_RightVector *= t);

	m_Width  = (int)(m_Width * t);
	m_Height = (int)(m_Height * t);

	setWidth(m_Width);
	setHeight(m_Height);

	setXRange (m_XRange *= t);
	setYRange (m_YRange *= t);

		// already done
//	temp_cam.setXScale (m_XScale *= t);
//	temp_cam.setYScale (m_YScale *= t);

	setNear (m_Near *= t);
	setFar (m_Far *= t);

	m_IsChanged = true;

	return (vuCamera *) (this);
}

vuCamera* operator* (float t, vuParallelCamera &cam)

{
	vuParallelCamera* temp_cam_vpac_cgs__ = (vuParallelCamera *) (cam.create_new ());

		// These four are the basic things that
		// are in vuCamera.
	temp_cam_vpac_cgs__->setPosition (cam.m_Position * t);
	temp_cam_vpac_cgs__->setUpVector (cam.m_UpVector * t);
	temp_cam_vpac_cgs__->setLookAtVector (cam.m_LookAtVector * t);
//	temp_cam.setRightVector (cam.m_RightVector * t);

	temp_cam_vpac_cgs__->setWidth ((int)(cam.m_Width * t));
	temp_cam_vpac_cgs__->setHeight ((int)(cam.m_Height * t));

	temp_cam_vpac_cgs__->setXRange (cam.m_XRange * t);
	temp_cam_vpac_cgs__->setYRange (cam.m_YRange * t);

		// already done
//	temp_cam.setXScale (cam.m_XScale * t);
//	temp_cam.setYScale (cam.m_YScale * t);

	temp_cam_vpac_cgs__->setNear (cam.m_Near * t);
	temp_cam_vpac_cgs__->setFar (cam.m_Far * t);

	return (vuCamera *) (temp_cam_vpac_cgs__);
}


vuCamera* vuParallelCamera::operator+ (vuParallelCamera &rhs)

{
	vuParallelCamera* temp_cam_vpac_cgs__ = (vuParallelCamera *) (create_new ());

		// These four are the basic things that
		// are in vuCamera.
	temp_cam_vpac_cgs__->setPosition (m_Position + rhs.getPosition ());
	temp_cam_vpac_cgs__->setUpVector (m_UpVector + rhs.getUpVector ());
	temp_cam_vpac_cgs__->setLookAtVector (m_LookAtVector + rhs.getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	temp_cam_vpac_cgs__->setWidth (m_Width + rhs.getWidth ());
	temp_cam_vpac_cgs__->setHeight (m_Height + rhs.getHeight ());

	temp_cam_vpac_cgs__->setXRange (m_XRange + rhs.getXRange ());
	temp_cam_vpac_cgs__->setYRange (m_YRange + rhs.getYRange ());

		// already done
//	temp_cam.setXScale (m_XScale * t);
//	temp_cam.setYScale (m_YScale * t);

	temp_cam_vpac_cgs__->setNear (m_Near + rhs.getNear ());
	temp_cam_vpac_cgs__->setFar (m_Far + rhs.getFar ());

	m_IsChanged = true;

	return (vuCamera* ) (temp_cam_vpac_cgs__);
}

vuCamera* vuParallelCamera::operator+= (vuParallelCamera &rhs)

{
		// These four are the basic things that
		// are in vuCamera.
	setPosition (m_Position += rhs.getPosition ());
	setUpVector (m_UpVector += rhs.getUpVector ());
	setLookAtVector (m_LookAtVector += rhs.getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	setWidth (m_Width += rhs.getWidth ());
	setHeight (m_Height += rhs.getHeight ());

	setXRange (m_XRange += rhs.getXRange ());
	setYRange (m_YRange + rhs.getYRange ());

		// already done
//	temp_cam.setXScale (m_XScale * t);
//	temp_cam.setYScale (m_YScale * t);

	setNear (m_Near += rhs.getNear ());
	setFar (m_Far += rhs.getFar ());

	m_IsChanged = true;

	return (vuCamera *) (this);
}

vuCamera* vuParallelCamera::operator+ (vuParallelCamera *rhs)

{
	vuParallelCamera* temp_cam_vpac_cgs__ = (vuParallelCamera *) (create_new ());

		// These four are the basic things that
		// are in vuCamera.
	temp_cam_vpac_cgs__->setPosition (m_Position + rhs->getPosition ());
	temp_cam_vpac_cgs__->setUpVector (m_UpVector + rhs->getUpVector ());
	temp_cam_vpac_cgs__->setLookAtVector (m_LookAtVector + rhs->getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	temp_cam_vpac_cgs__->setWidth (m_Width + rhs->getWidth ());
	temp_cam_vpac_cgs__->setHeight (m_Height + rhs->getHeight ());

	temp_cam_vpac_cgs__->setXRange (m_XRange + rhs->getXRange ());
	temp_cam_vpac_cgs__->setYRange (m_YRange + rhs->getYRange ());

		// already done
//	temp_cam.setXScale (m_XScale * t);
//	temp_cam.setYScale (m_YScale * t);

	temp_cam_vpac_cgs__->setNear (m_Near + rhs->getNear ());
	temp_cam_vpac_cgs__->setFar (m_Far + rhs->getFar ());

	m_IsChanged = true;

	return (vuCamera *) (temp_cam_vpac_cgs__);
}

vuCamera* vuParallelCamera::operator+= (vuParallelCamera *rhs)

{
		// These four are the basic things that
		// are in vuCamera.
	setPosition (m_Position += rhs->getPosition ());
	setUpVector (m_UpVector += rhs->getUpVector ());
	setLookAtVector (m_LookAtVector += rhs->getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	setWidth (m_Width += rhs->getWidth ());
	setHeight (m_Height += rhs->getHeight ());

	setXRange (m_XRange += rhs->getXRange ());
	setYRange (m_YRange += rhs->getYRange ());

		// already done
//	temp_cam.setXScale (m_XScale * t);
//	temp_cam.setYScale (m_YScale * t);

	setNear (m_Near += rhs->getNear ());
	setFar (m_Far += rhs->getFar ());

	m_IsChanged = true;

	return (vuCamera *) (this);
}

vuCamera* vuParallelCamera::set_equal_to_interp (vuCamera* cam1, vuCamera* cam2, float t1, float t2)

{
	vuParallelCamera *pcam1 = (vuParallelCamera *) (cam1);
	vuParallelCamera *pcam2 = (vuParallelCamera *) (cam2);

	m_Position = pcam1->getPosition () * t1;
	m_UpVector = pcam1->getUpVector () * t1;
	m_LookAtVector = pcam1->getLookAtVector () * t1;
	m_RightVector = pcam1->getRightVector () * t1;

	m_Width  = (int)(pcam1->m_Width *t1 + pcam2->m_Width*t2);
	m_Height = (int)(pcam1->m_Height*t1 + pcam2->m_Height*t2);

	m_XStep   = pcam1->m_XStep * t1;
	m_YStep   = pcam1->m_YStep * t1;

	m_XRange = pcam1->getXRange () * t1;
	m_YRange = pcam1->getYRange () * t1;
	m_Near = pcam1->getNear () * t1;
	m_Far = pcam1->getFar () * t1;

	m_Position += pcam2->getPosition () * t2;
	m_UpVector += pcam2->getUpVector () * t2;
	m_LookAtVector += pcam2->getLookAtVector () * t2;
	m_RightVector += pcam2->getRightVector () * t2;

	m_XRange += pcam2->getXRange () * t2;
	m_YRange += pcam2->getYRange () * t2;
	m_Near += pcam2->getNear () * t2;
	m_Far += pcam2->getFar () * t2;

	m_XStep   += pcam1->m_XStep * t2;
	m_YStep   += pcam1->m_YStep * t2;

	return (vuCamera *) (this);
}

vuParallelCamera* vuParallelCamera::operator= (vuParallelCamera *rhs)

{
	m_Position = rhs->getPosition ();
	m_UpVector = rhs->getUpVector ();
	m_LookAtVector = rhs->getLookAtVector ();
	m_RightVector = rhs->getRightVector ();

	m_Width   = rhs->getWidth ();
	m_Height  = rhs->getHeight ();
	m_XRange  = rhs->getXRange ();
	m_YRange  = rhs->getYRange ();
	m_Near    = rhs->getNear ();
	m_Far     = rhs->getFar ();
	m_XStep   = rhs->m_XStep;
	m_YStep   = rhs->m_YStep;

	m_IsChanged = true;

	return this;
}








