/** This file defines a basic camera for all of the other cameras
	in vuVolume to be derived from.

    The camera also depends on the width and height in pixels of the
    final image.

    Written by Steven Kilthau

    Modified Feb 2002
    By Chris Steinbach
    Modified to add the capabilities to record and restore states to the camera

    Modified Mar 2002
    By Chris Steinbach
    Modified to add the interpolation capabilities that will be needed by
    the Key Frame Animator class, also modifed to add the time storage
    capabilities as friends that will be needed by this.

    Also added the capabilities to create new cameras of the same type of
    this given nothing more than a pointer to the base class so that vuKeyFramer
    could work more easily.
*/

#ifndef _CAMERA_H_
#define _CAMERA_H_

#include <stdio.h>
//#include <stl.h>
#include <stream.h>
//#include <vector.h>

#include "vuDVector.h"
#include "vuHWTimer.h"
#include "vuSimpleTypes.h"
#include "vuVector.h"
#include "vuMisc/vuRay.h"

//!The camera class controls the parameters associated with a camera.
/*!This class also controls the OpenGL camera properties.
   The functionality of this basic camera can be extended by more
   specialized cameras like vuPerspectiveCamera. Other approaches
   like affine multiple centers of projection are also imaginable...

   This class also contains the functionality to record data and restore it later.
*/
class vuCamera
{
 public:

  enum vuCameraType {
    vuGENERIC_CAMERA,
    vuPARALLEL_CAMERA,
    vuPERSPECTIVE_CAMERA,
  };

  virtual vuCameraType getType(void) {
    return vuGENERIC_CAMERA;
  }

public:
    //!Default constructor.
    vuCamera();
    /**Copy constructor.

    	This will copy the state of the camera but will not preserve any recording or loaded file information
	for safety (ie, if c has a file loaded, the loaded file will not carry into the newly created vuCamera
	*/
    vuCamera(const vuCamera& c);
    //!Destructor
    virtual ~vuCamera();

    virtual void init(void) {};
    virtual vuRay getRay(float xpixel, float ypixel) { 
      return vuRay();
    };

    //!Assignment operator.
    virtual vuCamera& operator=(const vuCamera& rhs);

    //!Sets the position of the camera.
    void setPosition(const vuVector& pos);
    //!Returns the position of the camera.
    vuVector getPosition(void) const;
    //!Sets the camera's up vector.
    void setUpVector(const vuVector& up);
    //!Returns the camera's up vector.
    vuVector getUpVector(void) const;
    //!Sets the camera's look-at vector.
    void setLookAtVector(const vuVector& lookat);
    //!Returns the camera's look at vector.
    vuVector getLookAtVector(void) const;

    //!Sets the right vector
    void setRightVector (const vuVector &right);
    //!Returns the camera's right vector.
    vuVector getRightVector(void) const;

    //!Places a gluLookAt() function call for this camera.
    void gluLookAt(void);

    virtual void glInit(void) {}; //!< setup OpenGL camera settings
    /** Sets up the viewport for OpenGL rendering.
	This virtual function is implemented by the derived classes. */
    virtual void glViewport();

    int getWidth(void) const;       //!< Get the width (in pixels)
    void setWidth(const int width); //!< Set the width (in pixels)

    int getHeight(void) const;        //!< Get the height (in pixels)
    void setHeight(const int height); //!< Set the height (in pixels)


    //!Rotates the camera about the up vector.
    /*!Transforms always accur about the axes defining the local
      camera coordinate system.
    */
    void rotateAboutUp(float theta);
    //!Rotates the camera about the look-at vector.
    void rotateAboutLookAt(float theta);
    //!Rotates the camera about the right vector.
    void rotateAboutRight(float theta);
    //!Translates the camera by a position vector.
    void translateV(const vuVector& t);
    //!Translates the camera by a position.
    void translateXYZ(float x, float y, float z);
    //!Applies the given transformation matrix to the camera.
    void transform(const vuMatrix &m);
    //!Returns the viewing matrix
    vuMatrix getViewMat();

 public: // the following functions will be used to record basic camera information and should be used to store that
 		// data by the inherited classes.

    //!starts recording each position and direction that is rendered.
    /*! this will record to the file listed by record_to.

    	To properely use, you should run similar to as follows:

			camera.record (recordfile);
			while (takingpictures)

			{
				.
				.
				.
				camera.TakeSnapShot ();
				.
				.
				.
			}
			camera.stop_recording ();

	This is the proper usage for vuCamera and any camera derived from vuCamera (such as vuPerspectiveCamera and
	vuParallelCamera(.

    	and will store in the format:
		[BasicShot|[,AdvancedShot\n]]
		where basicShot is obtained from TakeSnapShotBasic ()
		and the advancedShot is obtained from the extra data in derived classes by functions in those derived classes.

	\return true if the recording was started successfully
	\return false if the recording was not successfully started

	NB - you may not start recording if you are in the middle of recording to any other file with this camera (ie, if you've
		successfully started recording and have not stop_recording (), then this will return false).
    */
    bool record (const char* record_to);

    //! saves the current state to Shot in the basic format.
    void TakeSnapShotBasic (char* Shot);

    /** This is overridden in derived classes so that the state of the current class is written to fp
    	in the approriate format.

	NB - refer to the comments associated with void vuCamera::record (char* record_to);
	*/
    virtual void TakeSnapShot ();

    /** This performs a TakeSnapShot except that it writes the shot to the char* Shot rather than the file that we are
    	currently recording to */
    virtual void TakeSnapShot (char* Shot);

    //! restores the camera to the state information in Shot
    int RestoreShotBasic (char* Shot);

    /** This will restore the next shot and return the position of the last character of the line
    */
    virtual int RestoreShot (char* Shot);

    /**	This will take the next line of the loaded file and set the current camera equal to that state

    	NB This must be overridden appropriately in any derived class!!!

    	\return length of shot 'consumed' if successful, < 0 otherwise
    	NB - You must call load () successfully before using this function
	NB - There IsNextStateAvailable must return true for this for to return success.
    */
    virtual int RestoreNextShot ();
    /**	This will take the previous line of the loaded file and set the current camera equal to that state

    	NB This must be overridden appropriately in any derived class!!!

    	\return length of shot 'consumed' if successful, < 0 otherwise

    	NB - You must call load () successfully before using this function
	NB - There IsNextStateAvailable must return true for this for to return success.
    */
    virtual int RestorePreviousShot ();
    /** stops recording and closes the file that the recording is saved to.

	NB - refer to the comments associated with void vuCamera::record (char* record_to);
	*/
    void stop_recording ();
    //! this will play from the file and return the time taken to play the file
    /*! this will play from the file listed by play_from and
    	will return the length of time taken to play the file.

	play_from will use the same format as record did to record it
	*/
//    float play (char* play_from);

    //! loads the file and returns success or failure
    /** loads the file so that RestoreNextShot will work.

    	The file must have the appropriate header (the string pointed to by get_id ()), otherwise, it will
	not be loaded.

	\return success or failure

	Each line is loaded line into lines (except for the header).

	NB - This function will fail if the header line of the file is not the apporpriate header for the type
		of camera that is being loaded into.

	The proper usage of this function is:

			camera.load (file_name);

			while (displaying pictures)

			{
				.
				.
				.
				camera.RestoreNextShot ();
				.
				.
				.
			}

			camera.clear_lines ();

	This proper usage will apply to any camera that is derived from vuCamera (such as vuPerspectiveCamera, vuParallelCamera).
*/
    bool load (char* load_from);

    /** this will remove every saved camera position from the buffer

    	Please refer to bool vuCamera::load for greater details about the buffer */
    void clear_lines ();

    /** this will return true if the first n characterds of id = get_id ()

    	NB - n = strlen get_id
	ie - this verifies that the header is the correct header for the type of camera that is being loaded */
    virtual bool verify_id (char* id);
    /** this will return a pointer to a string containing the appropriate header for the type of camera that this is

    	This is implemented to allow, say, a perspective camera to reject the notion of loading data that should be loaded by
	a parallel camera.  Type is important here.
	*/
    virtual char* get_id ();

    	//! This will set cam equal to the state stored in out.
    friend ostream& operator<<(ostream& out, vuCamera &cam);

    	//! This will set in to be the state that cam is in.
    friend istream& operator>>(istream& in, vuCamera &cam);

    	/** This will return whether or not there are any more states available in the buffer of the camera

		\return true if there are more states to be loaded in the buffer
		\return false if there are no more states in the buffer to be loaded

    	Please refer to bool vuCamera::load for greater details about the buffer */
    bool IsNextAvailable ();

    	/** This will return whether or not there are any more states available in the buffer of the camera

		\return true if there are previous states in the buffer
		\return false if there are no previous states in the buffer

	Please refer to bool vuCamera::load for greater details about the buffer */
    bool IsPreviousAvailable ();

    	/** This will return true if this camera is recording to a file, and this will
		return false otherwise. */
    bool IsRecording ();

    /** This function will load the cameras stored in the file "fname" and will
    	set vect to be the array of cameras in fname

	This will work if you recast a derived camera as a base class camera.*/
    virtual bool load_cameras (vuDVector <vuCamera> &vect, char *fname);

    /** This will open the file "record_to" for recording (cam is needed so that
        it can give the correct header information to this function),

	 This will return true if successful and false otherwise.

	 This works properely for all cameras properely derived from this class

	 This will work if you recast a derived camera as a base class camera.*/
    friend bool record_with_time (char* record_to, vuCamera &cam);

    /** This will stop recording to the file opened by record_with_time (...)
         and save the file.

	 This will return true if successful, and false otherwise.

	 This works properely for all cameras properely derived from this class.

	 This will work if you recast a derived camera as a base class camera.*/
    friend bool stop_recording (vuCamera &cam);

    /** This will take a SnapShot of the camera and record it to the file opened
    	with the function record_with_time (...) and willalso record the time.

	The format is time, path\n ...

	This will return true if successful, and false otherwise.

	This works correctly for all cameras properely derived from this
	class (ie, it'lkl record the correct data for a perspective camera
	passed into this recast as a vuCamera... ).

	This will work if you recast a derived camera as a base class camera.*/
    friend bool TakeSnapShotWithTime (float time, vuCamera &cam);

    /** This will load the frames with their times from the file "load from" into
    	the cameras in the vector cameras (it will first allocate those cameras from
	the create_new function of the camera cam, and then set those cameras equal
	to the proper camera).

	This returns true if successful and false if it fails to load.

    	This will work if you recast a derived camera as a base class camera. */
    friend bool load_WithTime (char* load_from, vuCamera *cam, vuDVector <vuCamera*> &cameras, vuDVector <float> &times);

 public:

    /** This returns an identifier that must be incorporated into the header file
    	of a timed recording if this recording also contains time values

	(ie, if times are associated with each camera, then this string will be in the
	file). */
    char* getWithTime_id ();

    /** This will verify that cmp is equal to the WithTime Identifier.

    	This will return whether or not the two are equal (true if they are, and false
	otherwise). */
    bool verify_time_id (char* cmp);

    /** returns a pointer to a new camera, of the same type as this.  This
    	must be overridden in all derived classes for the derived cameras to behave
	correctly. */
    virtual vuCamera* create_new ();

    virtual vuCamera* set_equal_to_interp (vuCamera* cam1, vuCamera* cam2, float t1, float t2);

//    void delete_me ();
		/* delete_me is not needed so long as you keep all
			of your destructors as virtual destructors
			and then delete the ptr */

  public:  // operators,
           // these are provided in a fashion so as to make it easy for the Key framer
	   // to use these operators without knowing what time of camera that it is
	   // working with (so that it can use the same code with all typea of
	   // cameras that are properly derived from this camera).

    /** Multiplication Operator,

        This will return the camera obtained by multiplying each component of
    	this camera with t

	It will create that camera, so you must delete it later.*/
    virtual vuCamera* operator* (float t);

    /** Addition/Assignment Operator,

        This will multiply each component of this camera by t and will return this
    	camera upon completion */
    virtual vuCamera* operator*= (float t);

    /** Addition Operator,

        This will perform a component wise addition of this and rhs and will return
    	a pointer to the result.

	 Note that this will create a new vuCamera, so you will have to deallocate it
	 later.*/
    virtual vuCamera* operator+ (vuCamera &rhs);

    /** Addition/Assignment operator,

        This will add each component of rhs to each component of this and return a pointer
    	to this camera...
    */
    virtual vuCamera* operator+= (vuCamera &rhs);

    /** Addition operator,

        This will add this and rhs (componentwise).

    	It will create the camera that it returns, you must deallocate this later. */
    virtual vuCamera* operator+ (vuCamera *rhs);

    /** Addition/Assignment operator,

        this will perform a component wise addition on this camera and the camera
    	pointed to by rhs, and then will store the result in this camera and return a
	pointer to this camera. */
    virtual vuCamera* operator+= (vuCamera *rhs);

    /** Assignment Operator,

        This will set this camera equal to the camera pointed to by rhs */
    virtual vuCamera* operator= (vuCamera *rhs);

    /** Multiplication Operator,

        This will create a new camera, and return a pointer to that camera.

    	It will then multiply each component of cam by t and return the a pointer
	to the camera storing the result of this multiplication. */
    friend vuCamera* operator*(float t, vuCamera &cam);

 public:

    /** This will return true if the camera has had any data changed since the last
    	call to this function, and false if no changes have been made to the camera
	since the last time that this camera was called. */
    bool IsChanged ();	// every function, every time...
    			// including the constructor

    /** This will restore the camera to the default state */
    virtual void set_defaults ();

 protected:
 	//! the position that the camera is located at
    vuVector m_Position;
    	//! the direction that 'up' is in for the screen
    vuVector m_UpVector;
    	//! the direction that the camera is pointed in
    vuVector m_LookAtVector;
    	//! the direction of 'right' for the camera to the screen
    vuVector m_RightVector;

    int m_Width;      //!< Width  in pixels
    int m_Height;     //!< Height in pixels

    bool m_IsChanged;	/**< This is true if the camera has been changed since the
    				last call to IsChanged (), and is false otherwise. */

 public:

    int m_IsRecording; //!< this is true if recording, false otherwise

    FILE *fp; //!< this is the file that is being recorded to

    int line_number; //!< this is the current line number (from [0..lines.Size ()])

//    vector <char*> lines; //!< these are the lines from the camera script
    vuDVector <char*> lines; //!< these are the lines from the camera script
};

/** this will return the position of the next '(' in the line

This is provided to make parsing the camera files easier */
int get_next_open (char* line)
;
/** this will return the position of the next ')' in the line

This is provided to make parsing the camera files easier */
int get_next_close (char* line);

/** This will imterpolate between cam1 and cam2 given that cam1 is at time t0, cam2
is at time tf, and the current time is t.

It will return a pointer to cam1, where the result will be stored.

This will work for all cameras properely derived from vuCamera */
vuCamera* interpolate (vuCamera *cam1, vuCamera *cam2, float t, float t0, float tf);
#endif




