#ifdef WIN32
#include <windows.h>
#endif

#include <GL/gl.h>
#include <stdlib.h>
#include <GL/glu.h>
#include <math.h>
#include "vuCamera.h"
#include "vuMatrix.h"

#define PI_OVER_180 0.01745329251994

vuCamera::vuCamera() : m_Position(0.0, 0.0, 256.0),
                   m_UpVector(0.0, 1.0, 0.0),
                   m_LookAtVector(0.0, 0.0, -1.0),
                   m_RightVector(1.0, 0.0, 0.0),
		   m_Width(320),
		   m_Height(240),
		   m_IsChanged(true),
		   m_IsRecording (0),
		   fp (NULL),
		   line_number (0)
{
}

void vuCamera::set_defaults ()

{
	vuVector m_Pos (0.0, 0.0, 256.0);
	m_Position = m_Pos;
	vuVector m_Up (0.0, 1.0, 0.0);
	m_UpVector = m_Up;
	vuVector m_Loo (0.0, 0.0, -1.0);
	m_LookAtVector = m_Loo;
	vuVector m_Rig (1.0, 0.0, 0.0);
	m_RightVector = m_Rig;

	m_IsChanged = true;
}

vuCamera::vuCamera(const vuCamera& c)
{
    m_Position      = c.m_Position;
    m_UpVector      = c.m_UpVector;
    m_LookAtVector  = c.m_LookAtVector;
    m_RightVector   = c.m_RightVector;

    m_Width         = c.m_Width;
    m_Height        = c.m_Height;

    fp = NULL;
    line_number = 0;

    m_IsRecording = 0;

    m_IsChanged = true;
}

vuCamera::~vuCamera()
{
	clear_lines ();
}

vuCamera& vuCamera::operator=(const vuCamera& rhs)
{
    if (this != &rhs)
    {
        m_Position      = rhs.m_Position;
        m_UpVector      = rhs.m_UpVector;
        m_LookAtVector  = rhs.m_LookAtVector;
        m_RightVector   = rhs.m_RightVector;
	m_Width         = rhs.m_Width;
	m_Height        = rhs.m_Height;

	m_IsRecording = 0;
    }

    m_IsChanged = true;

    return *this;
}

void vuCamera::setPosition(const vuVector& pos)
{
    m_Position = pos;

    m_IsChanged = true;
}

vuVector vuCamera::getPosition(void) const
{
    return m_Position;
}

void vuCamera::setUpVector(const vuVector& up)
{
    m_UpVector = up;
    m_UpVector.makeUnit();
    m_RightVector = m_LookAtVector.cross(m_UpVector);

    m_IsChanged = true;
}

vuVector vuCamera::getUpVector(void) const
{
    return m_UpVector;
}

void vuCamera::setLookAtVector(const vuVector& lookat)
{
    m_LookAtVector = lookat;
    m_LookAtVector.makeUnit();
    m_RightVector = m_LookAtVector.cross(m_UpVector);

    m_IsChanged = true;
}

vuVector vuCamera::getLookAtVector(void) const
{
    return m_LookAtVector;
}

vuVector vuCamera::getRightVector(void) const
{
    return m_RightVector;
}

void vuCamera::setRightVector (const vuVector &right)
{
  m_RightVector = right;
  m_RightVector.makeUnit ();
}

// Set up OpenGL viewport using width and height.
void vuCamera::gluLookAt(void)
{
    vuVector reference = m_Position + m_LookAtVector;
    ::gluLookAt(m_Position[0], m_Position[1], m_Position[2],
                reference[0],  reference[1],  reference[2], 
                m_UpVector[0], m_UpVector[1], m_UpVector[2]);
}

void vuCamera::glViewport()
{
    ::glViewport(0, 0, m_Width, m_Height);
}

// Get the pixel width
int vuCamera::getWidth(void) const
{
    return m_Width;
}

// Set the pixel width
void vuCamera::setWidth(int width)
{
    m_Width = (width < 0) ? -width : width;

    m_IsChanged = true;
}

// Get the pixel height
int vuCamera::getHeight(void) const
{
    return m_Height;
}

// Set the pixel height
void vuCamera::setHeight(int height)
{
    m_Height = (height < 0) ? -height : height;

    m_IsChanged = true;
}


void vuCamera::rotateAboutUp(float theta)
{
    vuMatrix rot;

    rot.makeRotate(m_UpVector, theta);

    transform(rot);
    /*
    m_LookAtVector = rot*m_LookAtVector;
    m_RightVector = rot*m_RightVector;

    m_LookAtVector.makeUnit();
    m_RightVector.makeUnit();*/
}

void vuCamera::rotateAboutLookAt(float theta)
{
    vuMatrix rot;

    rot.makeRotate(m_LookAtVector, theta);

    transform(rot);
/*    m_UpVector = rot*m_UpVector;
    m_RightVector = rot*m_RightVector;

    m_UpVector.makeUnit();
    m_RightVector.makeUnit();
*/}

void vuCamera::rotateAboutRight(float theta)
{
    vuMatrix rot;

    rot.makeRotate(m_RightVector, theta);

    transform(rot);

/*    m_UpVector = rot*m_UpVector;
    m_LookAtVector = rot*m_LookAtVector;

    m_UpVector.makeUnit();
    m_LookAtVector.makeUnit();
*/
}

void vuCamera::translateV(const vuVector& t)
{
    if (t[0])
        m_Position += m_RightVector*t[0];
    if (t[1])
        m_Position += m_UpVector*t[1];
    if (t[2])
        m_Position += m_LookAtVector*t[2];

    m_IsChanged = true;
}

void vuCamera::translateXYZ(float x, float y, float z)
{
    if (x)
        m_Position += m_RightVector*x;
    if (y)
        m_Position += m_UpVector*y;
    if (z)
        m_Position += m_LookAtVector*z;

    m_IsChanged = true;
}

void vuCamera::transform(const vuMatrix &m)
{
    m_UpVector = m*m_UpVector;
    m_LookAtVector = m*m_LookAtVector;
    m_RightVector = m*m_RightVector;

    m_UpVector.makeUnit();
    m_LookAtVector.makeUnit();
    m_RightVector.makeUnit();

    m_IsChanged = true;
}

vuMatrix vuCamera::getViewMat()
{
  vuMatrix cam;
  cam[3][3] = 1;
  memcpy(cam[0],m_RightVector.getData(),sizeof(float)*3);
  memcpy(cam[1],m_UpVector.getData(),sizeof(float)*3);
  memcpy(cam[2],m_LookAtVector.getData(),sizeof(float)*3);

  return cam;
}


bool vuCamera::record (const char* record_to)

{
	if (fp != NULL)
		return false;

	fp = fopen (record_to, "w");

	fputs (get_id (), fp);
	fputc ('\n', fp);

	m_IsRecording = 1;

	return true;
}

void vuCamera::TakeSnapShotBasic (char* Shot)

{
	if (m_IsRecording)

	{
		char temp [100];

		strcpy (Shot, "(\0");
		m_Position.save (temp);
		strcat (Shot, temp);
		strcat (Shot, "), (");

		m_UpVector.save (temp);
		strcat (Shot, temp);
		strcat (Shot, "), (");

		m_LookAtVector.save (temp);
		strcat (Shot, temp);
		strcat (Shot, "), (");

		m_RightVector.save (temp);
		strcat (Shot, temp);
		strcat (Shot, ") \0");
	}
}

void vuCamera::TakeSnapShot ()

{
	char Shot [1000];

	if (m_IsRecording)

	{
		TakeSnapShotBasic (Shot);

		strcat (Shot, "\n");

		fputs (Shot, fp);
	}
}

void vuCamera::TakeSnapShot (char* Shot)

{
	TakeSnapShotBasic (Shot);
}

int get_next_open (char* line)

{
	int position = 0;

	while ((line [position] != '(') && (line [position] != '\0'))
		position++;

	return position;
}

int get_next_close (char* line)

{
	int position = 0;

	while ((line [position] != ')') && (line [position] != '\0'))
		position++;

	return position;
}

int vuCamera::RestoreShotBasic (char* Shot)

{
	int position = -1;

	if (Shot [0] != '\0');

	{

		position = get_next_open (Shot) + 1;
		m_Position.load (&(Shot [position]));
		position += get_next_close (&(Shot [position])) + 1;

		position += get_next_open (&(Shot [position])) + 1;
		m_UpVector.load (&(Shot [position]));
		position += get_next_close (&(Shot [position])) + 1;

		position += get_next_open (&(Shot [position])) + 1;
		m_LookAtVector.load (&(Shot [position]));
		position += get_next_close (&(Shot [position])) + 1;

		position += get_next_open (&(Shot [position])) + 1;
		m_RightVector.load (&(Shot [position]));
		position += get_next_close (&(Shot [position])) + 1;
	}

	m_IsChanged = true;

	return position;
}

int vuCamera::RestoreShot (char* Shot)

{
	return RestoreShotBasic (Shot);
}

int vuCamera::RestoreNextShot ()

{
//	if (++line_number < int (lines.size ()))

	if (IsNextAvailable ())
		return RestoreShotBasic (lines [++line_number]);

	return -1;
}

int vuCamera::RestorePreviousShot ()

{
	if (IsPreviousAvailable ())
		return RestoreShotBasic (lines [--line_number]);

	return -1;
}

void vuCamera::stop_recording ()

{
	fclose (fp);

	m_IsRecording = 0;

	fp = NULL;
}
/*
float vuCamera::play (char* play_from)

{
	fp = fopen (play_from, "r");

	vuHWTimer timer;

	timer.getElapsedTimeFloat ();

	char temp [256];

	temp [0] = 'a';

	while (temp [0] != '\0')

	{
		fgets (temp, 256, fp);

		if (temp [0] != '\0');

		{
			int position = m_Position.load (temp);
			position += m_UpVector.load (&(temp [position]));
			position += m_LookAtVector.load (&(temp [position]));
			position += m_RightVector.load (&(temp [position]));

			gluLookAt ();
		}
	}

	fclose (fp);
	return timer.getElapsedTimeFloat ();
}*/

bool vuCamera::load (char* load_from)

{
	clear_lines ();

	char temp [1000];
	char* hithere;

	fp = fopen (load_from, "r");

	line_number = -1;

	if (fp == NULL)
		return false;

	fgets (temp, 100, fp);

	if (!(this->verify_id (temp)))
		return false;

	char* not_NULL = temp;

	while (not_NULL)

	{
		not_NULL = fgets (temp, 1000, fp);

		hithere = new char [strlen (temp) + 5];
		strcpy (hithere, temp);

		if (temp [0] != '\0')
			lines.add (hithere);
//			lines.push_back (hithere);
	}

	fclose (fp);

	fp = NULL;

	return true;
}

void vuCamera::clear_lines ()

{
	char *temp;

	for (unsigned int i = 0; i < lines.getLength (); i++)

	{
		temp = lines [i];
		delete [] temp;
	}

	lines.removeRange (0, lines.getLength () - 1);
/*	while (lines.size () > 0)

	{
		char* temp = lines [lines.size () - 1];
		lines.pop_back ();

		delete [] temp;
	}*/
}

bool vuCamera::verify_id (char* id)

{
	return (strncmp (get_id (), id, strlen (get_id ())) == 0);
}

// d refers to the data format (ie, if something changes, then  new format should be used so that you
// don't try to copy the old format into the new...
char id_string_cgs_xq1 [] = "vuBasicCameraD1.00___";

char* vuCamera::get_id ()

{
	return id_string_cgs_xq1;
}

ostream& operator<<(ostream& out, vuCamera &cam)

{
	char Shot [1024];

	cam.TakeSnapShot (Shot);

	out << Shot;

	cam.m_IsChanged = true;

	return out;
}

istream& operator>>(istream& in, vuCamera &cam)

{
	char Shot [1024];

	in >> Shot;

	cam.RestoreShot (Shot);

	return in;
}

bool vuCamera::IsNextAvailable ()

{
	return ((int (lines.getLength ()) > 0) && (line_number < int (lines.getLength () - 1)) && (line_number > -2));
}

bool vuCamera::IsPreviousAvailable ()

{
	return ((int (lines.getLength ()) > 0) && (line_number > 0) && (line_number < int (lines.getLength ()) + 2));
}

bool vuCamera::IsRecording ()

{
	return m_IsRecording;
}

bool vuCamera::load_cameras (vuDVector <vuCamera> &vect, char *fname)

{
	vuCamera cam;

	if (vect.getLength () > 0)
		vect.removeRange (dword (0), vect.getLength () - 1);

	cam.clear_lines ();

	if (!cam.load (fname))
		return false;

	while (cam.IsNextAvailable ())

	{
		vect.add (cam);

		cam.RestoreNextShot ();
	}

	vect.add (cam);

	return true;
}

bool record_with_time (char* record_to, vuCamera &cam)

{
	bool ret_val = cam.record (record_to);

	if (ret_val)

	{
		fputs (cam.getWithTime_id (), cam.fp);
		fputc ('\n', cam.fp);
	}

	return ret_val;
}

bool stop_recording (vuCamera &cam)

{
	cam.stop_recording ();

	return true;
}

bool TakeSnapShotWithTime (float time, vuCamera &cam)

{
	if (!cam.m_IsRecording)
		return false;

	char time_ascii [14];

	gcvt (time, 10, time_ascii);

	fputs (time_ascii, cam.fp);

	fputs (", ", cam.fp);

	cam.TakeSnapShot ();

	return true;
}

bool load_WithTime (char* load_from, vuCamera *cam, vuDVector <vuCamera*> &cameras, vuDVector <float> &times)

{
	cam->clear_lines ();

	char* temp;
	char* hithere;

	temp = new char [1000];

	times.removeRange (0, times.getLength () - 1);
	cameras.removeRange (0, cameras.getLength () - 1);

	cam->fp = fopen (load_from, "r");

	cam->line_number = -1;

	if (cam->fp == NULL)
		return false;

	fgets (temp, 100, cam->fp);

	if (!(cam->verify_id (temp)))
		return false;

	fgets (temp, 100, cam->fp);

	if (!(cam->verify_time_id (temp)))
		return false;

	char* not_NULL = temp;

	int position;
	float current_time;

	vuCamera *t_camera;

	while (not_NULL)

	{
		not_NULL = fgets (temp, 1000, cam->fp);

		position = clear_blanks (temp);

		current_time = atof (&(temp [position]));
		position += get_next_comma (&(temp [position])) + 1;
		position += clear_blanks (&(temp [position]));

		hithere = new char [strlen (&(temp [position])) + 5];
		strcpy (hithere, &(temp [position]));

		if (temp [0] != '\0')
			cam->lines.add (hithere);

		t_camera = cam->create_new ();
		t_camera->RestoreShot (hithere);
		times.add (current_time);

		cameras.add (t_camera);
	}

	fclose (cam->fp);

	cam->fp = NULL;

	return true;
}

char id_string_cgs_xq1_2 [] = "vuCamera_WithTime_id_1.00__";

char* vuCamera::getWithTime_id ()

{
	return id_string_cgs_xq1_2;
}

bool vuCamera::verify_time_id (char* cmp)

{
	return (strncmp (getWithTime_id (), cmp, strlen (getWithTime_id ())) == 0);
}

vuCamera* vuCamera::create_new ()

{
	vuCamera* ptr = new vuCamera [1];

	return ptr;
}
/*
void vuCamera::delete_me ()

{
	delete [] this;
}*/

vuCamera* vuCamera::operator* (float t)

{
	vuCamera *temp_cam_vc_cgs__ = create_new ();

	temp_cam_vc_cgs__->setPosition (m_Position * t);
	temp_cam_vc_cgs__->setUpVector (m_UpVector * t);
	temp_cam_vc_cgs__->setLookAtVector (m_LookAtVector * t);
//	temp_cam.setRightVector (m_RightVector * t);

	m_IsChanged = true;

	return temp_cam_vc_cgs__;
}

vuCamera* vuCamera::operator*= (float t)

{
	cout << "t:" << t << endl;

	setPosition (m_Position *= t);
	setUpVector (m_UpVector *= t);
	setLookAtVector (m_LookAtVector *= t);
//	temp_cam.setRightVector (m_RightVector *= t);

	m_IsChanged = true;

	return this;
}

vuCamera* operator* (float t, vuCamera &cam)

{
	vuCamera *temp_cam_vc_cgs__ = cam.create_new ();

	temp_cam_vc_cgs__->setPosition (cam.m_Position * t);
	temp_cam_vc_cgs__->setUpVector (cam.m_UpVector * t);
	temp_cam_vc_cgs__->setLookAtVector (cam.m_LookAtVector * t);
//	temp_cam.setRightVector (cam.m_RightVector * t);

	cam.m_IsChanged = true;

	return temp_cam_vc_cgs__;
}

vuCamera* vuCamera::operator+ (vuCamera &rhs)

{
	vuCamera *temp_cam_vc_cgs__ = create_new ();

	temp_cam_vc_cgs__->setPosition (m_Position + rhs.getPosition ());
	temp_cam_vc_cgs__->setUpVector (m_UpVector + rhs.getUpVector ());
	temp_cam_vc_cgs__->setLookAtVector (m_LookAtVector + rhs.getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	return temp_cam_vc_cgs__;
}

vuCamera* vuCamera::operator+= (vuCamera &rhs)

{
	setPosition (m_Position += rhs.getPosition ());
	setUpVector (m_UpVector += rhs.getUpVector ());
	setLookAtVector (m_LookAtVector += rhs.getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	m_IsChanged = true;

	return this;
}

vuCamera* vuCamera::operator+ (vuCamera *rhs)

{
	vuCamera *temp_cam_vc_cgs__ = create_new ();

	temp_cam_vc_cgs__->setPosition (m_Position + rhs->getPosition ());
	temp_cam_vc_cgs__->setUpVector (m_UpVector + rhs->getUpVector ());
	temp_cam_vc_cgs__->setLookAtVector (m_LookAtVector + rhs->getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	return temp_cam_vc_cgs__;
}

vuCamera* vuCamera::operator+= (vuCamera *rhs)

{
	setPosition (m_Position += rhs->getPosition ());
	setUpVector (m_UpVector += rhs->getUpVector ());
	setLookAtVector (m_LookAtVector += rhs->getLookAtVector ());
//	temp_cam.setRightVector (m_RightVector * t);

	m_IsChanged = true;

	return this;
}

vuCamera* vuCamera::operator= (vuCamera *rhs)

{
	m_Position = rhs->getPosition ();
	m_UpVector = rhs->getUpVector ();
	m_LookAtVector = rhs->getLookAtVector ();
	m_RightVector = rhs->getRightVector ();

	m_IsChanged = true;

	return this;
}

vuCamera* vuCamera::set_equal_to_interp (vuCamera* cam1, vuCamera* cam2, float t1, float t2)

{
	m_Position = cam1->getPosition () * t1;
	m_UpVector = cam1->getUpVector () * t1;
	m_LookAtVector = cam1->getLookAtVector () * t1;
	m_RightVector = cam1->getRightVector () * t1;

	m_Position += cam2->getPosition () * t2;
	m_UpVector += cam2->getUpVector () * t2;
	m_LookAtVector += cam2->getLookAtVector () * t2;
	m_RightVector += cam2->getRightVector () * t2;

	return this;
}

vuCamera* interpolate (vuCamera *cam1, vuCamera *cam2, float t, float t0, float tf)

	// Note that this will also extrapolate the data
	// because it might be of some use...

{
	float t1, t2;

	cout << "interpolate" << t << ',' << t0 << ',' << tf << endl;

	t2 = t - t0;
	t1 = tf - t;
	t1 /= (tf - t0);
	t2 /= (tf - t0);

	vuCamera* ncam1 = cam1->create_new ();

	ncam1->set_equal_to_interp (cam1, cam2, t1, t2);

//	*cam1 *= t1;
//	*cam2 *= t2;
//	*cam1 += cam2;

	return ncam1;
//	*m_camera = cam2;

}

bool vuCamera::IsChanged ()

{
	bool temp = m_IsChanged;
	m_IsChanged = false;

	return temp;
}

/*vuVector& vuVector::operator<<(const char* rhs)

{
	RestoreShot (rhs);
}*/

/*
int clear_blanks (char* line)
`
{
	int position = 0;

	while ((line [position] != ',') && (line [position] != '\0'))
		position++;

	return position;
}

int get_next_comma (char* line)

{
	int position = 0;

	while ((line [position] != ',') && (line [position] != '\0'))
		position++;

	return position;
}


*/




