//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.
//

#ifndef	VECTOR_HH
#define VECTOR_HH

#undef max
#undef min

//====================================================
// Vector class template library for basic algebraic operations
//
//  H. Lydia Deng, 01/24/1994
//	
//====================================================
// .NAME Vector class template
// .LIBRARY Base
// .HEADER c++ uitility classes
// .FOOTER CWP
// .INCLUDE defs.hh
// .FILE Vector.hh

// .SECTION Description
//  Vector<Type>
//   is a simple template class for Vectors. It manages 
//   to do algebraic operations of one-dimensional arrays
// 
// .SECTION Description
// Constructors:
//   Vector(): default constructor of length 0;
//   Vector(n): constructor of a vector with length n;
//   Vector(v): construct a new Vector as a copy of Vector v;
//   Vector(n, Type* p): construct a Vector with n elements 
//     of the array *p;
// Fetching the Vector information:
//   int size() const: returns size of the Vector;
//   Type v[i] const: returns the ith element as a constant;
//   Type max(): returns the largest elements of the Vector;
//   Type min(): returns the smallest elements of the Vector;
//   int indexMax(): returns the index of the largest elements;
//   int indexMin(): returns the index of the smallest elements;
//   Type sum(): returns the sum of all elements;
//   Type norm(p): returns the p norm of the Vector;
//   Type norm2S(): returns the square sum of the Vector;
//   Type* toPointer(): copy the Vector to an array, 
//     returns the pointer;
//   Vector<Type>* copy(i, n): copy to a Vector of length n 
//     starting from the ith element, returns a pointer to 
//     the new Vector;
//   Vector<Type>* copy(i): copy to a new Vector from the ith 
//     elemement to the last one, returns a pointer to the 
//     new Vector;
// Modifying the Vector;
//   Type& v[i]:  access the ith elemet of the writable Vector; 
//   Vector<Type>& linear(a, b): assign linear values to the
//     Vector, i.e. the ith element has the value of a+b*i;
//   Vector<Type>& chaSize(nlen): change the size to nlen;
//   Vector<Type>& addVal(a): append value a to the end;
//   Vector<Type>& addVal(v): append Vector v to the end;
//   Vector<Type>& addVal(a, i): insert a to the ith element;
//   Vector<Type>& normalize(): normalize the Vector;
// Arithmatics:
//   Operators -, =, +=, -=, *= /=, +, -, *, /,
//     arithmatics with other Vectors, Arrays, a single number;
//   Vector<Type> saxpy(a, u, v): u, v are Vectors of the same 
//     length and Type, returns a Vector a*u+v; 
// I/O operators: <<, >>
//
// .SECTION Caveats
// The implicit type conversion is not clean. A lot of warnings if
// compiled under gcc-2.7.0 and above.
// Other derived vectors should be created later.

#ifdef __GNUC__
#pragma interface
#endif

#include "defs.hh"
#include <stdio.h>
#include <iostream>

//*****************************************************************************
//definition of the Vector class and its member functions
//
//  H. Lydia Deng, 1/22/94 
//	add addVal, modify chaSize for Matrix class
//  H. Lydia Deng, 1/22/94 
//	changing a lot!
//  H. Lydia deng, 10/8/94
//****************************************************************************/
//@Man:
//@Memo:  a base and generic vector
/*@Doc: This class tends to be as generic as possible */

namespace coool
{

template <class Type>
class Vector {

  protected:
    int n;
    Type *a;

  public:
    //@ManMemo: Default constructor with length 0
    Vector();
    //@ManMemo:  constructor a vector of length $m$
    Vector(int m);
    //@ManMemo:  constructor a vector of the same as $v$
    Vector(const Vector<Type>& v);
    //@ManMemo:  constructor a vector of length $m$ with elements the same as first $m$ elements of arrays x
    Vector(int m,  Type* x);
    //@ManMemo:  constructor a vector of length $l$ with elements the same as first $m$ elements of arrays x,  the rest have the value of $x[m-1]$
    Vector(int l, int m,  Type* x);
    ~Vector();

    //@ManMemo: returns size of the vector
    int size() const;
    //@ManMemo: acess the ith element, non-writable
    Type operator[](int i) const; 
    //@ManMemo: acess the ith element, writable
    Type& operator[](int i);
    //@ManMemo: reset the vector to a different size
    Vector<Type>& resetSize(int newsize);
    

    //@ManMemo:  returns the conent as an array
    Type* toPointer() const;
    //@ManMemo:  assigns conents of array p to the vector
    Vector<Type>& setPointer(Type* p);
    //@ManMemo:  change size of the vector to $m$
    Vector<Type>& chaSize(int m);
    //@ManMemo:  assigns a linear related contents, $a[i] = iv + s*i$
    Vector<Type>&  linear(Type iv, Type s);

    //@ManMemo: simple operations
    int indexMax() const;
    //@ManMemo:  
    int indexMin() const;
    //@ManMemo:  
    Type max() const;
    //@ManMemo:  
    Type min() const;

    //@ManMemo:  
    Type sum() const;
    //@ManMemo:  
    Type norm(int p) const;
    //@ManMemo:  
    Type norm2S()const ;

    //@ManMemo:  
    Vector<Type>& normalize();
    //@ManMemo:  negate all components
    Vector<Type> operator-();
    //@ManMemo:  absolute value of all components
    Vector<Type> abs();

    //@ManMemo:overlodaing = operator, equating the vector to v
    Vector<Type>& operator=(const Vector<Type>& v);
//    //@ManMemo:overlodaing = operator, assigning m elements to the Vector
//    Vector<Type>& operator=(int m, const Vector<Type>& v);
    //@ManMemo:overloading = operator, assigning a constant c to the Vector
    Vector<Type>& operator=(Type c);
    //@ManMemo:overload = operator, assigning an array to the Vector
    Vector<Type>& operator=(Type *p);
    
    //@ManMemo:overlodaing += operator
    Vector<Type>& operator+=(const Vector<Type>& v);
    //@ManMemo:  
    Vector<Type>& operator+=(Type c);
    //@ManMemo:  
    Vector<Type>& operator+=(Type *p);
    
    //@ManMemo: overloading -= operator
    Vector<Type>& operator-=(const Vector<Type>& v);
    //@ManMemo:  
    Vector<Type>& operator-=(Type c);
    //@ManMemo:  
    Vector<Type>& operator-=(Type *p);
    
    //@ManMemo: overloading *= /= operators
    Vector<Type>& operator*=(Type c);
    //@ManMemo:  
    Vector<Type>& operator/=(Type c);

    //@ManMemo:joining elements to the vector
    Vector<Type>& addVal(Type);
    //@ManMemo:  
    Vector<Type>& addVal(const Vector<Type>&);
    //@ManMemo:  
    Vector<Type>& addVal(Type, int);
    //@ManMemo: delete the $j$th element of the Vector
    Vector<Type>& delVal(int j);
    
    operator Vector<int>() const
    {
       Vector<int> v(n); 
       for(int i=0; i<n; i++) v[i] = (int)a[i]; 
       return v;
    }
    operator Vector<long>() const
    {
       Vector<long> v(n); 
       for(int i=0; i<n; i++) v[i] = (long)a[i]; 
       return v;
    }
    operator Vector<float>() const
    {
       Vector<float> v(n); 
       for(int i=0; i<n; i++) v[i] = (float)a[i]; 
       return v;
    }
    operator Vector<double>() const
    {
       Vector<double> v(n); 
       for(int i=0; i<n; i++) v[i] = (double)a[i]; 
       return v;
    }

    Vector<Type>& constrain(const Vector<Type>& lb, const Vector<Type>& ub)
    {
	if(lb.size() == ub.size() && lb.size() == size()) {
	    for(int i=0; i<n; i++) {
		if(a[i] < lb.a[i])
		    a[i] = lb.a[i];
		else if(a[i] > ub.a[i])
		    a[i] = ub.a[i];
	    }
	    return *this;
	} else _THROWERROR("Vector<>::constrain dimensions don't agree.");
    }
    
    //@ManMemo:  
    size_t bfread(FILE *ifp)
    {return fread(a, sizeof(Type), n, ifp);}

    //@ManMemo:  
    size_t bfwrite(FILE *ofp);

    //@ManMemo: additional methods
    Type CircElem(int i) const;
    //@ManMemo:  
    Vector<Type>* copy(int);
    //@ManMemo:  
    Vector<Type>* copy(int, int);
    //@ManMemo:  
    int in(Type c) const;   // H.L. Deng, 07/25/95

    //@ManMemo:overloading insertion and extraction operators
template < class T >
    friend ostream& operator<< (ostream& ofp, const Vector<T>& v);

//@ManMemo:  
template < class T >
    friend istream& operator>> (istream& ifp,  Vector<T>& v);

//@ManMemo: comparing a vector with a number, ==
template < class T >
friend int operator== (const Vector<T>& u, T c);
    //@ManMemo: comparing a vector with a number, !=
template < class T >
friend int operator!= (const Vector<T>& u, T c);
    //@ManMemo: comparing two vectors, ==
template < class T >
friend int operator== (const Vector<T>& u, const Vector<T>& c);
    //@ManMemo: comparing two vectors, !=
template < class T >
friend int operator!= (const Vector<T>& u, const Vector<T>& c);

};

#ifdef __GNUC__
#pragma implementation
#endif

		//Constructors and Destructor, the default length is 0
template<class Type>
Vector<Type>::Vector()
: n(0), a(0) { };
 
template<class Type>
Vector<Type>::Vector(int m)
{
    
   n=m; 
    
   a=new Type[n]; 
    
   for(int i=0;   i<n;   i++) 
      a[i] = 0;
}
 
template<class Type>
Vector<Type>::Vector(const Vector<Type>& v)
{ 
    
   n = v.n; 
   a=new Type[n]; 
    
   for(int i=0;   i<n;   i++) 
      a[i] = v.a[i];
}
 
template<class Type>
Vector<Type>::Vector(int m,  Type* v)
{
   n = m; 
   a=new Type[n]; 
    
   for(int i=0; i<n; i++)
	 a[i] = v[i];
}
 
template<class Type>
Vector<Type>::Vector(int l, int m,  Type* v)
{

   n = l; 
   a=new Type[n];
    
   int i;
   for(i=0; i<Min(l,m); i++)
	 a[i] = v[i];
   if (l > m)
   {
      for (i=m; i<l; i++)
	 a[i] = a[m-1];
   }
   
}
 
template<class Type>
Vector<Type>::~Vector()
{
   delete [] a;
}

	//Fetch the size & Overloading [], change size of vector
template<class Type>
int Vector<Type>::size() const 
{ 
   return n;
}

//reset the size
template<class Type>
Vector<Type>& Vector<Type>::resetSize(int newsize)
{
   delete [] a;
   
   n = newsize;

   a = new Type[n];
   for (int i=0; i<n; i++)
      a[i] = 0;
   
   return *this;
}


 
template<class Type> // modifiable element
Type& Vector<Type>::operator[](int i) 
{ 
   return a[i];
}
        //access unwritable element, H.L. Deng, 05/29/96
template<class Type> 
Type Vector<Type>::operator[](int i) const 
{ 
   return a[i];
}
 
template<class Type>
Type* Vector<Type>::toPointer() const 	
{ 
   return a;
}
 
template<class Type>
Vector<Type>& Vector<Type>::setPointer(Type* p) 
{
   for(int i=0; i<n; i++) 
      a[i] = p[i];
   return *this;
}

 
template<class Type>
Vector<Type>&  Vector<Type>::linear(Type init, Type slope) 
{
   for (int i=0; i<n; i++)
      a[i] = init + slope*i;
    
   return *this;
}
			
			//simple operations
template<class Type>
int Vector<Type>::indexMax() const 
{ 
   Type m=a[0];   
   int k=0;
    
   for(int i=1;   i<n;   i++) 
      if ((m=Max(m,a[i]))==a[i]) k = i; 
   
   return k;
}
 
template<class Type>
int Vector<Type>::indexMin() const 
{ 
    
   Type m=a[0];   
  int k=0;
    
   for(int i=1;   i<n;   i++) 
      if ((m=Min(m,a[i]))==a[i]) k = i; 
    
   return k;
}

 
template<class Type>
Type Vector<Type>::max() const 
{ 
    
   int k = indexMax(); 
    
   return a[k];
}
 
template<class Type>
Type Vector<Type>::min() const 
{ 
    
   int k = indexMin(); 
    
   return a[k];
}

 
template<class Type>
Type Vector<Type>::sum() const 
{
    
   Type total = 0;
    
   for (int i=0;  i<n; i++) 
      total += a[i];
    
   return total;
}

 
template<class Type>
Type Vector<Type>::norm(int p) const 
{
    
   double sum = 0; 
    
   for(int i=0; i<n; i++)
      sum += pow(Abs(a[i]),p);
    
   return(Type(pow(sum,1./p)));
}


 
template<class Type>
Vector<Type>& Vector<Type>::normalize() 
{
    
   Type scale = this->norm(2);
    
   if (scale != 0) scale = 1/scale; 
    
   for (int i=0; i<n; i++)  
      a[i] *= scale;
    
   return *this;
}

 
template<class Type>
Vector<Type> Vector<Type>::operator-()
{
    
   Vector<Type> u(n); 
    
   for (int i=0; i<n; i++) 
      u[i] = -a[i]; 
    
   return u;
}

template<class Type>
Vector<Type> Vector<Type>::abs()
{
    
   Vector<Type> u(n);
    
   for (int i=0; i<n; i++)
      u[i] = ((a[i] > 0) ? a[i] : -a[i]); 
    
   return u;
}

/*
template<class Type>
Vector<float> Vector<Type>::operator Vector<float>() const
{
   Vector<float> v(n); 
   int i;
    
   for(i = 0; i < n;  i++)
      v[i] = (float)a[i]; 
    
   return v;
}

 
template<class Type>
Vector<double> Vector<Type>::operator Vector<double>() const
{
    
   Vector<double> v(n); 
   int i;
   
   for(i=0; i<n; i++)
      v[i] = (double)a[i]; 
    
   return v;
}

template<class Type>
Vector<int> Vector<Type>::operator Vector<int>() const
{
   Vector<int> v(n);
   int i;

   for(i=0; i<n; i++)
      v[i] = (int)a[i];
   
   return v;
}
 
template<class Type>
Vector<long> Vector<Type>::operator Vector<long>() const
{
   Vector<long> v(n);
   int i;
    
   for(i=0; i<n; i++) 
      v[i] = (long)a[i]; 
    
   return v;
}
*/
		//overloading = operator
template<class Type>
Vector<Type>& Vector<Type>::operator=(const Vector<Type>& v)
{
   if (n != v.n)
   {
      n = v.n;
      delete [] a;
      a = new Type[n];
   }
    
   for (int i=0; i<n; i++)
      a[i] = (Type) v[i];

   return *this;
}

/*
template<class Type>
Vector<Type>& Vector<Type>::operator=(int m, const Vector<Type>& v)
{

   for (int i=0; i<min(m,n); i++)
      a[i] = (Type) v[i];
   
   return *this;
}
*/

template<class Type>
Vector<Type>& Vector<Type>::operator=(Type c)
{
    
   for (int i=0; i<n; i++)
      a[i] = c; 
    
   return *this;
}
 
template<class Type>
Vector<Type>& Vector<Type>::operator=(Type *p)
{
    
   for (int i=0;   i<n;   i++) 
      a[i] = p[i]; 
    
   return *this;
}

			//overlodaing += operator
template<class Type>
Vector<Type>& Vector<Type>::operator+=(const Vector<Type>& v)
{
    
   if (n!=v.n) inValidSize();
    
   for (int i=0;   i<Min(v.n, n);   i++)
      a[i] += v.a[i]; 
    
   return *this;
}

 
template<class Type>
Vector<Type>& Vector<Type>::operator+=(Type c)
{
    
   for (int i=0;   i<n;   i++) 
      a[i] += c; 
    
   return *this;
}

 
template<class Type>
Vector<Type>& Vector<Type>::operator+=(Type *p)
{
    
   for (int i=0;   i<n;   i++) 
      a[i] += p[i]; 
    
   return *this;
}
			//overloading -= operator
template<class Type>
Vector<Type>& Vector<Type>::operator-=(const Vector<Type>& v)
{
    
   if (n!=v.n) inValidSize(); 
    
   for (int i=0;   i<Min(v.n, n);   i++)  
      a[i] -= v.a[i]; 
    
   return *this;
}
 
template<class Type>
Vector<Type>& Vector<Type>::operator-=(Type c)
{
    
   for (int i=0;   i<n;   i++) 
      a[i] -= c;  
    
   return *this;
}
 
template<class Type>
Vector<Type>& Vector<Type>::operator-=(Type *p)
{
    
   for (int i=0;   i<n;   i++) 
      a[i] -= p[i]; 
    
   return *this;
}

			//overloading *= /= operators
template<class Type>
Vector<Type>& Vector<Type>::operator*=(Type c)
{
    
   for (int i=0;   i<n;   i++) 
      a[i] *= c; 
    
   return *this;
}
 
template<class Type>
Vector<Type>& Vector<Type>::operator/=(Type c)
{
    
   assert(c != 0);  
 for(int i=0;   i<n;   i++) a[i] /= c;  
 return *this;}

	//overloading insertion and extraction operators
template<class Type>
ostream& operator<<(ostream& ofp, const Vector<Type>& v)
{
    
   ofp<<v.a[0]<<" ";
    
   for(int i=1;   i<v.n;   i++) 
   {
      if (i%5 == 0) ofp << "\n";
      ofp<<v.a[i]<<" ";
   }
    
   ofp<<endl;
    
   return ofp;
}

 
template<class Type>
istream& operator>>(istream& ifp,  Vector<Type>& v)
{
     
    for(int i=0;   i<v.n;   i++) 	
       if (!(ifp>>v.a[i])) break;    
     
    return ifp;
}

 
/*template<class Type>
size_t Vector<Type>::bfread(FILE *ifp)
{
   return fread(a, sizeof(Type), n, ifp);
}
*/

template<class Type>
size_t Vector<Type>::bfwrite(FILE *ofp) 
{
    
   return fwrite(a, sizeof(Type), n, ofp);
}

 
template<class Type>
Vector<Type>& Vector<Type>::chaSize(int m) 
{
    Vector<Type> v(n, a);
     
    delete [] a;  
    a = new Type[m];
     
    for(int i=0;   i<m;     i++) {
	if (i<n) a[i] = v[i]; else a[i] = 0; }
     
    n = m;  
     
    return *this;
}

//adding elements to the vector
template<class Type>
Vector<Type>& Vector<Type>::addVal( Type c)
{
     
    chaSize(n+1);     
    a[n-1] = c;      
     return *this;
}

 
template<class Type>
Vector<Type>& Vector<Type>::addVal(const Vector<Type>& u)
{
     
    int m = u.n, oldn = n; 
     
    chaSize(oldn+m);
     
    for (int i=0;   i < m;   i++) a[i+oldn] = u.a[i]; 
     
    return *this;
}

 
template<class Type>
Vector<Type>& Vector<Type>::addVal(Type c, int index)
{
     
    Vector<Type> u(n-index);
     
    for (int i=0;   i < n-index;   i++) u[i] = a[i+index];
     
    chaSize(index);
     
    addVal(c);   
    cerr << "length: "<<n <<endl;
     
    addVal(u);   
    cerr << "length: "<<n <<endl;
     
    return *this;
}

template<class Type>
Vector<Type>& Vector<Type>::delVal(int ielem)
{
   int i;
   Vector<Type> temp(*this);
   
   delete [] a;
   n --;
   a = new Type[n];
   
   for (i=0; i<ielem; i++)
      a[i] = temp[i];
   for (i=ielem; i<n; i++)
      a[i] = temp[i+1];
   
   return *this;
}

  
//inline functions
template<class Type>
inline Type operator*(const Vector<Type>& u, const Vector<Type>& v)
{
    
   int n = u.size();
    
   if (n != v.size()) inValidSize();
    
   Type sum = 0, temp1, temp2;
    
   for(int i=0;   i<n;   i++) {
	 temp1 = u[i];
	 temp2 = v[i];
	 sum += temp1*temp2;
      }
   
    
   return sum;
}

 
template<class Type>
inline Vector<Type> operator*( Type c,  const Vector<Type>& v){
     
    Vector<Type> u(v);  
 u *= c;  
 return u;}

 
template<class Type>
inline Vector<Type> operator*(const Vector<Type>& v,  Type c){
     
    Vector<Type> u(v);  
 u *= c;  
 return u;}

 
template<class Type>
inline Vector<Type> operator/(const Vector<Type>& v,  Type c){ 
     
    assert(c != 0);  
 Vector<Type> u(v);  
 u /= c;  
 return u;}
 
template<class Type>
inline Vector<Type> operator+(const Vector<Type>& v,  Type c){
     
    Vector<Type> w(v);  
 w += c;  
 return w; }
 
template<class Type>
inline Vector<Type> operator+( Type c,  const Vector<Type>& v){
     
    Vector<Type> w(v);  
 w += c;  
 return w; }
 
template<class Type>
inline Vector<Type> operator+(const Vector<Type>& v,  const Vector<Type>& u)
{
    
   Vector<Type> w(v); 
    
   w += u; 
    
   return w; 
}
 
template<class Type>
inline Vector<Type> operator-(const Vector<Type>& v,  Type c)
{
    
   Vector<Type> w(v); 
    
   w -= c; 
    
   return w; 
}
 
template<class Type>
inline Vector<Type> operator-( Type c,  const Vector<Type>& v)
{
    
   Vector<Type> w(v); 
    
   w -= c; 
    
   return -w; 
}
 
template<class Type>
inline Vector<Type> operator-(const Vector<Type>& v,  const Vector<Type>& u)
{
    
   Vector<Type> w(v); 
    
   w -= u; 
    
   return w; 
}
 
template<class Type>
inline Vector<Type> operator-(const Vector<Type>& v,  Type* p)
{
    
   Vector<Type> w(v); 
    
   w -= p; 
    
   return w; 
}

 
template<class Type>
Type Vector<Type>::norm2S() const 
{ 
    
   Vector<Type> v(*this);  
    
   return (v*v);
}

 
template<class Type>  //circular index of elements, 01/27/95
Type Vector<Type>::CircElem(int i) const
{
    
   int j;
    
   if (i<0) j = i + (Abs(i)/n +1)*n;       
   
    
   else j = i%n;  
    
   return a[j];
}

 
template<class Type>
inline Vector<Type> saxpy( Type factor,  const Vector<Type>& x,  const Vector<Type>& y)
{
    
   Vector<Type> v(factor*x);
    
   v += y;
    
   return v;
}

 
template<class Type> // 08/04/94, H.L. Deng
Vector<Type>* Vector<Type>::copy(int ibeg, int iend)
{
   int ilength = iend-ibeg+1;
    
   if (ilength <= 0) 
   {
      cerr << "Index is not correct!"<<endl; 
      return NULL;
   }
   
    
   Vector<Type>* u = new Vector<Type>(ilength);
    
   for (int i=0;   i<ilength;   i++) 
      u[0][i] = a[i+ibeg];
    
   return u;
}

 
template<class Type>
Vector<Type>* Vector<Type>::copy(int ilength)
{
    
   if (ilength <= 0)
   {
      cerr << "The length is not correct!"<<endl; 
      return NULL;
   }

    
   Vector<Type>* u = new Vector<Type>(ilength);
    
   for (int i=0;   i<ilength;   i++)
      u[0][i] = a[i];
   
    
   return u;
}

template<class T>
int operator!=(const Vector<T>& u, const Vector<T>& v)
{
   int i, n=u.size();
   if (n!=v.size()) return 1;
   
   for (int i=0; i<n; i++)
      if (u[i] != v[i]) return 1;
   
   return 0;
}

template<class Type> 
int operator==(const Vector<Type>& u, const Vector<Type>& v)
{
   int i, n=u.size();
   if (n!=v.size()) return 0;
   
   for (int i=0; i<n; i++)
      if (u[i] != v[i]) return 0;
   
   return 1;
}

template<class T>
int operator!=(const Vector<T>& u, T c)
{
   for (int i=0; i<u.size(); i++)
      if (u != c) return 1;

   return 0;
}

template<class Type>
int operator==(const Vector<Type>& u, Type c)
{
   for (int i=0; i<u.size(); i++)
      if (u != c) return 0;

   return 1;
}

template<class Type>  // H.L. Deng, 07/25/95
int Vector<Type>::in(Type c) const   
 {
     
    for (int i=0;   i<n;   i++) 
    {
       if (a[i] == c) return i;
       continue;
    }
     
    return -1;
}
 
}

#endif
