//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.



#ifndef SPAMATRIX_HH
#define SPAMATRIX_HH

//===========================
// Sparse-Matrix class template library for efficient algebraic operations
//
//  H. Lydia Deng,  01/24/1994
//===========================
// .NAME SpaMatrix class template, an inheritance of Matrix class
// .LIBRARY Base
// .HEADER c++ uitility classes
// .INCLUDE Matrix.hh
// .FILE SpaMatrix.hh

// .SECTION Description
// .B SpaMatrix<Type>
// is a derived sparse matrix class from Matrix
// it manages to do algebraic operations of two-dimensional sparse-matrix
// by considering the sparsity of the matrix
// it also include some basic row operations.
// 
// .SECTION Caveats

#include "Vector.hh"
#include "Matrix.hh"

namespace coool
{

static const char*  myName =  "sparse matrix class";

//*****************************************************************************
//definition of the sparse SpaMatrix using row index class and its member functions
//@Man:
//@Memo: a sparse matrix class template
/*@Doc:  This is a class for efficient representation and manipulation of 
  sparse matrix. 
*/

#ifdef __GNUC__
#pragma interface
#endif

template <class Type>
class SpaMatrix : public Matrix<Type> {
  private:
    using Matrix<Type>::nrow;
    using Matrix<Type>::ncol;
    int nz;
    //@ManMemo: 
    Vector<Type>* elem;
    //@ManMemo: 
    Vector<int>* icol;
    //@ManMemo: 
    Vector<int>* irow;
    public:			//constructors and destructor 
    //@ManMemo: 
    SpaMatrix(int,int,int); 
    //@ManMemo: 
    SpaMatrix(Vector<Type>&, Vector<int>&, Vector<int>&);
    //@ManMemo: 
    SpaMatrix(Vector<Type>&, int*, int*);
    //@ManMemo: 
    SpaMatrix(const SpaMatrix<Type>&);
    ~SpaMatrix();

    //@ManMemo: access the size description of the matrix
    const int nzSize() const;

    //@ManMemo: 
    const char* matrixType() const;

    //@ManMemo:the ith non-zero element
    Type& operator[](int );
    //@ManMemo:the ith row-vector 
    Vector<Type> rowVector(int i) const;	
    //@ManMemo:the jth coumn-vector 
    Vector<Type> colVector(int j) const;		

    //@ManMemo:sum across rows
    Type operator()(char*, int);
    //@ManMemo:sum across columns 
    Type operator()(int, char*);
    //@ManMemo:return the largest non-zero member of the row
    Type rowMax(int) const;	
    //@ManMemo:return the smallest non-zero member of the row
    Type rowMin(int) const;	
    //@ManMemo:return the largest non-zero member of the col
    Type colMax(int) const;	
    //@ManMemo:return the smallest non-zero member of the col
    Type colMin(int) const;	

    const Type nzElem(int i) const;
    //@ManMemo: 
    const int colPos(int i) const;
    //@ManMemo: 
    const int rowPos(int i) const;
    //@ManMemo: 
    const int nzRow(int i) const;

    //@ManMemo: count number of nonzeros 
    int rowCount(int) const;
    //@ManMemo: row-norm of the sparse matrix
    Type rowNorm(int ir, int p); 	
    //@ManMemo: row-2-norm squared of the sparse matrix
    Type rowNorm2(int ir);
    //@ManMemo: dot product of one row with a Vector
    Type rowDot(int, const Vector<Type>&);
    //@ManMemo: row scaling
    Vector<Type> rowScale(int, Type); 	

    //implicit type converting
    operator SpaMatrix<int>() const
    {
       int j, m = irow->size();
       Vector<int> mm(elem[0]);
       Vector<int> row(elem->size());
       for (int i=0; i<m; i++)
	  for (j=(*irow)[i]; j<(*irow)[i+1]; j++)  row[j] = i;
       SpaMatrix<int> A(mm,row,icol[0]);
       return A;
    }
    operator SpaMatrix<long>() const
    {
       int j, m = irow->size();
       Vector<long> mm(elem[0]);
       Vector<int> row(elem->size());
       for (int i=0; i<m; i++)
	  for (j=(*irow)[i]; (j<(*irow)[i+1]); j++)  row[j] = i;
       SpaMatrix<long> A(mm,row,icol[0]);
       return A;
    }
    operator SpaMatrix<float>() const
    {
       int j, m = irow->size();
       Vector<float> mm(elem[0]);
       Vector<int> row(elem->size());
       for (int i=0; i<m; i++)
	  for (j=(*irow)[i]; (j<(*irow)[i+1]); j++)  row[j] = i;
       SpaMatrix<float> A(mm,row,icol[0]);
       return A;
    }
    operator SpaMatrix<double>() const
    {
       int j, m = irow->size();
       Vector<double> mm(elem[0]);
       Vector<int> row(elem->size());
       for (int i=0; i<m; i++)
	  for (j=(*irow)[i]; (j<(*irow)[i+1]); j++)  row[j] = i;
       SpaMatrix<double> A(mm,row,icol[0]);
       return A;
    }
    //@ManMemo: 
    SpaMatrix<Type>&  	operator=(const SpaMatrix<Type>&);
    //@ManMemo: 
    SpaMatrix<Type>&  	operator+=(Type c);
    //@ManMemo: 
    SpaMatrix<Type>&  	operator-=(Type c);
    //@ManMemo: 
    SpaMatrix<Type>&  	operator*=(Type c);
    //@ManMemo: 
    SpaMatrix<Type>&  	operator/=(Type c);
    //@ManMemo: 
    SpaMatrix<Type> 	operator-();

    //@ManMemo: 
    Vector<Type> adotx(const Vector<Type>& x);
    //@ManMemo: 
    Vector<Type> atdotx(const Vector<Type>& x);

    //@ManMemo: 
template < class T >
friend Vector<T> operator* (const SpaMatrix<T>& A, const Vector<T>& v);
    //@ManMemo: write the sparse matrix to a stream
template < class T >
friend ostream& operator<< (ostream&, const SpaMatrix<T>&);
};

#ifdef __GNUC__
#pragma implementation
#endif

 
template<class Type>
SpaMatrix<Type>::SpaMatrix(int mm, int nn, int mn)
:Matrix<Type>(nn, mn) 
{
    
   nz = mm;
    
   elem = new Vector<Type>(nz);
    
   icol = new Vector<int>(nz);
    
   irow = new Vector<int>(nrow+1);
}

 
template<class Type>
SpaMatrix<Type>::SpaMatrix(Vector<Type>& x, Vector<int>& mm, Vector<int>& nn)
:Matrix<Type>(mm.max()+1, nn.max()+1) {
       
      nz = x.size();
       
      elem = new Vector<Type>(nz);
       
      icol = new Vector<int>(nz);
      
       
      for (int i=0;   i<nz;   i++)
      {
	 (*elem)[i] = x[i];
	 (*icol)[i] = nn[i];
      }

       
      irow = new Vector<int>(nrow+1);
       
      int i = 0; (* irow)[i]=0;
       
      for (int j=1;   j<nz;   j++)
	 if (mm[j] != i) { assert(mm[j]==i+1); i++; (*irow)[i]=j; }
       
      assert(i==nrow-1);
      (* irow)[nrow] = nz;
}

 
template<class Type>
SpaMatrix<Type>::SpaMatrix(Vector<Type>& x, int* mm, int* nn)
{
    
   nz = x.size();
    
   Vector<int> vm(nz), vn(nz);
    
   vm = mm;
    
   vn = nn;

    
   SpaMatrix<Type> A(x,vm,vn);
    
   ncol = A.ncol;  
 nrow = A.nrow;
    
   elem = new Vector<Type>(nz);
    
   icol = new Vector<int>(nz);
    
   irow = new Vector<int>(nrow+1);

   * elem = *(A.elem);
   * icol = *(A.icol);
   * irow = *(A.irow);
}

 
template<class Type>
SpaMatrix<Type>::SpaMatrix(const SpaMatrix<Type>& A)
:Matrix<Type>(A.nrow, A.ncol) 
{
    
   nz = A.nz;
    
   elem = new Vector<Type>(nz);
    
   icol = new Vector<int>(nz);
    
   irow = new Vector<int>(nrow+1);
   * elem = *(A.elem);
   * icol = *(A.icol);
   * irow = *(A.irow);
}

 
template<class Type>
SpaMatrix<Type>::~SpaMatrix(){  
 delete elem;  
 delete icol;  
 delete irow;}

    				// access the size description of the matrix
template<class Type>   		// return  number of nonzero elements
const int SpaMatrix<Type>::nzSize() const { 
return nz;}

 
template<class Type>
const char* SpaMatrix<Type>::matrixType() const 
{ return myName;}

				//access elements of the matrix
template<class Type>            //the ith non-zero element
Type& SpaMatrix<Type>::operator[](int i) { 
return elem[0][i];}

 
template<class Type>
Vector<Type> SpaMatrix<Type>::colVector(int ic)	const	//the ith col-vector
{
     
    Vector<Type> v(nrow);
     
    int j, iend;
     
    for(int k=0;   k<nrow;   k++) {
        v[k] = 0;
        j = (*irow)[k];
        iend = (*irow)[k+1];
        if (k==nrow-1) iend ++;
        while ((*icol)[j] < ic && j<iend) j++;
        if ((*icol)[j]==ic && (j<iend)) v[k] = (*elem)[j];
    }
     
    return v;
}
 
 
template<class Type>
Vector<Type> SpaMatrix<Type>::rowVector(int ir) const	//the ith row-vector
{
     
    int j = (*irow)[ir];
     
    int iend = (*irow)[ir+1]-1;
     
    if (ir == nrow-1)  iend = (*irow)[ir];
     
    Vector<Type> v(ncol);
    int k;
    for(k=0;  k<(*icol)[j];  k++)
	v[k] = 0;
     
    for(; ( k<=(*icol)[iend]);   k++) {
	if ((*icol)[j] == k) {
	    v[k] = (*elem)[j];
	    j++;
	}
	else v[k] = 0;
    }
     
    for(;   k<ncol;   k++)
	v[k] = 0;
    
     
    return v;
}

 
template<class Type>
const Type SpaMatrix<Type>::nzElem(int i) const { 
return (*elem)[i];}
 
template<class Type>
const int SpaMatrix<Type>::colPos(int i) const { 
return (*icol)[i];}
 
template<class Type>
const int SpaMatrix<Type>::rowPos(int i) const 
{
    
   int j=1; 
    
   while((*irow)[j] <= i) j++;
   if (j>nrow) j--;  
   return (--j);
}
 
template<class Type>
const int SpaMatrix<Type>::nzRow(int i) const {  
  return (*irow)[i];}

				//row operations
template<class Type>
int SpaMatrix<Type>::rowCount(int ir) const
{
    
   int k = (*irow)[ir+1]-(*irow)[ir]; 
    
   if (ir==nrow-1) k++;   
   return k;
}

 
template<class Type>
Type SpaMatrix<Type>::rowDot(int ir, const Vector<Type>& x) 
{
    
   assert(ncol==x.size()); 
    
   int n = (*irow)[ir]; 
    
   Type sum = 0;
    
   for (int i=n; ( i<(*irow)[ir+1]);  i++)
      sum += (*elem)[i]*x[(*icol)[i]];
    
   return sum; 
}

 
template<class Type>
Vector<Type> SpaMatrix<Type>::rowScale(int ir, Type c) 
{
    
   Vector<Type> v(ncol); 
    
   int n = (*irow)[ir];

    
   for (int i=0;   i<ncol;   i++)
   {
      if ((*icol)[n] == i) 
      {
	 v[i] = c*(*elem)[n]; n++; 
      }
      else v[i] = (Type)0; 
   }
    
   return v;
}

 
template<class Type>
Type SpaMatrix<Type>::rowNorm(int ir, int p) 	//row-norm of the sparse matrix
{
    
   int nelem = rowCount(ir); 
    
   Vector<Type> v(nelem);
    
   for (int i=0;   i<nelem;   i++) v[i] = (*elem)[irow[0][ir]+i]; 
    
   return v.norm(p); 
}
 
template<class Type>
Type SpaMatrix<Type>::rowNorm2(int ir) 	//row-2-norm squared of the sparse matrix
{
    
  Type p = rowNorm(ir,2);  
  return p*p;
}

				// Overloading operators
template<class Type>
SpaMatrix<Type>& SpaMatrix<Type>::operator=(const SpaMatrix<Type>& A){
 
  assert(nz==A.nz);  
  assert(nrow==A.nrow);  
  assert(ncol==A.ncol);

  *elem = *A.elem; 
  *icol = *A.icol; 
  *irow = *A.irow; 
 
  return *this;
}
 
template<class Type>
SpaMatrix<Type>&  SpaMatrix<Type>::operator+=(Type c){

  *elem += c;  
  return *this;
}
 
template<class Type>
SpaMatrix<Type>&  SpaMatrix<Type>::operator-=(Type c){

  *elem -= c;  
  return *this;
}
 
template<class Type>
SpaMatrix<Type>&  SpaMatrix<Type>::operator*=(Type c){

  *elem *= c;  
  return *this;
} 
 
template<class Type>
SpaMatrix<Type>&  SpaMatrix<Type>::operator/=(Type c){

  *elem /= c;  
  return *this;
}


				//friends functions
template<class Type>
Vector<Type> operator*(const SpaMatrix<Type>& A, const Vector<Type>& v) 
{
    
   assert(A.numOfCols()==v.size());
    
   int n = A.numOfRows();
    
   Vector<Type> u(n);

    
   int i, j;
    
   for(i=0;   i<n;   i++)
     for(j = A.nzRow(i); (j<A.nzRow(i+1));  j++)
       u[i] +=  A.nzElem(j)*v[A.colPos(j)];
    
   return u;
}

 
template<class Type>
Vector<Type> SpaMatrix<Type>::adotx(const Vector<Type>& x)
{
    
   SpaMatrix<Type> A(*this);
    
   return A*x;
}

 
template<class Type>
Vector<Type> SpaMatrix<Type>::atdotx(const Vector<Type>& x)
{
    
   if (x.size() != nrow) inValidSize();
    
   Vector<Type> v(ncol);

    
   int i, j = 0;
    
   for(i = 0;  i < nz;   i++)
   {
     if((i>=(*irow)[j+1])&&(i!=nz)) j++;
     v[(*icol)[i]] += (*elem)[i] * x[j];
   }
    
   return v;
}

 
template<class Type>			//sum across columns
Type SpaMatrix<Type>::operator() (int ir, char*)
{
     
    Type sum = 0;
     
    for (int j=(*irow)[ir]; ( j<(*irow)[ir+1]);   j++)
	sum += (*elem)[j];
     
    return sum;
}

 
template<class Type>			//sum across rows
Type SpaMatrix<Type>::operator() (char*, int ic)
{
     
    Type sum = 0;
     
    int k, iend = (*irow)[0];
	
     
    for (int j=0;   j<nrow;   j++){
	k = iend;
	iend = (*irow)[j+1];
	for(; (k<iend); k++)
	  if ((*icol)[k] == ic) { 
	    sum += (*elem)[k]; break;
	  }
    }
     
    return sum;
}

 
template<class Type>
Type SpaMatrix<Type>::rowMax(int ir) const //return the largest non-zero member of the row
{
     
    int j = (*irow)[ir]+1;
     
    Type m = (*elem)[j-1];
     
    for (; (j<(*irow)[ir+1]);  j++)
	m = Max(m, (*elem)[j]);
     
    return m;
}

 
template<class Type>
Type SpaMatrix<Type>::rowMin(int ir) const	//return the smallest non-zero member of the row
{
     
    int j = (*irow)[ir]+1;
     
    Type m = (*elem)[j-1];
     
    for (; (j<(*irow)[ir+1]);   j++)
	m = Min(m, (*elem)[j]);
     
    return m;
}

 
template<class Type>
Type SpaMatrix<Type>::colMax(int ic) const	//return the largest non-zero member of the col
{
     
    int k = 0;
     
    while (ic!=(*icol)[k]) k++;
     
    Type m = (*elem)[k];
     
    int ir = 0;
     
    while ((*irow)[ir] < k) ir++;
     
    int iend = (*irow)[ir+1];
     
    for (;   ir<nrow;   ir++) {
    	iend = (*irow)[ir+1];
	while (ic!=(*icol)[k] && k<iend) k++;
	if (k != iend) {
	    m = Max(m, (*elem)[k]);
	    k = iend;
	}
    }
     
    return m;
}
	
 
template<class Type>
Type SpaMatrix<Type>::colMin(int ic) const	//return the smallest non-zero member of the col
{
     
    int k = 0;
     
    while ( ic!=(*icol)[k]) k++;
     
    Type m = (*elem)[k];
     
    int ir = 0;
     
    while ((*irow)[ir] < k) ir++;
     
    int iend;
     
    for (;   ir<nrow;   ir++) {
	iend = (*irow)[ir+1];
	while (ic!=(*icol)[k] && k<iend) k++;
	if (k != iend ) {
	    m = Min(m, (*elem)[k]);
	    k = iend;
	    iend = (*irow)[ir+2];
	}
    }
     
    return m;
}


 
template<class Type>
SpaMatrix<Type> SpaMatrix<Type>::operator-(){
     
    SpaMatrix<Type> A(nz,nrow,ncol); 
    *(A.icol) = *icol; 
    *(A.irow) = *irow; 
    *(A.elem) = -(*elem);
     
    return A;
}
					//Overloading operators

template<class Type>
ostream& operator<<(ostream& osp, const SpaMatrix<Type>& A)
{
 
	int j, k = 0;
 
	for (int i=0;   i<A.nrow;   i++){
	  for(j=0; j<A.ncol; j++){
	    if((i<A.nrow-1 && k==(A.irow[0])[i+1]) || (A.icol[0])[k]!=j) osp<<"0  ";
	    else {osp<<(*A.elem)[k]<<"  "; k++; }
	  } osp<<endl;
	}
 
	return osp;
}

 
template <class Type>
inline SpaMatrix<Type>  operator+(const SpaMatrix<Type>& A, Type c){
     
  SpaMatrix<Type> B(A);  
  B += c;  
  return B;
}
 
template <class Type>
inline SpaMatrix<Type>  operator+(Type c, const SpaMatrix<Type>& A)
{ 
   SpaMatrix<Type> B(A);  
   B += c;  
   return B;
}
 
template <class Type>
inline SpaMatrix<Type>  operator-(const SpaMatrix<Type>& A, Type c){
     
  SpaMatrix<Type> B(A);  
  B -= c;  
  return B;
}
 
template <class Type>
inline SpaMatrix<Type>  operator-(Type c, const SpaMatrix<Type>& A){
     
  SpaMatrix<Type> B(A);  
  B -= c;  
  return -B;
}
 
template <class Type>
inline SpaMatrix<Type>  operator*(const SpaMatrix<Type>& A, Type c)
{
   SpaMatrix<Type> B(A);  
   B *= c;  
   return B;
}
 
template <class Type>
inline SpaMatrix<Type>  operator*(Type c, const SpaMatrix<Type>& A)
{ 
   SpaMatrix<Type> B(A);  
   B *= c;  
   return B;
}
 
template <class Type>
inline SpaMatrix<Type>  operator/(const SpaMatrix<Type>& A, Type c)
{
   SpaMatrix<Type> B(A);  
   B /= c;  
   return B;
}
 
}

#endif

