//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.
//
//=============================
// Definition of the Simplex class
// The flexible polyhedron algorithm
// author:  Wenceslau Gouveia
// modified:  H. Lydia Deng, 02/23/94,  /03/14/94
//=============================

// .NAME ConjugateGradient class
// .LIBRARY Base
// .HEADER Optimization Algorithms
// .INCLUDE defs.hh
// .FILE CG.hh

// .SECTION Description
// .B Simplex()
// The downhill simplex -do not confuse this one with the simplex algorithm used
// in linear programming- is a direct search method that do not require information
// on the derivatives of the objective function (Nelder, J. and Mead, R., The SIMPLEX 
// method, Computer Journal, 7, 1087 - 1092). The basic idea is to build a polyhedron
// of dimension n+1, where n is the dimension of the objective function, with trial 
// solutions to the problem assigned to each one of its vertexes. During the optimization
// process this polyhedron is distorted to "move" in the direction of the best solution.
// 
// .SECTION Description
// Public Operations
// Constructors: 
//		Simplex(ObjectiveFunction* f, Model<double>*models, int iter, 
//			      double alpha, double beta, double gamma)
// 		Here:
//			f: Defines a pointer to the objective function
//		        models: Defines a pointer to the n+1 initial models required by the Simplex class,
//				     where n is the dimension of the objective function
//			iter: Maximum number of iterations
//		 	tol: Minimum accepted module of the gradient at optimum solution
//			alpha: Reflection parameter (I recommend gamma = 1)
//			beta: contraction parameter ( I recommend beta = .5)
//			gamma: Expansion parameter (I recommend gamma = 2)
//
// Methods:
//		Model<>ConjugateGradient::optimizer(double atol)
//		Here:
//			atol:  Stopping criterion. It defines te mimimum size of the polyhedron
//				 around the optimum solution. Notice that as the optimization go the
//				 polyhedron tends to shrink arounf the minimizer.
//
//			The optimum model is returned by the function.
//
// .SECTION Caveats
// The fact that n+1 initial guesses have to be provided to the Simplex algorithm
// almost rules out its application in large optimization problems. Also the Powell's
// derivative-free conjugate gradient tend to be a more efficient algorithm. The inclusion 
// of the Simplex algorithm in the COOOL library is mainly for completeness 
// purposes.

#ifndef SIMPLEX_HH
#define SIMPLEX_HH

#include "NonQuaOptima.hh"


//@Man:
//@Memo: non-gradient based optimizer
/*@Doc:  
  The downhill simplex. Do not confuse this one with the
  simplex algorithm used in linear programming- is a direct
  search method that do not require information
  on the derivatives of the objective function (Nelder, J. 
  and Mead, R., The SIMPLEX method, Computer Journal, 7, 
  1087 - 1092). The basic idea is to build a polyhedron
  of dimension n+1, where n is the dimension of the objective
  function, with trial solutions to the problem assigned to 
  each one of its vertexes. During the optimization
  process this polyhedron is distorted to "move" in the 
  direction of the best solution.

  The fact that n+1 initial guesses have to be provided to the
  Simplex algorithm almost rules out its application in large
  optimization problems. Also the Powell's derivative-free 
  conjugate gradient tend to be a more efficient algorithm. 
  The inclusion of the Simplex algorithm in the COOOL
  library is mainly for completeness purposes.

*/


namespace coool 
{
    using namespace coool;

class Simplex : public NonQuadraticOptima {
  private:
    void			formPsum();
    double			tryNewPoint(int ihigh, const double lever);

    int				nd;
    Model<double>*		models;
    Vector<double>*		psum;
    Vector<double>*		fv;
    double			value;

  public:
    //@ManMemo: maximum number of iterations
    int				maxiters;
    //@ManMemo: reflection parameter (recommend value alpha = 1)
    double			alpha;
    //@ManMemo: contraction parameter (recommend value beta = .5)
    double			beta;
    //@ManMemo: expansion parameter (recommended value gamma = 2)
    double			gamma;

    //@ManMemo: constructor
    Simplex(///pointer to the objective function
	    ObjectiveFunction* f,
	    ///a pointer to the $n+1$ initial models, where $n$ is dimension of model space
	    Model<double>* models,
	    ///maximum number of iterations
	    int iter, 
	    ///reflection parameter (recommend value, alpha = 1)
	    double alpha, 
	    ///contraction parameter (recommend value, beta = .5)
	    double beta, 
	    ///expansion parameter (recommend value, gamma = 2)
	    double gamma,
	    ///vebose or quiet
	    int verbose);
    //@ManMemo: constructor
    Simplex(ObjectiveFunction* f, Model<double>* models, int iter, double alpha, double beta, double gamma);
    
    ~Simplex();

    Model<double>	optimizer(Model<double>&) {return 0;}
    Model<long>		optimizer(Model<long>&) {return 0;}

    //@ManMemo: search with Simplex with stopping criterion
    Model<double>	optimizer(const double atol);
    /*@Doc: $atol$ defines the mimimum size of
           the polyhedron around the optimum solution. 
           Notice that as the optimization go the
           polyhedron tends to shrink arounf the minimizer. */
    //@ManMemo: search with Simplex with stopping criterion
    Model<double>	optimizer(Vector<double>& atollist);
    /*@Doc: $atollist$ defines the mimimum sizes of
      the polyhedron around the optimum solution. 
      Notice that as the optimization go the
      polyhedron tends to shrink arounf the minimizer. */

    //@ManMemo:
    double		bestValue() 	{return value;}
    //@ManMemo:
    int			evaluations() 	{return fp->iterations();}
    //@ManMemo:
    void reset(Vector<double>& lambda);
};
 
}

#endif



