//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.
//
//=========================
// Definition for the Powel's conjugate gradient class 
// Powell's derivative-free conjugate gradient class
// author:  Wenceslau Gouveia
// modified: H. Lydia Deng, 02/23/94, 03/15/94
//=========================

// .NAME PowellOptima class
// .LIBRARY Base
// .HEADER Optimization Algorithms
// .INCLUDE defs.hh
// .FILE Powell.hh

// .SECTION Description
// .B Powell()
// Powell's method can be considered a derivative-free version of the conjugate gradient
// algorithm (Powell, M., Computer Journal, 7, 155-162). Here the objective function
// is minimized from an initial model along a set of conjugate directiions generatied by the 
// procedure without resorting to the gradient of the objective function.
// 
// .SECTION Description
// Public Operations
// Constructors: 
//		PowellOptima(LineSearch* ls,  int iter, double tol, double changeOf, double delta);
// 		Here:
//			ls: Defines the line search used in the optimization (At the present version you
//			     should use the BrentLineSearch procedure)
//			iter: Maximum number of iterations
//			tol: Tolerance used in the line search
//			changeOf: This is the minimum relative change of the objective function value
//				      to stop the optimization (stopping criterion)
//			delta: Initial step used in line search
// Methods:
//		Model<double> optimizer(Model<double>&model0);
//		Here:
//			model0:  Initial model for the Powell's conjugate gradient procedure
//
//			The optimum model is returned by the function.
//
// .SECTION Caveats
// Although an interesting algorithm, Powell's method has demonstrated to have a high
// computational cost, due the excessive number of line search required to compute 
// the conjugate directions (see Applied Nonlinear Programming, by David Himmelblau for 
// details. It seems that the standard conjugate gradient by Hesteness and Stiefel with 
// numerical derivatives is a more efficient procedure.

#ifndef POWELL_OPTIMA_HH
#define POWELL_OPTIMA_HH

#include "LSearchOptima.hh"

//@Man:
//@Memo: Powell Optima
/*@Doc:
  Powell's method can be considered a derivative-free version
  of the conjugate gradient algorithm (Powell, M., Computer
  Journal, 7, 155-162). Here the objective function
  is minimized from an initial model along a set of conjugate
  directiions generatied by the procedure without resorting
  to the gradient of the objective function.

  Although an interesting algorithm, Powell's method has 
  demonstrated to have a high computational cost, due 
  the excessive number of line search required to compute 
  the conjugate directions (see Applied Nonlinear Programming, 
  by David Himmelblau for details. It seems that the standard
  conjugate gradient by Hesteness and Stiefel with 
  numerical derivatives is a more efficient procedure.

*/

namespace coool 
{
    using namespace coool;

class PowellOptima : public LineSearchOptima 
{
   private:
   double 			delta;
   double 			change;
   
   public:
   
   //@ManMemo: a constructor
    PowellOptima(///defines the line search used in the optimization
		 LineSearch* ls,  
		 ///maximum number of iterations
		 int iter, 
		 ///tolerance used in the line search
		 double tol, 
		 ///minimum relative change of function value to stop the optimization (stopping criterion) 
		 double changOf, 
		 ///initial step used in line search
		 double delta, 
		 ///vebose or quiet
		 int vebose);
   //@ManMemo: a constructor
   PowellOptima(LineSearch* ls,  int iter, double tol, double changeOf, double delta);
   

    ~PowellOptima(){;}

   //@ManMemo: Powell's search starting from m0, returns an optimum model
   Model<double> optimizer(Model<double>& m0);
   //@ManMemo: Powell's search starting from m0, returns an optimum model
   Model<long> optimizer(Model<long>& m0);

   //@ManMemo:
   const char* className() const;
};
 
}

#endif

		
		




















