//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.

#ifndef MODEL_HH
#define MODEL_HH

#ifdef __GNUC__
#pragma interface
#endif

#include <iostream>
using namespace std;

#include "Vector.hh"

// .NAME Model - a simple class for model parameters
// .INCLUDE c++/Model.hh
// .FILE Model.cc
// .SECTION Description
// .B Model
//=================
// H. Lydia Deng,  02/17/94
//============================

//@Man:
//@Memo: a simple class for a model space
/*@Doc:

 This is a class for a model space and models in this space. 
It should have been a derived class of Vector<Type>, but it is standalone 
because of historical reasons (I didn't know about inheritence back then. :).

 There are several choices for the Model Space. You could construct a 
n-D infinite space $R^n$, n-D hypercube (continuous with lower and upper 
bound at each directions), n-D lattice (discrete Model space), or n-D space
with irregular boundary (tested by the user-provide function, $cstraints(Vector\&)$.

 After the model space is constructed, it will take care of the updating of thenew models that guarentees all models belong to the model space.

*/

namespace coool 
{
    using namespace coool;
    
template <class Type>
class Model {

private:
	int	ndim;
	Type	deltmod;
	int     (*tp)(Vector<Type> &);
	
	Vector<Type>*	mp;
	Vector<Type>*	maxParam;
	Vector<Type>*	minParam;
	
public:
	//@ManMemo: the default constructor
	Model();
	//@ManMemo: construct a model space with dimension n
	Model(int n);
	//@ManMemo: construct a discrete model space with boundary and initial values
	Model(/// upper boundary of the model space
	      const Vector<Type>& maxp, 
	      /// lower boundary of the model space
	      const Vector<Type>& minp, 
	      /// initial values of the model parameters
	      const Vector<Type>& ini_mod, 
	      /// discrete step size in the model space
	      Type delta);
	//@ManMemo: construct a discrete model space with boundary and constraints
	Model(/// upper boundary of the model space
	      const Vector<Type>& maxp, 
	      /// lower boundary of the model space
	      const Vector<Type>& minp, 
	      /// point to the function which tests if the constraints are statisfied
	      int (*cstraints)(Vector<Type>&));
	//@ManMemo: construct a continuous model space with boundary and initial values
	Model(const Vector<Type>& maxp, const Vector<Type>& minp, const Vector<Type>& ini_mod);
	//@ManMemo: construct a continuous model space with boundary
	Model(const Vector<Type>& maxp, const Vector<Type>& minp);
	//@ManMemo: construct a continuous model space with initial values 
	Model(const Vector<Type>& ini_mod);
	//@ManMemo: construct an n-D models space with constraints
	Model(const int n, int (*cstraints)(Vector<Type>&));
	//@ManMemo: construct a model space identical to that of Model m
	Model(const Model<Type>& m);
	~Model();

	//@ManMemo: 
	int modSize() const;
	//@ManMemo: 
	Type oneMax(int i) const;
	//@ManMemo: 
	Type oneMin(int i) const;
	//@ManMemo: 
	Type operator[](int) const;
	//@ManMemo: 
	Vector<Type> modMax() const;
	//@ManMemo: 
    	Vector<Type> modMin() const;
	//@ManMemo: 
    	void setModMax(const Vector<Type>& v);
	//@ManMemo: 
    	void setModMin(const Vector<Type>& v);

	//@ManMemo: 
	Vector<Type>  modParam() const;
	//@ManMemo: 
	Type&	grid();

	//@ManMemo: 
	Type norm(int );
	//@ManMemo: 
	Model<Type> normalize();
	//@ManMemo: 
	Type	range(int);
	//@ManMemo: 
	Type& operator[](int);
	//@ManMemo: 
	Model<Type> update(Type, Type, const Vector<Type>&) const;

 
	operator Model<int>() const
	{
	   Vector<int> mm(mp[0]);
	   Model<int> m(mm);

	   if (maxParam != NULL || minParam != NULL) {
		 Vector<int> maxBound(maxParam[0]);
		 Vector<int> minBound(minParam[0]);
		 Model<int> m1(maxBound,minBound,mm,(int)deltmod);
		 return m1;
	      }
	   return m;
	}

	operator Model<long>() const
	{
	   Vector<long> mm(mp[0]);
	   Model<long> m(mm);
	   
	   if (maxParam != NULL || minParam != NULL) {
		 Vector<long> maxBound(maxParam[0]);
		 Vector<long> minBound(minParam[0]);
		 Model<long> m1(maxBound,minBound,mm, (long)deltmod);
		 return m1;
	      }
	   return m;
	}

	operator Model<float>() const
	{
	   Vector<float> mm(mp[0]);
	   Model<float> m(mm);

	   if (maxParam != NULL || minParam != NULL) {
		 Vector<float> maxBound(maxParam[0]);
		 Vector<float> minBound(minParam[0]);
		 Model<float> m1(maxBound,minBound,mm, (float)deltmod);
		 return m1;
	      }
	   return m;
	}

	operator Model<double>() const
	{
	   Vector<double> mm(mp[0]);
	   Model<double> m(mm);

	   if (maxParam != NULL || minParam != NULL) {
		 Vector<double> maxBound(maxParam[0]);
		 Vector<double> minBound(minParam[0]);
		 Model<double> m1(maxBound,minBound,mm, (double)deltmod);
		 return m1;
	      }
	   return m;
	}

	//@ManMemo: 
	Model<Type>& 	operator=(const Vector<Type>&);
	//@ManMemo: 
	Model<Type>& 	operator=(const Model<Type>&);
	//@ManMemo: 
	Model<Type>& 	operator=(Type);
	//@ManMemo: 
	Model<Type>& 	operator*=( Type);
	//@ManMemo: 
	Model<Type>& 	operator/=( Type);
	//@ManMemo: 
	Model<Type>& 	operator+=(const  Vector<Type>&);
	//@ManMemo: 
	Model<Type>& 	operator-=(const  Vector<Type>&);
	//@ManMemo: 
	Model<Type>& 	operator+=(const  Model<Type>&);
	//@ManMemo: 
	Model<Type>& 	operator-=(const  Model<Type>&);
	//@ManMemo: 
	Model<Type>&	chaSize(int);

	//@ManMemo: binary read from which ifp points to
	size_t bfread(FILE *ifp);
	//@ManMemo: binary write to which ofp points to
	size_t bfwrite(FILE *ofp);
	
	//@ManMemo: 
    friend ostream& operator<< (ostream& os, const Model<Type>& d) {
		os << d.mp[0] << endl;
 		return os;
	}
	//@ManMemo: 
    friend istream& operator>> (istream& is, Model<Type>& d) {
		is >> d.mp[0]; 
		return is;
	}
	//@ManMemo: 
template < class T >
    friend Model<T> operator/ (const Model<T>&, T);
	//@ManMemo: 
template < class T >
    friend Model<T> operator* (const Model<T>&, T);
    //@ManMemo: 
template < class T >
    friend Model<T> operator* (T,  const Model<T>&);
    //@ManMemo: 
template < class T >
    friend Model<T> operator+ (const Model<T>&, const Model<T>&);
    //@ManMemo: 
template < class T >
    friend Model<T> operator+ (const Model<T>&, const Vector<T>&);
    //@ManMemo: 
template < class T >
    friend Model<T> operator+ (const Vector<T>&, const Model<T>&);
    //@ManMemo: 
template < class T >
    friend Model<T> operator- (const Model<T>&,const  Model<T>&);
    //@ManMemo: 
template < class T >
    friend Model<T> operator- (const Model<T>&, const Vector<T>&);
    //@ManMemo: 
template < class T >
    friend Model<T> operator- (const Vector<T>&, const Model<T>&);

    //@ManMemo: comparing two Models, ==
template < class T >
    friend int operator== (const Model<T>& m1, const Model<T>& m2);
    //@ManMemo: comparing two Models, !=
template < class T >
    friend int operator!= (const Model<T>& m1, const Model<T>& m2);
};

#ifdef __GNUC__
#pragma implementation
#endif

 
template<class Type>
int Model<Type>::modSize() const { 
return ndim;}
 
template<class Type>
Type Model<Type>::operator[](int i) const{ 
return (*mp)[i];}
 
template<class Type>
Vector<Type> Model<Type>::modMax() const 
{ 
if (maxParam != NULL) return maxParam[0];  
 else return -999; }
 
template<class Type>
Vector<Type> Model<Type>::modMin() const 
{ 
if (minParam != NULL) return minParam[0];  
 else return -999; }
 
template<class Type>
void Model<Type>::setModMax(const Vector<Type>& v) { 
maxParam[0] = v;}
 
template<class Type>
void Model<Type>::setModMin(const Vector<Type>& v) { 
minParam[0] = v;}

 
template<class Type>
Vector<Type>  Model<Type>::modParam() const { 
return *mp;}
 
template<class Type>
Type&	Model<Type>::grid() {  
 return deltmod;}

 
template<class Type>
Type	Model<Type>::norm(int p){  
 return mp->norm(p);}
 
template<class Type>
Model<Type> Model<Type>::normalize()
{
    
   Model<Type> m(*this);
    
   m.mp[0] = (m.mp)->normalize();
    
   return m;
}

 
template<class Type>
Type& Model<Type>::operator[](int i){ 
return (*mp)[i];}


	// binary I/O
template<class Type>
size_t Model<Type>::bfread(FILE *ifp)
{
    
   return mp->bfread(ifp);
}
	
 
template<class Type>
size_t Model<Type>::bfwrite(FILE *ofp)
{
    
   return mp->bfwrite(ofp);
}
	

 
template<class Type>
Model<Type>& Model<Type>::operator=(const Vector<Type>& v) 
{
    
   mp[0] = v;
    
   return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::chaSize(int m) {
     
    ndim	=	m;
     
    mp->chaSize(m);
     
    maxParam->chaSize(m);
     
    minParam->chaSize(m);

     
    return *this;
}
    
 
template<class Type>
Model<Type>& Model<Type>::operator=(Type c) { 
 
	mp[0] = c;
 
	return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::operator*=(Type c) { 
 
	mp[0] *= c;
 
	return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::operator/=(Type c) { 
       
      if (c==0) 
      {
	 cerr<<"WARNING: divided by a zero. \n";
	 return *this;
      }
      
       
      mp[0] /= c;
       
      return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::operator+=(const Vector<Type>& v){
 
	mp[0] += v;
 
	return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::operator+=(const Model<Type>& m){
 
	mp[0] += m.mp[0];
 
	return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::operator-=(const Vector<Type>& v){
 
	mp[0] -= v;
 
	return *this;
}

 
template<class Type>
Model<Type>& Model<Type>::operator-=(const Model<Type>& m){
 
	mp[0] -= m.mp[0];
 
	return *this;
}

 
template<class Type>
Model<Type> operator+(const Model<Type>& m, const Vector<Type>& v){
 
	Model<Type> m1(m);
 
	m1 += v;
 
	return m1;
}

 
template<class Type>
Model<Type> operator+(const Vector<Type>& v, const Model<Type>& m){
 
	Model<Type> m1(m);
 
	m1 += v;
 
	return m1;
}

 
template<class Type>
Model<Type> operator-(const Model<Type>& m,  const Vector<Type>& v){
 
	Model<Type> m1(m);
 
	m1 -= v;
 
	return m1;
}

 
template<class Type>
Model<Type> operator-(const Vector<Type>& v,  const Model<Type>& m){
 
	Model<Type> m1(m);
 
	m1 -= v;
 
	return m1;
}

 
template<class Type>
Model<Type> operator+(const Model<Type>& m, const  Model<Type>& m1){
 
	Model<Type> m2(m);
 
	m2 += m1;
 
	return m2;
}

 
template<class Type>
Model<Type> operator-(const Model<Type>& m, const Model<Type>& m1){
 
	Model<Type> m2(m);
 
	m2 -= m1;
 
	return m2;
}

 
template<class Type>
Model<Type> operator/(const Model<Type>& m,  Type c){
 
	Model<Type> m1(m);
 
	m1 /= c;
 
	return m1;
}

 
template<class Type>
Model<Type> operator*(const Model<Type>& m,  Type c){
 
	Model<Type> m1(m);
 
	m1 *= c;
 
	return m1;
}

 
template<class Type>
Model<Type> operator*(Type c, const Model<Type>& m)
{
 
   Model<Type> m1(m);
   
   m1 *= c;
   
   return m1;
}

 
template<class Type>
Model<Type>& Model<Type>::operator=(const Model<Type>& m) 
{ 
   ndim = m.ndim;
	
   mp[0] = m.mp[0]; 
 
   deltmod = m.deltmod;
 
   if( m.maxParam == NULL) maxParam = NULL;
   else 
   {
      if (maxParam == NULL) maxParam = new Vector<Type>(ndim);
      maxParam[0] = (m.maxParam)[0];
   }

   if( m.minParam == NULL) minParam = NULL;
   else 
   {
      if (minParam == NULL) minParam = new Vector<Type>(ndim);
      minParam[0] = (m.minParam)[0];
   }

   tp = m.tp;
   
   return *this;
}

 
template <class Type>
Model<Type>::Model(const Vector<Type>& maxp, const Vector<Type>& minp, const Vector<Type>& initmod, Type deltmod)
{
   ndim = initmod.size();
   deltmod = (Type) deltmod;

   mp = new Vector<Type>(ndim);
   mp[0] = initmod;
 
   maxParam = new Vector<Type>(ndim);
   minParam = new Vector<Type>(ndim);
   maxParam[0] = maxp;
   minParam[0] = minp;

   tp = NULL;
}

template <class Type>
Model<Type>::Model(const Vector<Type>& maxp, const Vector<Type>& minp, int (*cstraints)(Vector<Type>&))
{
   ndim = maxp.size();
   deltmod = (Type)0;
   
   mp = new Vector<Type>(ndim);

   maxParam = new Vector<Type>(ndim);
   minParam = new Vector<Type>(ndim);
   maxParam[0] = maxp;
   minParam[0] = minp;
   
   tp = cstraints;
}

template <class Type>
Model<Type>::Model(const Vector<Type>& maxp, const Vector<Type>& minp, const Vector<Type>& initmod)
{
 
   ndim = initmod.size();
   
   deltmod = (Type)0;
	
   mp = new Vector<Type>(ndim);
   mp[0] = initmod;

   maxParam = new Vector<Type>(ndim);
   minParam = new Vector<Type>(ndim);
   maxParam[0] = maxp;
   minParam[0] = minp;
	
   tp = NULL;
}

 
template <class Type>
Model<Type>::Model(const Vector<Type>& maxp, const Vector<Type>& minp)
{
   ndim = Min(maxp.size(),minp.size());
   deltmod = (Type)0;
	
   mp = new Vector<Type>(ndim);
 
   maxParam = new Vector<Type>(ndim);
   minParam = new Vector<Type>(ndim);
   maxParam[0] = maxp;
   minParam[0] = minp;

   tp = NULL;
}

template <class Type>
Model<Type>::Model(int n, int (*cstraints)(Vector<Type> &))
{
   ndim = n;
   deltmod = (Type)0;
	
   mp = new Vector<Type>(ndim);
 
   maxParam = NULL;
   minParam = NULL;

   tp = cstraints;
}

 
template <class Type>
Model<Type>::Model(int n)
{
   ndim = n;
   deltmod = (Type)0;
	
   mp = new Vector<Type>(ndim);
 
   maxParam = NULL;
   minParam = NULL;

   tp = NULL;
}

 
template <class Type>
Model<Type>::Model()
{
   ndim = 1;
   deltmod = (Type)0;
	
   mp = new Vector<Type>(ndim);
 
   maxParam = NULL;
   minParam = NULL;

   tp = NULL;
}

 
template <class Type>
Model<Type>::Model(const Vector<Type>& initmod)
{
   ndim = initmod.size();
 
   deltmod = (Type)0;
 
   mp = new Vector<Type>(ndim);
   mp[0] = initmod;
 
   maxParam = NULL;
   minParam = NULL;

   tp = NULL;
}

 
template <class Type>
Model<Type>::Model(const Model<Type>& initmod)
{
   ndim = initmod.ndim;
   deltmod = initmod.deltmod;
	
   mp = new Vector<Type>(ndim);
   mp[0] = initmod.mp[0];
	
   if (initmod.maxParam != NULL) 
   {
      maxParam	=	new Vector<Type>(ndim);
      maxParam[0] = 	(initmod.maxParam)[0];
   }
 
   else	maxParam = NULL;

   if (initmod.minParam != NULL) 
   {
      minParam	=	new Vector<Type>(ndim);
      minParam[0] = 	(initmod.minParam)[0];
   }
   else	minParam = NULL;

   tp = initmod.tp;
}

 
template<class Type>
Model<Type>::~Model()
{
     
    if (maxParam != NULL)  delete maxParam;
    if (minParam != NULL)  delete minParam;
 
    delete mp;
}

 
template<class Type> //change the way to handle boundary condition 11/17/95
Model<Type> Model<Type>::update(Type alpha, Type beta, const Vector<Type>& dmod) const
{
     
    Model<Type> newmodel(*this);
     
    Vector<Type> direction(dmod);
     
    if (direction.size() != ndim) {
	cerr << "WARNING: the length of the direction is different from dimensions." << endl;
	direction.chaSize(ndim);
    }
     
	// check to see if the new model is out of the upper bound
    Type diff=beta;
    int inout, icount=0;
     
    newmodel.mp[0] = alpha * mp[0] + diff * direction;

    if (tp != NULL)
    {
       inout = (*tp)(newmodel.mp[0]);
       if (inout != 1)
       {
	  diff += inout*diff/2;
	  icount ++;
	  
	  newmodel.mp[0] = alpha * mp[0] + diff * direction;
	  
	  inout = (*tp)(newmodel.mp[0]);
	  while (inout != 1 || icount <= 5)
	  {
	     if (icount > 10)
	     {
		newmodel.mp[0] = mp[0];
		return newmodel;
	     }
	     
	     diff += inout*diff/2;
	     icount ++;
	     
	     newmodel.mp[0] = alpha*mp[0] + diff*direction;
	     inout = (*tp)(newmodel.mp[0]);
	  }
       }
    }
       
    if ((maxParam != NULL) || (minParam != NULL))
    {
       for (int i=0; i<ndim; i++){

	     if ((diff = newmodel.mp[0][i]-maxParam[0][i])>0)
		newmodel.mp[0][i] = maxParam[0][i];
		
	     if ((diff = newmodel.mp[0][i]-minParam[0][i])<0)
		newmodel.mp[0][i] = minParam[0][i];
	  }
    }
    
       
    return newmodel;
 }

 
template<class Type>
Type Model<Type>::oneMax(int i) const { 
 
	if (maxParam != NULL) return (*maxParam)[i]; 
 
	else	{
	    cerr << "No boundary specified!"<< endl;
	    return -999;
	}
}

 
template<class Type>
Type Model<Type>::oneMin(int i) const { 
 
	if (minParam != NULL) return (*minParam)[i]; 
 
	else {
	    cerr << "No boundary specified!"<< endl;
	    return -999;
	}
}

 
template<class Type>
Type Model<Type>::range(int i) { 
 
	if (minParam != NULL && maxParam != NULL) return (maxParam[0][i]-minParam[0][i]); 
 
	else {
	    cerr << "No boundary specified!"<< endl;
	    return -999;
	}
}

template<class Type>
int operator ==(const Model<Type>& m1, const Model<Type>& m2)
{
   if (m1.mp[0] != m2.mp[0]) return 0;

   if (m1.maxParam != NULL && 
       m1.maxParam[0] != m2.maxParam[0]) return 0;
   if (m1.minParam != NULL &&
       m1.minParam[0] != m2.minParam[0]) return 0;
   if (m1.tp != NULL && m2.tp != NULL &&
       m1.tp[0] != m2.tp[0]) return 0;
   
   if (m1.deltmod != m2.deltm) return 0;
   
   return 1;
}

template<class Type>
int operator !=(const Model<Type>& m1, const Model<Type>& m2)
{
   if (m1.mp[0] != m2.mp[0]) return 1;

   if (m1.maxParam != NULL && 
       m1.maxParam[0] != m2.maxParam[0]) return 1;
   if (m1.minParam != NULL &&
       m1.minParam[0] != m2.minParam[0]) return 1;
   if (m1.tp != NULL && m2.tp != NULL &&
       m1.tp[0] != m2.tp[0]) return 1;
   
   if (m1.deltm != m2.delm) return 1;
   
   return 0;
}
 
}

#endif
