//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.
//
#ifndef MISFIT_FCN_HH
#define MISFIT_FCN_HH

#include "Vector.hh"
#include "Forward.hh"
#include "ObjFcn.hh"

//=============================================================
// Author: H. Lydia Deng
//=============================================================
//@Man:
//@Memo: A misfit objective function 
/*@Doc: 
  MisFitFcn
  is a derived class of objective function which measures
  the least-square error between the observed and calculated 
  data set. The calculated data set are passed by the Forward
  operator which needs to be defined when constructing the
  MisFitFcn class.
  
  MisFitFcn currently only takes Models either double or 
  long integer. It is mostly because I tried to avoid using
  templates, due to the guliness of template features in
  G++. Hopefully, this could be changed soon.
 
*/

namespace coool 
{
    using namespace coool;

class MisFitFcn: public ObjectiveFunction {
 private:
    Forward*		forOp;
    Vector<double>*	refData;
    Vector<double>*	errData;
    int			lp;
    int 		ndata;
    int			isUpdated;

    Vector<double> modeling(const Model<double>&);
    Vector<double> modeling(const Model<long>&);

 protected:
    //@ManMemo: 
    double 	realPerformance(const Model<long>&);
    //@ManMemo: 
    double 	realPerformance(const Model<double>&);

 public:
    //@ManMemo: 
    MisFitFcn();
    //@ManMemo: Constructor
    MisFitFcn(///dimension of the Model space
	      int n, 
	      ///pointer to a Forward operator class 
	      Forward *p, 
	      ///pointer to the data Vector
	      Vector<double>* d, 
	      ///$l_p$ norm. $l_p=2$, least-square criterian
	      int lp);
    ~MisFitFcn();
    //@ManMemo: returns lp
    int	getOrder() const {return lp;}
    //@ManMemo: returns number of iterations
    int	experiments() const {return ndata;}

    //@ManMemo: returns name of the current class
    const char* 	className() const;
    //@ManMemo: returns name of the Forward class
    const char* 	forwardName() const;

    //@ManMemo: evaluate gradient Vector for Model $m$ 
    Vector<double>*	getGradient(const Model<double>& m);
    //@ManMemo: 
    Vector<double>*	getGradient(const Model<long>&);

    //@ManMemo: evaluate lp-norm error for Model $m$
    Vector<double>	updateError(const Model<double>& m);
    //@ManMemo: 
    Vector<double>	updateError(const Model<long>& m);
    //@ManMemo: returns the observed data Vector
    Vector<double>	observedData() const{ return refData[0];}
    //@ManMemo: returns the error Vector between observed and calculated
    Vector<double>	currentError()  const{ return errData[0];}
    //@ManMemo: Forward operator applied to double vector $v$
    Vector<double>	operateOn(const Vector<double>&);
    //@ManMemo: Forward operator applied to long vector $v$
    Vector<double>	operateOn(const Vector<long>&);
    //@ManMemo: 
    double       	operateOn(int, const Vector<double>&);
    //@ManMemo: 
    double       	operateOn(int, const Vector<long>&);
    //@ManMemo: adjoint of the Forward operator operates on Vector v; mostly used when the Forward is a LinForward
    Vector<double>	adjointOperation(const Vector<double> &);

    //@ManMemo: 
    Vector<double>	getOneOperator(int);
    //@ManMemo: 
    Vector<double>	getRefData();
};
 
}

#endif
