//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.

#ifndef LIST_HH
#define LIST_HH

#ifdef __GNUC__
#pragma interface
#endif

#include <iostream.h>
#include "Vector.hh"

// .NAME List - a simple class for lists of numbers
// .LIBRARY c++ts
// .HEADER c++ utility classes
// .INCLUDE c++/List.hh
// .FILE List.cc

// .SECTION Description
// .B List
// is a simple array of floats that supports formatted i/o
// with curly braces around the list.
// It also supports convenient extension and combination of
// arrays of Types.
//=================
// Modified from Martin Smith's DList.
// H. Lydia Deng,  01/23/94
//============================

//@Man:
//@Memo: a simple class for a List.
/*@Doc:
  This class was modified after the class DList from Martin Smith.

*/

namespace coool 
{
    using namespace coool;


template<class Type>
class List {
   protected:
   //@ManMemo: add values to the List
   void addvals(const int n, const Type* dv);
   //@ManMemo: check if i is a valid index
    void ensureSubs(const int i);
  private:
    int		l;
    Type*	d;
  public:
   //@ManMemo: default constructor
    List();
   //@ManMemo: copy constructor
    List(const List<Type>& );
    ~List();
   //@ManMemo: returns the $i$th element of the List, un-writable
    const Type& operator[](const int i) const;
   //@ManMemo: returns the $i$th element of the List, writable
    Type& operator[](const int );
	
   //@ManMemo: normalize the List
    List<Type>  normalize();
   //@ManMemo: reset the List elements to be zero
    List<Type>& reset();

   //@ManMemo: returns the length of the List
    int	length();
   //@ManMemo: returns the last element of the List
    Type last();
   //@ManMemo: returns the first element of the List
    Type first();

   //@ManMemo: copy the current List to a Vector
    Vector<Type> ToVect();
   //@ManMemo: copy the first $m$ elements to a Vector 
    Vector<Type> ToVect(int m);

   //@ManMemo: append an element to the current List
    List<Type>& operator+=(const Type );
   //@ManMemo: append a List $list$ to the current List
    List<Type>& operator+=(const List<Type>& list);
   //@ManMemo: assign the List $list$ to the current List
    List<Type>& operator=(const List<Type>& );

   //@ManMemo: check if the List $list$ is the same as the current list
    int operator==(const List<Type>& list );
   //@ManMemo: check if $c$ is one of the elements of the current list. Returns the position index if is in, otherwise, returns -1.
    int In(Type c);
    

   //@ManMemo: a friend function, assign the elements of $v$ to a List
template < class T >
friend  List<T> ToList (Vector<T>& v);
   //@ManMemo: a friend function, stream output
template < class T >
friend	ostream& operator << (ostream& , const List<T>& );
   //@ManMemo: a friend function, stream input
template < class T >
friend  istream& operator >> (istream& , List<T>& );
};


#ifdef __GNUC__
#pragma implementation
#endif

//static int isws(int c)
//{
//    return ((c == ' ') || (c == '\t') || (c == '\n'));
//}

static const int b_inc = 1024;

template<class Type>
List<Type>::List() : l(0), d(0) {};
template<class Type>
List<Type>::List(const List<Type>& x)
{
    l = x.l;
    d = new Type[l];
    for(int i = 0; i < l; i++)
	d[i] = x[i];
}
template<class Type>
List<Type>::~List() {delete [] d; d = NULL; }
template<class Type>
const Type& List<Type>::operator[](const int i) const {return d[i];};
template<class Type>
Type& List<Type>::operator[](const int i) {ensureSubs(i); return d[i];};
	
template<class Type>
int List<Type>::length() {return l;}
template<class Type>
Type List<Type>::last()  {return d[l-1];}
template<class Type>
Type List<Type>::first() {return d[0];}

template<class Type>
void List<Type>::addvals(const int n, const Type* dv)
{
    Type* ds = d;
    d = new Type[l + n];
    int i;
    for(i = 0; i < l; i++)
	d[i] = ds[i];
    delete [] ds;
    for(i = l; i < l + n; i++)
	d[i] = dv[i - l];
    l += n;
}

template<class Type>
void List<Type>::ensureSubs(const int newi)
{
    if(newi < l) return;
    Type* nd = new Type[newi + 1];
    int i;
    for(i = 0; i < l; i++)
	nd[i] = d[i];
    for(; i <= newi; i++)
	nd[i] = 0;
    delete [] d;
    d = nd;
    l = newi + 1;
}

template<class Type>
List<Type>& List<Type>::reset()
{
   delete [] d;
   if (d != NULL) d = NULL;
   
   l = 0;
}

template<class Type>
List<Type> List<Type>::normalize() {
    List<Type> v(*this);
    Type scale = d[0];
    if (d[0] == 0) {
	cerr << "Zero first element, cannot be normed! \n";
	return v;
    }
    else {
	for (int i=0; i<l; i++)
	    v.d[i] /= scale;
	return v;
    }
}

// Description:
// Add a single value to the list.
template<class Type>
List<Type>& List<Type>::operator+=(const Type d)
{
    addvals(1, &d);
    return *this;
}
// Description:
// Add the contents of another List to the list.
template<class Type>
List<Type>& List<Type>::operator+=(const List<Type>& d)
{
    addvals(d.l, d.d);
    return *this;
}

template<class Type>
List<Type>& List<Type>::operator=(const List<Type>& e)
{
    if(this == &e) return *this;
    delete [] d;
    l = e.l;
    d = new Type[l];
    for(int i = 0; i < l; i++)
	d[i] = e[i];
    return *this;
}

//check to see if c is in the list, return the position, otherwise return -1
template<class Type>
int List<Type>::In(Type c)
{
   for (int i=0; i<l; i++)
   {
      if (d[i] == c) return c;
   }
}

// compare two List<Type>, return 0 or 1
template<class Type>
int List<Type>::operator==(const List<Type>& a)
{
   if (l != a.l) return 0;
   for (int i=0; i<l; i++)
   {
      if (d[i] != a[i]) return 0;
      continue;
   }
   
   return 1;
}

template<class Type>
Vector<Type> List<Type>::ToVect()
{
    Vector<Type> v(l);
    for (int i=0; i<l; i++) v[i] = d[i];
    return v;
}

template<class Type>
Vector<Type> List<Type>::ToVect(int m)
{
    Vector<Type> v(m);
    int i;
    for (i=0; i<m && i<l; i++) v[i] = d[i];
    if (m>l) 
	for (i = l; i<m; i++) v[i] = 0;
    return v;
}

template<class Type>
List<Type> ToList(Vector<Type>& v)
{
    List<Type> list;
    for (int i=0; i<v.size(); i++) list += v[i];
    return list;
}

// Description:
// Write a formatted brace-wrapped copy of the list.
template<class Type>
ostream& operator <<(ostream& os, const List<Type>& d)
{
    os << "{";
    os << d.d[0];
    for(int i = 1; i < d.l; i++) {
	if(i%5 == 0) os << "\n";
	else os << " ";
	os << d.d[i];
    }
    os << "} \n";
    return os;
}
// Description:
// Read a formatted nrace-wrapped list into memory.
template<class Type>
istream& operator >>(istream& is, List<Type>& d)
{
    char c;
    int used, available;
    Type* dp;

    used = 0;
    available = b_inc;
    dp = new Type[available];
    while(is && (c = is.get()) != '{')
	;
    while(is >> c) {
	if(c == '}') break;
	is.putback(c);
	if(!(is >> dp[used++])) {
	    cerr << "bad List input  used = " << used << endl;
	    exit(1);
	}
	if(used >= available) {
	    available += b_inc;
	    Type* nd = dp;
	    dp = new Type[available];
	    for(int i = 0; i < used; i++)
		dp[i] = nd[i];
	    delete nd;
	}
    }
    d.l = used;
    delete d.d;
    d.d = new Type[used];
    for(int j = 0; j < used; j++)
	d.d[j] = dp[j];
    delete dp;
    return is;
}
 
}

#endif
