//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.
//
//=============================
// Definition of the LineSearchOptima class
// General information on the convergence of the optimization modules
// author:  H. Lydia Deng
//=============================

// .NAME LineSearchOptima class
// .LIBRARY Base
// .HEADER Reports on the optimization
// .INCLUDE defs.hh
// .FILE CG.hh

// .SECTION Description
// .B LineSearchOptima()
// All optimization classes are inherited from the LineSearchOptima class. This class
// ables the user to retrieve important information from the convergence of the 
// optimization algorithms. 
// 
// .SECTION Description
// Public Operations
// Constructors: 
//		LineSearchOptima(LineSearch* search)
// 		Here:
//			search: Defines the search direction.
// Methods:
//		virtual Model<double> optimizer(Model<double>&model0) 
//		Here:
//			model0:  Initial model for the optimization procedure.
//
//			The optimum model is returned by the function.
//
//		const char*	objName():	Returns the name of the objective function
//		int num		Iterations():	Returns the number of iterations
//		int	currentNumSearches() :	Returns the number of unidimensional searches
//								done so far
//		int	oneNumSearches(int i):	Returns the i-th search direction
//		List<int> allNumSearches(): 	Returns all search directions used so far
		
#ifndef LINE_SEARCH_OPTIMA_HH
#define LINE_SEARCH_OPTIMA_HH

#include "NonQuaOptima.hh"
#include "LineSearch.hh"

//@Man:
//@Memo: an abstract class for optimization algorithms using line searched
/*@Doc: LineSearchOptima
  All optimization classes are inherited from the 
  LineSearchOptima class. This class ables the user to 
  retrieve important information from the convergence
  of the optimization algorithms. 

*/

namespace coool 
{
    using namespace coool;

class LineSearchOptima : public NonQuadraticOptima {
  protected:
    int			iterNum;
    LineSearch*		ls;
 public:
    //@ManMemo: a constructor with a pointer to the LineSearch object
    LineSearchOptima(LineSearch* ls);
    //@ManMemo: a constructor with a pointer to the LineSearch object
    LineSearchOptima(LineSearch*, int verbose);
    virtual ~LineSearchOptima();

    //@ManMemo: 
    virtual Model<double> optimizer(Model<double>&) = 0;
    //@ManMemo: 
    virtual Model<long> optimizer(Model<long>&) = 0;

    //@ManMemo:  the name of the objective function
    const char*	objName() 	{ return ls->objName();}
    //@ManMemo:  the number of iterations
    int numIterations()		{ return iterNum;}

    //@ManMemo: the number of uni-dimensional searches done so far
    int	currentNumSearches() 	{ return ls->searchIterations();}
    //@ManMemo: the number of iterations for the i-th search
    int	oneNumSearches(int i) 	{ return  (ls->allSearchIterations())[i];}
    //@ManMemo: the number of iterations for all search so far
    List<int> allNumSearches()	{ return ls->allSearchIterations();}
};
 
}

#endif

