//============================================================
// COOOL           version 1.1           ---     Nov,  1995
//   Center for Wave Phenomena, Colorado School of Mines
//============================================================
//
//   This code is part of a preliminary release of COOOL (CWP
// Object-Oriented Optimization Library) and associated class 
// libraries. 
//
// The COOOL library is a free software. You can do anything you want
// with it including make a fortune.  However, neither the authors,
// the Center for Wave Phenomena, nor anyone else you can think of
// makes any guarantees about anything in this package or any aspect
// of its functionality.
//
// Since you've got the source code you can also modify the
// library to suit your own purposes. We would appreciate it 
// if the headers that identify the authors are kept in the 
// source code.
//
//======================================================================
// Definition of the cubic line search class
// Armijo and Goldstein's line search algorithm
// author:  Doug Hart, Adapted to the COOOL library by Wenceslau Gouveia
// Modified to fit into new classes.  H. Lydia Deng, 02/21/94, 03/15/94
//========================================================================

// .NAME CubicLineSearch class
// .LIBRARY Base
// .HEADER Optimization Algorithms
// .INCLUDE defs.hh
// .FILE CubicLineSearch.hh

// .SECTION Description
// .B CubicLineSearch()
// This class implements the efficient line search procedure described in Dennis and
// Schbabel's book entitled "Numerical Methods for Unconstrained and Nonlinear
// Equations. The objective is to perform a unidimensional search for a minimum point
// along a specified direction in a multidimensional space.
// 
// .SECTION Description
// Public Operations
// Constructors: 
//		CubicLineSearch(ObjectiveFunction*f, int iter, double delta)
// 		Here:
//			f: Defines a pointer to the objective function.
//			iter: Maximum number of iterations
//		 	delta: This parameter is not used by the line search itself. Rather it is 
//				  used in the numerical computation of the derivatives using 
//				  centered differences. For example the derivative of f(x) at the point x0 
//				  would be given by (f(x0 - delta) - f(x0 + delta) / 2 * delta)
// Methods:
//		Model<> search(Model<double>&model0, Vector<double>& direction, 
//					   double descent, double lambda)
//		Here:
//			model0:  Initial model for the line search
//			direction: Search direction
//			descent: dor product of search direction and gradient 
//			lambda: This parameter controls the accuraccy of the line search. Lambda =
//				      .25 is a good choice.
//
//			The minimizer along the search direction is returned by the function.
//
// .SECTION Caveats
// This procedure seems to be fairly robust. It has worked for a fairly broad class of
// problems from optimization of standard test functons in optimization theory and
// to hard geophysical problems as stacking power optimization and amplitude 
// seismogram inversion

#ifndef CUBIC_LINE_SEARCH_HH
#define CUBIC_LINE_SEARCH_HH

#include "LineSearch.hh"

//@Man:
//@Memo: CubicLineSearch class
/*@Doc:
  CubicLineSearch() class implements the efficient line search procedure
  described in Dennis and Schbabel's book entitled "Numerical
  Methods for Unconstrained and Nonlinear Equations. The 
  objective is to perform a unidimensional search for a minimum
  point along a specified direction in a multidimensional 
  space.

  This procedure seems to be fairly 
  robust. It has worked for a fairly broad class of 
  problems from optimization of standard test functons 
  in optimization theory and to hard geophysical problems as 
  stacking power optimization and amplitude seismogram inversion.
  But we never know ...
*/

namespace coool 
{
    using namespace coool;

class CubicLineSearch : public LineSearch
{
   
   public:
   //@ManMemo: a constructor with the default delta
   CubicLineSearch(ObjectiveFunction* f, int iter);
   //@ManMemo: a constructor with the specified delta
   CubicLineSearch(///pointer to the objective function
		   ObjectiveFunction* f, 
		   ///maximum number of iterations
		   int iter, 
		   ///pointer to the vector of step size for FD
		   Vector<double>* delta);
   /*@Doc: The parameter $delta$ is not used by the line search 
         itself. Rather it is used in the numerical computation
         of the derivatives using centered differences. For
         example the derivative of f(x) at the point x0 would be
         given by 
         \[f(x0 - delta) - f(x0 + delta) / 2 * delta\]
	 */
   ~CubicLineSearch();

   //@ManMemo: search for the minimum model along a 1-D direction
    Model<double> search(///initial model for the line search
			 Model<double>& m0, 
			 ///search direction
			 Vector<double>& direction, 
			 ///dot product of search direction and gradient
			 double descent, 
			 ///a parameter
			 double lambda);
   /*@Doc: The parameter $lambda$ controls the accuraccy of the  
     line search. $lambda = .25$ is a good choice.
    The minimizer along the search direction is returned 
    by the function.  */
   //@ManMemo: search for the minimum model along a 1-D direction
    Model<long> search(Model<long>&, Vector<double>&, double, double);
};
 
}

#endif

