// *****************************************************************
// UGA        version        1.0        --       November       1992
// *****************************************************************

// This  code is part of a preliminary release of UGA (uni-processor
// genetic  algorithm)  and  associated  class  libraries.   It  was
// developed  on  Sun Sparcstations using AT&T's C++ compiler.  Ver-
// sions which will compile using the GNU C++  compiler  on  various
// architectures are in preparation.

// This software was developed and tested at  various  institutions,
// including  Amoco  Production  Company's  Tulsa  Research  Center,
// Princeton University, The University of Paris, and Los Alamos Na-
// tional Laboratories.  The authors wish to express their gratitude
// to these institutions for encouraging this work.

// You get this code for free.

// You can do anything you want with it, including make a fortune.

// Neither the authors, Amoco, nor anyone else you can imagine makes
// any  promises or guarantees about anything in this package or any
// aspect of its functionality.

// Authors:  Martin L. Smith, Terri L. Fischer

// For    further    information    contact    John    Scales     at
// jscales@dix.mines.colorado.edu,  telephone:   303-273-3408,  fax:
// 303-273-3478.  

// This may look like C code, but it is really -*- C++ -*-

#include "AString.hh"
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>

namespace coool 
{
    using namespace coool;


static const int	default_chunk_size = 8;
static const char*	myNameIs = "AString";
static const char*	empty_string = "";

// private methods
void AString::bigEnough(const int size)
{
    if(size < 0){
	cerr << className() << "bigEnough: arg size = " << size
	    << " is less than zero." << endl;
	exit(1);
    	}
    
    if(size >= 0 && mem < (size + 1)){
	char* tmp = contents;
	
	// have to do division before multiplication because
	// we're doing integer arithmetic and we want the division
	// step to truncate the answer
	
	int num_chunks = ((size + 1) + (chunksize() - 1)) / chunksize();
	mem = num_chunks * chunksize();
	contents = new char[mem];
	contents[0] = '\0';
	
	if(tmp != NULL)
	    strcpy(contents, tmp);
	delete tmp;
    }
}

void AString::initVars() {
    length(0);
    mem = 0;
    contents = NULL;
    chunksize(default_chunk_size);
}

int AString::inRange(const long i) const {
    return(i >= 0 && i < length());
}

void AString::length(const int new_length){
    len = new_length >= 0 ? new_length : -1;
}

// protected methods

// Description:
// Converts a single char into an AString.  Be warned that it will
// also convert numbers (\fIe.g.\fP 7) into an AString as well.
AString::AString(const char c){
    initVars();
    copy(c);
}

// Description:
// Called by the equality operators. Returns an integer greater than,
// equal to, or less than 0, according as the current AString is
// lexicographically greater than, equal to, or less than the AString
// argument \fIas\fP.
int AString::compare(const AString& as) const{
    if(this == &as)		// they're the same AString
	return 0;
    int asIsEmpty	= as.empty();
    int thisIsEmpty	= empty();
    
    if(asIsEmpty && thisIsEmpty) // both are empty
	return 0;		 // contents == as.contents
    else if(asIsEmpty || thisIsEmpty) // only one is empty
	return asIsEmpty ? 1 : -1;
    else			      // neither are empty
	return strcmp(contents, as.contents);
}

// Description:
// Called by the equality operators. Returns an integer greater than,
// equal to, or less than 0, according as the current AString is
// lexicographically greater than, equal to, or less than the const char*
// argument \fIs\fP.
int AString::compare(const char* s) const {
    if(contents == s)
	return 0;		// they're the same pointer
    
    int sIsEmpty = (s == NULL ? True : (strlen(s) == 0 ? TRUE : FALSE));
    int thisIsEmpty = empty();

    if(sIsEmpty && thisIsEmpty) // both are empty
	return 0;		 // s == as.s
    else if(sIsEmpty || thisIsEmpty) // only one is empty
	// sIsEmpty ? contents > s : contents < s
	return sIsEmpty ? 1 : -1;
    else			     // neither are empty
	return strcmp(contents, s);
}

// Description:
// Appends the contents of the AString argument \fIas\fP to the end of
// the current AString.
void AString::concat(const AString& as){
    if(!as.empty()){
	int this_length	= length();
	int as_length	= as.length();
	int m		= this_length + as_length;
	bigEnough(m);		// make bigEnough before assigning pointer
	char* start_pointer = &contents[this_length];
	length(m);

	// do strncpy (instead of strcpy) in case as is same AString as this
	strncpy(start_pointer, as.contents, as_length);
	contents[m] = '\0';
    }
}

// Description:
// Appends the contents of the const char* argument \fIs\fP to the end
// of the current AString.
void AString::concat(const char* s){
    if(s != NULL){
	int this_length	= length();
	int s_length	= strlen(s);
	int m		= this_length + s_length;
	bigEnough(m);		// make bigEnough before assigning pointer
	char* start_pointer = &contents[this_length];
	length(m);
	strncpy(start_pointer, s, s_length);
	contents[m] = '\0';
    }
}
	
// Description:
// Appends the const char argument \fIc\fP to the end of the current
// AString.
void AString::concat(const char& c){
    int m = length();
    bigEnough(m + 1);
    contents[m++] = c;
    contents[m] = '\0';
    length(m);
}

// Description:
// Replaces the old contents of the current AString with the contents
// of the AString argument \fIas\fP.
void AString::copy(const AString& as){
    if(this != &as)		// if they're not the same AString
	if(!as.empty()){
	    length(as.length());
	    bigEnough(length());
	    strcpy(contents, as.contents);
	}
}

// Description:
// Replaces the old contents of the current AString with the contents
// of the const char* argument \fIcp\fP.
void AString::copy(const char* cp){
    if(contents != cp)		// if they're not the same pointer
	if(cp != NULL){
	    length(strlen(cp));
	    bigEnough(length());
	    strcpy(contents, cp);
	}
}

// Description:
// Replaces the old contents of the current AString with the const
// char argument \fIc\fP.
void AString::copy(const char& c){
    length(1);
    bigEnough(length());
    contents[0] = c;
    contents[1] = '\0';
}

// Description:
// Returns 1 if the variable \fIcontents\fP is equal to NULL and 0
// otherwise.
int AString::isNull() const {
    return(contents == NULL);
}

// public methods

// Description:
// Default constructor.
AString::AString(){
    initVars();
}

// Description:
// Converts a const char* into an AString.
AString::AString(const char* cp){
    initVars();
    copy(cp);
}

// copy constructor
AString::AString(const AString& as){
    initVars();
    copy(as);
}

AString::~AString(){
    delete contents;
}

// Description:
// Returns the size of memory allocation hunks.
int AString::chunksize() const {
    return(chunk_size > 0 ? chunk_size : default_chunk_size);
}

// Description:
// Changes the size of memory allocation hunks.
void AString::chunksize(const int new_size){
    if(new_size > 0)
	chunk_size = new_size;
}

// Description:
// Returns 1 if the string is empty, 0 otherwise.
int AString::empty() const {
    return (length() == 0);
}

// Description:
// Returns the number of characters in the string.
// Always non-negative.
int AString::length() const{
    if(len == -1){
	AString* that = (AString*)(this); // cast away const-ness
	that->length(isNull() ? 0 : strlen(contents));
    }
    return(len);
}

// Description:
// Assigns one AString to another. The user doesn't have to worry
// about there being enough space to hold the new string.
AString& AString::operator=(const AString& as){
    if(this != &as){				// check for s = s
	delete contents;			// zap the current guy
	initVars();
	chunksize(as.chunksize());
	copy(as);
    }
    return *this;
}

// Description:
// Assigns a char* to an AString. The user doesn't have to worry
// about there being enough space to hold the new string.
AString& AString::operator=(const char* s){
    delete contents;			// zap the current guy
    initVars();
    copy(s);
    return *this;
}

// Description:
// Returns the (read-only) value of the \fIi\fPth character,
// counting from zero.
const char& AString::operator[](const long i) const{
    if(!inRange(i)){
	cerr << className() << "[" << i << "], i is out of range" << endl;
	exit(1);
    }
    return(contents[i]);
}

// Description:
// Add an AString to the end of the current AString.
AString& AString::operator+=(const AString& as){
    concat(as);
    return *this;
}

// Description:
// Add a char* to the end of the current AString.
AString& AString::operator+=(const char* s){
    concat(s);
    return *this;
}

// Description:
// Return a (read-only) character pointer to the current AString.
AString::operator const char*() const{
    if(isNull())
	return empty_string;
    else
	return(contents);
}

const char* AString::className() const{
    return(myNameIs);
}

// Description:
// Return a lower-case'd version of the current AString.
AString AString::asLowerCase() const
{
    AString t = *this;
    for(int i = 0; i < len; i++)
	t.contents[i] = tolower(t.contents[i]);
    return t;
}

// friend functions

// Description:
// Return an AString formed by joining two AStrings.
AString operator+(const AString& as, const AString& bs){
    AString tmp;
    tmp.bigEnough(as.length() + bs.length());
    tmp.concat(as);
    tmp.concat(bs);
    return tmp;
}

// Description:
// Return an AString formed by joining an AString and a const char*.
AString operator+(const AString& as, const char* s){
    AString tmp;
    tmp.bigEnough(as.length() + (s == NULL ? 0 : strlen(s)));
    tmp.concat(as);
    tmp.concat(s);
    return tmp;
}

// Description:
// Return an AString formed by joining a const char* and an AString.
AString operator+(const char* s, const AString& as){
    AString tmp;
    tmp.bigEnough((s == NULL ? 0 : strlen(s)) + as.length());
    tmp.concat(s);
    tmp.concat(as);
    return tmp;
}

// equality
int operator==(const AString& as, const AString& bs){
    return(as.compare(bs) == 0);
}

int operator==(const AString& as, const char* s){
    return(as.compare(s) == 0);
}

int operator==(const char* s, const AString& as){
    return(as.compare(s) == 0);
}

// inequality
int operator!=(const AString& as, const AString& bs){
    return(as.compare(bs) != 0);
}

int operator!=(const AString& as, const char* s){
    return(as.compare(s) != 0);
}

int operator!=(const char* s, const AString& as){
    return(as.compare(s) != 0);
}

// greater than
int operator>(const AString& as, const AString& bs){
    return(as.compare(bs) > 0);
}

int operator>(const AString& as, const char* s){
    return(as.compare(s) > 0);
}

int operator>(const char* s, const AString& as){
    return(as.compare(s) < 0);	// reverse the order 
}

// greater than or equal to
int operator>=(const AString& as, const AString& bs){
    return(as.compare(bs) >= 0);
}

int operator>=(const AString& as, const char* s){
    return(as.compare(s) >= 0);
}

int operator>=(const char* s, const AString& as){
    return(as.compare(s) <= 0);	// reverse the order
}

// less than
int operator<(const AString& as, const AString& bs){
    return(as.compare(bs) < 0);
}

int operator<(const AString& as, const char* s){
    return(as.compare(s) < 0);
}

int operator<(const char* s, const AString& as){
    return(as.compare(s) > 0);	// reverse the order
}

// less than or equal to
int operator<=(const AString& as, const AString& bs){
    return(as.compare(bs) <= 0);
}

int operator<=(const AString& as, const char* s){
    return(as.compare(s) <= 0);
}

int operator<=(const char* s, const AString& as){
    return(as.compare(s) >= 0);	// reverse the order
}

ostream& operator<<(ostream& fp, const AString& as)
{
   fp << as.contents;

   return fp;
}

istream& operator>>(istream& fp, AString& as)
{
   fp >> as.contents;

   return fp;
}


// functions

#include <stdarg.h>		// to get varargs

// Description:
// Format, according to \fIfmt\fP the arguments to ASFormat() and
// into an AString.  Uses printf(3) style arguments.
AString ASFormat(char* fmt, ...){
    va_list	arg_list;
    char	s[1024];
    
    va_start(arg_list, fmt);
    (void) vsprintf(s, fmt, arg_list);
    va_end(arg_list);

    AString as = s;
    return as;
}

// Description:
// Convert an unsigned char into an AString.
AString toAS(unsigned char uc){
    return ASFormat("%c", uc);
}

// Description:
// Convert a char into an AString.
AString toAS(char c){
    return ASFormat("%c", c);
}

// Description:
// Convert an unsigned short into its AString representation.
AString toAS(unsigned short um){
    return ASFormat("%hu", um);
}

// Description:
// Convert a short into its AString representation.
AString toAS(short m){
    return ASFormat("%hd", m);
}

// Description:
// Convert a unsigned int into its AString representation.
AString toAS(unsigned int um){
    return ASFormat("%u", um);
}

// Description:
// Convert a int into its AString representation.
AString toAS(int m){
    return ASFormat("%d", m);
}

// Description:
// Convert a long into its AString representation.
AString toAS(unsigned long um){
    return ASFormat("%lu", um);
}

// Description:
// Convert a long into its AString representation.
AString toAS(long m){
    return ASFormat("%d", m);
}

// Description:
// Convert a float or float into its AString representation.
AString toAS(double d){
    return ASFormat("%g", d);
}
 
}
