## @package index
#  Main Module of the server-side application.
#
#  Handles communication between FLASK Server and Javascript FrontEnd

# own modules
import clustering
import readData
import wordCloudHandler

# python modules
import logging
import time
import json
import numpy as np

# flask
from flask import Flask, render_template, request
from flask import send_file, send_from_directory

## the flask application
app = Flask(__name__)

## contains the original data from the disk
data = {}
## edgeCollections of data
edgeCollections = {} 
## List of all nodes
nodeList = []
## normalized nodes features for clustering
normalizedNodes = {}  
## the complete edge matrices for the edge collections after clustering
edgeMatrices = {} 
## supernode IDs from the user selected supernodes
selectedSuperNodeIDs = [] 

## current cluster features
clusterFeatures = []
## result of the clustering
clusterResult = {}
## indicates if we need to do a new clustering
reCluster = True
## the number of clusters that needs to be computed
numberOfClusters = 0
## the current attribute fro the word cloud
wordCloudAttribute = "cast"
## indicates if there is a clustering process running at this moment
stableClustering = False

# set a 'SECRET_KEY' to enable the Flask session cookies
app.config['SECRET_KEY'] = 'bestsecretkeyevermadeinhumanhistory'
app.config['SEND_FILE_MAX_AGE_DEFAULT'] = 0


@app.route("/index")
## Reads the initial data, does some preprocessing and provides the html for the browser.
def main():
    global data
    global normalizedNodes
    global edgeCollections
    global nodeList

    data = readData.readAllData()
    nodeList = [None] * len(data["nodes"])
    normalizedNodes = readData.normalizeNodeValues(data, nodeList)
    edgeCollections = readData.buildEdgeCollections(data)
    return render_template('index.html')


@app.route('/getData')
## provides the client with the initial data
def getData():
    print("started data send")
    global data
    if data is None:
        return "Data is not available."
    else:
        print("send data to JS")
        return json.dumps(data)



@app.route('/setClusterFeatures', methods=['POST'])
## recieves the cluster features and number of clusters from the servers
#
# if one of the 2 parameters differs from the last set ones, reCluster is set true
def setClusterFeatures():
    global clusterFeatures
    global numberOfClusters
    global reCluster
    newClusterFeatures = request.form.getlist('features[]')
    newNumberOfCluster = int(request.form.get('noClusters'))

    # we only need to do another clustering, if there are new clsuter feature or the nubmer of clusters changed
    if clusterFeatures != newClusterFeatures or numberOfClusters != newNumberOfCluster:
        reCluster = True
    else:
        reCluster = False

    clusterFeatures = newClusterFeatures
    numberOfClusters = newNumberOfCluster

    print("recieved clusterFeatures")
    return "recieved clusterFeatures"


@app.route('/getSupernodes')
## perfroms the clustering and sends the results to the client
def getSupernodes():
    print("started clustering")
    global data
    global clusterFeatures
    global numberOfClusters
    global edgeCollections
    global edgeMatrices
    global stableClustering
    global clusterResult
    stableClustering = False

    if reCluster:
        edgeMatrices = {}
        lowDetailSuperNodeEdgeMatrices = {}
        clusterResult = clustering.clusterData(data, normalizedNodes, clusterFeatures, numberOfClusters, "kmeans",
                                               edgeMatrices, lowDetailSuperNodeEdgeMatrices, edgeCollections)
    if clusterResult is None:
        return "could not compute supernodes"
    else:
        stableClustering = True
        print("send clusters to JS")
        return json.dumps(clusterResult)


@app.route('/setSelectedSupernodes', methods=['POST'])
## recieves the IDs of the selected supernodes from the client
def setSelectedSupernodes():
    global selectedSuperNodeIDs
    selectedSuperNodeIDs = request.form.getlist('selectedSNodes[]')
    print("recieved selectedSuperNodes")
    return "recieved selectedSuperNodes"


@app.route('/getEdgeMatricesOfSelectedSupernodes')
## provides the client with the edge matrices for the selected supernodes
def getEdgeMatricesOfSelectedSupernodes():
    print("started edgeMatrices send")

    global edgeMatrices
    global edgeCollections
    global stableClustering
    global selectedSuperNodeIDs

    selectedEdgeMatrices = None
    if stableClustering:
        selectedEdgeMatrices = clustering.getEdgeMatrixSubSets(
            edgeMatrices, edgeCollections, selectedSuperNodeIDs)

    if selectedEdgeMatrices is None:
        return "server could not send edgeMatrices, because clustering is not finished!"
    else:
        print("send edgeMatrices to JS")
        return json.dumps(selectedEdgeMatrices)


@app.route('/setWordCloudAttribute', methods=['POST'])
## revieves the word cloud attribute from the server
def setWordCloudAttribute():
    global wordCloudAttribute

    wordCloudAttribute = request.form.get('wCAttribute')

    print("recieved wordCloudAttribute")
    return "recieved wordCloudAttribute"


@app.route('/getWordCloudsOfSelectedSupernodes')
## computes the word clouds and saves them in a local folder
def getWordCloudsOfSelectedSupernodes():
    print("started wordcloud generation")

    global stableClustering
    global clusterResult
    global selectedSuperNodeIDs
    global wordCloudAttribute
    global nodeList

    if stableClustering:
        selectedSuperNodes = []

        for supernode in clusterResult["supernodes"]:
            if str(supernode["id"]) in selectedSuperNodeIDs:
                selectedSuperNodes.append(supernode)

        wordCloudHandler.generateWordClouds(
            nodeList, selectedSuperNodes, wordCloudAttribute)

    if selectedSuperNodeIDs is None:
        return "server could not send wordCloudIDs, because clustering is not finished!"
    else:
        print("send wordCloudIDs to JS")
        return json.dumps(selectedSuperNodeIDs)


@app.route('/get_wordCloud/<path:path>')
## provides the client with the word cloud image
def get_image(path):

    return send_from_directory("data/wordCloud/", path, mimetype='image/png')


if __name__ == '__main__':
    app.run()