﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using SharpDX.Direct3D11;
using SharpDX.D3DCompiler;
using SharpDX;
using SharpDX.DXGI;

using Device = SharpDX.Direct3D11.Device;
using System.Windows.Forms;

namespace VoluRen
{
    /// <summary>
    /// Singleton
    /// Management-Klasse verwaltet das Laden von Volumes und die Zuordnung der passenden
    /// Transferfunktion
    /// </summary>
    public class VolumeManagement
    {
        private static VolumeManagement _instance;

        private ShaderResourceView _filteredGradientTexSrv;

        private List<TransferFunction> _transferFunctions;

        private ShaderResourceView _transferfunctionSrv;

        private float _rotateX;
        private float _rotateY;

        private VolumeManagement() 
        {
            _filteredGradientTexSrv = null;
            _transferfunctionSrv = null;
            _rotateX = 0.0f;
            _rotateY = 0.0f;

            _transferFunctions = new List<VoluRen.TransferFunction>();

            #region aneurism volume
            List<TransferFunction.ControlPoint> col = new List<TransferFunction.ControlPoint>(){
                new TransferFunction.ControlPoint(0,new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(28, new Vector4(1f, 0f, 0f, 1f)),
                new TransferFunction.ControlPoint(255, new Vector4(1f, 0f, 0f, 1f)),
	        };

            _transferFunctions.Add(new VoluRen.TransferFunction(col, col));
            #endregion

            #region scorpion volume
            col = new List<TransferFunction.ControlPoint>(){
		        new TransferFunction.ControlPoint(0,new Vector4(0f, 0f, 0f, 1f)),
		        new TransferFunction.ControlPoint(81,new Vector4(0.91f, 0.7f, 0.61f, 1.0f)),	
		        new TransferFunction.ControlPoint(255, new Vector4(0f, 1.0f, 0.0f, 1.0f)),
	        };

            List<TransferFunction.ControlPoint> alpha = new List<TransferFunction.ControlPoint>(){
		        new TransferFunction.ControlPoint(0,new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(100,new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(110,new Vector4(0f, 0f, 0f, 0.5f)),
                new TransferFunction.ControlPoint(111,new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(129,new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(130,new Vector4(0f, 0f, 0f, 1f)),
		        new TransferFunction.ControlPoint(255, new Vector4(0f, 0f, 0f, 1f)),
	        };

            _transferFunctions.Add(new VoluRen.TransferFunction(col, alpha));
            #endregion

            #region skull volume
            col = new List<TransferFunction.ControlPoint>(){
                new TransferFunction.ControlPoint(0,new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(50, new Vector4(0f, 0f, 0f, 0f)),
                new TransferFunction.ControlPoint(51, new Vector4(0f, 0f, 1f, 0.8f)),
                new TransferFunction.ControlPoint(255, new Vector4(0f, 0f, 1f, 1f)),
	        };

            _transferFunctions.Add(new VoluRen.TransferFunction(col, col));
            #endregion

            #region vismale volume
            col = new List<TransferFunction.ControlPoint>(){
		        new TransferFunction.ControlPoint(0,new Vector4(0.91f, 0.7f, 0.61f, 1.0f)),	
		        new TransferFunction.ControlPoint(80,new Vector4(0.91f, 0.7f, 0.61f, 1.0f)),	
		        new TransferFunction.ControlPoint(82,new Vector4(1.0f,   1.0f,  1.0f, 1.0f)),	
		        new TransferFunction.ControlPoint(255, new Vector4(1.0f,   1.0f,  1.0f, 1.0f)),	
	        };

            alpha = new List<TransferFunction.ControlPoint>(){
		        new TransferFunction.ControlPoint(0,new Vector4(0f, 0f, 0f, 0f)),
	            new TransferFunction.ControlPoint(50,new Vector4(0f, 0f, 0f, 0f)),
		        new TransferFunction.ControlPoint(55,new Vector4(0f, 0f, 0f, 0.8f)),	
		        new TransferFunction.ControlPoint(60,new Vector4(0f, 0f, 0f, 0.2f)),
                new TransferFunction.ControlPoint(80,new Vector4(0f, 0f, 0f, 0.05f)),
                new TransferFunction.ControlPoint(82,new Vector4(0f, 0f, 0f, 0.9f)),
		        new TransferFunction.ControlPoint(255, new Vector4(0f, 0f, 0f, 1f)),
	        };

            _transferFunctions.Add(new VoluRen.TransferFunction(col, alpha));
            #endregion

            ModelMatrix = Matrix.Identity;
            InverseModelMatrix = Matrix.Identity;
        }

        /// <summary>
        /// Liefert die einzige Instanz der Klasse
        /// </summary>
        public static VolumeManagement Instance
        {
            get 
            {
                if (_instance == null)
                {
                    _instance = new VolumeManagement();
                }
                return _instance;
            }
        }

        /// <summary>
        /// Wird beim Laden von Volumensdaten aufgerufen
        /// Lädt ein byte-Array aus der über Meta-Daten identifizierten Datei
        /// Übergibt das byte-Array an einen ComputeShader, welcher die Gradienten (CentralDifference) berechnet
        /// Die errechneten Gradienten werden an einen ComputeShader übergeben, welcher diese mit einem 3x3x3 Gauss-Filter filtert
        /// Je nach Index in den Meta-Daten wird die zugehörige Transferfunktion gesetzt
        /// </summary>
        /// <param name="meta">Volumensmetadaten</param>
        /// <param name="device">D3D11 Device</param>
        public void ProcessVolume(VolumeMetaData meta, Device device)
        {
            ShaderFlags flags = ShaderFlags.EnableStrictness;
#if DEBUG
            flags |= ShaderFlags.Debug;
#endif

            var blob = ShaderBytecode.CompileFromFile("resources\\shaders\\gradient.hlsl", "ComputeGradient", "cs_5_0", flags, EffectFlags.None);
            ComputeShader gradientCShader = new ComputeShader(device, blob);

            blob = ShaderBytecode.CompileFromFile("resources\\shaders\\filter.hlsl", "Filter", "cs_5_0", flags, EffectFlags.None);
            ComputeShader filterCShader = new ComputeShader(device, blob);

            //Load Volume from File
            byte[] vol = File.ReadAllBytes(meta.FilePath);

            //Create Volume Texture
            Texture3DDescription descTex = new Texture3DDescription();
            descTex.Width = meta.Width;
            descTex.Height = meta.Height;
            descTex.Depth = meta.Depth;
            descTex.MipLevels = 1;
            descTex.Format = SharpDX.DXGI.Format.R8_UNorm;
            descTex.Usage = ResourceUsage.Default;
            descTex.BindFlags = BindFlags.ShaderResource;
            descTex.CpuAccessFlags = CpuAccessFlags.None;

            DataBox[] data;
            using (var ds = DataStream.Create<byte>(vol,true, true))
            {
                data = new DataBox[1];
                data[0] = new DataBox(ds.DataPointer, meta.Width, meta.Width * meta.Height);
            }

            Texture3D volTex = new Texture3D(device, descTex,data);
            ShaderResourceView volTexSrv = new ShaderResourceView(device, volTex);

            //Create Gradient Texture
            Texture3DDescription descTex2 = new Texture3DDescription();
            descTex2.Width = meta.Width;
            descTex2.Height = meta.Height;
            descTex2.Depth = meta.Depth;
            descTex2.MipLevels = 1;
            descTex2.Format = SharpDX.DXGI.Format.R32G32B32A32_Float;
            descTex2.Usage = ResourceUsage.Default;
            descTex2.BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess;
            descTex2.CpuAccessFlags = CpuAccessFlags.None;


            Texture3D gradientTex = new Texture3D(device, descTex2, null);
            ShaderResourceView gradientTexSrv = new ShaderResourceView(device, gradientTex);

            UnorderedAccessViewDescription descUav = new UnorderedAccessViewDescription();
            descUav.Dimension = UnorderedAccessViewDimension.Texture3D;
            descUav.Format = SharpDX.DXGI.Format.Unknown;
            descUav.Texture3D.WSize = meta.Depth;
            UnorderedAccessView gradientUav = new UnorderedAccessView(device, gradientTex, descUav);

            //Create final Filtered Gradient Texture
            descTex = new Texture3DDescription();
            descTex.Width = meta.Width;
            descTex.Height = meta.Height;
            descTex.Depth = meta.Depth;
            descTex.MipLevels = 1;
            descTex.Format = SharpDX.DXGI.Format.R32G32B32A32_Float;
            descTex.Usage = ResourceUsage.Default;
            descTex.BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess;
            descTex.CpuAccessFlags = CpuAccessFlags.None;

            Texture3D filteredGradientTex = new Texture3D(device, descTex);
            _filteredGradientTexSrv = new ShaderResourceView(device, filteredGradientTex);

            descUav = new UnorderedAccessViewDescription();
            descUav.Dimension = UnorderedAccessViewDimension.Texture3D;
            descUav.Format = SharpDX.DXGI.Format.Unknown;
            descUav.Texture3D.WSize = meta.Depth;
            UnorderedAccessView filteredGradientUav = new UnorderedAccessView(device, filteredGradientTex, descUav);

            QueryDescription qdesc = new QueryDescription();
            qdesc.Type = QueryType.Event;
            qdesc.Flags = 0;

            Query q = new Query(device, qdesc);
            var res = 0;

            //Compute Gradients
            device.ImmediateContext.ComputeShader.Set(gradientCShader);
            device.ImmediateContext.ComputeShader.SetShaderResource(0, volTexSrv);
            device.ImmediateContext.ComputeShader.SetUnorderedAccessView(0, gradientUav, 0);
            device.ImmediateContext.Dispatch(meta.Width / 8, meta.Height / 8, meta.Depth / 8);

            device.ImmediateContext.End(q);
            while (device.ImmediateContext.GetData(q, 0, out res) == false) { }

            device.ImmediateContext.ComputeShader.SetShaderResource(0, null);
            device.ImmediateContext.ComputeShader.SetUnorderedAccessView(0, null, 0);
            device.ImmediateContext.ComputeShader.Set(null);

            //Filter Gradients
            device.ImmediateContext.ComputeShader.Set(filterCShader);
            device.ImmediateContext.ComputeShader.SetShaderResource(0, gradientTexSrv);
            device.ImmediateContext.ComputeShader.SetUnorderedAccessView(0, filteredGradientUav, 0);
            device.ImmediateContext.Dispatch(meta.Width / 8, meta.Height / 8, meta.Depth / 8);

            device.ImmediateContext.End(q);
            while (device.ImmediateContext.GetData(q, 0, out res) == false) { }

            device.ImmediateContext.ComputeShader.SetShaderResource(0, null);
            device.ImmediateContext.ComputeShader.SetUnorderedAccessView(0, null, 0);
            device.ImmediateContext.ComputeShader.Set(null);

            blob.Dispose();
            gradientCShader.Dispose();
            filterCShader.Dispose();

            volTex.Dispose();
            volTexSrv.Dispose();
            gradientTex.Dispose();
            gradientTexSrv.Dispose();
            gradientUav.Dispose();
            filteredGradientTex.Dispose();
            filteredGradientUav.Dispose();
            q.Dispose();

            var description1D = new Texture1DDescription()
            {
                Width = TransferFunction.SIZE,
                ArraySize = 1,
                Format = Format.R32G32B32A32_Float,
                MipLevels = 1,
                Usage = ResourceUsage.Default,
                BindFlags = BindFlags.ShaderResource,
                CpuAccessFlags = CpuAccessFlags.None,
            };

            using (var ds = DataStream.Create<Vector4>(_transferFunctions[meta.TransferIndex].InterpolatedColors, true, true))
            {
                data = new DataBox[1];
                data[0] = new DataBox(ds.DataPointer);
            }

            Texture1D transfertex = new Texture1D(device, description1D, data);
            _transferfunctionSrv = new ShaderResourceView(device, transfertex);

            transfertex.Dispose();

            ModelMatrix = Matrix.Identity;
        }

        /// <summary>
        /// ShaderResourceView der aktiven Transferfunction (get)
        /// </summary>
        public ShaderResourceView TransferFunctionSrv
        {
            get
            {
                return _transferfunctionSrv;
            }
        }

        /// <summary>
        /// ShaderResourceView des geladenen Volumens (gefilterte Gradienten) (get)
        /// </summary>
        public ShaderResourceView FilteredGradientTextureSrv
        {
            get
            {
                return _filteredGradientTexSrv;
            }
        }

        /// <summary>
        /// ModelMatrix des Volumens (get)
        /// </summary>
        public Matrix ModelMatrix
        {
            get;
            private set;
        }

        /// <summary>
        /// Inverse ModelMatrix des Volumens (get)
        /// </summary>
        public Matrix InverseModelMatrix
        {
            get;
            private set;
        }

        /// <summary>
        /// Aktualisiert die ModelMatrix des Volumens. Erlaubt es das Volumen im Raum zu rotieren
        /// Rotation anhand der Mausbewegung, bei gedrückter ALT-Taste
        /// </summary>
        public void UpdateModelMatrix()
        {
            float rotationSpeed = 0.01f;

            if (InputManagement.Instance.MouseDelta.X != 0 || InputManagement.Instance.MouseDelta.Y != 0)
            {
                if (InputManagement.Instance.KeyPressed == Keys.Menu)
                {
                    _rotateX -= rotationSpeed * InputManagement.Instance.MouseDelta.X;
                    _rotateY += rotationSpeed * InputManagement.Instance.MouseDelta.Y;
                }
            }

            Matrix rotationMatrix = Matrix.RotationX(_rotateY) * Matrix.RotationY(_rotateX);

            ModelMatrix = rotationMatrix;

            Matrix invrotationMatrix;
            Matrix.Invert(ref rotationMatrix, out invrotationMatrix);

            InverseModelMatrix = invrotationMatrix;
        }

        /// <summary>
        /// Speichert die Meta-Daten eines Volumens (Dateipfad, Breite/Höhe/Tiefe, Index laut ListView)
        /// </summary>
        public class VolumeMetaData
        {
            /// <summary>
            /// Erzeugt ein MetaDaten-Objekt
            /// </summary>
            /// <param name="idx">Index laut ListView</param>
            /// <param name="path">Dateipfad</param>
            /// <param name="width">Breite</param>
            /// <param name="height">Höhe</param>
            /// <param name="depth">Tiefe</param>
            public VolumeMetaData(int idx, string path, int width, int height, int depth)
            {
                FilePath = path;
                Width = width;
                Height = height;
                Depth = depth;
                TransferIndex = idx;
            }

            /// <summary>
            /// Index der zugehörigen Transferfunktion (Index laut ListView) (get)
            /// </summary>
            public int TransferIndex { get; private set; }

            /// <summary>
            /// Dateipfad des Volumens (get)
            /// </summary>
            public string FilePath { get; private set; }

            /// <summary>
            /// Breite des Volumens (get)
            /// </summary>
            public int Width { get; private set; }

            /// <summary>
            /// Höhe des Volumens (get)
            /// </summary>
            public int Height { get; private set; }

            /// <summary>
            /// Tiefe des Volumens (get)
            /// </summary>
            public int Depth { get; private set; }
        }
    }
}
