﻿using SharpDX;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VoluRen
{
    /// <summary>
    /// Klasse zur Erstellen von Transferfunktionen
    /// </summary>
    public class TransferFunction
    {
        /// <summary>
        /// Kapselt einen Kontrollpunkt einer Transferfunktion
        /// </summary>
        public class ControlPoint
        {
            /// <summary>
            /// Konstruktor
            /// </summary>
            /// <param name="pos">Position (Iso-Wert) - 0-255</param>
            /// <param name="col">Farbwert (RGBA) - 0.0f-1.0f</param>
            public ControlPoint(int pos, Vector4 col)
            {
                Color = col;
                Position = pos;
            }

            /// <summary>
            /// Farbwert des Kontrollpunkts (get)
            /// </summary>
            public Vector4 Color { get; private set; }

            /// <summary>
            /// Position/Iso-Wert des Kontrollpunkt (get)
            /// </summary>
            public int Position { get; private set; }
        }

        /// <summary>
        /// Anzahl der Werte einer Transferfunktion
        /// </summary>
        public static int SIZE = 256;

        private Vector4[] _colors;
        private Vector4[] _alphas;
        private Vector4[] _final;

        private List<ControlPoint> _controlPoints;
        private List<ControlPoint> _alphaPoints;

        /// <summary>
        /// Erzeugt eine Transferfunktion
        /// Interpoliert die Werte der Kontrollpunkte linear (Farbe und Alpha extra)
        /// und fügt die interpolierten Werte zu einer Farbpalette zusammen
        /// </summary>
        /// <param name="color">Kontrollpunkte für die Farbwerte</param>
        /// <param name="alpha">Kontrollpunkte für die Alpha-Werte</param>
        public TransferFunction(List<ControlPoint> color, List<ControlPoint> alpha)
        {
            _controlPoints = color;
            _alphaPoints = alpha;

            Interpolate(_controlPoints, false);
            Interpolate(_alphaPoints, true);
            Merge();
        }

        /// <summary>
        /// Merged die interpolierten Farbwerte mit den interpolierten Alpha-Werten
        /// </summary>
        private void Merge()
        {
            _final = new Vector4[SIZE];

            for (int i = 0; i < SIZE; i++)
            {
                _final[i] = new Vector4(_colors[i].X, _colors[i].Y, _colors[i].Z, _alphas[i].W);
            }
        }

        /// <summary>
        /// Interpoliert die übergebenen Kontrollpunkte linear
        /// </summary>
        /// <param name="cp">Kontrollpunkte</param>
        /// <param name="alpha">true = Alpha-Werte werden interpoliert, false = Farb-Werte werden interpoliert</param>
        private void Interpolate(List<ControlPoint> cp, bool alpha)
        {
            Vector4[] interp = new Vector4[SIZE];
            int startpoint = 0;
            for (int i = 0; i < cp.Count - 1; i++)
            {
                int dist = cp[i + 1].Position - cp[i].Position;

                for (int j = 1; j <= dist; j++)
                {
                    interp[startpoint + j] = Vector4.Lerp(cp[i].Color, cp[i + 1].Color, (float)j / (float)dist);
                }

                startpoint+=dist;
            }

            if (alpha)
                _alphas = interp;
            else
                _colors = interp;
        }

        /// <summary>
        /// Linear interpolierte Farbwerte der Transferfunktion (get)
        /// </summary>
        public Vector4[] InterpolatedColors
        {
            get
            {
                return _final;
            }
        }
    }
}
