﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using SharpDX;
using SharpDX.D3DCompiler;
using SharpDX.Direct3D;
using SharpDX.Direct3D11;
using SharpDX.DXGI;
using Buffer = SharpDX.Direct3D11.Buffer;
using Device = SharpDX.Direct3D11.Device;

namespace VoluRen
{
    /// <summary>
    /// SliceBased-Renderer ohne Depth-Of-Field-Effekt
    /// Verzichtet auf die Render-To-Texture Schritte des DOF-Renderers um Framerate zu erhöhen
    /// </summary>
    public class SliceBasedRenderer : IRenderer
    {
        private const float SAMPLINGDIST = 0.01f;

        private BufferWrapper<StructVertexColor> _vertexBufferWrapper2;
        private BufferWrapper<ushort> _indexBufferWrapper2;

        private BufferWrapper<StructVertex> _vertexBufferWrapper;
        private BufferWrapper<Projections> _projectionsBufferWrapper;
        private BufferWrapper<PerFrameSlice> _perFrameSliceBufferWrapper;

        private Device _device;
        private Camera _camera;

        private VertexShader _sliceVS;
        private PixelShader _slicePS;
        private InputLayout _sliceLayout;

        private VertexShader _colorcubeVS;
        private PixelShader _colorcubePS;
        private InputLayout _colorcubeLayout;

        private BoundingBox _boundingBox;
        private BoundingBox _boundingBoxView;

        private Slice _slice;

        private StructVertex[] _vertices;

        private Matrix _normalizeMatrix;

        private BlendState _btfBlendState;

        private RasterizerState _cullNoneState;
        private SamplerState _trilinearSamperState;

        /// <summary>
        /// Initialisiert den Renderer
        /// Erzeugt Shader, RenderTargets/ShaderResources,...
        /// </summary>
        /// <param name="device">D3D11 Device</param>
        /// <param name="camera">Kamera-Objekt</param>
        public void Init(Device device, Camera camera)
        {
            _device = device;
            _camera = camera;

            _normalizeMatrix = new Matrix(0.5f, 0, 0, 0.0f, 0, 0.5f, 0, 0.0f, 0, 0, 0.5f, 0.0f, 0.5f, 0.5f, 0.5f, 1.0f);

            ShaderFlags flags = ShaderFlags.EnableStrictness;
#if DEBUG
            flags |= ShaderFlags.Debug;
#endif

            var blob = ShaderBytecode.CompileFromFile("resources\\shaders\\Slice.hlsl", "VertexShaderFunction", "vs_5_0", flags, EffectFlags.None);
            var inputsig = ShaderSignature.GetInputSignature(blob);
            _sliceVS = new VertexShader(device, blob);

            blob = ShaderBytecode.CompileFromFile("resources\\shaders\\Slice.hlsl", "PixelShaderFunction", "ps_5_0", flags, EffectFlags.None);
            _slicePS = new PixelShader(device, blob);

            _sliceLayout = new InputLayout(device, inputsig, new[]{
					new InputElement("POSITION", 0, Format.R32G32B32_Float, 0),
				});

            blob = ShaderBytecode.CompileFromFile("resources\\shaders\\ColorCube.hlsl", "VertexShaderFunction", "vs_5_0", flags, EffectFlags.None);
            inputsig = ShaderSignature.GetInputSignature(blob);
            _colorcubeVS = new VertexShader(device, blob);

            blob = ShaderBytecode.CompileFromFile("resources\\shaders\\ColorCube.hlsl", "PixelShaderFunction", "ps_5_0", flags, EffectFlags.None);
            _colorcubePS = new PixelShader(device, blob);


            _colorcubeLayout = new InputLayout(device, inputsig, new[]{
					new InputElement("POSITION", 0, Format.R32G32B32_Float, 0),
                    new InputElement("COLOR",0,Format.R32G32B32A32_Float,12,0),
				});

            blob.Dispose();
            inputsig.Dispose();

            BufferDescription vertexbufferdesc = new SharpDX.Direct3D11.BufferDescription
            {
                BindFlags = BindFlags.VertexBuffer | BindFlags.IndexBuffer,
                CpuAccessFlags = CpuAccessFlags.Write,
                OptionFlags = ResourceOptionFlags.None,
                StructureByteStride = 0,
                Usage = ResourceUsage.Dynamic
            };

            _boundingBox = new BoundingBox();
            StructVertexColor[] vertices = new StructVertexColor[(int)BoundingBox.Corner.CornerCOUNT];
            for (int i = 0; i < (int)BoundingBox.Corner.CornerCOUNT; i++)
            {
                vertices[i] = new StructVertexColor();
                vertices[i].Position = _boundingBox.DrawVertices[i];
                vertices[i].Color = new SharpDX.Color4(0.7f, 0.7f, 0.7f, 0.2f);
            }
            vertexbufferdesc.SizeInBytes = Marshal.SizeOf(typeof(StructVertexColor)) * vertices.Count();

            _vertexBufferWrapper2 = new BufferWrapper<StructVertexColor>(device, vertexbufferdesc);
            _vertexBufferWrapper2.ArrayValue = vertices;

            vertexbufferdesc.SizeInBytes = Marshal.SizeOf(typeof(ushort)) * _boundingBox.LineIndices.Count();
            _indexBufferWrapper2 = new BufferWrapper<ushort>(device, vertexbufferdesc);
            _indexBufferWrapper2.ArrayValue = _boundingBox.LineIndices;

            _slice = new Slice(_camera.ViewDirection, 0.0f);

            _vertices = new StructVertex[18];
            vertexbufferdesc.SizeInBytes = Marshal.SizeOf(typeof(StructVertex)) * 18;
            _vertexBufferWrapper = new BufferWrapper<StructVertex>(device, vertexbufferdesc);
            _projectionsBufferWrapper = new BufferWrapper<Projections>(device);
            _perFrameSliceBufferWrapper = new BufferWrapper<PerFrameSlice>(device);

            _trilinearSamperState = new SamplerState(_device, StateManagement.Instance.SamplerLinearClamp);

            BlendStateDescription bdesc = StateManagement.Instance.BackToFront;
            RenderTargetBlendDescription rtbdesc = StateManagement.Instance.RTBackToFront;
            bdesc.RenderTarget[0] = rtbdesc;
            _btfBlendState = new BlendState(_device, bdesc);

            _cullNoneState = new RasterizerState(_device, StateManagement.Instance.RCullNoneSolid);

            device.ImmediateContext.InputAssembler.PrimitiveTopology = PrimitiveTopology.TriangleList;
            _device.ImmediateContext.Rasterizer.State = _cullNoneState;
        }

        /// <summary>
        /// Berechnet die BoundingBox in ViewSpace
        /// Wird für die Schnitt-Berechnung des Slicers benötigt
        /// </summary>
        /// <param name="nearPt">Punkt am Nähesten zur Kamera</param>
        /// <param name="farPt">Punkt am weitesten weg von der Kamera</param>
        /// <returns>ViewSpace-BoundingBox</returns>
        private BoundingBox CalculateViewSpaceBB(out Vector3 nearPt, out Vector3 farPt)
        {
            Vector3[] bbviewcorners = new Vector3[(int)BoundingBox.Corner.CornerCOUNT];
            Vector3 worldTemp = Vector3.Zero;

            nearPt = new Vector3(float.MaxValue,float.MaxValue,float.MaxValue);
            farPt = new Vector3(float.MinValue, float.MinValue, float.MinValue);

            for (int i = 0; i < (int)BoundingBox.Corner.CornerCOUNT; i++)
            {
                worldTemp = (Vector3)Vector3.Transform(_boundingBox.Vertices[i], VolumeManagement.Instance.ModelMatrix);
                bbviewcorners[i] = (Vector3)Vector3.Transform(worldTemp, _camera.ViewMatrix);

                if (bbviewcorners[i].Z < nearPt.Z)
                    nearPt = bbviewcorners[i];

                if (bbviewcorners[i].Z > farPt.Z)
                    farPt = bbviewcorners[i];
            }
            
            return new BoundingBox(bbviewcorners);
        }

        /// <summary>
        /// Erzeugt die ProxyGeometry für jeden einzelnen Slice und zeichnet die Slices (Slice-Shader)
        /// </summary>
        /// <param name="startZ">Start der Iteration durch das Volumen</param>
        /// <param name="slicecnt">Anzahl der Slices, die erzeugt werden müssen</param>
        public void CreateProxyAndDraw(float startZ, int slicecnt)
        {
            List<Vector3> intersections = null;
            for (int i = 0; i < slicecnt; i++)
            {
                _slice.Distance = startZ - i * SAMPLINGDIST;

                intersections = _slice.CreateProxyGeometry(_boundingBoxView);

                if (intersections != null)
                    DrawSlice(intersections);
            }
        }

        /// <summary>
        /// Zeichnet den aktuellen Frame
        /// </summary>
        /// <param name="args">Zeitdaten (hier ungenutzt -> können null sein)</param>
        public void Draw(TimeEventArgs args)
        {
            _device.ImmediateContext.OutputMerger.SetBlendState(_btfBlendState);

            Vector3 nearPtView, farPtView;

            _boundingBoxView = CalculateViewSpaceBB(out nearPtView, out farPtView);
            float fulldistance = Math.Abs(farPtView.Z - nearPtView.Z);
            int totalSliceCnt = (int)(Math.Round(fulldistance / SAMPLINGDIST));

            if (InputManagement.Instance.DrawBB)
                DrawBoundingBox();

            CreateProxyAndDraw(farPtView.Z, totalSliceCnt);
        }

        /// <summary>
        /// Zeichnet die BoundingBox mit einer LineList
        /// </summary>
        private void DrawBoundingBox()
        {
            _device.ImmediateContext.InputAssembler.PrimitiveTopology = PrimitiveTopology.LineList;

            _device.ImmediateContext.InputAssembler.InputLayout = _colorcubeLayout;
            _device.ImmediateContext.InputAssembler.SetVertexBuffers(0, new VertexBufferBinding(_vertexBufferWrapper2.Buffer, Marshal.SizeOf(typeof(StructVertexColor)), 0));
            _device.ImmediateContext.InputAssembler.SetIndexBuffer(_indexBufferWrapper2.Buffer, Format.R16_UInt, 0);

            _projectionsBufferWrapper.Value = new Projections()
            {
                Model = Matrix.Transpose(Matrix.Identity),
                View = Matrix.Transpose(_camera.ViewMatrix),
                Projection = Matrix.Transpose(_camera.ProjectionMatrix),
            };
            _device.ImmediateContext.VertexShader.Set(_colorcubeVS);
            _device.ImmediateContext.VertexShader.SetConstantBuffer(0, _projectionsBufferWrapper.Buffer);
            _device.ImmediateContext.PixelShader.Set(_colorcubePS);
            _device.ImmediateContext.DrawIndexed(24, 0, 0);

            _device.ImmediateContext.InputAssembler.PrimitiveTopology = PrimitiveTopology.TriangleList;
        }

        /// <summary>
        /// Zeichnet den aktuellen Slice/ProxyGeometry
        /// </summary>
        /// <param name="intersections">Vertexliste in ViewSpace</param>
        private void DrawSlice(List<Vector3> intersections)
        {
            for (int i = 0; i < intersections.Count; i++)
            {
                _vertices[i].Position = (Vector3)Vector3.Transform(intersections[i], _camera.InverseViewMatrix);
                _vertices[i].Position = (Vector3)Vector3.Transform(_vertices[i].Position, VolumeManagement.Instance.InverseModelMatrix);
            }

            _vertexBufferWrapper.ArrayValue = _vertices;
            _device.ImmediateContext.InputAssembler.SetVertexBuffers(0, new VertexBufferBinding(_vertexBufferWrapper.Buffer, Marshal.SizeOf(typeof(StructVertex)), 0));

            _projectionsBufferWrapper.Value = new Projections()
            {
                Model = Matrix.Transpose(VolumeManagement.Instance.ModelMatrix),
                View = Matrix.Transpose(_camera.ViewMatrix),
                Projection = Matrix.Transpose(_camera.ProjectionMatrix),
                Normalize = Matrix.Transpose(_normalizeMatrix),
            };

            _perFrameSliceBufferWrapper.Value = new PerFrameSlice()
            {
                Alpha = InputManagement.Instance.Alpha,
            };

            _device.ImmediateContext.InputAssembler.InputLayout = _sliceLayout;
            _device.ImmediateContext.VertexShader.Set(_sliceVS);
            _device.ImmediateContext.VertexShader.SetConstantBuffer(0, _projectionsBufferWrapper.Buffer);
            _device.ImmediateContext.PixelShader.Set(_slicePS);
            _device.ImmediateContext.PixelShader.SetConstantBuffer(1, _perFrameSliceBufferWrapper.Buffer);
            _device.ImmediateContext.PixelShader.SetSampler(0, _trilinearSamperState);
            _device.ImmediateContext.PixelShader.SetShaderResource(0, VolumeManagement.Instance.FilteredGradientTextureSrv);
            _device.ImmediateContext.PixelShader.SetShaderResource(1, VolumeManagement.Instance.TransferFunctionSrv);

            _device.ImmediateContext.Draw(intersections.Count, 0);
        }

        /// <summary>
        /// Disposed die erzeugten DirectX-Objekte
        /// </summary>
        public void Dispose()
        {
            _vertexBufferWrapper2.Dispose();
            _indexBufferWrapper2.Dispose();

            _vertexBufferWrapper.Dispose();
            _projectionsBufferWrapper.Dispose();
            _perFrameSliceBufferWrapper.Dispose();

            _sliceVS.Dispose();
            _slicePS.Dispose();
            _sliceLayout.Dispose();

            _colorcubeVS.Dispose();
            _colorcubePS.Dispose();
            _colorcubeLayout.Dispose();

            _btfBlendState.Dispose();
            _cullNoneState.Dispose();
            _trilinearSamperState.Dispose();

            _device.Dispose();
        }
    }
}
