﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using SharpDX;

namespace VoluRen
{
    /// <summary>
    /// Kapselt ein Plane-Objekt, welches verwendet wird, um die ProxyGeometry
    /// des SliceBased-Renderers zu berechnen
    /// </summary>
    public class Slice
    {
        private Plane _slice;

        /// <summary>
        /// Konstruktor
        /// erzeugt eine Plane welche Parallel zur ViewPlane ist
        /// </summary>
        /// <param name="normal">ViewPlane-Normale</param>
        /// <param name="distance">Entfernung in Richtung ViewPlane-Normalen</param>
        public Slice(Vector3 normal, float distance)
        {
            _slice = new Plane(normal, distance);
        }

        /// <summary>
        /// Normalvektor des Slices (get/set)
        /// </summary>
        public Vector3 Normal
        {
            get
            {
                return _slice.Normal;
            }

            set
            {
                _slice.Normal = value;
            }
        }

        /// <summary>
        /// Distanz in Richtung des Normalvektors (get/set)
        /// </summary>
        public float Distance
        {
            get
            {
                return _slice.D;
            }

            set
            {
                _slice.D = value;
            }
        }

        /// <summary>
        /// Erzeugt die ProxyGeometry des BoundingBox/Plane-Schnittes
        /// 1. Schnittpunkt der 8 Seiten mit der Plane werden berechnet
        /// 2. Polygon wird mithilfe der Methode Triangulate erzeugt
        /// 3. Vertices des Polygons werden zurückgeliefert
        /// Quelle: http://http.developer.nvidia.com/GPUGems/gpugems_ch39.html
        /// </summary>
        /// <param name="bb">BoundingBox, mit der die Plane geschnitten werden soll</param>
        /// <returns>Vertices der erzeugten ProxyGeometry</returns>
        public List<Vector3> CreateProxyGeometry(BoundingBox bb)
        {
            List<Vector3> intersections = new List<Vector3>();
            List<Vector3> intersections2 = new List<Vector3>();
            List<Vector3> intersections3 = new List<Vector3>();

            Vector3 bbmin = bb.Vertices[(int)BoundingBox.Corner.MinMinMin];
            Vector3 bbmax = bb.Vertices[(int)BoundingBox.Corner.MaxMaxMax];

            int[] valuesx = {0,2,4,6};
            int[] valuesy = {0,1,4,5};
            int[] valuesz = {0,1,2,3};

            Vector3 origin;
            Ray ray;
            float pt;

            // X - Axis
            Vector3 dir = Vector3.Normalize(bb.Vertices[(int)BoundingBox.Corner.MaxMinMin] - bb.Vertices[(int)BoundingBox.Corner.MinMinMin]);
            foreach( var val in valuesx )
            {
                origin = bb.Vertices[val];
                ray = new Ray(origin, dir);

                if (Collision.RayIntersectsPlane(ref ray, ref _slice, out pt))
                    if (pt >= 0.0f && pt <= 2.0f)
                        intersections.Add(origin+dir*pt);
            }
            
            // Y-Axis
            dir = Vector3.Normalize(bb.Vertices[(int)BoundingBox.Corner.MinMaxMin] - bb.Vertices[(int)BoundingBox.Corner.MinMinMin]);
            foreach (var val in valuesy)
            {
                origin = bb.Vertices[val];
                ray = new Ray(origin, dir);

                if (Collision.RayIntersectsPlane(ref ray, ref _slice, out pt))
                    if (pt >= 0.0f && pt <= 2.0f)
                        intersections.Add(origin + dir * pt);
            }

            // Z-Axis
            dir = Vector3.Normalize(bb.Vertices[(int)BoundingBox.Corner.MinMinMax] - bb.Vertices[(int)BoundingBox.Corner.MinMinMin]);
            foreach (var val in valuesz)
            {
                origin = bb.Vertices[val];
                ray = new Ray(origin, dir);

                if (Collision.RayIntersectsPlane(ref ray, ref _slice, out pt))
                    if (pt >= 0.0f && pt <= 2.0f)
                        intersections.Add(origin + dir * pt);
            }

            if (intersections.Count > 0)
                return Triangulate(intersections.ToArray());

            return null;
        }

        /// <summary>
        /// Erzeugt anhand der Schnittpunkt das Proxypolygon
        /// 1. Ermittle Mittelpunkt aller Schnittpunkte
        /// 2. Sortiere die Vertices anhand ihres Winkels zum Mittelpunkt
        /// 3. Erzeuge Dreiecke/Polygone
        /// </summary>
        /// <param name="intersections"></param>
        /// <returns></returns>
        private List<Vector3> Triangulate(Vector3[] intersections)
        {
            List<Vector3> intersectionPoints = new List<Vector3>();

            int isecCnt = intersections.Length;

            //get center
            Vector3 center = Vector3.Zero;

            foreach(var isec in intersections)
                center+=isec;
            center = center / isecCnt;

            float[] angles = new float[intersections.Count()];
            for(int i=0;i<isecCnt;i++)
            {
                angles[i] = (float)Math.Atan2((double)(intersections[i].Y - center.Y), (double)(intersections[i].X - center.X));
            }

            //sort by angle
            Array.Sort(angles, intersections);

            //create triangles
            for (int i = 0; i < isecCnt; i++)
            {
                intersectionPoints.Add(center);
                intersectionPoints.Add(intersections[i]);
                intersectionPoints.Add(intersections[(i + 1) % isecCnt]);
            }

            return intersectionPoints;
        }
    }
}
