﻿using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Drawing;

using SharpDX;
using SharpDX.Windows;
using SharpDX.DXGI;
using SharpDX.Direct3D11;
using SharpDX.Direct3D;
using SharpDX.D3DCompiler;

using MetroFramework.Forms;

using Device = SharpDX.Direct3D11.Device;
using Color = SharpDX.Color;
using System.Collections.Generic;

namespace VoluRen
{
    /// <summary>
    /// Hauptfenster der Anwendung
    /// Abonniert UserControl-Events und gibt diese an das InputManagement weiter
    /// Schleift den Render-Aufruf des Launchers an den jeweilig aktiven Renderer durch
    /// </summary>
    public partial class MainForm : MetroForm
    {
        /// <summary>
        /// RenderTargetView des BackBuffers
        /// </summary>
        public static RenderTargetView _renderView;

        private Device _device;
        private DeviceContext _context;
        private SwapChain _swapChain;
        private Texture2D _backBuffer;
        private Color _clearColor;

        private IRenderer _renderer;
        private IRenderer _renderer2;

        private Camera _camera;
        private TimeEventArgs _lastTime;
        private Stopwatch _clock;

        private bool _mousePressed = false;
        private Keys _keyPressed;
        private double _lastCheck;
        private int _frameCounter;

        /// <summary>
        /// Initialisiert die Hauptkomponenten der Anwendung:
        /// Hauptfenster
        /// D3D11-Device, SwapChain
        /// Renderer-Komponenten
        /// Kamera
        /// Timing-Komponente
        /// </summary>
        public MainForm()
        {
            InitializeComponent();

            _clearColor = Color.Black;

            var desc = new SwapChainDescription()
            {
                BufferCount = 1,
                ModeDescription = new ModeDescription(_renderControl.ClientSize.Width, _renderControl.ClientSize.Height, new Rational(60, 1), Format.R8G8B8A8_UNorm),
                IsWindowed = true,
                OutputHandle = _renderControl.Handle,
                SampleDescription = new SampleDescription(1, 0),
                SwapEffect = SwapEffect.Discard,
                Usage = Usage.RenderTargetOutput
            };

            DeviceCreationFlags flags = DeviceCreationFlags.None;
#if DEBUG
            flags = DeviceCreationFlags.Debug;
#endif

            Device.CreateWithSwapChain(DriverType.Hardware, flags, desc, out _device, out _swapChain);
            _context = _device.ImmediateContext;

            var factory = _swapChain.GetParent<Factory>();
            factory.MakeWindowAssociation(_renderControl.Handle, WindowAssociationFlags.IgnoreAll); 

            _backBuffer = Texture2D.FromSwapChain<Texture2D>(_swapChain, 0);
            _renderView = new RenderTargetView(_device, _backBuffer);

            _context.Rasterizer.SetViewports(new Viewport(0, 0, _renderControl.ClientSize.Width, _renderControl.ClientSize.Height, 0.0f, 1.0f));

            _context.OutputMerger.SetTargets(_renderView);

            _camera = new Camera(_renderControl.Width, _renderControl.Height);
            _renderer = new SliceBasedDOFRenderer();
            _renderer.Init(_device,_camera);

            _renderer2 = new SliceBasedRenderer();
            _renderer2.Init(_device, _camera);

            _lastTime = new TimeEventArgs();
            _clock = new Stopwatch();
            _clock.Start();
            _lastCheck = 0.0f;
            _frameCounter = 0;
        }

        /// <summary>
        /// Cleart den BackBuffer, ruft die Draw-Methode des aktive Renderers auf
        /// und swapt den BackBuffer/FrontBuffer
        /// aktiver Renderer: falls DOF aktiv: SliceBasedDOFRenderer, sonst SliceBasedRenderer
        /// </summary>
        public void Render()
        {
            _context.ClearRenderTargetView(_renderView, _clearColor);

            TimeEventArgs currentTime = new TimeEventArgs();
            currentTime.TotalTime = _clock.Elapsed;
            currentTime.DeltaTime = _lastTime != null ? currentTime.TotalTime - _lastTime.TotalTime : TimeSpan.Zero;
            _lastTime = currentTime;

            if (_renderControl.Focused)
            {
                InputManagement.Instance.Update(_keyPressed);

                if (!_mousePressed)
                    InputManagement.Instance.MouseDelta = Vector2.Zero;

                _camera.Update(currentTime.DeltaTime);
                VolumeManagement.Instance.UpdateModelMatrix();
            }

            if (VolumeManagement.Instance.FilteredGradientTextureSrv != null)
            {
                _lastCheck += currentTime.DeltaTime.TotalSeconds;
                _frameCounter++;

                if (InputManagement.Instance.EnableDOF)
                    _renderer.Draw(currentTime);
                else
                    _renderer2.Draw(currentTime);

                if (_lastCheck > 0.1f)
                {
                    //_fpsLabel.Text = String.Format("{0:0}", (float)_frameCounter / _lastCheck);
                    _lastCheck = 0.0f;
                    _frameCounter = 0;
                }
            } 

            _swapChain.Present(0, PresentFlags.None);
        }

        /// <summary>
        /// Disposed die erzeugten DirectX-Objekte und Renderer
        /// </summary>
        public void CleanUp()
        {
            _renderView.Dispose();
            _backBuffer.Dispose();
            _context.ClearState();
            _context.Flush();
            
            _context.Dispose();
            _swapChain.Dispose();

            _renderer.Dispose();
            _renderer2.Dispose();

            if (VolumeManagement.Instance.FilteredGradientTextureSrv != null)
                VolumeManagement.Instance.FilteredGradientTextureSrv.Dispose();

            if (VolumeManagement.Instance.TransferFunctionSrv != null)
                VolumeManagement.Instance.TransferFunctionSrv.Dispose();

            //DeviceDebug dd = new DeviceDebug(_device);
            //dd.ReportLiveDeviceObjects(ReportingLevel.Detail);
            _device.Dispose();

            _clock.Stop();
        }

        private void _renderControl_KeyDown(object sender, KeyEventArgs e)
        {
            if (_renderControl.Focused)
            {
                _keyPressed = e.KeyCode;
            }
        }

        private void _loadVolButton_Click(object sender, EventArgs e)
        {
            FileLoaderForm form = new FileLoaderForm();
            form.ShowDialog();

            VolumeManagement.VolumeMetaData vol = form.GetSelectedVolume();
            
            form.Dispose();

            if (vol != null)
            {
                VolumeManagement.Instance.ProcessVolume(vol, _device);
            }
        }

        private void _renderControl_MouseEnter(object sender, EventArgs e)
        {
            _renderControl.Focus();
        }

        private void _renderControl_MouseLeave(object sender, EventArgs e)
        {
            _loadVolButton.Focus();
        }

        private void _renderControl_Resize(object sender, EventArgs e)
        {
            _camera.UpdateProjectionMatrix(_renderControl.ClientSize.Width, _renderControl.ClientSize.Height);

            _renderView.Dispose();
            _backBuffer.Dispose();

            _swapChain.ResizeBuffers(1, _renderControl.ClientSize.Width, _renderControl.ClientSize.Height, SharpDX.DXGI.Format.R8G8B8A8_UNorm, SharpDX.DXGI.SwapChainFlags.None);

            _backBuffer = Texture2D.FromSwapChain<Texture2D>(_swapChain, 0);
            _renderView = new RenderTargetView(_device, _backBuffer);

            _context.OutputMerger.SetTargets(_renderView);
            _context.Rasterizer.SetViewports(new Viewport(0, 0, _renderControl.ClientSize.Width, _renderControl.ClientSize.Height, 0.0f, 1.0f));

            if (_renderer is SliceBasedDOFRenderer)
                ((SliceBasedDOFRenderer)_renderer).CreateRenderTarget(_renderControl.ClientSize.Width, _renderControl.ClientSize.Height);

            if (_renderer2 is SliceBasedDOFRenderer)
                ((SliceBasedDOFRenderer)_renderer2).CreateRenderTarget(_renderControl.ClientSize.Width, _renderControl.ClientSize.Height);
        }

        private void MainForm_Shown(object sender, EventArgs e)
        {
            Cursor.Position = Point.Add(Point.Empty,new Size(_renderControl.ClientSize.Width / 2, _renderControl.ClientSize.Height / 2));

            if(_renderer is SliceBasedDOFRenderer)
                ((SliceBasedDOFRenderer)_renderer).CreateRenderTarget(_renderControl.ClientSize.Width, _renderControl.ClientSize.Height);

            if (_renderer2 is SliceBasedDOFRenderer)
                ((SliceBasedDOFRenderer)_renderer2).CreateRenderTarget(_renderControl.ClientSize.Width, _renderControl.ClientSize.Height);

            InputManagement.Instance.Alpha = _alphaSlider.Value;
            InputManagement.Instance.FocusPlanePosition = (float)_focusSlider.Value / 100.0f;
            InputManagement.Instance.EnableDOF = _dofCheckBox.Checked;
            InputManagement.Instance.R = (float)_blurSlider.Value / 5.0f;
            InputManagement.Instance.DrawBB = _bbCheckBox.Checked;
        }


        private void _renderControl_MouseDown(object sender, MouseEventArgs e)
        {
            _mousePressed = false;

            if (e.Button == MouseButtons.Left)
                _mousePressed = true;
        }

        private void _renderControl_MouseUp(object sender, MouseEventArgs e)
        {
            _mousePressed = false;
        }

        private void _renderControl_KeyUp(object sender, KeyEventArgs e)
        {
            _keyPressed = Keys.None;
        }

        private void _alphaSlider_Scroll(object sender, ScrollEventArgs e)
        {
            InputManagement.Instance.Alpha = e.NewValue;
        }

        private void _focusSlider_Scroll(object sender, ScrollEventArgs e)
        {
            InputManagement.Instance.FocusPlanePosition = (float)e.NewValue / 100.0f;
        }

        private void _dofCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            InputManagement.Instance.EnableDOF = _dofCheckBox.Checked;
        }

        private void _blurSlider_Scroll(object sender, ScrollEventArgs e)
        {
            InputManagement.Instance.R = (float)e.NewValue / 5.0f;
        }

        private void _bbCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            InputManagement.Instance.DrawBB = _bbCheckBox.Checked;
        }
    }
}
