﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using SharpDX;
using SharpDX.Direct3D11;

namespace VoluRen
{
    /// <summary>
    /// Interface, welches jeder Renderer (SliceBased, SliceBasedDOF) implementiert
    /// </summary>
    public interface IRenderer : IDisposable
    {
        /// <summary>
        /// Initialisiert den Renderer
        /// </summary>
        /// <param name="device">D3D11 Device</param>
        /// <param name="camera">Kamera-Objekt</param>
        void Init(Device device, Camera camera);

        /// <summary>
        /// Zeichnet den aktuellen Frame
        /// </summary>
        /// <param name="args">EventArgs, enthält Zeitdaten (Gesamtzeit, Framezeit)</param>
        void Draw(TimeEventArgs args);
    }

    /// <summary>
    /// EventArgs, welches Zeitdaten speichert um framerateunabhängige Interaktion/Animation zu gewährleisten
    /// </summary>
    public class TimeEventArgs : EventArgs
    {
        /// <summary>
        /// Default-Konstruktor
        /// </summary>
        public TimeEventArgs()
        {
            TotalTime = TimeSpan.Zero;
        }

        /// <summary>
        /// Die Gesamtlaufzeit des Programmes
        /// </summary>
        public TimeSpan TotalTime { get; set; }

        /// <summary>
        /// Aktuelle Framezeit
        /// </summary>
        public TimeSpan DeltaTime { get; set; }
    }

    /// <summary>
    /// Kapselt MVP-Matrizen + Normalizationmatrix für die Übergabe an den Shader
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct Projections
    {
        /// <summary>
        /// ModelMatrix
        /// </summary>
        public Matrix Model;

        /// <summary>
        /// ViewMatrix
        /// </summary>
        public Matrix View;

        /// <summary>
        /// ProjectionMatrix
        /// </summary>
        public Matrix Projection;
        
        /// <summary>
        /// NormalizationMatrix (-1/+1 -> 0/1)
        /// </summary>
        public Matrix Normalize;
    }

    /// <summary>
    /// Kapselt VertexPositionen und VertexColors für die Übergabe an den Shader
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct StructVertexColor
    {
        /// <summary>
        /// Konstruktor
        /// </summary>
        /// <param name="pos">Vertex-Position</param>
        /// <param name="col">Vertex-Color</param>
        public StructVertexColor(Vector3 pos, Color4 col)
        {
            Position = pos;
            Color = col;
        }

        /// <summary>
        /// Vertex-Position
        /// </summary>
        public Vector3 Position;

        /// <summary>
        /// Vertex-Color
        /// </summary>
        public Color4 Color;
    }

    /// <summary>
    /// Kapselt VertexPositionen für die Übergabe an den Shader
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct StructVertex
    {
        /// <summary>
        /// Konstruktor
        /// </summary>
        /// <param name="pos">Vertex-Position</param>
        public StructVertex(Vector3 pos)
        {
            Position = pos;
        }

        /// <summary>
        /// Vertex-Position
        /// </summary>
        public Vector3 Position;
    }

    /// <summary>
    /// Kapselt den benutzerdefinierten Alpha-Wert für die Übergabe an den Slice-Shader
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct PerFrameSlice
    {
        /// <summary>
        /// Konstruktor
        /// </summary>
        /// <param name="alpha">Alpha-Wert</param>
        public PerFrameSlice(float alpha)
        {
            Alpha = alpha;
            padding = Vector3.Zero;
        }

        /// <summary>
        /// benutzerdefinierter Alpha-Wert
        /// </summary>
        public float Alpha;

        /// <summary>
        /// Padding...Buffergröße muss Vielfaches von 16 sein!
        /// </summary>
        private Vector3 padding;
    }

    /// <summary>
    /// Kapselt den TextureSpace-CircleOfConfusion und ein CompositingFlag (BTF/FTB)
    /// für die Übergabe an den DepthOfField-Shader (CompositingShader)
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct PerFrameDOF
    {
        /// <summary>
        /// Konstruktor
        /// </summary>
        /// <param name="ct">TextureSpace-CircleOfConfusion</param>
        /// <param name="btf">CompositingFlag (BTF = 1.0f / FTB = 0.0f)</param>
        public PerFrameDOF(Vector2 ct, float btf)
        {
            CoCTexSpace = ct;
            BackToFront = btf;
            padding = 0.0f;
        }

        /// <summary>
        /// TextureSpace-CircleOfConfusion
        /// </summary>
        public Vector2 CoCTexSpace;

        /// <summary>
        /// CompositingFlag (BTF = 1.0f / FTB = 0.0f)
        /// </summary>
        public float BackToFront;

        /// <summary>
        /// Padding...Buffergröße muss Vielfaches von 16 sein!
        /// </summary>
        private float padding;
    }
}
