﻿using SharpDX;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace VoluRen
{
    /// <summary>
    /// Klasse zur Abbildung einer Freefly-Kamera
    /// </summary>
	public class Camera
	{
        private const float FOV = (float)Math.PI / 4;
        private const float ZNEAR = 0.01f;
        private const float ZFAR = 1000.0f;

		private Matrix m_viewMatrix;
        private Matrix m_inverseViewMatrix;
		private float _rotateX;
		private float _rotateY;
        private Vector3 _move;

        /// <summary>
        /// Konstruktor, erzeugt das Kameraobjekt
        /// </summary>
        /// <param name="width">Viewport-Breite</param>
        /// <param name="height">Viewport-Höhe</param>
        public Camera(int width, int height)
		{
            ProjectionMatrix = Matrix.PerspectiveFovLH(FOV, (float)((float)width / (float)height), ZNEAR, ZFAR);

            _rotateX = -(float)Math.PI / 8;
            _rotateY = -(float)Math.PI / 24;
            _move = Vector3.Zero;

            Position = new Vector3(-2.0f, 0.5f, 5.0f);
            ViewDirection = new Vector3(0.0f, 0.0f, -1.0f);

            UpdateViewMatrix();
		}

        /// <summary>
        /// ViewMatrix der Kamera (get)
        /// </summary>
		public Matrix ViewMatrix
		{
			get
			{
				return m_viewMatrix;
			}
		}

        /// <summary>
        /// ProjektionMatrix (get)
        /// </summary>
        public Matrix ProjectionMatrix
        {
            get;
            private set;
        }

        /// <summary>
        /// Richtung, in der die Kamera schaut (get)
        /// </summary>
        public Vector3 ViewDirection
        {
            get;
            private set;
        }

        /// <summary>
        /// Position der Kamera im Raum (get)
        /// </summary>
        public Vector3 Position
        {
            get;
            private set;
        }

        /// <summary>
        /// Inverse ViewMatrix der Kamera (get)
        /// </summary>
        public Matrix InverseViewMatrix
        {
            get
            {
                return m_inverseViewMatrix;
            }
        }

        /// <summary>
        /// Updated die Kameraparameter. Wird für jedes Frame aufgerufen
        /// </summary>
        /// <param name="timedelta">Framezeit um framerateunabhängige Interaktion zu gewährleisten</param>
		public void Update(TimeSpan timedelta)
		{
            float rotationSpeed = 0.01f;
            float moveSpeed = (float)timedelta.TotalSeconds * 5.0f;

            if (InputManagement.Instance.MouseDelta.X != 0 || InputManagement.Instance.MouseDelta.Y != 0)
            {
                if (InputManagement.Instance.KeyPressed != Keys.Menu)
                {
                    _rotateX += rotationSpeed * InputManagement.Instance.MouseDelta.X;
                    _rotateY -= rotationSpeed * InputManagement.Instance.MouseDelta.Y;
                }
            }

            switch (InputManagement.Instance.KeyPressed)
            {
                case Keys.W:
                    _move = new Vector3(0, 0, -moveSpeed);
                    break;
                case Keys.A:
                    _move = new Vector3(moveSpeed, 0, 0);
                    break;
                case Keys.S:
                    _move = new Vector3(0, 0, moveSpeed);
                    break;
                case Keys.D:
                    _move = new Vector3(-moveSpeed, 0, 0);
                    break;
                default:
                    _move = Vector3.Zero;
                    break;
            }

			UpdateViewMatrix();
		}

        /// <summary>
        /// Updated die Projektionsmatrix bei Fenster-Resize
        /// </summary>
        /// <param name="width">neue Viewport-Breite</param>
        /// <param name="height">neue Viewport-Höhe</param>
        public void UpdateProjectionMatrix(int width, int height)
        {
            ProjectionMatrix = Matrix.PerspectiveFovLH(FOV, (float)((float)width / (float)height), ZNEAR, ZFAR);
        }

		private void UpdateViewMatrix()
		{
			Matrix rotationMatrix = Matrix.RotationX(_rotateY) * Matrix.RotationY(_rotateX);
            Vector3 moveVector = (Vector3)Vector3.Transform(_move, rotationMatrix);
            Position += moveVector;

            Vector3 viewPlaneNormal = (Vector3)Vector3.Transform(ViewDirection, rotationMatrix);
            Vector3 rotatedUpVector = (Vector3)Vector3.Transform(new Vector3(0, 1, 0), rotationMatrix);

            m_viewMatrix = Matrix.LookAtLH(Position, Position+viewPlaneNormal, rotatedUpVector);
            Matrix.Invert(ref m_viewMatrix, out m_inverseViewMatrix);
		}
	}
}
