﻿using SharpDX;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;

using Direct3D11 = SharpDX.Direct3D11;
using SharpDX.Direct3D11;

namespace VoluRen
{
    /// <summary>
    /// Wrapper-Klasse zum leichteren Erstellen von Buffer-Objekten
    /// </summary>
    /// <typeparam name="T">Datentyp der Struktur, welche im Buffer geschrieben werden soll</typeparam>
	public class BufferWrapper<T> : IDisposable
		where T : struct
	{
		private Direct3D11.Device _device;
		private Direct3D11.Buffer _buffer;
		private DataStream _dataStream;
        private T _bufferValue;
        private T[] _bufferArrayValue;

        /// <summary>
        /// Das erzeugte Buffer-Objekt (get)
        /// </summary>
		public Direct3D11.Buffer Buffer { get { return _buffer; } }

        /// <summary>
        /// Konstruktor
        /// Erzeugt einen ConstantBuffer mit Default-Usage
        /// </summary>
        /// <param name="device">D3D11 Device</param>
		public BufferWrapper(Direct3D11.Device device)
			: this(device, new Direct3D11.BufferDescription
			{
				Usage = Direct3D11.ResourceUsage.Default,
				BindFlags = Direct3D11.BindFlags.ConstantBuffer,
				CpuAccessFlags = Direct3D11.CpuAccessFlags.None,
				OptionFlags = Direct3D11.ResourceOptionFlags.None,
				StructureByteStride = 0,
                SizeInBytes = Marshal.SizeOf(typeof(T))
			})
		{
		}

        /// <summary>
        /// Konstruktor
        /// Erzeugt einen Buffer mit den Merkmalen aus der Übergebenen Description
        /// </summary>
        /// <param name="device">D3D11 Device</param>
        /// <param name="desc">BufferDescription</param>
		public BufferWrapper(SharpDX.Direct3D11.Device device, SharpDX.Direct3D11.BufferDescription desc)
		{
			this._device = device;
			_buffer = new Direct3D11.Buffer(device, desc);
			_dataStream = new DataStream(desc.SizeInBytes, true, true);
		}

        /// <summary>
        /// Daten, die in den Buffer geschrieben werden, z.B. Vertices (get/set)
        /// </summary>
        public T[] ArrayValue
		{
			get { return _bufferArrayValue; }
			set
			{
                _bufferArrayValue = value;

                if (_buffer.Description.Usage == Direct3D11.ResourceUsage.Dynamic)
                {
                    DataStream stream;
                    _device.ImmediateContext.MapSubresource(_buffer, MapMode.WriteDiscard, MapFlags.None, out stream);
                    {
                        IntPtr iptr = stream.DataPointer;
                        stream.Position = 0;
                        stream.WriteRange<T>(value,0,value.Length);
                    }
                    _device.ImmediateContext.UnmapSubresource(_buffer, 0);
                }
                else
                {
                    IntPtr iptr = _dataStream.DataPointer;
                    for (int i = 0; i < value.Count(); i++)
                    {
                        Marshal.StructureToPtr(value[i], iptr, true);
                        iptr = new IntPtr(iptr.ToInt32() + Marshal.SizeOf(typeof(T)));
                    }
                    var dataBox = new DataBox(_dataStream.DataPointer);
                    _device.ImmediateContext.UpdateSubresource(dataBox, _buffer, 0);
                }
			}
		}

        /// <summary>
        /// Daten, die in den Buffer geschrieben werden, z.B. Projektionsmatrizen (get/set)
        /// </summary>
        public T Value
        {
            get
            {
                return _bufferValue;
            }

            set
            {
                _bufferValue =  value;

                if (_buffer.Description.Usage == Direct3D11.ResourceUsage.Dynamic)
                {
                    DataStream stream;
                    _device.ImmediateContext.MapSubresource(_buffer, MapMode.WriteDiscard, MapFlags.None, out stream);
                    {
                        IntPtr iptr = stream.DataPointer;
                        stream.Position = 0;
                        stream.Write<T>(value);
                    }
                    _device.ImmediateContext.UnmapSubresource(_buffer, 0);
                }
                else
                {
                    Marshal.StructureToPtr(value, _dataStream.DataPointer, true);
                    var dataBox = new DataBox(_dataStream.DataPointer);
                    _device.ImmediateContext.UpdateSubresource(dataBox, _buffer, 0);
                }
            }
        }

        /// <summary>
        /// Disposed die erzeugten DirectX-Objekte
        /// </summary>
        public void Dispose()
        {
            _dataStream.Dispose();
            _buffer.Dispose();
            _device.Dispose();
            _buffer = null;
            _dataStream = null;
        }
    }
}
