/**
 * Resolve.wgsl
 * 
 * Resolves the K-buffer by reading sorted transparent fragments and compositing
 * them back-to-front to produce the final blended color.
 */

struct UniformData
{
    viewMatrix: mat4x4f,
    projMatrix: mat4x4f,
    camPos: vec4f,
    camDir: vec4f,
    clearColor: vec4f,
    helperLineColor: vec4f,
    kBufferInfo: vec4f,
    
    dirLightDirection: vec4f,
    dirLightColor: vec4f,
    ambLightColor: vec4f,
    materialLightResponse: vec4f,
    
    vertexColorMin: vec4f,
    vertexColorMax: vec4f,
    vertexAlphaBounds: vec4f,
    vertexRadiusBounds: vec4f,
    
    billboardClippingEnabled: u32,
    billboardShadingEnabled: u32,
    vertexColorMode: u32,
    vertexAlphaMode: u32,
    vertexRadiusMode: u32,
    
    vertexAlphaInvert: u32,
    vertexRadiusInvert: u32,
    dataMaxLineLength: f32,
    dataMaxVertexAdjacentLineLength: f32,
}

@group(0) @binding(0) var<uniform> uniforms: UniformData;
@group(0) @binding(1) var<storage, read> kBuffer: array<vec2u>;

struct VertexOutput
{
    @builtin(position) position: vec4f,
    @location(0) texCoords: vec2f,
}

@vertex
fn vs_main(@builtin(vertex_index) vertexID: u32) -> VertexOutput
{
    var positions = array<vec2f, 6>(
        vec2f(0.0, 0.0),
        vec2f(0.0, 1.0),
        vec2f(1.0, 1.0),
        vec2f(1.0, 1.0),
        vec2f(1.0, 0.0),
        vec2f(0.0, 0.0)
    );
    
    var output: VertexOutput;
    output.texCoords = positions[vertexID];
    output.position = vec4f(positions[vertexID] * 2.0 - 1.0, 0.0, 1.0);
    return output;
}

fn listPos(i: u32, coord: vec2i) -> u32
{
    let imgSize = vec3i(uniforms.kBufferInfo.xyz);
    return u32(coord.x + coord.y * imgSize.x + i32(i) * (imgSize.x * imgSize.y));
}

fn unpackUnorm4x8(packed: u32) -> vec4f
{
    return vec4f(
        f32(packed & 0xFFu),
        f32((packed >> 8u) & 0xFFu),
        f32((packed >> 16u) & 0xFFu),
        f32(packed >> 24u)
    ) / 255.0;
}

//Hardcoding this may not be the best
//esp. since we could go up to 32
const K_MAX: u32 = 16u;

struct Sample
{
    color: vec4f,
    depth: f32,
}

@fragment
fn fs_main(input: VertexOutput) -> @location(0) vec4f
{
    let coord = vec2i(input.position.xy);
    
    var samples: array<Sample, 16>;
    for (var i = 0u; i < K_MAX; i++)
    {
        samples[i].color = vec4f(0.0);
        samples[i].depth = 0.0;
    }
    
    var sampleCount = 0u;
    let K = u32(uniforms.kBufferInfo.z);
    
    for (var i = 0u; i < K; i++)
    {
        let value = kBuffer[listPos(i, coord)];
        
        if (value.x == 0xFFFFFFFFu && value.y == 0xFFFFFFFFu)
        {
            break;
        }
        
        samples[i].color = unpackUnorm4x8(value.y);
        samples[i].depth = bitcast<f32>(value.x);
        sampleCount++;
    }
    
    if (sampleCount == 0u)
    {
        discard;
    }
    
    var color = vec3f(0.0);
    var alpha = 1.0;
    
    for (var i = 0u; i < sampleCount; i++)
    {
        let idx = sampleCount - i - 1u;
        color = color * samples[idx].color.a + samples[idx].color.rgb;
        alpha *= samples[idx].color.a;
    }
    
    alpha = 1.0 - alpha;
    return vec4f(color / alpha, alpha);
}