/**
 * ClearKBuffer.wgsl
 * 
 * Compute shader that initializes the K-buffer for order-independent transparency.
 * Sets all depth and color entries to sentinel values indicating empty slots.
 */

struct UniformData
{
    viewMatrix: mat4x4f,
    projMatrix: mat4x4f,
    camPos: vec4f,
    camDir: vec4f,
    clearColor: vec4f,
    helperLineColor: vec4f,
    kBufferInfo: vec4f,
    
    dirLightDirection: vec4f,
    dirLightColor: vec4f,
    ambLightColor: vec4f,
    materialLightResponse: vec4f,
    
    vertexColorMin: vec4f,
    vertexColorMax: vec4f,
    vertexAlphaBounds: vec4f,
    vertexRadiusBounds: vec4f,
    
    billboardClippingEnabled: u32,
    billboardShadingEnabled: u32,
    vertexColorMode: u32,
    vertexAlphaMode: u32,
    vertexRadiusMode: u32,
    
    vertexAlphaInvert: u32,
    vertexRadiusInvert: u32,
    dataMaxLineLength: f32,
    dataMaxVertexAdjacentLineLength: f32,
}

@group(0) @binding(0) var<uniform> uniforms: UniformData;
@group(0) @binding(1) var<storage, read_write> kBuffer: array<atomic<u32>>;

@compute @workgroup_size(16, 16, 1)
fn main(@builtin(global_invocation_id) dispatchID: vec3u)
{
    let coord = vec2i(dispatchID.xy);
    let imgSize = vec3i(uniforms.kBufferInfo.xyz);

    if (coord.x >= imgSize.x || coord.y >= imgSize.y)
    {
        return;
    }

    let K = u32(imgSize.z);
    for (var i = 0u; i < K; i++)
    {
        let baseIdx = (coord.x + coord.y * imgSize.x + i32(i) * (imgSize.x * imgSize.y)) * 2;
        atomicStore(&kBuffer[baseIdx], 0xFFFFFFFFu);
        atomicStore(&kBuffer[baseIdx + 1], 0xFFFFFFFFu);
    }
}