#include "Volume.h"

#include <math.h>

#include <QDebug> 


//-------------------------------------------------------------------------------------------------
// Voxel
//-------------------------------------------------------------------------------------------------

Voxel::Voxel()
{
	setValue(0.0f);
}

Voxel::Voxel(const Voxel &other)
{
	setValue(other.getValue());
}

Voxel::Voxel(const float value)
{
	setValue(value);
}

Voxel::~Voxel()
{
}

void Voxel::setValue(const float value)
{
	m_Value = value;
}

const float Voxel::getValue() const
{
	return m_Value;
};

const bool Voxel::operator==(const Voxel &other) const
{
	return (getValue() == other.getValue());
};

const bool Voxel::operator!=(const Voxel &other) const
{
	return !(*this == other);
};

const bool Voxel::operator>(const Voxel &other) const
{
	return getValue() > other.getValue();
};

const bool Voxel::operator>=(const Voxel &other) const
{
	return getValue() >= other.getValue();
};

const bool Voxel::operator<(const Voxel &other) const
{
	return getValue() < other.getValue();
};

const bool Voxel::operator<=(const Voxel &other) const
{
	return getValue() <= other.getValue();
};

const Voxel& Voxel::operator+=(const Voxel &other)
{
	m_Value += other.m_Value;
	return *this;
};

const Voxel& Voxel::operator-=(const Voxel &other)
{
	m_Value -= other.m_Value;
	return *this;
};

const Voxel& Voxel::operator*=(const float &value)
{
	m_Value *= value;
	return *this;
};

const Voxel& Voxel::operator/=(const float &value)
{
	m_Value /= value;
	return *this;
};

const Voxel Voxel::operator+(const Voxel &other) const
{
	Voxel voxNew = *this;
	voxNew += other;
	return voxNew;
};

const Voxel Voxel::operator-(const Voxel &other) const
{
	Voxel voxNew = *this;
	voxNew -= other;
	return voxNew;
};

const Voxel Voxel::operator*(const float &value) const
{
	Voxel voxNew = *this;
	voxNew *= value;
	return voxNew;
};

const Voxel Voxel::operator/(const float &value) const
{
	Voxel voxNew = *this;
	voxNew /= value;
	return voxNew;
};


//-------------------------------------------------------------------------------------------------
// Volume
//-------------------------------------------------------------------------------------------------

Volume::Volume()
	: m_Width(1), m_Height(1), m_Depth(1), m_Size(0), m_Voxels(1)
{
}

Volume::~Volume()
{
}

const Voxel& Volume::voxel(const int x, const int y, const int z) const
{
	return m_Voxels[x + y*m_Width + z*m_Width*m_Height];
}

const Voxel& Volume::voxel(const int i) const
{
	return m_Voxels[i];
}

const Voxel* Volume::voxels() const
{
	return &(m_Voxels.front());
};

const int Volume::width() const
{
	return m_Width;
};

const int Volume::height() const
{
	return m_Height;
};

const int Volume::depth() const
{
	return m_Depth;
};

const int Volume::size() const
{
	return m_Size;
};


//-------------------------------------------------------------------------------------------------
// Volume File Loader
//-------------------------------------------------------------------------------------------------

bool Volume::loadFromFile(QString filename, QProgressBar* progressBar)
{
	if(filename.toStdString().find("Engine") != std::string::npos)
		return loadFromFileAlt(filename, progressBar, 256, 256, 256);
	if (filename.toStdString().find("Teddybear") != std::string::npos)
		return loadFromFileAlt(filename, progressBar, 128, 128, 64);
	if (filename.toStdString().find("CT_Head_large") != std::string::npos)
		return loadFromFileAlt(filename, progressBar, 512, 512, 106);
	if (filename.toStdString().find("Head_MRI") != std::string::npos)
		return loadFromFileAlt(filename, progressBar, 256, 256, 320);

	// load file
	FILE *fp = NULL;
	fopen_s(&fp, filename.toStdString().c_str(), "rb");
	if (!fp)
	{
		std::cerr << "+ Error loading file: " << filename.toStdString() << std::endl;
		return false;
	}

	// progress bar

	progressBar->setRange(0, m_Size + 10);
	progressBar->setValue(0);


	// read header and set volume dimensions

	unsigned short uWidth, uHeight, uDepth;
	fread(&uWidth, sizeof(unsigned short), 1, fp);
	fread(&uHeight, sizeof(unsigned short), 1, fp);
	fread(&uDepth, sizeof(unsigned short), 1, fp);
	
	m_Width = int(uWidth);
	m_Height = int(uHeight);
	m_Depth = int(uDepth);

	int slice = m_Width * m_Height;
	m_Size = slice * m_Depth;
	m_Voxels.resize(m_Size);


	// read volume data

	// read into vector before writing data into volume to speed up process
	std::vector<unsigned short> vecData;
	vecData.resize(m_Size);
	fread((void*)&(vecData.front()), sizeof(unsigned short), m_Size, fp);
	fclose(fp);

	progressBar->setValue(10);


	// store volume data

	for (int i = 0; i < m_Size; i++)
	{
		// data is converted to FLOAT values in an interval of [0.0 .. 1.0];
		// uses 4095.0f to normalize the data, because only 12bit are used for the
		// data values, and then 4095.0f is the maximum possible value
		const float value = float(vecData[i]) / 4095.0f;
		m_Voxels[i] = Voxel(value);
		
		progressBar->setValue(10 + i);
	}

	progressBar->setValue(0);

	qDebug() << "Loaded VOLUME with dimensions " << QString::number(m_Width) << " x " << QString::number(m_Height) << " x " << QString::number(m_Depth);

	return true;
}

bool Volume::loadFromFileAlt(QString filename, QProgressBar* progressBar, int width, int height, int depth) {

	// load file
	FILE* fp = NULL;
	fopen_s(&fp, filename.toStdString().c_str(), "rb");
	if (!fp)
	{
		std::cerr << "+ Error loading file: " << filename.toStdString() << std::endl;
		return false;
	}

	// progress bar

	progressBar->setRange(0, m_Size + 10);
	progressBar->setValue(0);


	// read header and set volume dimensions

	unsigned short uWidth = width;
	unsigned short uHeight = height;
	unsigned short uDepth = depth;

	m_Width = int(uWidth);
	m_Height = int(uHeight);
	m_Depth = int(uDepth);

	int slice = m_Width * m_Height;
	m_Size = slice * m_Depth;
	m_Voxels.resize(m_Size);


	// read volume data

	// read into vector before writing data into volume to speed up process
	std::vector<unsigned char> vecData;
	vecData.resize(m_Size);
	fread((void*)&(vecData.front()), sizeof(unsigned char), m_Size, fp);
	fclose(fp);

	progressBar->setValue(10);


	// store volume data

	for (int i = 0; i < m_Size; i++)
	{
		// data is converted to FLOAT values in an interval of [0.0 .. 1.0];
		// uses 4095.0f to normalize the data, because only 12bit are used for the
		// data values, and then 4095.0f is the maximum possible value
		const float value = float(vecData[i]) / 256.0f;
		m_Voxels[i] = Voxel(value);

		progressBar->setValue(10 + i);
	}

	progressBar->setValue(0);

	qDebug() << "Loaded VOLUME with dimensions " << QString::number(m_Width) << " x " << QString::number(m_Height) << " x " << QString::number(m_Depth);

	return true;
}
/*

bool Volume::loadFromFileStanford(QString filename, QProgressBar* progressBar, int width, int height, int depth) {

	m_Voxels.resize(width * height * depth);
	m_Width = int(width);
	m_Height = int(height);
	m_Depth = int(depth);
	m_Size = width * height * depth;
	for (int i = 0; i < depth; i++) {
		FILE* fp = NULL;
		fopen_s(&fp, (i+1) + "", "rb");
		for (int j = 0; j < width * height; j++)
		{
			std::vector<unsigned short> vecData;
			vecData.resize(width * height);
			fread((void*)&(vecData.front()), sizeof(unsigned short), width * height, fp);
			fclose(fp);
			const float value = float(vecData[j]) / 65536.0f;
			m_Voxels[i+j] = Voxel(value);

			progressBar->setValue(10 + i);
		}
	}

	qDebug() << "Loaded VOLUME with dimensions " << QString::number(m_Width) << " x " << QString::number(m_Height) << " x " << QString::number(m_Depth);

	return true;
}*/