#include "TransferFunctionWidget.h"
#include <QDebug> 


#include "MainWindow.h"


TransferFunctionWidget::TransferFunctionWidget(QWidget* parent, MainWindow* mainWindow, QComboBox* colorPicker, QComboBox* pointPicker, bool is_medium) : QWidget(parent), m_mainWindow(mainWindow), m_colorPicker(colorPicker), m_pointPicker(pointPicker)
{
	setupPoints(is_medium);
	mapPoints();
	this->setMinimumSize(200, 150);
	m_colorPicker->addItem("Red");
	m_colorPicker->addItem("Green");
	m_colorPicker->addItem("Blue");
	if (is_medium) {
		m_colorPicker->addItem("Refraction Index");
	} else
	{
		m_colorPicker->addItem("Alpha");
	}
	
	//m_colorPicker->setMinimumSize(100, 100);
	//m_colorPicker->setGeometry(QRect(10, 20, 111, 22));
	m_colorPicker->show();
	m_colorPicker->setVisible(true);

	for (int i = 0; i < 5; i++) {
		m_pointPicker->addItem("Point " + QString::number(i + 1));
	}

	QObject::connect(m_pointPicker, QOverload<int>::of(&QComboBox::currentIndexChanged), this, &TransferFunctionWidget::pointSelectionChanged);
	QObject::connect(m_colorPicker, QOverload<int>::of(&QComboBox::currentIndexChanged), this, &TransferFunctionWidget::colorSelectionChanged);
}


TransferFunctionWidget::~TransferFunctionWidget()
{
}


void TransferFunctionWidget::paintEvent(QPaintEvent* e) {

	QPainter painter(this);
	painter.drawLine(20, 10, 20, 150);
	painter.drawLine(10, 140, 200, 140);


	QPen redPointPen;
	redPointPen.setWidth(8);
	redPointPen.setCapStyle(Qt::RoundCap);
	redPointPen.setColor(Qt::red);

	QPen greenPointPen;
	greenPointPen.setWidth(8);
	greenPointPen.setCapStyle(Qt::RoundCap);
	greenPointPen.setColor(Qt::green);

	QPen bluePointPen;
	bluePointPen.setWidth(8);
	bluePointPen.setCapStyle(Qt::RoundCap);
	bluePointPen.setColor(Qt::blue);

	QPen alphaPointPen;
	alphaPointPen.setWidth(8);
	alphaPointPen.setCapStyle(Qt::RoundCap);
	alphaPointPen.setColor(Qt::gray);


	QPen linePen;
	linePen.setWidth(1);
	linePen.setCapStyle(Qt::FlatCap);
	linePen.setColor(Qt::black);

	QPolygon redLine;
	QPolygon blueLine;
	QPolygon greenLine;
	QPolygon alphaLine;
	for (int i = 0; i < 5; i++) {
		redLine << m_redPoints[i];
		blueLine << m_bluePoints[i];
		greenLine << m_greenPoints[i];
		alphaLine << m_alphaPoints[i];
	}



	//Draw red points
	painter.setPen(redPointPen);
	painter.drawPoint(m_redPoints[0]);
	painter.drawPoint(m_redPoints[1]);
	painter.drawPoint(m_redPoints[2]);
	painter.drawPoint(m_redPoints[3]);
	painter.drawPoint(m_redPoints[4]);


	//Draw blue points
	painter.setPen(bluePointPen);
	painter.drawPoint(m_bluePoints[0]);
	painter.drawPoint(m_bluePoints[1]);
	painter.drawPoint(m_bluePoints[2]);
	painter.drawPoint(m_bluePoints[3]);
	painter.drawPoint(m_bluePoints[4]);


	//Draw green points
	painter.setPen(greenPointPen);
	painter.drawPoint(m_greenPoints[0]);
	painter.drawPoint(m_greenPoints[1]);
	painter.drawPoint(m_greenPoints[2]);
	painter.drawPoint(m_greenPoints[3]);
	painter.drawPoint(m_greenPoints[4]);

	//Draw alpha points
	painter.setPen(alphaPointPen);
	painter.drawPoint(m_alphaPoints[0]);
	painter.drawPoint(m_alphaPoints[1]);
	painter.drawPoint(m_alphaPoints[2]);
	painter.drawPoint(m_alphaPoints[3]);
	painter.drawPoint(m_alphaPoints[4]);


	//Draw lines
	painter.setPen(linePen);
	painter.drawPolyline(greenLine);
	painter.drawPolyline(blueLine);
	painter.drawPolyline(redLine);
	painter.drawPolyline(alphaLine);


}


void TransferFunctionWidget::setupPoints(bool is_medium) {
	if (is_medium) {
		m_redPoints[0] = QPoint(xStartValue, yMaxValue);
		m_redPoints[1] = QPoint(xStartValue + 20, yMaxValue);
		m_redPoints[2] = QPoint(xStartValue + 80, yMaxValue);
		m_redPoints[3] = QPoint(xStartValue + 100, yMaxValue);
		m_redPoints[4] = QPoint(xMaxValue, yMaxValue);

		//m_bluePoints[0] = QPoint(xStartValue + 1, yStartValue - 20);
		m_bluePoints[0] = QPoint(xStartValue, yMaxValue);
		m_bluePoints[1] = QPoint(xStartValue + 20, yMaxValue);
		m_bluePoints[2] = QPoint(xStartValue + 80, yMaxValue);
		m_bluePoints[3] = QPoint(xStartValue + 100, yMaxValue);
		m_bluePoints[4] = QPoint(xMaxValue, yMaxValue);


		//m_greenPoints[0] = QPoint(xStartValue + 1, yStartValue - 40);
		m_greenPoints[0] = QPoint(xStartValue, yMaxValue);
		m_greenPoints[1] = QPoint(xStartValue + 20, yMaxValue);
		m_greenPoints[2] = QPoint(xStartValue + 80, yMaxValue);
		m_greenPoints[3] = QPoint(xStartValue + 100, yMaxValue);
		m_greenPoints[4] = QPoint(xMaxValue, yMaxValue);

		//m_alphaPoints[0] = QPoint(xStartValue + 1, yStartValue);
		m_alphaPoints[0] = QPoint(xStartValue, yStartValue);
		m_alphaPoints[1] = QPoint(xStartValue + 20, yStartValue - 10);
		m_alphaPoints[2] = QPoint(xStartValue + 80, yStartValue - 40);
		m_alphaPoints[3] = QPoint(xStartValue + 100, yStartValue - 30);
		m_alphaPoints[4] = QPoint(xMaxValue, yStartValue - 70);
	}
	else {
		//m_redPoints[0] = QPoint(xStartValue + 1, yStartValue);
		m_redPoints[0] = QPoint(xStartValue, yStartValue);
		m_redPoints[1] = QPoint(xStartValue + 20, yStartValue - 20);
		m_redPoints[2] = QPoint(xStartValue + 80, yStartValue - 30);
		m_redPoints[3] = QPoint(xStartValue + 100, yStartValue - 40);
		m_redPoints[4] = QPoint(xMaxValue, yMaxValue);

		//m_bluePoints[0] = QPoint(xStartValue + 1, yStartValue - 20);
		m_bluePoints[0] = QPoint(xStartValue, yStartValue);
		m_bluePoints[1] = QPoint(xStartValue + 20, yStartValue - 100);
		m_bluePoints[2] = QPoint(xStartValue + 80, yStartValue - 30);
		m_bluePoints[3] = QPoint(xStartValue + 100, yStartValue - 70);
		m_bluePoints[4] = QPoint(xMaxValue, yMaxValue + 20);


		//m_greenPoints[0] = QPoint(xStartValue + 1, yStartValue - 40);
		m_greenPoints[0] = QPoint(xStartValue, yStartValue);
		m_greenPoints[1] = QPoint(xStartValue + 20, yStartValue - 70);
		m_greenPoints[2] = QPoint(xStartValue + 80, yStartValue - 50);
		m_greenPoints[3] = QPoint(xStartValue + 100, yStartValue - 20);
		m_greenPoints[4] = QPoint(xMaxValue, yMaxValue + 20);

		//m_alphaPoints[0] = QPoint(xStartValue + 1, yStartValue);
		m_alphaPoints[0] = QPoint(xStartValue, yStartValue);
		m_alphaPoints[1] = QPoint(xStartValue + 20, yMaxValue);
		m_alphaPoints[2] = QPoint(xStartValue + 80, yStartValue - 6);
		m_alphaPoints[3] = QPoint(xStartValue + 100, yStartValue - 10);
		m_alphaPoints[4] = QPoint(xMaxValue, yMaxValue + 100);
	}
}


void TransferFunctionWidget::mouseMoveEvent(QMouseEvent* event)
{
	QPoint point = event->pos();


	switch (m_selectedColor)
	{
	case 0:
		updatePoints(&m_redPoints[0], point);
		break;
	case 1:
		updatePoints(&m_greenPoints[0], point);
		break;
	case 2:
		updatePoints(&m_bluePoints[0], point);
		break;
	case 3:
		updatePoints(&m_alphaPoints[0], point);
		break;
	}

	update();
	m_mainWindow->updateGL(this);

}

void TransferFunctionWidget::updatePoints(QPoint* points, QPoint mousePos) {
	//std::cout << "Position y: " << mousePos.y() << " Position x: " << mousePos.x() << std::endl;
	if (mousePos.y() > yStartValue) {
		mousePos.setY(yStartValue);
	}

	if (mousePos.y() < yMaxValue) {
		mousePos.setY(yMaxValue);
	}

	if (mousePos.x() > xMaxValue) {
		mousePos.setX(xMaxValue);
	}
	if (mousePos.x() < xStartValue) {
		mousePos.setX(xStartValue + 1);
	}


	if (m_selectedPoint == 0) {

		//points[m_selectedPoint] = QPoint(xStartValue, mousePos.y()); // let first checkpoint be static
	}
	else if (m_selectedPoint == 4) {
		points[m_selectedPoint] = QPoint(xMaxValue, mousePos.y());
	}
	else
	{
		
		switch (m_selectedColor)
		{
			float old_y;
		case 0:
			points[m_selectedPoint] = QPoint(mousePos);
			
			old_y = m_alphaPoints[m_selectedPoint].y();
			m_alphaPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_bluePoints[m_selectedPoint].y();
			m_bluePoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_greenPoints[m_selectedPoint].y();
			m_greenPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			break;
		case 1:
			points[m_selectedPoint] = QPoint(mousePos);
			old_y = m_alphaPoints[m_selectedPoint].y();
			m_alphaPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_bluePoints[m_selectedPoint].y();
			m_bluePoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_redPoints[m_selectedPoint].y();
			m_redPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			break;
		case 2:
			points[m_selectedPoint] = QPoint(mousePos);
			old_y = m_alphaPoints[m_selectedPoint].y();
			m_alphaPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_redPoints[m_selectedPoint].y();
			m_redPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_greenPoints[m_selectedPoint].y();
			m_greenPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			break;
		case 3:
			points[m_selectedPoint] = QPoint(mousePos);
			old_y = m_redPoints[m_selectedPoint].y();
			m_redPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_bluePoints[m_selectedPoint].y();
			m_bluePoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			old_y = m_greenPoints[m_selectedPoint].y();
			m_greenPoints[m_selectedPoint] = QPoint(mousePos.x(), old_y);
			break;
		}
			
			
	}

	mapPoints();
	//map points


}

void TransferFunctionWidget::mapPoints() {
	m_redValues[5];
	m_greenValues[5];
	m_blueValues[5];
	m_alphaValues[5];

	for (int i = 0; i < 5; i++) {
		m_redValues[i] = normalizePoint(m_redPoints[i]);
		//qDebug() << "Rot 1: " << QString::number(m_redValues[i].x()) << QString::number(m_redValues[i].y());
	}

	for (int i = 0; i < 5; i++) {
		m_greenValues[i] = normalizePoint(m_greenPoints[i]);
		//qDebug() << "green 1: " << QString::number(m_greenValues[i].x()) << QString::number(m_greenValues[i].y());

	}

	for (int i = 0; i < 5; i++) {
		m_blueValues[i] = normalizePoint(m_bluePoints[i]);
		//qDebug() << "blue 1: " << QString::number(m_blueValues[i].x()) << QString::number(m_blueValues[i].y());

	}

	for (int i = 0; i < 5; i++) {
		m_alphaValues[i] = normalizePoint(m_alphaPoints[i]);
		//qDebug() << "alpha 1: " << QString::number(m_alphaValues[i].x()) << QString::number(m_alphaValues[i].y());

	}


}

QVector2D TransferFunctionWidget::normalizePoint(QPoint point) {


	//in float 
	float y = 1.0 - (point.y() - (float)yMaxValue) / ((float)yStartValue - (float)yMaxValue);
	float x = (point.x() - (float)xStartValue) / ((float)xMaxValue - (float)xStartValue);

	QVector2D normalizedVector = QVector2D(x, y);


	return normalizedVector;

}

void TransferFunctionWidget::pointSelectionChanged(int value)
{
	m_selectedPoint = value;
	update();
	//TODO call update
	//m_glWidget->update();

}

void TransferFunctionWidget::colorSelectionChanged(int value)
{
	m_selectedColor = value;
	update();
	//TODO call update
	//m_glWidget->update();

}