#include "TransferFunction.h"
#include <iostream>
TransferFunction::TransferFunction(std::vector<glm::vec4> controlPoints, std::vector<glm::vec4> colors)
{
	for (int i = 0; i < controlPoints.size(); i++)
	{
		glm::vec4 cp = controlPoints.at(i);
		m_controlPointsList.push_back(new ControlPoint(controlPoints[i], colors[i]));
	}
}

/* Constructor of TF for alpha values */
TransferFunction::TransferFunction(std::vector<glm::vec2> controlPoints)
{
	for (int i = 0; i < controlPoints.size(); i++)
	{
		glm::vec2 cp = controlPoints.at(i);
		m_controlPointsList.push_back(new ControlPoint(cp.x, cp.y));
	}
}

/* construct a control point for color value */
TransferFunction::ControlPoint::ControlPoint(glm::vec3 pos, glm::vec4 col)
{
	m_coord = glm::vec4(pos.x, pos.y, pos.z, 1.0f);
	//m_color = glm::vec4(col.r, col.g, col.b, 1.0f);
	m_color = glm::vec4(col.r, col.g, col.b, col.a);
}

/* construct a control point for the alpha value */
TransferFunction::ControlPoint::ControlPoint(float alpha, float value)
{
	m_color = glm::vec4(0.0f, 0.0f, 0.0f, alpha);
	m_value = value;
}

void TransferFunction::setColorForControlPoints(std::vector<glm::vec4> cpColor)
{
	m_controlPointsColor = cpColor;
}


void TransferFunction::computePreintegrationTable(GLuint texture_handle)
{

	std::vector<glm::vec4> interpPoints(256);

	computeBezierInterp(interpPoints);


	



	//https://raphaelmenges.github.io/assets/theses/raphaelmenges_bachelor.pdf seite 45...
	//...
	// 
	// 
	// numerical integration of the transfer function
	std::vector<glm::vec4> preintegration(256);

	preintegration[0] = interpPoints[0];

	for (int i = 1; i < 256; i++)
	{
		preintegration[i] = preintegration[i - 1] + interpPoints[i];
	}
	// end

	// 2D texture from preintegrated values
	std::vector<glm::vec4> textureData(256 * 256);

	glm::vec4 tmp;

	for (int x = 0; x < 256; x++)
	{
		for (int y = 0; y < 256; y++)
		{
			if (x > y)
			{
				tmp = preintegration[x] - preintegration[y];
				
				tmp *= (1.0f / float(x - y));
				textureData[x + 256 * y] = tmp;
			}
			if (x < y)
			{
				//std::cout << "y.r: " << preintegration[y].r << " x.r: " << preintegration[x].r  <<  std::endl;
				tmp = preintegration[y] - preintegration[x];
				//std::cout << "y-x:.r " << tmp.x << std::endl;
				//tmp *= (1.0f / float(y - x));
				//std::cout << "y-x: " << float(y - x) << std::endl;
				tmp = tmp * (1.0f / float(y - x));
				//std::cout << "y-x:.r scaled" << tmp.x << std::endl;
				textureData[x + 256 * y] = tmp;
			}
			else
			{
				textureData[x + 256 * y] = interpPoints[x];
			}
		}
	}
	
	glBindTexture(GL_TEXTURE_2D, texture_handle);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, 256, 256, 0, GL_RGBA, GL_FLOAT, reinterpret_cast<GLfloat*> (&(textureData[0])));
	//glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	//glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glBindTexture(GL_TEXTURE_2D, 0);

}


void TransferFunction::computeBezierInterp(std::vector<glm::vec4>& interpPoints)
{

	//https://github.com/raphaelmenges/Voraca/blob/master/src/Transferfunction.cpp updateFunction() --> cubic bezier interpolation
	// calculate steps between control points
	// consider 4 fixed cpoints
	int stepNr = m_controlPointsList.size() - 1;
	std::vector<int> stepsBetweenCPoints(stepNr);
	int bezierSteps = 0;
	int steps;

	for (int i = 0; i < stepNr; i++)
	{
		steps = static_cast<int>((m_controlPointsList[i + 1]->getVec3().x - m_controlPointsList[i]->getVec3().x) * BEZIER_STEPS);
		steps = glm::max(steps, MIN_BEZIER_STEPS);
		stepsBetweenCPoints[i] = steps;
		bezierSteps += steps;
	}

	// create bezier curve
	std::vector<glm::vec3> bezierCurve(bezierSteps);
	bezierSteps = 0;


	ControlPoint* pLeft;
	ControlPoint* pRight;
	glm::vec2 leftPointPos, leftControlPointPos, rightControlPointPos, rightPointPos;
	float leftIndex;
	float rightIndex;
	float t1;
	float t2;
	float a;
	float b;
	float c;
	float d;
	for (int i = 0; i < stepNr; i++)
	{

		pLeft = m_controlPointsList[i];
		pRight = m_controlPointsList[i + 1];

		leftPointPos = pLeft->getXY();
		leftControlPointPos = pRight->getXY();
		rightControlPointPos = pLeft->getXY();
		rightPointPos = pRight->getXY();
		leftIndex = static_cast<float>(i);
		rightIndex = static_cast<float>(i + 1);

		// steps between two consecutive control points
		steps = stepsBetweenCPoints[i];

		// for each control point, do bezier interpolation
		for (int j = 0; j < steps; j++)
		{
			t1 = static_cast<float>(j) / (steps - 1);
			t2 = 1 - t1;

			a = t2 * t2 * t2;
			b = 3 * t1 * t2 * t2;
			c = 3 * t1 * t1 * t2;
			d = t1 * t1 * t1;

			bezierCurve[bezierSteps + j].x = a * leftPointPos.x + b * leftControlPointPos.x + c * rightControlPointPos.x + d * rightPointPos.x;
			bezierCurve[bezierSteps + j].y = a * leftPointPos.y + b * leftControlPointPos.y + c * rightControlPointPos.y + d * rightPointPos.y;
			bezierCurve[bezierSteps + j].z = a * leftIndex + b * leftIndex + c * rightIndex + d * rightIndex;
		}

		bezierSteps += steps;
	}

	// calculate color with bezier curve
	int positionInBezier = 0;
	int positionInTexture = 0;
	float indexLeft, indexRight;
	int index;
	float stepSize = 1.0f / 256; // 256 for texture resolution
	glm::vec4 leftValue, rightValue;
	for (int i = 0; i < 256; i++)
	{
		if (i == 255) {
			interpPoints[i] = m_controlPointsList[m_controlPointsList.size() - 1]->m_color;
			return;
		}
		while ((bezierCurve[positionInBezier].x) <= positionInTexture)
		{
			positionInBezier++;
		}

		indexLeft = bezierCurve[positionInBezier - 1].z;
		indexRight = bezierCurve[positionInBezier].z;
		index = static_cast<int>((indexLeft + indexRight) / 2.0f);

		
		pLeft = m_controlPointsList[index];
		pRight = m_controlPointsList[index + 1];

		// calculate interpolation
		t1 = ((indexLeft + indexRight) / 2.0f) - static_cast<float>(index);
		t2 = 1 - t1;

		// get color values
		leftValue = pLeft->m_color;
		rightValue = pRight->m_color;

		glm::vec4 color = (t2)*leftValue + (t1)*rightValue;

		// fill result vector
		interpPoints[i] = color; //glm::vec4(color, 1);

		positionInTexture += stepSize;
		positionInBezier += 1;
	}
	// DEBUG
	interpPoints[0] = glm::vec4(0.0);

	// fill result vector
	/*for (int i = 0; i < interpPoints.size(); i++)
	{
		interpPoints[i] = glm::vec4(bezierCurve[i].x, bezierCurve[i].y, bezierCurve[i].z, 1);
	}*/
}